/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <fstream>
#include <cstdlib>
#include <cstring>

// GTLCore Headers
#include <GTLCore/PixelDescription.h>
#include <GTLCore/Debug.h>
#include <GTLCore/Region.h>
#include <GTLCore/Type.h>

// GTLImageIO Headers
#include <GTLImageIO/ImageDC.h>
#include <GTLImageIO/ImageDCRegistry.h>

// OpenShiva Headers
#include <OpenShiva/Debug.h>
#include <GTLCore/Image.h>
#include <OpenShiva/Kernel.h>
#include <OpenShiva/Version.h>

void printVersion()
{
  std::cout << OpenShiva::LibraryShortName << " - " << OpenShiva::LibraryName << " - " << OpenShiva::LibraryVersionString << std::endl;
  std::cout << OpenShiva::LibraryCopyright << std::endl;
  std::cout << "Shiva Version : " << OpenShiva::LanguageVersion << std::endl;
}
void printHelp()
{
  std::cout << "Usage : shiva [option] fileName.shiva [input0.png input1.png ...] output.png" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << "  -S --asm-source         print the assembly source code generated after the execution of the kernel" << std::endl;
  std::cout << "  -w --width [w]          define the width of the output" << std::endl;
  std::cout << "  -h --height [h]         define the height of the output" << std::endl;
//   std::cout << "  -L --module-dir   add a location where to find modules" << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}
#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

int main(int argc, char** argv)
{
  GTLCore::String fileName = "";
  GTLCore::String outputFileName = "";
  GTLCore::String output = "";
  std::list< GTLCore::String > fileNames;
  bool showAssembly = false;
  int width = 800;
  int height = 600;
  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-S","--asm-source")) {
      showAssembly = true;
    } else if(ARG_IS("-L", "--module-dir")) {
      if( ai == argc )
      {
        std::cerr << "Expected directory after -L --module-dir." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
//         OpenShiva::ModulesManager::instance()->addDirectory(argv[ai]);
      }
    } else if(ARG_IS("-w", "--width")) {
      if( ai == argc )
      {
        std::cerr << "Expected width after -w --width." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        width = GTLCore::String( argv[ai] ).toInt();
      }
    } else if(ARG_IS("-h", "--height")) {
      if( ai == argc )
      {
        std::cerr << "Expected width after -h --height." << std::endl;
        return EXIT_FAILURE;
      } else {
        ++ai;
        height = GTLCore::String( argv[ai] ).toInt();
      }
    } else {
      fileNames.push_back( argv[ai] );
    }
  }
  if( fileNames.size() > 1 )
  {
    fileName = *( fileNames.begin() );
    fileNames.pop_front();
    outputFileName = *( --fileNames.end() );
    fileNames.pop_back();
  }
  if( fileName == "")
  {
    std::cerr << "Invalid command line parameters." << std::endl;
    printHelp();
  } else {
    GTLCore::String source;
    std::ifstream in;
    in.open(fileName.c_str() );
    if(not in)
    {
      std::cerr << "Impossible to open file " << fileName << std::endl;
      return EXIT_FAILURE;
    }
    GTLCore::String str;
    std::getline(in,str);
    while ( in ) {
      source += str;
      source += "\n";
      std::getline(in,str);
    }
    int channelsCount = 3;
    GTLCore::PixelDescription pixel( GTLCore::Type::UnsignedInteger8, channelsCount );
    OpenShiva::Kernel p(fileName, channelsCount);
    p.setSource( source );
    p.compile();
    GTLCore::String errMsg;
    if(not p.isCompiled())
    {
      std::cerr << "Error: " << std::endl << p.compilationErrorsMessage() << std::endl;
      return EXIT_FAILURE;
    }
    GTLCore::Image image(width, height, pixel );
    std::list<GTLCore::AbstractImage*> images;
    for( std::list< GTLCore::String >::iterator it = fileNames.begin();
         it != fileNames.end(); ++it )
    {
      const GTLImageIO::ImageDC* decoder = GTLImageIO::ImageDCRegistry::instance()->decoder( *it );
      if( not decoder )
      {
        std::cerr << "Can't find decoder for " << *it << std::endl;
        return EXIT_FAILURE;
      }
      GTLCore::AbstractImage* image = decoder->decode( *it, 0, &errMsg );
      if( not image )
      {
        std::cerr << "Can't read image " << *it << " : " << errMsg << std::endl;
      }
      images.push_back( image );
    }
    p.evaluatePixeles( GTLCore::Region(0,0, width, height), images, &image );
    if( showAssembly )
    {
      std::cout << p.asmSourceCode();
    }
    const GTLImageIO::ImageDC* encoder = GTLImageIO::ImageDCRegistry::instance()->encoder( outputFileName );
    if( not encoder )
    {
      std::cerr << "Can't find encoder for " << outputFileName << std::endl;
      return EXIT_FAILURE;
    }
    if( not encoder->encode( &image, GTLCore::Region(0, 0, width, height), outputFileName, &errMsg ) )
    {
      std::cerr << "Can't encode " << outputFileName << " : " << errMsg << std::endl;
      return EXIT_FAILURE;
    }
  }
  return EXIT_SUCCESS;
}
