/*************************************************************************
 *
 *  $RCSfile: intn.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: th $ $Date: 2001/06/15 13:11:39 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

// Ist dafuer da, um schnell zu testen, welche Formate NT vorgibt
// #define IMPL_NO_DEFAULT_TABLES

#include <limits.h>
#include <string.h>

#define private public

#ifndef _TOOLSIN_HXX
#include <toolsin.hxx>
#endif
#ifndef _IMPSTRG_HXX
#include <impstrg.hxx>
#endif
#ifndef _INTN_HXX
#include <intn.hxx>
#endif
#ifndef _INTNTAB_HXX
#include <intntab.hxx>
#endif
#ifndef _DEBUG_HXX
#include <debug.hxx>
#endif

// =======================================================================

DBG_NAME( International );
DBG_NAME( LanguageTable );
DBG_NAME( FormatTable );

// =======================================================================

static xub_Unicode* ImplAddUNum( xub_Unicode* pBuf, ULONG nNumber )
{
    // Temp-Buffer mit Zahlen fuellen
    xub_Unicode aTempBuf[30];
    xub_Unicode* pTempBuf = aTempBuf;
    do
    {
        *pTempBuf = (xub_Unicode)(nNumber % 10) + '0';
        pTempBuf++;
        nNumber /= 10;
    }
    while ( nNumber );

    // Temp-Buffer in uebergebenen Buffer uebertragen
    do
    {
        pTempBuf--;
        *pBuf = *pTempBuf;
        pBuf++;
    }
    while ( pTempBuf != aTempBuf );

    return pBuf;
}

// -----------------------------------------------------------------------

static xub_Unicode* ImplAddUNum( xub_Unicode* pBuf, ULONG nNumber, int nMinLen )
{
    // Temp-Buffer mit Zahlen fuellen
    xub_Unicode aTempBuf[30];
    xub_Unicode* pTempBuf = aTempBuf;
    do
    {
        *pTempBuf = (xub_Unicode)(nNumber % 10) + '0';
        pTempBuf++;
        nNumber /= 10;
        nMinLen--;
    }
    while ( nNumber );

    // Den String auf die minimale Laenge mit Nullen fuellen
    while ( nMinLen > 0 )
    {
        *pBuf = '0';
        pBuf++;
        nMinLen--;
    }

    // Temp-Buffer in uebergebenen Buffer uebertragen
    do
    {
        pTempBuf--;
        *pBuf = *pTempBuf;
        pBuf++;
    }
    while ( pTempBuf != aTempBuf );

    return pBuf;
}

// -----------------------------------------------------------------------

static xub_Unicode* ImplAdd2UNum( xub_Unicode* pBuf, USHORT nNumber, int bLeading )
{
    DBG_ASSERT( nNumber < 100, "ImplAdd2Num() - Number >= 100" );

    if ( nNumber < 10 )
    {
        if ( bLeading )
        {
            *pBuf = '0';
            pBuf++;
        }
        *pBuf = nNumber + '0';
    }
    else
    {
        USHORT nTemp = nNumber % 10;
        nNumber /= 10;
        *pBuf = nNumber + '0';
        pBuf++;
        *pBuf = nTemp + '0';
    }

    pBuf++;
    return pBuf;
}

// -----------------------------------------------------------------------

inline xub_Unicode* ImplAddString( xub_Unicode* pBuf, xub_Unicode c )
{
    *pBuf = c;
    pBuf++;
    return pBuf;
}

// -----------------------------------------------------------------------

inline xub_Unicode* ImplAddString( xub_Unicode* pBuf, const xub_Unicode* pCopyBuf, USHORT nLen )
{
    memcpy( pBuf, pCopyBuf, nLen*sizeof( xub_Unicode ) );
    return pBuf+nLen;
}

// -----------------------------------------------------------------------

inline xub_Unicode* ImplAddString( xub_Unicode* pBuf, const XubString& rStr )
{
    return ImplAddString( pBuf, rStr.GetBuffer(), rStr.Len() );
}

// =======================================================================

LanguageTable::LanguageTable()
{
    DBG_CTOR( LanguageTable, NULL );

    nRefCount           = 0;
    eLanguage           = LANGUAGE_DONTKNOW;
    pTypeProc           = NULL;
    pLowerProc          = NULL;
    pUpperProc          = NULL;
    pCompareProc        = NULL;
    pIndexProc          = NULL;

    USHORT i;
    for ( i = 0; i < DAYOFWEEK_COUNT; i++ )
        pDayText[i] = new XubString;
    for ( i = 0; i < DAYOFWEEK_COUNT; i++ )
        pAbbrevDayText[i] = new XubString;
    for ( i = 0; i < MONTH_COUNT; i++ )
        pMonthText[i] = new XubString;
    for ( i = 0; i < MONTH_COUNT; i++ )
        pAbbrevMonthText[i] = new XubString;
    for ( i = 0; i < FOLLOWTEXT_COUNT; i++ )
        pFollowingText[i] = new XubString;

    meQuotationMarkStart        = APOSTOPHE_QUOTE;
    meQuotationMarkEnd          = APOSTOPHE_QUOTE;
    meQuotationMarkStart2       = APOSTOPHE_QUOTE;
    meQuotationMarkEnd2         = APOSTOPHE_QUOTE;
    meDoubleQuotationMarkStart  = QUOTATION_MARK;
    meDoubleQuotationMarkEnd    = QUOTATION_MARK;
    meDoubleQuotationMarkStart2 = QUOTATION_MARK;
    meDoubleQuotationMarkEnd2   = QUOTATION_MARK;
}

// -----------------------------------------------------------------------

LanguageTable::LanguageTable( const LanguageTable& rTable )
{
    DBG_CTOR( LanguageTable, NULL );

    nRefCount           = 0;
    eLanguage           = rTable.eLanguage;
    pTypeProc           = rTable.pTypeProc;
    pLowerProc          = rTable.pLowerProc;
    pUpperProc          = rTable.pUpperProc;
    pCompareProc        = rTable.pCompareProc;
    pIndexProc          = rTable.pIndexProc;

    USHORT i;
    for ( i = 0; i < DAYOFWEEK_COUNT; i++ )
        pDayText[i] = new XubString( *(rTable.pDayText[i]) );
    for ( i = 0; i < DAYOFWEEK_COUNT; i++ )
        pAbbrevDayText[i] = new XubString( *(rTable.pAbbrevDayText[i]) );
    for ( i = 0; i < MONTH_COUNT; i++ )
        pMonthText[i] = new XubString( *(rTable.pMonthText[i]) );
    for ( i = 0; i < MONTH_COUNT; i++ )
        pAbbrevMonthText[i] = new XubString( *(rTable.pAbbrevMonthText[i]) );
    for ( i = 0; i < FOLLOWTEXT_COUNT; i++ )
        pFollowingText[i] = new XubString( *(rTable.pFollowingText[i]) );

    meQuotationMarkStart        = rTable.meQuotationMarkStart;
    meQuotationMarkEnd          = rTable.meQuotationMarkEnd;
    meQuotationMarkStart2       = rTable.meQuotationMarkStart2;
    meQuotationMarkEnd2         = rTable.meQuotationMarkEnd2;
    meDoubleQuotationMarkStart  = rTable.meDoubleQuotationMarkStart;
    meDoubleQuotationMarkEnd    = rTable.meDoubleQuotationMarkEnd;
    meDoubleQuotationMarkStart2 = rTable.meDoubleQuotationMarkStart2;
    meDoubleQuotationMarkEnd2   = rTable.meDoubleQuotationMarkEnd2;
}

// -----------------------------------------------------------------------

LanguageTable::~LanguageTable()
{
    DBG_DTOR( LanguageTable, NULL );

    USHORT i;
    for ( i = 0; i < DAYOFWEEK_COUNT; i++ )
        delete pDayText[i];
    for ( i = 0; i < DAYOFWEEK_COUNT; i++ )
        delete pAbbrevDayText[i];
    for ( i = 0; i < MONTH_COUNT; i++ )
        delete pMonthText[i];
    for ( i = 0; i < MONTH_COUNT; i++ )
        delete pAbbrevMonthText[i];
    for ( i = 0; i < FOLLOWTEXT_COUNT; i++ )
        delete pFollowingText[i];
}

// =======================================================================

FormatTable::FormatTable()
{
    DBG_CTOR( FormatTable, NULL );

    eLanguage                   = LANGUAGE_DONTKNOW;
    eWeekStart                  = MONDAY;
    eWeekCountStart             = WEEKCOUNT_FIRSTDAY;
    eDateFormat                 = DMY;
    cDateSep                    = '/';
    bDateDayLeadingZero         = TRUE;
    bDateMonthLeadingZero       = TRUE;
    bDateCentury                = FALSE;
    eLongDateFormat             = DMY;
    eLongDateDayOfWeekFormat    = DAYOFWEEK_NONE;
    bLongDateDayLeadingZero     = TRUE;
    eLongDateMonthFormat        = MONTH_NORMAL;
    bLongDateCentury            = TRUE;
    eTimeFormat                 = HOUR_24;
    cTimeSep                    = ':';
    cTime100SecSep              = '.';
    bTimeLeadingZero            = FALSE;
    cNumThousandSep             = ',';
    bNumThousandSep             = TRUE;
    cNumDecimalSep              = '.';
    bNumLeadingZero             = TRUE;
    nNumDigits                  = 2;
    bNumTrailingZeros           = TRUE;
    nCurrPositiveFormat         = 0;
    nCurrNegativeFormat         = 1;
    nCurrDigits                 = 2;
    cCurrZeroChar               = '0';
    cListSep                    = ';';
    nPercentFormat              = 0;
    eMeasurementSystem          = MEASURE_METRIC;
}

// -----------------------------------------------------------------------

FormatTable::FormatTable( const FormatTable& rTable ) :
                aLongDateDayOfWeekSep( rTable.aLongDateDayOfWeekSep ),
                aLongDateDaySep( rTable.aLongDateDaySep ),
                aLongDateMonthSep( rTable.aLongDateMonthSep ),
                aLongDateYearSep( rTable.aLongDateYearSep ),
                aTimeAM( rTable.aTimeAM ),
                aTimePM( rTable.aTimePM ),
                aTimeStr( rTable.aTimeStr ),
                aCurrSymbol( rTable.aCurrSymbol ),
                aCurrBankSymbol( rTable.aCurrBankSymbol ),
                aCurrSymbol2( rTable.aCurrSymbol2 ),
                aCurrBankSymbol2( rTable.aCurrBankSymbol2 )
{
    DBG_CTOR( FormatTable, NULL );

    eLanguage                   = rTable.eLanguage;
    eWeekStart                  = rTable.eWeekStart;
    eWeekCountStart             = rTable.eWeekCountStart;
    eDateFormat                 = rTable.eDateFormat;
    cDateSep                    = rTable.cDateSep;
    bDateDayLeadingZero         = rTable.bDateDayLeadingZero;
    bDateMonthLeadingZero       = rTable.bDateMonthLeadingZero;
    bDateCentury                = rTable.bDateCentury;
    eLongDateFormat             = rTable.eLongDateFormat;
    eLongDateDayOfWeekFormat    = rTable.eLongDateDayOfWeekFormat;
    bLongDateDayLeadingZero     = rTable.bLongDateDayLeadingZero;
    eLongDateMonthFormat        = rTable.eLongDateMonthFormat;
    bLongDateCentury            = rTable.bLongDateCentury;
    eTimeFormat                 = rTable.eTimeFormat;
    cTimeSep                    = rTable.cTimeSep;
    cTime100SecSep              = rTable.cTime100SecSep;
    bTimeLeadingZero            = rTable.bTimeLeadingZero;
    cNumThousandSep             = rTable.cNumThousandSep;
    bNumThousandSep             = rTable.bNumThousandSep;
    cNumDecimalSep              = rTable.cNumDecimalSep;
    bNumLeadingZero             = rTable.bNumLeadingZero;
    nNumDigits                  = rTable.nNumDigits;
    bNumTrailingZeros           = rTable.bNumTrailingZeros;
    nCurrPositiveFormat         = rTable.nCurrPositiveFormat;
    nCurrNegativeFormat         = rTable.nCurrNegativeFormat;
    nCurrDigits                 = rTable.nCurrDigits;
    cCurrZeroChar               = rTable.cCurrZeroChar;
    cListSep                    = rTable.cListSep;
    nPercentFormat              = rTable.nPercentFormat;
    eMeasurementSystem          = rTable.eMeasurementSystem;
}

// -----------------------------------------------------------------------

FormatTable::~FormatTable()
{
    DBG_DTOR( FormatTable, NULL );
}

// =======================================================================

void ImplDeleteIntnData( InternationalData* pData )
{
    if ( pData->nRefCount )
        pData->nRefCount--;
    else
    {
        if ( !pData->bStaticLang )
            delete pData->pLanguage;
        if ( !pData->bStaticFormat )
            delete pData->pFormat;
        delete pData;
    }
}

// -----------------------------------------------------------------------

inline void ImplIncRefCount( International* pIntn )
{
    // Laeuft der Referenzzaehler ueber
    if ( pIntn->pData->nRefCount == USHRT_MAX )
        ImplCopyIntnData( pIntn );
    else
        pIntn->pData->nRefCount++;
}

// =======================================================================

void International::Init( LanguageType eLanguage, LanguageType eFormat )
{
    DBG_CTOR( International, NULL );

    pData            = new InternationalData;
    pData->nRefCount = 0;

    // Language-Tabelle anlegen
    LanguageTable* pLanguageTable;
    pLanguageTable = ImplGetLanguageTable( eLanguage );
    // Falls keine existiert, dann mit einer aehnlichen probieren
    if ( !pLanguageTable )
        pLanguageTable = ImplGetLanguageTable( GetNeutralLanguage( eLanguage ) );
    // Falls keine existiert, dann System-Einstellungen
    if ( !pLanguageTable )
        pLanguageTable = ImplGetLanguageTable( LANGUAGE_SYSTEM );
    // Wenn Sprache uebereinstimmt, cachen wir die Tabelle, ansonsten
    // machen wir eine Kopie
    if ( pLanguageTable->eLanguage == eLanguage )
    {
        pData->pLanguage = pLanguageTable;
        pData->bStaticLang = TRUE;
    }
    else
    {
        // Tabelle kopieren
        pData->pLanguage = new LanguageTable( *pLanguageTable );
        pData->pLanguage->eLanguage = eLanguage;
        pData->bStaticLang = FALSE;
    }

    // Format-Tabelle anlegen
    FormatTable* pFormatTable;
    pFormatTable = ImplGetFormatTable( eFormat );
    // Falls keine existiert, dann mit einer aehnlichen probieren
    if ( !pFormatTable )
        pFormatTable = ImplGetFormatTable( GetNeutralLanguage( eFormat ) );
    // Falls keine existiert, dann System-Einstellungen
    if ( !pFormatTable )
        pFormatTable = ImplGetFormatTable( LANGUAGE_SYSTEM );
    // Wenn Sprache uebereinstimmt, cachen wir die Tabelle, ansonsten
    // machen wir eine Kopie
    if ( pLanguageTable->eLanguage == eLanguage )
    {
        pData->pFormat = pFormatTable;
        pData->bStaticFormat = TRUE;
    }
    else
    {
        // Tabelle kopieren
        pData->pFormat = new FormatTable( *pFormatTable );
        pData->pFormat->eLanguage = eFormat;
        pData->bStaticFormat = FALSE;
    }
}

// -----------------------------------------------------------------------

International::International( const International& rIntn )
{
    DBG_CTOR( International, NULL );
    DBG_CHKOBJ( &rIntn, International, NULL );

    pData = rIntn.pData;
    ImplIncRefCount( this );
}

// -----------------------------------------------------------------------

International::~International()
{
    DBG_DTOR( International, NULL );

    ImplDeleteIntnData( pData );
}

// -----------------------------------------------------------------------

XubString International::GetDate( const Date& rDate ) const
{
    DBG_CHKTHIS( International, NULL );

    xub_Unicode aBuf[20];
    xub_Unicode* pBuf = aBuf;
    USHORT  nDay    = rDate.GetDay();
    USHORT  nMonth  = rDate.GetMonth();
    USHORT  nYear   = rDate.GetYear();
    USHORT  nYearLen;

    if ( IsDateCentury() )
        nYearLen = 4;
    else
    {
        nYearLen = 2;
        nYear %= 100;
    }

    if ( GetDateFormat() == DMY )
    {
        pBuf = ImplAdd2UNum( pBuf, nDay, IsDateDayLeadingZero() );
        pBuf = ImplAddString( pBuf, GetDateSep() );
        pBuf = ImplAdd2UNum( pBuf, nMonth, IsDateMonthLeadingZero() );
        pBuf = ImplAddString( pBuf, GetDateSep() );
        pBuf = ImplAddUNum( pBuf, nYear, nYearLen );
    }
    else if ( GetDateFormat() == MDY )
    {
        pBuf = ImplAdd2UNum( pBuf, nMonth, IsDateMonthLeadingZero() );
        pBuf = ImplAddString( pBuf, GetDateSep() );
        pBuf = ImplAdd2UNum( pBuf, nDay, IsDateDayLeadingZero() );
        pBuf = ImplAddString( pBuf, GetDateSep() );
        pBuf = ImplAddUNum( pBuf, nYear, nYearLen );
    }
    else
    {
        pBuf = ImplAddUNum( pBuf, nYear, nYearLen );
        pBuf = ImplAddString( pBuf, GetDateSep() );
        pBuf = ImplAdd2UNum( pBuf, nMonth, IsDateMonthLeadingZero() );
        pBuf = ImplAddString( pBuf, GetDateSep() );
        pBuf = ImplAdd2UNum( pBuf, nDay, IsDateDayLeadingZero() );
    }

    return XubString( aBuf, (USHORT)(ULONG)(pBuf-aBuf) );
}

// -----------------------------------------------------------------------

XubString International::GetLongDate( const Date& rDate ) const
{
    DBG_CHKTHIS( International, NULL );

    XubString   aDate;
    xub_Unicode     aBuf[20];
    xub_Unicode*    pBuf;

    // Wochentag anhaengen
    if ( GetLongDateDayOfWeekFormat() == DAYOFWEEK_SHORT )
    {
        aDate += GetAbbrevDayText( rDate.GetDayOfWeek() );
        aDate += GetLongDateDayOfWeekSep();
    }
    else if ( GetLongDateDayOfWeekFormat() == DAYOFWEEK_LONG )
    {
        aDate += GetDayText( rDate.GetDayOfWeek() );
        aDate += GetLongDateDayOfWeekSep();
    }

    // Tag ermitteln
    pBuf = ImplAdd2UNum( aBuf, rDate.GetDay(), IsLongDateDayLeadingZero() );
    XubString aDay( aBuf, (USHORT)(ULONG)(pBuf-aBuf) );
    aDay += GetLongDateDaySep();

    // Monat ermitteln
    XubString aMonth;
    if ( GetLongDateMonthFormat() == MONTH_NORMAL )
    {
        pBuf = ImplAdd2UNum( aBuf, rDate.GetMonth(), FALSE );
        aMonth = XubString( aBuf, (USHORT)(ULONG)(pBuf-aBuf) );
    }
    else if ( GetLongDateMonthFormat() == MONTH_ZERO )
    {
        pBuf = ImplAdd2UNum( aBuf, rDate.GetMonth(), TRUE );
        aMonth = XubString( aBuf, (USHORT)(ULONG)(pBuf-aBuf) );
    }
    else if ( GetLongDateMonthFormat() == MONTH_SHORT )
        aMonth = GetAbbrevMonthText( rDate.GetMonth() );
    else
        aMonth = GetMonthText( rDate.GetMonth() );
    aMonth += GetLongDateMonthSep();

    // Jahr ermitteln
    USHORT nYear = rDate.GetYear();
    if ( IsLongDateCentury() )
        pBuf = ImplAddUNum( aBuf, nYear, 4 );
    else
        pBuf = ImplAddUNum( aBuf, nYear % 100, 2 );
    XubString aYear( aBuf, (USHORT)(ULONG)(pBuf-aBuf) );
    aYear += GetLongDateYearSep();

    if ( GetLongDateFormat() == DMY )
    {
        aDate += aDay;
        aDate += aMonth;
        aDate += aYear;
    }
    else if ( GetLongDateFormat() == MDY )
    {
        aDate += aMonth;
        aDate += aDay;
        aDate += aYear;
    }
    else
    {
        aDate += aYear;
        aDate += aMonth;
        aDate += aDay;
    }

    return aDate;
}

// -----------------------------------------------------------------------

XubString International::GetDuration( const Time& rTime,
                                     BOOL bSec, BOOL b100Sec ) const
{
    DBG_CHKTHIS( International, NULL );

    xub_Unicode aBuf[20];
    xub_Unicode* pBuf = aBuf;

    if ( rTime < Time( 0 ) )
        pBuf = ImplAddString( pBuf, ' ' );

    if ( IsTimeLeadingZero() )
        pBuf = ImplAddUNum( pBuf, rTime.GetHour(), 2 );
    else
        pBuf = ImplAddUNum( pBuf, rTime.GetHour() );
    pBuf = ImplAddString( pBuf, GetTimeSep() );
    pBuf = ImplAdd2UNum( pBuf, rTime.GetMin(), TRUE );
    if ( bSec )
    {
        pBuf = ImplAddString( pBuf, GetTimeSep() );
        pBuf = ImplAdd2UNum( pBuf, rTime.GetSec(), TRUE );

        if ( b100Sec )
        {
            pBuf = ImplAddString( pBuf, GetTime100SecSep() );
            pBuf = ImplAdd2UNum( pBuf, rTime.Get100Sec(), TRUE );
        }
    }

    return XubString( aBuf, (USHORT)(ULONG)(pBuf-aBuf) );
}

// -----------------------------------------------------------------------

XubString International::GetTime( const Time& rTime,
                                 BOOL bSec, BOOL b100Sec ) const
{
    DBG_CHKTHIS( International, NULL );

    xub_Unicode aBuf[20];
    xub_Unicode* pBuf = aBuf;
    USHORT  nHour = rTime.GetHour();

    if ( GetTimeFormat() == HOUR_12 )
    {
        nHour %= 12;
        // 0.00 -> 12.00
        if ( !nHour )
            nHour = 12;
    }
    else
        nHour %= 24;

    pBuf = ImplAdd2UNum( pBuf, nHour, IsTimeLeadingZero() );
    pBuf = ImplAddString( pBuf, GetTimeSep() );
    pBuf = ImplAdd2UNum( pBuf, rTime.GetMin(), TRUE );
    if ( bSec )
    {
        pBuf = ImplAddString( pBuf, GetTimeSep() );
        pBuf = ImplAdd2UNum( pBuf, rTime.GetSec(), TRUE );

        if ( b100Sec )
        {
            pBuf = ImplAddString( pBuf, GetTime100SecSep() );
            pBuf = ImplAdd2UNum( pBuf, rTime.Get100Sec(), TRUE );
        }
    }

    XubString aStr( aBuf, (USHORT)(ULONG)(pBuf-aBuf) );

    if ( GetTimeFormat() == HOUR_12 )
    {
        if ( (rTime.GetHour() % 24) >= 12 )
            aStr += GetTimePM();
        else
            aStr += GetTimeAM();
    }
    else
        aStr += GetTimeStr();

    return aStr;
}

// -----------------------------------------------------------------------

xub_Unicode* ImplAddFormatNum( xub_Unicode* pBuf, const International& rIntn,
                               long nNumber, USHORT nDigits )
{
    xub_Unicode aNumBuf[32];
    xub_Unicode* pNumBuf;
    USHORT  nNumLen;
    USHORT  i = 0;
    BOOL    bNeg;

    // Ist Zahl negativ
    if ( nNumber < 0 )
    {
        nNumber *= -1;
        bNeg = TRUE;
        *pBuf = '-';
        pBuf++;
    }
    else
        bNeg = FALSE;

    // Zahl umwandeln
    pNumBuf = ImplAddUNum( aNumBuf, (ULONG)nNumber );
    nNumLen = (USHORT)(ULONG)(pNumBuf-aNumBuf);
    pNumBuf = aNumBuf;

    // Zahl kleiner als Nachkommastellen
    if ( nNumLen <= nDigits )
    {
        // Soll bei ,0 die Nachkommastellen entfernt werden
        if ( !nNumber && !rIntn.IsNumTrailingZeros() )
        {
            *pBuf = '0';
            pBuf++;
        }
        else
        {
            // LeadingZero, dann 0 einfuegen
            if ( rIntn.IsNumLeadingZero() )
            {
                *pBuf = '0';
                pBuf++;
            }

            // Dezimaltrenner anhaengen
            *pBuf = rIntn.GetNumDecimalSep();
            pBuf++;

            // Gegebenenfalls mit Nullen auffuellen
            while ( i < (nDigits-nNumLen) )
            {
                *pBuf = '0';
                pBuf++;
                i++;
            }

            // Nachkommastellen anhaengen
            while ( nNumLen )
            {
                *pBuf = *pNumBuf;
                pBuf++;
                pNumBuf++;
                nNumLen--;
            }
        }
    }
    else
    {
        // Zahl in Buffer einfuegen (exklusiv Nachkommastellen)
        USHORT nNumLen2 = nNumLen-nDigits;
        while ( i < nNumLen2 )
        {
            *pBuf = *pNumBuf;
            pBuf++;
            pNumBuf++;
            i++;

            // Muss Tausendertrenner eingefuegt werden
            if ( !((nNumLen2-i)%3) && rIntn.IsNumThousandSep() && (i < nNumLen2) )
            {
                *pBuf = rIntn.GetNumThousandSep();
                pBuf++;
            }
        }

        // Wenn Nachkommastellen vorhanden, dann anhaengen
        if ( nDigits )
        {
            *pBuf = rIntn.GetNumDecimalSep();
            pBuf++;

            BOOL bNullEnd = TRUE;
            while ( i < nNumLen )
            {
                if ( *pNumBuf != '0' )
                    bNullEnd = FALSE;

                *pBuf = *pNumBuf;
                pBuf++;
                pNumBuf++;
                i++;
            }

           // Soll bei ,0 die Nachkommastellen entfernt werden
           if ( !rIntn.IsNumTrailingZeros() && bNullEnd )
               pBuf -= nDigits+1;
        }
    }

    return pBuf;
}

// -----------------------------------------------------------------------

XubString International::GetNum( long nNumber, USHORT nDigits ) const
{
    DBG_CHKTHIS( International, NULL );

    xub_Unicode aBuf[45];
    xub_Unicode* pBuf = ImplAddFormatNum( aBuf, *this, nNumber, nDigits );
    return XubString( aBuf, (USHORT)(ULONG)(pBuf-aBuf) );
}

// -----------------------------------------------------------------------

XubString International::GetCurr( long nNumber, USHORT nDigits ) const
{
    DBG_CHKTHIS( International, NULL );

    xub_Unicode         aBuf[75];
    xub_Unicode         aNumBuf[45];
    xub_Unicode*        pNumBuf;
    xub_Unicode*        pStartBuf;
    xub_Unicode*        pBuf;
    const XubString& rSymbol = GetCurrSymbol();
    USHORT          nNumLen;
    xub_Unicode         cZeroChar = GetCurrZeroChar();

    if ( rSymbol.Len() < (sizeof(aBuf)-50) )
        pStartBuf = aBuf;
    else
        pStartBuf = new xub_Unicode[50+rSymbol.Len()];
    pBuf = pStartBuf;

    BOOL bNeg;
    if ( nNumber < 0 )
    {
        bNeg = TRUE;
        nNumber *= -1;
    }
    else
        bNeg = FALSE;

    // Zahl umwandeln
    pNumBuf = ImplAddFormatNum( aNumBuf, *this, nNumber, nDigits );
    nNumLen = (USHORT)(ULONG)(pNumBuf-aNumBuf);

    // Zeros durch Ersatzzeichen ersetzen
    if ( (cZeroChar != '0') && nDigits && IsNumTrailingZeros() )
    {
        xub_Unicode* pTempBuf;
        USHORT  i;
        BOOL    bZero = TRUE;

        pTempBuf = aNumBuf+nNumLen-nDigits;
        i = 0;
        do
        {
            if ( *pTempBuf != '0' )
            {
                bZero = FALSE;
                break;
            }

            pTempBuf++;
            i++;
        }
        while ( i < nDigits );

        if ( bZero )
        {
            pTempBuf = aNumBuf+nNumLen-nDigits;
            i = 0;
            do
            {
                *pTempBuf = cZeroChar;
                pTempBuf++;
                i++;
            }
            while ( i < nDigits );
        }
    }

    if ( !bNeg )
    {
        switch( GetCurrPositiveFormat() )
        {
            case 0:
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                break;
            case 1:
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, rSymbol );
                break;
            case 2:
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, ' ' );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                break;
            case 3:
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, ' ' );
                pBuf = ImplAddString( pBuf, rSymbol );
                break;
        }
    }
    else
    {
        switch( GetCurrNegativeFormat() )
        {
            case 0:
                pBuf = ImplAddString( pBuf, '(' );
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, ')' );
                break;
            case 1:
                pBuf = ImplAddString( pBuf, '-' );
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                break;
            case 2:
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, '-' );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                break;
            case 3:
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, '-' );
                break;
            case 4:
                pBuf = ImplAddString( pBuf, '(' );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, ')' );
                break;
            case 5:
                pBuf = ImplAddString( pBuf, '-' );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, rSymbol );
                break;
            case 6:
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, '-' );
                pBuf = ImplAddString( pBuf, rSymbol );
                break;
            case 7:
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, '-' );
                break;
            case 8:
                pBuf = ImplAddString( pBuf, '-' );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, ' ' );
                pBuf = ImplAddString( pBuf, rSymbol );
                break;
            case 9:
                pBuf = ImplAddString( pBuf, '-' );
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, ' ' );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                break;
            case 10:
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, ' ' );
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, '-' );
                break;
            case 11:
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, ' ' );
                pBuf = ImplAddString( pBuf, '-' );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                break;
            case 12:
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, ' ' );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, '-' );
                break;
            case 13:
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, '-' );
                pBuf = ImplAddString( pBuf, ' ' );
                pBuf = ImplAddString( pBuf, rSymbol );
                break;
            case 14:
                pBuf = ImplAddString( pBuf, '(' );
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, ' ' );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, ')' );
                break;
            case 15:
                pBuf = ImplAddString( pBuf, '(' );
                pBuf = ImplAddString( pBuf, aNumBuf, nNumLen );
                pBuf = ImplAddString( pBuf, ' ' );
                pBuf = ImplAddString( pBuf, rSymbol );
                pBuf = ImplAddString( pBuf, ')' );
                break;
        }
    }

    XubString aNumber( aBuf, (USHORT)(ULONG)(pBuf-aBuf) );

    if ( pStartBuf != aBuf )
        delete pStartBuf;

    return aNumber;
}

// -----------------------------------------------------------------------

XubString International::GetPercent( long nPercent ) const
{
    String aPercentStr = String::CreateFromInt32( nPercent );

    switch ( GetPercentFormat() )
    {
        case 0:
            aPercentStr.Append( '%' );
            break;
        case 1:
            aPercentStr.AppendAscii( " %" );
            break;
        case 2:
            aPercentStr.Insert( '%', 0 );
            break;
    }

    return aPercentStr;
}

// -----------------------------------------------------------------------

International& International::operator=( const International& rIntn )
{
    DBG_CHKTHIS( International, NULL );
    DBG_CHKOBJ( &rIntn, International, NULL );

    // Sind es nicht die gleichen Daten
    if ( pData != rIntn.pData )
    {
        ImplDeleteIntnData( pData );
        pData = rIntn.pData;
        ImplIncRefCount( this );
    }

    return *this;
}

// -----------------------------------------------------------------------

LanguageType International::GetNeutralLanguage( LanguageType eLang )
{
    USHORT nLang = (USHORT)eLang;

    // User-Languages nicht verarbeiten
    if ( (nLang & 0x03FF) >= 0x0200 )
        return eLang;

    nLang &= 0x03FF;

    LanguageType ePrimLang = (LanguageType)(nLang | 0x0400);

    switch ( ePrimLang )
    {
        case LANGUAGE_CHINESE_TRADITIONAL:
            eLang = LANGUAGE_CHINESE;
            break;
        case LANGUAGE_ENGLISH_US:
            eLang = LANGUAGE_ENGLISH;
            break;
        case LANGUAGE_NORWEGIAN_BOKMAL:
            eLang = LANGUAGE_NORWEGIAN;
            break;
        case LANGUAGE_PORTUGUESE_BRAZILIAN:
            eLang = LANGUAGE_PORTUGUESE;
            break;

        default:
            eLang = ePrimLang;
            break;
    }

    return eLang;
}

// -----------------------------------------------------------------------

LanguageType International::GetRealLanguage( LanguageType eLang )
{
    if ( eLang == LANGUAGE_SYSTEM )
        eLang = GetSystemLanguage();
    if ( eLang == LANGUAGE_DONTKNOW )
        eLang = LANGUAGE_ENGLISH_US;
    return eLang;
}

// =======================================================================

ImplRegisterLanguageData* ImplGetFirstLanguageData()
{
#ifndef BOOTSTRAP
    TOOLSINDATA* pToolsData = ImplGetToolsInData();
#else
    TOOLSINDATA* pToolsData = 0x0;
#endif

    // Falls schon initialisiert, muss es min. LANGUAGE_SYSTEM geben
    if ( !pToolsData->mpFirstLangData )
    {
#ifdef IMPL_NO_DEFAULT_TABLES
#undef STD_LANGUAGE
#define STD_LANGUAGE 3
        static LanguageType aStdLangTab[STD_LANGUAGE] =
        {
            LANGUAGE_SYSTEM,
            LANGUAGE_ENGLISH,
            LANGUAGE_ENGLISH_US
        };
#else
        static LanguageType aStdLangTab[STD_LANGUAGE] =
        {
            LANGUAGE_SYSTEM,
            LANGUAGE_DANISH,
            LANGUAGE_DUTCH,
            LANGUAGE_DUTCH_BELGIAN,
            LANGUAGE_ENGLISH,
            LANGUAGE_ENGLISH_US,
            LANGUAGE_ENGLISH_UK,
            LANGUAGE_ENGLISH_AUS,
            LANGUAGE_ENGLISH_CAN,
            LANGUAGE_ENGLISH_NZ,
            LANGUAGE_ENGLISH_EIRE,
            LANGUAGE_ENGLISH_BELIZE,
            LANGUAGE_ENGLISH_TRINIDAD,
            LANGUAGE_ENGLISH_ZIMBABWE,
            LANGUAGE_ENGLISH_PHILIPPINES,
            LANGUAGE_FINNISH,
            LANGUAGE_FRENCH,
            LANGUAGE_FRENCH_BELGIAN,
            LANGUAGE_FRENCH_CANADIAN,
            LANGUAGE_FRENCH_SWISS,
            LANGUAGE_FRENCH_MONACO,
            LANGUAGE_GERMAN,
            LANGUAGE_GERMAN_SWISS,
            LANGUAGE_GERMAN_AUSTRIAN,
            LANGUAGE_ITALIAN,
            LANGUAGE_ITALIAN_SWISS,
            LANGUAGE_NORWEGIAN,
            LANGUAGE_NORWEGIAN_BOKMAL,
            LANGUAGE_PORTUGUESE,
            LANGUAGE_PORTUGUESE_BRAZILIAN,
            LANGUAGE_SPANISH,
            LANGUAGE_SPANISH_MODERN,
            LANGUAGE_SWEDISH
        };
#endif

        ImplRegisterLanguageData* pData;
        ImplRegisterLanguageData* pNewData;

        // LANGUAGE_SYSTEM muss immer vorhanden sein
        pData               = new ImplRegisterLanguageData;
        pData->eLanguage    = LANGUAGE_SYSTEM;
        pData->pTable       = NULL;
        pData->bStandard    = FALSE;
        pData->bSystem      = TRUE;
        pToolsData->mpFirstLangData = pData;

        // Standardtabellen eintragen
        USHORT  i;
        USHORT  j;
        BOOL    bFound;
        for ( i = 1; i < STD_LANGUAGE; i++ )
        {
            pNewData            = new ImplRegisterLanguageData;
            pNewData->eLanguage = aStdLangTab[i];
            pNewData->pTable    = NULL;
            pNewData->bStandard = TRUE;
            pNewData->bSystem   = FALSE;
            pData->pNext = pNewData;
            pData = pNewData;
        }

        // Systemtabellen dazutragen
        USHORT nSystemTab = ImplGetSystemLanguageCount();
        for ( i = 0; i < nSystemTab; i++ )
        {
            LanguageType eLanguage = GetSystemLanguage( i );

            // Erstmal schauen, ob nicht schon entsprechende Standard-Tabelle
            // vorhanden ist
            bFound = FALSE;
            for ( j = 0; j < STD_LANGUAGE; j++ )
            {
                if ( eLanguage == aStdLangTab[j] )
                {
                    bFound = TRUE;
                    break;
                }
            }

            // Wenn nicht, dann dazutragen
            if ( !bFound )
            {
                pNewData            = new ImplRegisterLanguageData;
                pNewData->eLanguage = eLanguage;
                pNewData->pTable    = NULL;
                pNewData->bStandard = FALSE;
                pNewData->bSystem   = TRUE;
                pData->pNext = pNewData;
                pData = pNewData;
            }
        }

        pData->pNext = NULL;
    }

    return pToolsData->mpFirstLangData;
}

// -----------------------------------------------------------------------

LanguageTable* ImplGetLanguageTable( LanguageType eLanguage )
{
    ImplRegisterLanguageData* pData = ImplGetFirstLanguageData();
    while ( pData )
    {
        // Language suchen
        if ( pData->eLanguage == eLanguage )
        {
            // Falls Tabelle noch nicht erzeugt wurde, dann erzeugen
            if ( !pData->pTable )
            {
                // Bei System wird eine Kopie angelegt, sonst Default-Tabelle
                if ( pData->bSystem && !pData->bStandard )
                {
                    LanguageType eCopyLanguage = International::GetNeutralLanguage( pData->eLanguage );
                    // Bei Language-System die Systemsprache abfragen
                    if ( eLanguage == LANGUAGE_SYSTEM )
                    {
                        LanguageType eSystemLanguage = GetSystemLanguage();
                        if ( eSystemLanguage != LANGUAGE_DONTKNOW )
                            eCopyLanguage = eSystemLanguage;
                    }
                    else
                    {
                        // Wenn gleiche Sprache, dann ENGLISH nehmen
                        if ( eLanguage == eCopyLanguage )
                            eCopyLanguage = LANGUAGE_ENGLISH;
                    }
                    LanguageTable* pCopyTable = ImplGetLanguageTable( eCopyLanguage );
                    if ( !pCopyTable )
                        pCopyTable = ImplGetLanguageTable( LANGUAGE_ENGLISH );
                    pData->pTable = new LanguageTable( *pCopyTable );
                }
                else
                    pData->pTable = new LanguageTable;
                pData->pTable->eLanguage = pData->eLanguage;

                // Falls als Standard registriert, die Standardwerte eintragen
                if ( pData->bStandard )
                    ImplUpdateStandardLanguage( pData->eLanguage, pData->pTable );
                // Falls als System registriert, die Systemwerte eintragen
                if ( pData->bSystem )
                    ImplUpdateSystemLanguage( pData->eLanguage, pData->pTable );
            }

            return pData->pTable;
        }

        pData = pData->pNext;
    }

    return NULL;
}

// =======================================================================

ImplRegisterFormatData* ImplGetFirstFormatData()
{
#ifndef BOOTSTRAP
    TOOLSINDATA* pToolsData = ImplGetToolsInData();
#else
    TOOLSINDATA* pToolsData = 0x0;
#endif

    // Falls schon initialisiert, muss es min. LANGUAGE_SYSTEM geben
    if ( !pToolsData->mpFirstFormatData )
    {
#ifdef IMPL_NO_DEFAULT_TABLES
#undef STD_FORMAT
#define STD_FORMAT 2
        static LanguageType aStdFormatTab[STD_FORMAT] =
        {
            LANGUAGE_SYSTEM,
            LANGUAGE_ENGLISH_US
        };
#else
        static LanguageType aStdFormatTab[STD_FORMAT] =
        {
            LANGUAGE_SYSTEM,
            LANGUAGE_AFRIKAANS,
            LANGUAGE_SPANISH_ARGENTINA,
            LANGUAGE_ENGLISH_AUS,
            LANGUAGE_GERMAN_AUSTRIAN,
            LANGUAGE_DUTCH_BELGIAN,
            LANGUAGE_FRENCH_BELGIAN,
            LANGUAGE_PORTUGUESE_BRAZILIAN,
            LANGUAGE_ENGLISH_CAN,
            LANGUAGE_FRENCH_CANADIAN,
            LANGUAGE_CATALAN,
            LANGUAGE_SPANISH_CHILE,
            LANGUAGE_SPANISH_COLOMBIA,
            LANGUAGE_SPANISH_ECUADOR,
            LANGUAGE_DANISH,
            LANGUAGE_DUTCH,
            LANGUAGE_ENGLISH_EIRE,
            LANGUAGE_FINNISH,
            LANGUAGE_FRENCH,
            LANGUAGE_GERMAN,
            LANGUAGE_GREEK,
            LANGUAGE_SPANISH_GUATEMALA,
            LANGUAGE_ICELANDIC,
            LANGUAGE_INDONESIAN,
            LANGUAGE_ITALIAN,
            LANGUAGE_FRENCH_LUXEMBOURG,
            LANGUAGE_GERMAN_LUXEMBOURG,
            LANGUAGE_GERMAN_LIECHTENSTEIN,
            LANGUAGE_HUNGARIAN,
            LANGUAGE_ENGLISH_JAMAICA,
            LANGUAGE_JAPANESE,
            LANGUAGE_SPANISH_MEXICAN,
            LANGUAGE_NORWEGIAN,
            LANGUAGE_NORWEGIAN_BOKMAL,
            LANGUAGE_NORWEGIAN_NYNORSK,
            LANGUAGE_ENGLISH_NZ,
            LANGUAGE_SPANISH_PANAMA,
            LANGUAGE_SPANISH_PARAGUAY,
            LANGUAGE_SPANISH_PERU,
            LANGUAGE_PORTUGUESE,
            LANGUAGE_SPANISH,
            LANGUAGE_SPANISH_MODERN,
            LANGUAGE_SWEDISH,
            LANGUAGE_FRENCH_SWISS,
            LANGUAGE_GERMAN_SWISS,
            LANGUAGE_ITALIAN_SWISS,
            LANGUAGE_TURKISH,
            LANGUAGE_ENGLISH_UK,
            LANGUAGE_ENGLISH_US,
            LANGUAGE_SPANISH_URUGUAY,
            LANGUAGE_SPANISH_VENEZUELA
        };
#endif

        ImplRegisterFormatData* pData;
        ImplRegisterFormatData* pNewData;

        // LANGUAGE_SYSTEM muss immer vorhanden sein
        pData               = new ImplRegisterFormatData;
        pData->eLanguage    = LANGUAGE_SYSTEM;
        pData->pTable       = NULL;
        pData->bStandard    = FALSE;
        pData->bSystem      = TRUE;
        pToolsData->mpFirstFormatData = pData;

        // Standardtabellen eintragen
        USHORT  i;
        USHORT  j;
        BOOL    bFound;
        for ( i = 1; i < STD_FORMAT; i++ )
        {
            pNewData            = new ImplRegisterFormatData;
            pNewData->eLanguage = aStdFormatTab[i];
            pNewData->pTable    = NULL;
            pNewData->bStandard = TRUE;
            pNewData->bSystem   = FALSE;
            pData->pNext = pNewData;
            pData = pNewData;
        }

        // Systemtabellen dazutragen
        USHORT nSystemTab = ImplGetSystemFormatCount();
        for ( i = 0; i < nSystemTab; i++ )
        {
            LanguageType eLanguage = GetSystemFormatLanguage( i );

            // Erstmal schauen, ob nicht schon entsprechende Standard-Tabelle
            // vorhanden ist
            bFound = FALSE;
            for ( j = 0; j < STD_FORMAT; j++ )
            {
                if ( eLanguage == aStdFormatTab[j] )
                {
                    bFound = TRUE;
                    break;
                }
            }

            // Wenn nicht, dann dazutragen
            if ( !bFound )
            {
                pNewData            = new ImplRegisterFormatData;
                pNewData->eLanguage = eLanguage;
                pNewData->pTable    = NULL;
                pNewData->bStandard = FALSE;
                pNewData->bSystem   = TRUE;
                pData->pNext = pNewData;
                pData = pNewData;
            }
        }

        pData->pNext = NULL;
    }

    return pToolsData->mpFirstFormatData;
}

// -----------------------------------------------------------------------

FormatTable* ImplGetFormatTable( LanguageType eLanguage )
{
    ImplRegisterFormatData* pData = ImplGetFirstFormatData();
    while ( pData )
    {
        // Language suchen
        if ( pData->eLanguage == eLanguage )
        {
            // Falls Tabelle noch nicht erzeugt wurde, dann erzeugen
            if ( !pData->pTable )
            {
                // Bei System wird eine Kopie angelegt, sonst Default-Tabelle
                if ( pData->bSystem && !pData->bStandard )
                {
                    LanguageType eCopyLanguage = International::GetNeutralLanguage( pData->eLanguage );
                    // Bei Language-System die Systemsprache abfragen
                    if ( eLanguage == LANGUAGE_SYSTEM )
                    {
                        LanguageType eSystemLanguage = GetSystemFormatLanguage();
                        if ( eSystemLanguage != LANGUAGE_DONTKNOW )
                            eCopyLanguage = eSystemLanguage;
                    }
                    else
                    {
                        // Wenn gleiche Sprache, dann ENGLISH_US nehmen
                        if ( eLanguage == eCopyLanguage )
                            eCopyLanguage = LANGUAGE_ENGLISH_US;
                    }
                    FormatTable* pCopyTable = ImplGetFormatTable( eCopyLanguage );
                    if ( !pCopyTable )
                        pCopyTable = ImplGetFormatTable( LANGUAGE_ENGLISH_US );
                    pData->pTable = new FormatTable( *pCopyTable );
                }
                else
                    pData->pTable = new FormatTable;
                pData->pTable->eLanguage = pData->eLanguage;

                // Falls als Standard registriert, die Standardwerte eintragen
                if ( pData->bStandard )
                    ImplUpdateStandardFormat( pData->eLanguage, pData->pTable );
                // Falls als System registriert, die Systemwerte eintragen
                if ( pData->bSystem )
                    ImplUpdateSystemFormat( pData->eLanguage, pData->pTable );
            }

            return pData->pTable;
        }

        pData = pData->pNext;
    }

    return NULL;
}

// =======================================================================

void ImplDeleteIntnListData()
{
    // Falls International-Tabellen angelegt wurden, dann Updaten
#ifndef BOOTSTRAP
    TOOLSINDATA* pToolsData = ImplGetToolsInData();
#else
    TOOLSINDATA* pToolsData = 0x0;
#endif
    if ( pToolsData->mpFirstLangData )
    {
        ImplRegisterLanguageData* pTemp;
        ImplRegisterLanguageData* pData = pToolsData->mpFirstLangData;
        while ( pData )
        {
            pTemp = pData->pNext;
            if ( pData->pTable )
                delete pData->pTable;
            delete pData;
            pData = pTemp;
        }

        pToolsData->mpFirstLangData = NULL;
    }
    if ( pToolsData->mpFirstFormatData )
    {
        ImplRegisterFormatData* pTemp;
        ImplRegisterFormatData* pData = pToolsData->mpFirstFormatData;
        while ( pData )
        {
            pTemp = pData->pNext;
            if ( pData->pTable )
                delete pData->pTable;
            delete pData;
            pData = pTemp;
        }

        pToolsData->mpFirstFormatData = NULL;
    }
}

// =======================================================================

void UpdateInternationalSystemTables()
{
    // Systemtabellen updaten
    ImplUpdateSystemLanguage( LANGUAGE_SYSTEM, NULL );
    ImplUpdateSystemFormat( LANGUAGE_SYSTEM, NULL );

    // Falls International-Tabellen angelegt wurden, dann Updaten
#ifndef BOOTSTRAP
    TOOLSINDATA* pToolsData = ImplGetToolsInData();
#else
    TOOLSINDATA* pToolsData = 0x0;
#endif
    if ( pToolsData->mpFirstLangData )
    {
        ImplRegisterLanguageData* pData = pToolsData->mpFirstLangData;
        while ( pData )
        {
            if ( pData->bSystem && pData->pTable )
                ImplUpdateSystemLanguage( pData->eLanguage, pData->pTable );

            pData = pData->pNext;
        }
    }
    if ( pToolsData->mpFirstFormatData )
    {
        ImplRegisterFormatData* pData = pToolsData->mpFirstFormatData;
        while ( pData )
        {
            if ( pData->bSystem && pData->pTable )
                ImplUpdateSystemFormat( pData->eLanguage, pData->pTable );

            pData = pData->pNext;
        }
    }
}
