/*************************************************************************
 *
 *  $RCSfile: b3dprint.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:30:10 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#pragma hdrstop

#ifndef _B3D_B3DPRINT_HXX
#include "b3dprint.hxx"
#endif

#ifndef _B3D_B3DTEX_HXX
#include "b3dtex.hxx"
#endif

#ifndef _SV_POLY_HXX
#include <vcl/poly.hxx>
#endif

#ifndef _INC_FLOAT
#include <float.h>
#endif

/*************************************************************************
|*
|* Konstruktor B3dPrimitive
|*
\************************************************************************/

B3dPrimitive::B3dPrimitive(B3dPrimitiveType eTy, ULONG nInd)
:	eType(eTy),
	nIndex(nInd),
	pLeft(NULL),
	pRight(NULL),
	pSame(NULL),
	nMaterialIndex(NO_MATERIAL_INDEX),
	bIsVisible(TRUE),
	bIsLeftDone(FALSE),
	bIsRightDone(FALSE),
	bIsOutputDone(FALSE),
	bIsSameDone(FALSE)
{
}

/*************************************************************************
|*
|* Primitiv zuruecksetzen
|*
\************************************************************************/

void B3dPrimitive::Reset()
{
	pLeft = pRight = pSame = NULL;
	SetNothingDone();
	bIsVisible = TRUE;
	nMaterialIndex=NO_MATERIAL_INDEX;
}

/*************************************************************************
|*
|* Alles was einen Pfad besitzt auf NICHT ERLEDIGT setzen
|*
\************************************************************************/

void B3dPrimitive::SetNothingDone(BOOL bNew)
{
	bIsLeftDone = pLeft ? bNew : !bNew;
	bIsRightDone = pRight ? bNew : !bNew;
	bIsSameDone = pSame ? bNew : !bNew;
	bIsOutputDone = bNew;
}

/*************************************************************************
|*
|* Bucket fuer Primitive
|*
\************************************************************************/

BASE3D_IMPL_BUCKET(B3dPrimitive, Bucket)

/*************************************************************************
|*
|* Bucket fuer Base3DBSPLocal
|*
\************************************************************************/

BASE3D_IMPL_BUCKET(Base3DBSPLocal, Bucket)

/*************************************************************************
|*
|* Konstruktor Base3DPrinter
|*
\************************************************************************/

Base3DPrinter::Base3DPrinter(OutputDevice* pOutDev)
:	Base3DCommon(pOutDev),
	pBSPTreeRoot(NULL),
	fMinTriangleSize(0.0),
	fMinLineLength(0.0),
	nMinColorDistance(0L),
	nMaterialFrontIndex(0),
	nMaterialBackIndex(1),
	bBackMaterialChanged(FALSE),
	bFrontMaterialChanged(FALSE),
	nLocalVarIndex(0),
	aEntityBucket(14),		// 16K
	aPrimitiveBucket(15),	// 32K
	aMaterialBucket(12),	// 4K
	aLocalVarBucket(12)		// 4K
{
	// Darstellungsqualitaet beim Drucker etwas hoeher ansetzen
	SetDisplayQuality(200);
}

/*************************************************************************
|*
|* Destruktor Base3DPrinter
|*
\************************************************************************/

Base3DPrinter::~Base3DPrinter()
{
}

/*************************************************************************
|*
|* Typbestimmung
|*
\************************************************************************/

UINT16 Base3DPrinter::GetBase3DType()
{
	return BASE3D_TYPE_PRINTER;
}

/*************************************************************************
|*
|* Start der Szenenbeschreibung:
|* Buffer leeren
|*
\************************************************************************/

void Base3DPrinter::StartScene()
{
	// buffer leeren
	EmptyBuckets();

	// call parent
	Base3DCommon::StartScene();
}

/*************************************************************************
|*
|* Ende der Szenenbeschreibung:
|* Alle Zwischengespeicherten Daten ausgeben
|* Buffer leeren
|*
\************************************************************************/

void Base3DPrinter::EndScene()
{
	// Primitive ausgeben
	if(aPrimitiveBucket.Count())
	{
		// Darstellungsqualitaet einstellen
		UINT8 nQuality = GetDisplayQuality();
		if(nQuality >= 50)
		{
			SetMinimalColorDistance( (206 - (nQuality - 50)) );
		}
		else
		{
			SetMinimalColorDistance(0);
		}

		// Alle Primitive vom Typ Dreieck in den BSPTree einsortieren.
		ULONG nIndex = aPrimitiveBucket.Count();
		ULONG a;
		for(a=0; a < nIndex; a++)
		{
			B3dPrimitive& aCurrent = aPrimitiveBucket[a];
			if(aCurrent.IsVisible()
				&& aCurrent.GetType() == B3dPrimitiveTriangle)
				AddTriangleToBSPTree(pBSPTreeRoot, &aCurrent);
		}

		// Alle Primitive vom Typ Dreieck sind in den BSPTree eingeordnet.
		// Es muessen jetzt noch die Linien und Punkte einsortiert werden,
		// wobei Linien evtl. noch an den Ebenen des BSPTrees geschnitten
		// werden.
		for(a=0;a<nIndex;a++)
		{
			B3dPrimitive& aCurrent = aPrimitiveBucket[a];
			if(aCurrent.IsVisible())
			{
				switch(aCurrent.GetType())
				{
					case B3dPrimitivePoint :
					{
						AddPointToBSPTree(pBSPTreeRoot, &aCurrent);
						break;
					}
					case B3dPrimitiveLine :
					{
						AddLineToBSPTree(pBSPTreeRoot, &aCurrent);
						break;
					}
				}
			}
		}

		// BSPTree ausgeben
		// Ausgabereihenfolge ermitteln. Dies muss ohne
		// Rekursion realisiert werden, da sonst
		// der Stack evtl. ueberlaeuft.
		B3dPrimitive* pCurrent = pBSPTreeRoot;
		BOOL bStepFinished = FALSE;
		BOOL bLeftHasPrio = TRUE;
		pCurrent->SetParent(NULL);
		pCurrent->SetNothingDone();

		do
		{
			bStepFinished = FALSE;
			bLeftHasPrio = TRUE;

			// Die eigentliche Bedingung fuer davor/dahinter:
			// if((pCurrent->Normal() * Vector3D(0.0, 0.0, 1.0)) > 0.0)
			// ist hier auf die Folgende reduziert (siehe Skalarprodukt):
			// if(pCurrent->Normal().Z() < 0.0)
			if(pCurrent->GetType() == B3dPrimitiveTriangle
				&& pCurrent->Normal().Z() < 0.0)
				bLeftHasPrio = FALSE;

			if(bLeftHasPrio)
			{
				// Links
				if(pCurrent->GetLeft() && !pCurrent->IsLeftDone())
				{
					pCurrent->GetLeft()->SetParent(pCurrent);
					pCurrent->SetLeftDone();
					pCurrent = pCurrent->GetLeft();
					pCurrent->SetNothingDone();
					bStepFinished = TRUE;
				}
			}
			else
			{
				// Rechts
				if(pCurrent->GetRight() && !pCurrent->IsRightDone())
				{
					pCurrent->GetRight()->SetParent(pCurrent);
					pCurrent->SetRightDone();
					pCurrent = pCurrent->GetRight();
					pCurrent->SetNothingDone();
					bStepFinished = TRUE;
				}
			}

			// Selbst ausgeben
			if(!bStepFinished && !pCurrent->IsOutputDone())
			{
				if(pCurrent->IsVisible())
					PrintPrimitive(pCurrent);
				pCurrent->SetOutputDone();
			}

			// Same
			if(!bStepFinished && pCurrent->GetSame() && !pCurrent->IsSameDone())
			{
				pCurrent->GetSame()->SetParent(pCurrent);
				pCurrent->SetSameDone();
				pCurrent = pCurrent->GetSame();
				pCurrent->SetNothingDone();
				bStepFinished = TRUE;
			}

			if(!bStepFinished && bLeftHasPrio)
			{
				// Rechts
				if(pCurrent->GetRight() && !pCurrent->IsRightDone())
				{
					pCurrent->GetRight()->SetParent(pCurrent);
					pCurrent->SetRightDone();
					pCurrent = pCurrent->GetRight();
					pCurrent->SetNothingDone();
					bStepFinished = TRUE;
				}
			}
			else
			{
				// Links
				if(pCurrent->GetLeft() && !pCurrent->IsLeftDone())
				{
					pCurrent->GetLeft()->SetParent(pCurrent);
					pCurrent->SetLeftDone();
					pCurrent = pCurrent->GetLeft();
					pCurrent->SetNothingDone();
					bStepFinished = TRUE;
				}
			}

			// Zurueck
			if(!bStepFinished)
			{
#ifdef DBG_UTIL
				ULONG nWeight = 1;
				if(pCurrent->GetLeft())
					nWeight += pCurrent->GetLeft()->GetWeight();
				if(pCurrent->GetRight())
					nWeight += pCurrent->GetRight()->GetWeight();
				if(pCurrent->GetSame())
					nWeight += pCurrent->GetSame()->GetWeight();
				pCurrent->SetWeight(nWeight);
#endif
				if(pCurrent->GetParent() || pCurrent->IsAllDone())
					pCurrent = pCurrent->GetParent();
			}
		} while(pCurrent);
	}

	// buffer leeren
	EmptyBuckets();

	// call parent
	Base3DCommon::EndScene();
}

/*************************************************************************
|*
|* Primitiv tatsaechlich ausgeben
|*
\************************************************************************/

void Base3DPrinter::PrintPrimitive(B3dPrimitive* pPrimitive)
{
	if(pPrimitive->IsVisible())
	{
		ULONG nInd = pPrimitive->GetIndex();
		switch(pPrimitive->GetType())
		{
			case B3dPrimitivePoint :
			{
				Print3DPoint(pPrimitive, nInd);
				break;
			}
			case B3dPrimitiveLine :
			{
				Print3DLine(pPrimitive, nInd, nInd+1);
				break;
			}
			case B3dPrimitiveTriangle :
			{
				Print3DTriangle(pPrimitive, nInd, nInd+1, nInd+2);
				break;
			}
		}
	}
}

/*************************************************************************
|*
|* Zeichenfunktionen, alle Objekte sind geclippt und tiefensortiert.
|* Gib einen Punkt auf dem Drucker aus
|*
\************************************************************************/

void Base3DPrinter::Print3DPoint(B3dPrimitive* pPrimitive, ULONG nInd)
{
	B3dEntity& rEntity = aEntityBucket[nInd];

	// Geometrie holen
	rEntity.ToDeviceCoor(GetTransformationSet());

	// Farbe holen
	GetOutputDevice()->SetLineColor( rEntity.Color() );

	// Zeichnen
	GetOutputDevice()->DrawPixel(
		Point((long)(rEntity.Point().X() + 0.5),
		(long)(rEntity.Point().Y() + 0.5)));
}

/*************************************************************************
|*
|* Gib eine Linie auf dem Drucker aus
|*
\************************************************************************/

void Base3DPrinter::Print3DLine(B3dPrimitive* pPrimitive, ULONG nInd1,
	ULONG nInd2)
{
	B3dEntity& rEntity1 = aEntityBucket[nInd1];
	B3dEntity& rEntity2 = aEntityBucket[nInd2];

	// ColorModel fuer diese Punkte anwenden, falls Normale vorhanden
	// Danach Normale als ungueltig markieren, da nur noch die berechnete
	// Farbe bei Aufteilungen weiter interpoliert wird
	if(rEntity1.IsNormalUsed() && rEntity2.IsNormalUsed())
	{
		rEntity1.Color() = SolveColorModel(
			aMaterialBucket[pPrimitive->GetMaterialIndex()],
			rEntity1.Normal(), rEntity1.Point().GetVector3D());
		rEntity2.Color() = SolveColorModel(
			aMaterialBucket[pPrimitive->GetMaterialIndex()],
			rEntity2.Normal(), rEntity2.Point().GetVector3D());

		if(GetShadeModel() != Base3DPhong)
		{
			// Normalen deaktivieren und damit auf
			// Gouraud-shading zurueckgehen
			rEntity1.SetNormalUsed(FALSE);
			rEntity2.SetNormalUsed(FALSE);
		}
	}

	// Geometrie holen
	rEntity1.ToDeviceCoor(GetTransformationSet());
	rEntity2.ToDeviceCoor(GetTransformationSet());

	// MinimalLineLength einstellen
	if(rEntity1.Color() != rEntity2.Color())
	{
		// splitting notwendig, stelle die Parameter ein
		// Default: 3 Milimeter
		Size aSize(MINIMAL_SPLIT_SIZE, 0);
		aSize = GetOutputDevice()->LogicToLogic(
			aSize, MapMode(MAP_MM), GetOutputDevice()->GetMapMode());
		SetMinimalLineLength((double)aSize.Width());
	}
	else
	{
		// kein splitting
		SetMinimalLineLength(0.0);
	}

	// Ausgeben
	Print3DLine(pPrimitive, rEntity1, rEntity2);
}

double Base3DPrinter::CalcLengthOfLine(B3dEntity& rEntity1, B3dEntity& rEntity2)
{
	Vector3D aVec(rEntity2.Point().GetVector3D()
		- rEntity1.Point().GetVector3D());
	return aVec.GetLength();
}

void Base3DPrinter::Print3DLine(B3dPrimitive* pPrimitive, B3dEntity& rEntity1,
	B3dEntity& rEntity2)
{
	// Linie splitten?
	if(DoSplitLine(pPrimitive, rEntity1, rEntity2))
	{

		// Linie weiter aufsplitten
		B3dEntity aNew;

		rEntity1.ForceEqualBase(GetTransformationSet(), rEntity2);
		aNew.CalcMiddle(rEntity1, rEntity2);

		// ColorModel fuer neuen Punkt anwenden, falls Normale vorhanden
		if(aNew.IsNormalUsed())
		{
			// Geometrie nach ClipCoordinates
			aNew.To3DCoor(GetTransformationSet());
			aNew.Color() = SolveColorModel(
				aMaterialBucket[pPrimitive->GetMaterialIndex()],
				aNew.Normal(), aNew.Point().GetVector3D());
			// Geometrie nach DeviceCoordinates
			aNew.ToDeviceCoor(GetTransformationSet());
		}

		Print3DLine(pPrimitive, rEntity1, aNew);
		Print3DLine(pPrimitive, aNew, rEntity2);
	}
	else
	{
		// Farbe holen
		B3dColor aCol;
		aCol.CalcMiddle(rEntity1.Color(), rEntity2.Color());
		GetOutputDevice()->SetLineColor( aCol );

		// Zeichnen
		GetOutputDevice()->DrawLine(
			Point((long)(rEntity1.Point().X() + 0.5),
			(long)(rEntity1.Point().Y() + 0.5)),
			Point((long)(rEntity2.Point().X() + 0.5),
			(long)(rEntity2.Point().Y() + 0.5)));
	}
}

/*************************************************************************
|*
|* Testet, ob die Linie gesplittet werden soll
|*
\************************************************************************/

BOOL Base3DPrinter::DoSplitLine(B3dPrimitive* pPrimitive, B3dEntity& rEntity1,
	B3dEntity& rEntity2)
{
	BOOL bDoSplit = (fMinLineLength != 0.0
		&& nMinColorDistance != 0L
		&& CalcLengthOfLine(rEntity1, rEntity2) > fMinLineLength);

	if(bDoSplit)
	{
		// Farbabstand kleiner als nMinColorDistance?
		bDoSplit = rEntity1.Color().GetDistance(rEntity2.Color()) > nMinColorDistance;
	}
	return bDoSplit;
}

/*************************************************************************
|*
|* Gib ein Dreieck auf dem Drucker aus
|*
\************************************************************************/

void Base3DPrinter::Print3DTriangle(B3dPrimitive* pPrimitive, ULONG nInd1,
	ULONG nInd2, ULONG nInd3)
{
	// Alle Punkte in DeviceKoor umwandeln, somit wird auch
	// die zu berechnende Flaeche in DeviceKoordinaten angegeben.
	B3dEntity& rEntity1 = aEntityBucket[nInd1];
	B3dEntity& rEntity2 = aEntityBucket[nInd2];
	B3dEntity& rEntity3 = aEntityBucket[nInd3];

	// ColorModel fuer diese Punkte anwenden, falls Normale vorhanden
	// Danach Normale als ungueltig markieren, da nur noch die berechnete
	// Farbe bei Aufteilungen weiter interpoliert wird
	if(rEntity1.IsNormalUsed())
	{
		rEntity1.Color() = SolveColorModel(
			aMaterialBucket[pPrimitive->GetMaterialIndex()],
			rEntity1.Normal(), rEntity1.Point().GetVector3D());
		rEntity2.Color() = SolveColorModel(
			aMaterialBucket[pPrimitive->GetMaterialIndex()],
			rEntity2.Normal(), rEntity2.Point().GetVector3D());
		rEntity3.Color() = SolveColorModel(
			aMaterialBucket[pPrimitive->GetMaterialIndex()],
			rEntity3.Normal(), rEntity3.Point().GetVector3D());

		if(GetShadeModel() != Base3DPhong)
		{
			// Normalen deaktivieren und damit auf
			// Gouraud-shading zurueckgehen
			rEntity1.SetNormalUsed(FALSE);
			rEntity2.SetNormalUsed(FALSE);
			rEntity3.SetNormalUsed(FALSE);
		}
	}

	// Geometrie holen
	rEntity1.ToDeviceCoor(GetTransformationSet());
	rEntity2.ToDeviceCoor(GetTransformationSet());
	rEntity3.ToDeviceCoor(GetTransformationSet());

	// TriangleSize einstellen
	if(rEntity1.Color() != rEntity2.Color()
		|| rEntity2.Color() != rEntity3.Color()
		|| rEntity3.Color() != rEntity1.Color())
	{
		// splitting notwendig, stelle die Parameter ein
		// Default: 3 Quadratmilimeter
		Size aSize(MINIMAL_SPLIT_SIZE, MINIMAL_SPLIT_SIZE);
		aSize = GetOutputDevice()->LogicToLogic(
			aSize, MapMode(MAP_MM), GetOutputDevice()->GetMapMode());
		SetMinimalTriangleSize((double)aSize.Width());
	}
	else
	{
		// kein splitting
		SetMinimalTriangleSize(0.0);
	}

	// Ausgeben
	Print3DTriangle(pPrimitive, rEntity1, rEntity2, rEntity3);
}

double Base3DPrinter::CalcSizeOfTriangle(B3dEntity& rEntity1,
	B3dEntity& rEntity2, B3dEntity& rEntity3)
{
	Vector3D aVec1(rEntity2.Point().GetVector3D()
		- rEntity1.Point().GetVector3D());
	Vector3D aVec2(rEntity2.Point().GetVector3D()
		- rEntity3.Point().GetVector3D());
	double l1 = aVec1.GetLength();
	double l2 = aVec2.GetLength();
	return (l1 * l2) / 2.0;
}

void Base3DPrinter::Print3DTriangle(B3dPrimitive* pPrimitive, B3dEntity& rEntity1,
	B3dEntity& rEntity2, B3dEntity& rEntity3)
{
	// Dreieck splitten?
	if(DoSplitTriangle(pPrimitive, rEntity1, rEntity2, rEntity3))
	{
		// Dreieck weiter aufsplitten
		B3dEntity aNew1, aNew2, aNew3;

		rEntity1.ForceEqualBase(GetTransformationSet(), rEntity2, rEntity3);
		aNew1.CalcMiddle(rEntity1, rEntity2);
		aNew2.CalcMiddle(rEntity2, rEntity3);
		aNew3.CalcMiddle(rEntity3, rEntity1);

		// ColorModel fuer neue Punkte anwenden, falls Normale vorhanden
		if(aNew1.IsNormalUsed())
		{
			// Geometrie nach ClipCoordinates
			aNew1.To3DCoor(GetTransformationSet());
			aNew2.To3DCoor(GetTransformationSet());
			aNew3.To3DCoor(GetTransformationSet());

			aNew1.Color() = SolveColorModel(
				aMaterialBucket[pPrimitive->GetMaterialIndex()],
				aNew1.Normal(), aNew1.Point().GetVector3D());
			aNew2.Color() = SolveColorModel(
				aMaterialBucket[pPrimitive->GetMaterialIndex()],
				aNew2.Normal(), aNew2.Point().GetVector3D());
			aNew3.Color() = SolveColorModel(
				aMaterialBucket[pPrimitive->GetMaterialIndex()],
				aNew3.Normal(), aNew3.Point().GetVector3D());

			// Geometrie nach DeviceCoordinates zurueck
			aNew1.ToDeviceCoor(GetTransformationSet());
			aNew2.ToDeviceCoor(GetTransformationSet());
			aNew3.ToDeviceCoor(GetTransformationSet());
		}

		Print3DTriangle(pPrimitive, rEntity1, aNew1, aNew3);
		Print3DTriangle(pPrimitive, rEntity2, aNew2, aNew1);
		Print3DTriangle(pPrimitive, rEntity3, aNew3, aNew2);
		Print3DTriangle(pPrimitive, aNew1, aNew2, aNew3);
	}
	else
	{
		// Primitiv ausgeben
		Point aPointArray[3] = {
			Point((long)(rEntity1.Point().X() + 0.5),
				(long)(rEntity1.Point().Y() + 0.5)),
			Point((long)(rEntity2.Point().X() + 0.5),
				(long)(rEntity2.Point().Y() + 0.5)),
			Point((long)(rEntity3.Point().X() + 0.5),
			(long)(rEntity3.Point().Y() + 0.5))
		};
		Polygon aPolygon(3, aPointArray);

		// Farbe(n) holen
		GetOutputDevice()->SetLineColor();
		B3dColor aCol;
		aCol.CalcMiddle(rEntity1.Color(), rEntity2.Color(), rEntity3.Color());
		GetOutputDevice()->SetFillColor( aCol );

		// Zeichnen
		GetOutputDevice()->DrawPolygon(aPolygon);
	}
}

BOOL Base3DPrinter::DoSplitTriangle(B3dPrimitive* pPrimitive,
	B3dEntity& rEntity1, B3dEntity& rEntity2, B3dEntity& rEntity3)
{
	BOOL bDoSplit = (fMinTriangleSize != 0.0
		&& nMinColorDistance != 0L
		&& CalcSizeOfTriangle(rEntity1, rEntity2, rEntity3) > fMinTriangleSize);

	if(bDoSplit)
	{
		// Farbabstand kleiner als nMinColorDistance?
		bDoSplit =
			((rEntity1.Color().GetDistance(rEntity2.Color())
				> nMinColorDistance)
			|| (rEntity2.Color().GetDistance(rEntity3.Color())
				> nMinColorDistance)
			|| (rEntity3.Color().GetDistance(rEntity1.Color())
				> nMinColorDistance));
	}
	return bDoSplit;
}

/*************************************************************************
|*
|* Primitivenfunktionen, alle Objekte sind geclippt
|* Die Primitive werden zwischengespeichert und dann in den BSPTree
|* einsortiert. Die Ausgabe erflogt bei EndScene().
|*
\************************************************************************/

void Base3DPrinter::Clipped3DPoint(UINT32 nInd)
{
	// Hole referentzierten Punkt
	B3dEntity& rEntity = aBuffers[nInd];

	// Erzeuge Primitiv
	NewPointPrimitive(rEntity);
}

void Base3DPrinter::Clipped3DLine(UINT32 nInd1, UINT32 nInd2)
{
	// Hole alle referentzierten Punkte
	B3dEntity& rEntity1 = aBuffers[nInd1];
	B3dEntity& rEntity2 = aBuffers[nInd2];

	// Erzeuge Primitiv
	NewLinePrimitive(rEntity1, rEntity2);
}

void Base3DPrinter::Clipped3DTriangle(UINT32 nInd1, UINT32 nInd2, UINT32 nInd3)
{
	// Hole alle referentzierten Punkte
	B3dEntity& rEntity1 = aBuffers[nInd1];
	B3dEntity& rEntity2 = aBuffers[nInd2];
	B3dEntity& rEntity3 = aBuffers[nInd3];

	// Erzeuge Primitiv
	NewTrianglePrimitive(rEntity1, rEntity2, rEntity3);
}

/*************************************************************************
|*
|* Sortiere das Primitiv in den BSPTree ein. Bei Punkten unproblematisch
|*
\************************************************************************/

void Base3DPrinter::AddPointToBSPTree(B3dPrimitive* pCurrent,
	B3dPrimitive* pNew)
{
	if(aLocalVarBucket.Count() <= nLocalVarIndex)
		aLocalVarBucket.Append();
	Base3DBSPLocal& aVar = aLocalVarBucket[nLocalVarIndex++];

	if(pCurrent)
	{
		// Schleife zum Durchlaufen des Baumes ohne Rekursion
		aVar.bSamePlane = FALSE;
		while(!aVar.bSamePlane)
		{
			if(pCurrent->GetType() == B3dPrimitiveTriangle)
			{
				aVar.fScalar = pCurrent->Scalar(aEntityBucket);

				// Seite feststellen
				aVar.fZwi = aVar.fScalar +
					(aEntityBucket[pNew->GetIndex()].Point().GetVector3D().Scalar(pCurrent->Normal()));

				aVar.bSide1 = (aVar.fZwi > 0.0) ? TRUE : FALSE;
				if(fabs(aVar.fZwi) > SMALL_DVALUE)
					aVar.bSamePlane = FALSE;
				else
					aVar.bSamePlane = TRUE;

				if(!aVar.bSamePlane)
				{
					if(aVar.bSide1)
					{
						// komplett links
						if(pCurrent->GetLeft())
						{
							pCurrent = pCurrent->GetLeft();
						}
						else
						{
							pCurrent->SetLeft(pNew);
							nLocalVarIndex--;
							return;
						}
					}
					else
					{
						// komplett rechts
						if(pCurrent->GetRight())
						{
							pCurrent = pCurrent->GetRight();
						}
						else
						{
							pCurrent->SetRight(pNew);
							nLocalVarIndex--;
							return;
						}
					}
				}
			}
			else
			{
				// Schleife verlassen, falls kein Dreieck
				aVar.bSamePlane = TRUE;
			}
		}

		// Linearer Teil
		if(pCurrent->GetType() == B3dPrimitiveTriangle)
		{
			if(aVar.bSamePlane)
			{
				// Selbe Ebene wie pCurrent, vorne einhaengen
				if(pCurrent->GetSame())
				{
					pNew->SetSame(pCurrent->GetSame());
				}
				pCurrent->SetSame(pNew);
			}
		}
		else
		{
			// pCurrent ist selbst keine Ebene mehr, den Punkt
			// nur noch hier eintragen
			if(pCurrent->GetLeft())
			{
				if(pCurrent->GetRight())
				{
					// Hinten anhaengen
					// damit Punkte als letztes gezeichnet werden
					while(pCurrent->GetSame())
						pCurrent = pCurrent->GetSame();
					pCurrent->SetSame(pNew);
				}
				else
				{
					pCurrent->SetRight(pNew);
				}
			}
			else
			{
				pCurrent->SetLeft(pNew);
			}
		}
	}
	else
	{
		// No root yet, set current primitive as root
		pBSPTreeRoot = pNew;
	}
	nLocalVarIndex--;
}

/*************************************************************************
|*
|* Sortiere das Primitiv in den BSPTree ein. Dabei kann es zu Schnitten
|* der Linie und damit zu neuen Primitiven kommen.
|*
\************************************************************************/

void Base3DPrinter::AddLineToBSPTree(B3dPrimitive* pCurrent, B3dPrimitive* pNew)
{
	if(aLocalVarBucket.Count() <= nLocalVarIndex)
		aLocalVarBucket.Append();
	Base3DBSPLocal& aVar = aLocalVarBucket[nLocalVarIndex++];

	if(pCurrent)
	{
		// Schleife zum Durchlaufen des Baumes ohne Rekursion
		aVar.bSameSide = TRUE;
		while(aVar.bSameSide)
		{
			if(pCurrent->GetType() == B3dPrimitiveTriangle)
			{
				aVar.fScalar = pCurrent->Scalar(aEntityBucket);

				// Seite feststellen
				aVar.nIndex = pNew->GetIndex();
				aVar.fZwi = aVar.fScalar +
					(aEntityBucket[aVar.nIndex++].Point().GetVector3D().Scalar(pCurrent->Normal()));

				aVar.bSameSide = TRUE;
				aVar.bSamePlane = TRUE;
				if(fabs(aVar.fZwi) > SMALL_DVALUE)
				{
					aVar.bSamePlane = FALSE;
					aVar.bSide = aVar.bSide1 = (aVar.fZwi > 0.0) ? TRUE : FALSE;
				}

				aVar.fZwi = aVar.fScalar +
					(aEntityBucket[aVar.nIndex].Point().GetVector3D().Scalar(pCurrent->Normal()));

				if(fabs(aVar.fZwi) > SMALL_DVALUE)
				{
					aVar.bSide2 = (aVar.fZwi > 0.0) ? TRUE : FALSE;
					if(aVar.bSamePlane)
					{
						aVar.bSide = aVar.bSide2;
						aVar.bSamePlane = FALSE;
					}
					else
					{
						if(aVar.bSide2 != aVar.bSide1)
							aVar.bSameSide = FALSE;
					}
				}

				if(aVar.bSamePlane)
				{
					// Selbe Ebene wie pCurrent, hinten anhaengen
					// damit Linien als letztes gezeichnet werden
					while(pCurrent->GetSame())
						pCurrent = pCurrent->GetSame();
					pCurrent->SetSame(pNew);
					nLocalVarIndex--;
					return;
				}
				else if(aVar.bSameSide)
				{
					if(aVar.bSide)
					{
						// komplett links
						if(pCurrent->GetLeft())
						{
							pCurrent = pCurrent->GetLeft();
						}
						else
						{
							pCurrent->SetLeft(pNew);
							nLocalVarIndex--;
							return;
						}
					}
					else
					{
						// komplett rechts
						if(pCurrent->GetRight())
						{
							pCurrent = pCurrent->GetRight();
						}
						else
						{
							pCurrent->SetRight(pNew);
							nLocalVarIndex--;
							return;
						}
					}
				}
			}
			else
			{
				// Schleife verlassen, falls kein Dreieck
				aVar.bSameSide = FALSE;
			}
		}

		// Linearer Teil
		if(pCurrent->GetType() == B3dPrimitiveTriangle)
		{
			// auf beiden Seiten, schneide das neue Primitiv an der Ebene
			// pCurrent und fuege die neuen Primitive rechts und links ein
			aVar.nIndex = pNew->GetIndex();
			B3dEntity& rEnt1 = aEntityBucket[aVar.nIndex++];
			B3dEntity& rEnt2 = aEntityBucket[aVar.nIndex];
			aVar.aVec1 = Vector3D(rEnt2.Point().GetVector3D()
				- rEnt1.Point().GetVector3D());

			// Schnittparameter Linie mit Ebene ( [0.0 .. 1.0[ )
			aVar.fCut = pCurrent->Normal().Scalar(aVar.aVec1);
			if(fabs(aVar.fCut) > SMALL_DVALUE)
			{
				aVar.fCut = -((aVar.fScalar + (pCurrent->Normal().Scalar(rEnt1.Point().GetVector3D()))) / aVar.fCut);

				if(aVar.fCut > SMALL_DVALUE
					&& aVar.fCut < 1.0 - SMALL_DVALUE)
				{
					// Schnitt ist innerhalb des Parameterbereiches der
					// Linie. Neuen Punkt anlegen
					B3dEntity aNew1;
					rEnt1.ForceEqualBase(GetTransformationSet(), rEnt2);
					aNew1.CalcInBetween(rEnt1, rEnt2, aVar.fCut);

					aVar.nIndex = NewLinePrimitive(rEnt1,
						aNew1, pNew->GetMaterialIndex());
					AddPartialLine(pCurrent, aVar.bSide1,
						&aPrimitiveBucket[aVar.nIndex]);
					aVar.nIndex = NewLinePrimitive(aNew1,
						rEnt2, pNew->GetMaterialIndex());
					AddPartialLine(pCurrent, aVar.bSide2,
						&aPrimitiveBucket[aVar.nIndex]);
				}
			}
		}
		else
		{
			// pCurrent ist selbst keine Ebene mehr, die Linie
			// nur noch hier eintragen
			if(pCurrent->GetLeft())
			{
				if(pCurrent->GetRight())
				{
					// Hinten anhaengen
					// damit Linien als letztes gezeichnet werden
					while(pCurrent->GetSame())
						pCurrent = pCurrent->GetSame();
					pCurrent->SetSame(pNew);
				}
				else
				{
					pCurrent->SetRight(pNew);
				}
			}
			else
			{
				pCurrent->SetLeft(pNew);
			}
		}
	}
	else
	{
		// No root yet, set current primitive as root
		pBSPTreeRoot = pNew;
	}
	nLocalVarIndex--;
}

/*************************************************************************
|*
|* Sortiere eine aus dem Schneiden stammende Teillinie ein
|*
\************************************************************************/

void Base3DPrinter::AddPartialLine(B3dPrimitive* pCurrent, BOOL bSide,
	B3dPrimitive* pPart)
{
	if(bSide)
	{
		if(pCurrent->GetLeft())
		{
			AddLineToBSPTree(pCurrent->GetLeft(), pPart);
		}
		else
		{
			pCurrent->SetLeft(pPart);
		}
	}
	else
	{
		if(pCurrent->GetRight())
		{
			AddLineToBSPTree(pCurrent->GetRight(), pPart);
		}
		else
		{
			pCurrent->SetRight(pPart);
		}
	}
}

/*************************************************************************
|*
|* Sortiere das Primitiv in den BSPTree ein. Dabei kann es zu Schnitten
|* zwischen den Primitiven und damit zu neuen Primitiven kommen.
|*
\************************************************************************/

void Base3DPrinter::AddTriangleToBSPTree(B3dPrimitive* pCurrent,
	B3dPrimitive* pNew)
{
	if(aLocalVarBucket.Count() <= nLocalVarIndex)
		aLocalVarBucket.Append();
	Base3DBSPLocal& aVar = aLocalVarBucket[nLocalVarIndex++];

	if(pCurrent)
	{
		// Schleife zum Durchlaufen des Baumes ohne Rekursion
		aVar.bSameSide = TRUE;
		while(aVar.bSameSide)
		{
			aVar.fScalar = pCurrent->Scalar(aEntityBucket);
			aVar.nIndex = pNew->GetIndex();
			aVar.bSamePlane = TRUE;

			// Erster Punkt
			aVar.fZwi = aVar.fScalar +
				(aEntityBucket[aVar.nIndex++].Point().GetVector3D().Scalar(pCurrent->Normal()));

			if(fabs(aVar.fZwi) > SMALL_DVALUE)
			{
				aVar.bSide = aVar.bSide1 = (aVar.fZwi > 0.0) ? TRUE : FALSE;
				aVar.bSamePlane = FALSE;
			}

			// Zweiter Punkt
			aVar.fZwi = aVar.fScalar +
				(aEntityBucket[aVar.nIndex++].Point().GetVector3D().Scalar(pCurrent->Normal()));

			if(fabs(aVar.fZwi) > SMALL_DVALUE)
			{
				aVar.bSide2 = (aVar.fZwi > 0.0) ? TRUE : FALSE;
				if(!aVar.bSamePlane)
				{
					// erster Punkt lag bereits nicht auf der Ebene
					if(aVar.bSide2 != aVar.bSide1)
						aVar.bSameSide = FALSE;
				}
				else
				{
					aVar.bSamePlane = FALSE;
					// in bSide auch die aktuelle Seite merken
					aVar.bSide = aVar.bSide2;
				}
			}

			// Dritter Punkt
			aVar.fZwi = aVar.fScalar +
				(aEntityBucket[aVar.nIndex].Point().GetVector3D().Scalar(pCurrent->Normal()));

			if(fabs(aVar.fZwi) > SMALL_DVALUE)
			{
				aVar.bSide3 = (aVar.fZwi > 0.0) ? TRUE : FALSE;
				if(!aVar.bSamePlane)
				{
					// erster od. zweiter Punkt od. beide lagen
					// nicht auf der Ebene
					// Weiterer Vergleich ist nur interessant, wenn
					// die vorherigen Punkte auf derselben Seite lagen
					if(aVar.bSameSide)
					{
						if(aVar.bSide3 != aVar.bSide)
							aVar.bSameSide = FALSE;
					}
				}
				else
				{
					aVar.bSamePlane = FALSE;
					// in bSide auch die aktuelle Seite merken
					aVar.bSide = aVar.bSide3;
				}
			}

			if(aVar.bSamePlane)
			{
				// Selbe Ebene wie pCurrent, vorne einhaengen
				if(pCurrent->GetSame())
				{
					pNew->SetSame(pCurrent->GetSame());
				}
				pCurrent->SetSame(pNew);
				nLocalVarIndex--;
				return;
			}
			else if(aVar.bSameSide)
			{
				// komplett auf einer Seite
				if(aVar.bSide)
				{
					// komplett links
					if(pCurrent->GetLeft())
					{
						pCurrent = pCurrent->GetLeft();
					}
					else
					{
						pCurrent->SetLeft(pNew);
						nLocalVarIndex--;
						return;
					}
				}
				else
				{
					// komplett rechts
					if(pCurrent->GetRight())
					{
						pCurrent = pCurrent->GetRight();
					}
					else
					{
						pCurrent->SetRight(pNew);
						nLocalVarIndex--;
						return;
					}
				}
			}
		}

		// Linearer Teil
		// Nicht mehr auf derselben Seite
		// Schneide das neue Primitiv an der Ebene
		// pCurrent und fuege die neuen Primitive rechts und links ein
		aVar.nIndex = pNew->GetIndex();
		B3dEntity& rEnt1 = aEntityBucket[aVar.nIndex++];
		B3dEntity& rEnt2 = aEntityBucket[aVar.nIndex++];
		B3dEntity& rEnt3 = aEntityBucket[aVar.nIndex];

		aVar.aVec1 = Vector3D(rEnt2.Point().GetVector3D()
			- rEnt1.Point().GetVector3D());
		aVar.aVec2 = Vector3D(rEnt3.Point().GetVector3D()
			- rEnt2.Point().GetVector3D());
		aVar.aVec3 = Vector3D(rEnt1.Point().GetVector3D()
			- rEnt3.Point().GetVector3D());
		aVar.bCut1 = aVar.bCut2 = aVar.bCut3 = FALSE;
		aVar.nNumCuts = 0;

		// Schnittparameter 1.Linie mit Ebene ( [0.0 .. 1.0[ )
		aVar.fCut = pCurrent->Normal().Scalar(aVar.aVec1);
		if(fabs(aVar.fCut) > SMALL_DVALUE)
		{
			aVar.fCut1 = -((aVar.fScalar + (pCurrent->Normal().Scalar(rEnt1.Point().GetVector3D()))) / aVar.fCut);

			if(aVar.fCut1 > -SMALL_DVALUE
				&& aVar.fCut1 < 1.0 - SMALL_DVALUE)
			{
				aVar.bCut1 = TRUE;
				aVar.nNumCuts++;
			}
		}

		// Schnittparameter 2.Linie mit Ebene ( [0.0 .. 1.0[ )
		aVar.fCut = pCurrent->Normal().Scalar(aVar.aVec2);
		if(fabs(aVar.fCut) > SMALL_DVALUE)
		{
			aVar.fCut2 = -((aVar.fScalar + (pCurrent->Normal().Scalar(rEnt2.Point().GetVector3D()))) / aVar.fCut);

			if(aVar.fCut2 > -SMALL_DVALUE
				&& aVar.fCut2 < 1.0 - SMALL_DVALUE)
			{
				aVar.bCut2 = TRUE;
				aVar.nNumCuts++;
			}
		}

		// Schnittparameter 3.Linie mit Ebene ( [0.0 .. 1.0[ )
		aVar.fCut = pCurrent->Normal().Scalar(aVar.aVec3);
		if(fabs(aVar.fCut) > SMALL_DVALUE)
		{
			aVar.fCut3 = -((aVar.fScalar + (pCurrent->Normal().Scalar(rEnt3.Point().GetVector3D()))) / aVar.fCut);

			if(aVar.fCut3 > -SMALL_DVALUE
				&& aVar.fCut3 < 1.0 - SMALL_DVALUE)
			{
				aVar.bCut3 = TRUE;
				aVar.nNumCuts++;
			}
		}

		if(aVar.nNumCuts == 2)
		{
			if(aVar.bCut1)
			{
				// Neuen Punkt 1 ausfuellen
				B3dEntity aNew1;
				rEnt1.ForceEqualBase(GetTransformationSet(), rEnt2);
				aNew1.CalcInBetween(rEnt1, rEnt2, aVar.fCut1);

				if(aVar.bCut2) {
					// Neuen Punkt 2 ausfuellen
					B3dEntity aNew2;
					rEnt2.ForceEqualBase(GetTransformationSet(), rEnt3);
					aNew2.CalcInBetween(rEnt2, rEnt3, aVar.fCut2);

					// Durch Linien 1,2
					if(aVar.fCut1 < SMALL_DVALUE)
					{
						// Durch Punkt1, Linie2
						aVar.nIndex = NewTrianglePrimitive(rEnt1,
							rEnt2, aNew2, pNew->GetMaterialIndex());
						AddPartialTriangle(pCurrent, aVar.bSide2,
							&aPrimitiveBucket[aVar.nIndex]);
						aVar.nIndex = NewTrianglePrimitive(rEnt1,
							aNew2, rEnt3, pNew->GetMaterialIndex());
						AddPartialTriangle(pCurrent, aVar.bSide3,
							&aPrimitiveBucket[aVar.nIndex]);
					}
					else
					{
						// Durch beide Linien
						aVar.nIndex = NewTrianglePrimitive(aNew1,
							rEnt2, aNew2, pNew->GetMaterialIndex());
						AddPartialTriangle(pCurrent, aVar.bSide2,
							&aPrimitiveBucket[aVar.nIndex]);
						aVar.nIndex = NewQuadPrimitive(aNew2, rEnt3,
							rEnt1, aNew1, pNew->GetMaterialIndex());
						AddPartialTriangle(pCurrent, aVar.bSide3,
							&aPrimitiveBucket[aVar.nIndex++]);
						AddPartialTriangle(pCurrent, aVar.bSide3,
							&aPrimitiveBucket[aVar.nIndex]);
					}
				}
				else
				{
					// Neuen Punkt 3 ausfuellen
					B3dEntity aNew3;
					rEnt3.ForceEqualBase(GetTransformationSet(), rEnt1);
					aNew3.CalcInBetween(rEnt3, rEnt1, aVar.fCut3);

					// Durch Linien 1,3
					if(aVar.fCut3 < SMALL_DVALUE)
					{
						// Durch Punkt3, Linie1
						aVar.nIndex = NewTrianglePrimitive(rEnt3,
							rEnt1, aNew1, pNew->GetMaterialIndex());
						AddPartialTriangle(pCurrent, aVar.bSide1,
							&aPrimitiveBucket[aVar.nIndex]);
						aVar.nIndex = NewTrianglePrimitive(rEnt3,
							aNew1, rEnt2, pNew->GetMaterialIndex());
						AddPartialTriangle(pCurrent, aVar.bSide2,
							&aPrimitiveBucket[aVar.nIndex]);
					}
					else
					{
						// Durch beide Linien
						aVar.nIndex = NewTrianglePrimitive(aNew3,
							rEnt1, aNew1, pNew->GetMaterialIndex());
						AddPartialTriangle(pCurrent, aVar.bSide1,
							&aPrimitiveBucket[aVar.nIndex]);
						aVar.nIndex = NewQuadPrimitive(aNew1, rEnt2,
							rEnt3, aNew3, pNew->GetMaterialIndex());
						AddPartialTriangle(pCurrent, aVar.bSide2,
							&aPrimitiveBucket[aVar.nIndex++]);
						AddPartialTriangle(pCurrent, aVar.bSide2,
							&aPrimitiveBucket[aVar.nIndex]);
					}
				}
			}
			else
			{
				// Neue Punkte 2,3 ausfuellen
				rEnt1.ForceEqualBase(GetTransformationSet(), rEnt2, rEnt3);
				B3dEntity aNew2;
				aNew2.CalcInBetween(rEnt2, rEnt3, aVar.fCut2);
				B3dEntity aNew3;
				aNew3.CalcInBetween(rEnt3, rEnt1, aVar.fCut3);

				// Durch Linien 2,3
				if(aVar.fCut2 < SMALL_DVALUE)
				{
					// Durch Punkt2, Linie3
					aVar.nIndex = NewTrianglePrimitive(rEnt2,
						rEnt3, aNew3, pNew->GetMaterialIndex());
					AddPartialTriangle(pCurrent, aVar.bSide3,
						&aPrimitiveBucket[aVar.nIndex]);
					aVar.nIndex = NewTrianglePrimitive(rEnt2,
						aNew3, rEnt1, pNew->GetMaterialIndex());
					AddPartialTriangle(pCurrent, aVar.bSide1,
						&aPrimitiveBucket[aVar.nIndex]);
				}
				else
				{
					// Durch beide Linien
					aVar.nIndex = NewTrianglePrimitive(aNew2,
						rEnt3, aNew3, pNew->GetMaterialIndex());
					AddPartialTriangle(pCurrent, aVar.bSide3,
						&aPrimitiveBucket[aVar.nIndex]);
					aVar.nIndex = NewQuadPrimitive(aNew3, rEnt1,
						rEnt2, aNew2, pNew->GetMaterialIndex());
					AddPartialTriangle(pCurrent, aVar.bSide1,
						&aPrimitiveBucket[aVar.nIndex++]);
					AddPartialTriangle(pCurrent, aVar.bSide1,
						&aPrimitiveBucket[aVar.nIndex]);
				}
			}
			// Altes Primitiv ist vollstaendig ersetzt durch neue
			// und wird NICHT in den BSPTree eingefuegt.
			pNew->SetVisible(FALSE);
		}
		else
		{
			// Nicht auf einer Seite der Ebene, aber != 2 Schnitte ?!?
			// Dies sollte nicht vorkommen
			if(aVar.nNumCuts != 0)
			{
				DBG_ERROR(
					"Triangle on one side in BSPTree, but cuts != 2");
			}
		}
	}
	else
	{
		// No root yet, set current primitive as root
		pBSPTreeRoot = pNew;
	}
	nLocalVarIndex--;
}

/*************************************************************************
|*
|* Sortiere ein aus dem Schneiden stammendes Teildreieck ein
|*
\************************************************************************/

void Base3DPrinter::AddPartialTriangle(B3dPrimitive* pCurrent, BOOL bSide,
	B3dPrimitive* pPart)
{
	if(bSide)
	{
		if(pCurrent->GetLeft())
		{
			AddTriangleToBSPTree(pCurrent->GetLeft(), pPart);
		}
		else
		{
			pCurrent->SetLeft(pPart);
		}
	}
	else
	{
		if(pCurrent->GetRight())
		{
			AddTriangleToBSPTree(pCurrent->GetRight(), pPart);
		}
		else
		{
			pCurrent->SetRight(pPart);
		}
	}
}

/*************************************************************************
|*
|* Lege neues Primitiv vom Typ Punkt an
|*
\************************************************************************/

ULONG Base3DPrinter::NewPointPrimitive(B3dEntity& rEnt1, ULONG nMatInd)
{
	ULONG nRetval = aPrimitiveBucket.Count();
	aPrimitiveBucket.Append();
	B3dPrimitive& rNew = aPrimitiveBucket[nRetval];

	rNew.Reset();
	rNew.SetType(B3dPrimitivePoint);
	rNew.SetIndex(aEntityBucket.Count());

	if(rEnt1.IsNormalUsed())
	{
		if(nMatInd == NO_MATERIAL_INDEX)
		{
			// Material muss gemerkt werden
			// Bei Linien immer das Vordergrundmaterial benutzen
			rNew.SetMaterialIndex(GetMaterialIndex(Base3DMaterialFront));
		}
		else
		{
			// Bereits vorhandenen MaterialIndex benutzen
			rNew.SetMaterialIndex(nMatInd);
		}
		// Vektor normalisieren
		rEnt1.Normal().Normalize();
	}

	aEntityBucket.Append(rEnt1);

	return nRetval;
}

/*************************************************************************
|*
|* Lege neues Primitiv vom Typ Linie an
|*
\************************************************************************/

ULONG Base3DPrinter::NewLinePrimitive(B3dEntity& rEnt1, B3dEntity& rEnt2,
	ULONG nMatInd)
{
	ULONG nRetval = aPrimitiveBucket.Count();
	aPrimitiveBucket.Append();
	B3dPrimitive& rNew = aPrimitiveBucket[nRetval];

	rNew.Reset();
	rNew.SetType(B3dPrimitiveLine);
	rNew.SetIndex(aEntityBucket.Count());

	if(rEnt1.IsNormalUsed())
	{
		if(nMatInd == NO_MATERIAL_INDEX)
		{
			// Material muss gemerkt werden
			// Bei Linien immer das Vordergrundmaterial benutzen
			rNew.SetMaterialIndex(GetMaterialIndex(Base3DMaterialFront));
		}
		else
		{
			// Bereits vorhandenen MaterialIndex benutzen
			rNew.SetMaterialIndex(nMatInd);
		}
		// Vektoren normalisieren
		rEnt1.Normal().Normalize();
		rEnt2.Normal().Normalize();
	}

	aEntityBucket.Append(rEnt1);
	aEntityBucket.Append(rEnt2);

	return nRetval;
}

/*************************************************************************
|*
|* Lege neues Primitiv vom Typ Dreieck an
|*
\************************************************************************/

ULONG Base3DPrinter::NewTrianglePrimitive(B3dEntity& rEnt1,
	B3dEntity& rEnt2, B3dEntity& rEnt3, ULONG nMatInd)
{
	ULONG nRetval = aPrimitiveBucket.Count();
	aPrimitiveBucket.Append();
	B3dPrimitive& rNew = aPrimitiveBucket[nRetval];

	rNew.Reset();
	rNew.SetType(B3dPrimitiveTriangle);
	rNew.SetIndex(aEntityBucket.Count());

	if(rEnt1.IsNormalUsed())
	{
		// Vektoren normalisieren
		rEnt1.Normal().Normalize();
		rEnt2.Normal().Normalize();
		rEnt3.Normal().Normalize();
	}

	aEntityBucket.Append(rEnt1);
	aEntityBucket.Append(rEnt2);
	aEntityBucket.Append(rEnt3);

	// Normale vorberechnen, wird ab jetzt haeufig gebraucht
	rNew.Normal() = rEnt1.PlaneNormal();

	// Falls OHNE BackfaceCulling gerendert wird, muessen die geometrischen
	// Normalen alle in die gleiche Z-Richtung zeigen, um die Sortierung
	// im BSPTree zu garantieren.
	if(rNew.Normal().Z() < 0.0)
		rNew.Normal() = -rNew.Normal();

	if(rEnt1.IsNormalUsed())
	{
		if(nMatInd == NO_MATERIAL_INDEX)
		{
			// Material muss gemerkt werden
			ULONG nNewIndex = GetMaterialIndex(Base3DMaterialFront);
			if(rNew.Normal().Z() < 0.0 && (GetLightGroup() && GetLightGroup()->GetModelTwoSide()))
				nNewIndex = GetMaterialIndex(Base3DMaterialBack);
			rNew.SetMaterialIndex(nNewIndex);
		}
		else
		{
			// Bereits vorhandenen MaterialIndex benutzen
			rNew.SetMaterialIndex(nMatInd);
		}
	}
	return nRetval;
}

/*************************************************************************
|*
|* Ueberladene Materialfunktionen, um auf Materialaenderungen reagieren
|* zu koennen
|*
\************************************************************************/

void Base3DPrinter::SetMaterial(Color rNew, Base3DMaterialValue eVal,
	Base3DMaterialMode eMode)
{
	if(GetMaterial(eVal, eMode) != rNew)
	{
		if(eMode == Base3DMaterialFrontAndBack
			|| eMode == Base3DMaterialFront)
			bFrontMaterialChanged = TRUE;
		if(eMode == Base3DMaterialFrontAndBack
			|| eMode == Base3DMaterialBack)
			bBackMaterialChanged = TRUE;

		// call parent
		Base3D::SetMaterial(rNew, eVal, eMode);
	}
}

void Base3DPrinter::SetShininess(UINT16 nExponent,
	Base3DMaterialMode eMode)
{
	if(GetShininess(eMode) != nExponent)
	{
		if(eMode == Base3DMaterialFrontAndBack
			|| eMode == Base3DMaterialFront)
			bFrontMaterialChanged = TRUE;
		if(eMode == Base3DMaterialFrontAndBack
			|| eMode == Base3DMaterialBack)
			bBackMaterialChanged = TRUE;

		// call parent
		Base3D::SetShininess(nExponent, eMode);
	}
}

/*************************************************************************
|*
|* Gebe den richtigen Index auf das Material zurueck
|*
\************************************************************************/

ULONG Base3DPrinter::GetMaterialIndex(Base3DMaterialMode eMode)
{
	if(eMode == Base3DMaterialFront
		|| eMode == Base3DMaterialFrontAndBack)
	{
		if(bFrontMaterialChanged)
		{
			nMaterialFrontIndex = aMaterialBucket.Count();
			aMaterialBucket.Append(GetMaterialObject(eMode));
			bFrontMaterialChanged = FALSE;
		}
		return nMaterialFrontIndex;
	}
	else
	{
		if(bBackMaterialChanged)
		{
			nMaterialBackIndex = aMaterialBucket.Count();
			aMaterialBucket.Append(GetMaterialObject(eMode));
			bBackMaterialChanged = FALSE;
		}
		return nMaterialBackIndex;
	}
}

/*************************************************************************
|*
|* Lege zwei neue Primitive vom Typ Dreieck an, waehle als Trennlinie
|* die kuerzeste Entfernung
|*
\************************************************************************/

ULONG Base3DPrinter::NewQuadPrimitive(B3dEntity& rEnt1, B3dEntity& rEnt2,
	B3dEntity& rEnt3, B3dEntity& rEnt4, ULONG nMatInd)
{
	ULONG nRetval;

	Vector3D aVec31(rEnt3.Point().GetVector3D()
		- rEnt1.Point().GetVector3D());
	Vector3D aVec42(rEnt4.Point().GetVector3D()
		- rEnt2.Point().GetVector3D());

	if(aVec31.GetLength() < aVec42.GetLength())
	{
		nRetval = NewTrianglePrimitive(rEnt1, rEnt2, rEnt3, nMatInd);
		NewTrianglePrimitive(rEnt3, rEnt4, rEnt1, nMatInd);
	}
	else
	{
		nRetval = NewTrianglePrimitive(rEnt2, rEnt3, rEnt4, nMatInd);
		NewTrianglePrimitive(rEnt4, rEnt1, rEnt2, nMatInd);
	}
	return nRetval;
}

/*************************************************************************
|*
|* Berechne Schnitt zwischen den in einer Ebene liegenden Linien
|* rPnt1,rVec1 und rPnt2,rvec2. Der Rueckgabewert ist TRUE, wenn der
|* Schnitt im Bereich [0.0 .. 1.0[ im parametrisierten Bereich der
|* ersten Linie liegt. In rFac1 wird der Schnittfaktor im Bezug auf die
|* erste, in rFac2 in Bezug auf die zweite
|* Linie zurueckgegeben. Bei return FALSE sind diese Werte
|* undefiniert!
|*
\************************************************************************/

BOOL Base3DPrinter::GetCutFactor(double& rFac1,double& rFac2,
	const Vector3D& rPnt1, const Vector3D& rVec1, const Vector3D& rPnt2,
	const Vector3D& rVec2)
{
	// Berechne Schnittfaktor fuer 1.Linie
	rFac1 = (rVec1.Y() * rVec2.X()) - (rVec1.X() * rVec2.Y());
	if(fabs(rFac1) < SMALL_DVALUE)
	{
		// Schnitt in XY-Ebene ist parallel, weiche auf YZ-Ebene aus
		rFac1 = (rVec1.Z() * rVec2.Y()) - (rVec1.Y() * rVec2.Z());
		if(fabs(rFac1) > SMALL_DVALUE)
		{
			// Berechne Schnittfaktor fuer Dreieckskante in YZ-Ebene
			rFac1 = (rVec2.Y() * (rPnt2.Z() - rPnt1.Z())
				+ rVec2.Z() * (rPnt1.Y() - rPnt2.Y())) / rFac1;
		}
		else
		{
			// Schnitt in XZ-Ebene ebenfalls parallel, keine
			// Loesung moeglich
			return FALSE;
		}
	}
	else
	{
		// Berechne Schnittfaktor fuer Dreieckskante in XY-Ebene
		if(rFac1 != 0.0)
			rFac1 = (rVec2.X() * (rPnt2.Y() - rPnt1.Y())
				+ rVec2.Y() * (rPnt1.X() - rPnt2.X())) / rFac1;
	}
	if(rFac1 > -SMALL_DVALUE && rFac1 - 1.0 < -SMALL_DVALUE)
	{
		// Es gibt einen vernuenftigen Schnitt innerhalb des
		// Wertebereichs der Dreieckskante [0.0 .. 1.0[, berechne
		// Schnittfaktor fuer 2.Linie (Schnittlinie) als Rueckgabewert
		if(rVec2.X() > rVec2.Y() && rVec2.X() > rVec2.Z())
		{
			// X
			if(rVec2.X() != 0.0)
				rFac2 = (rPnt1.X() + rFac1 * rVec1.X() - rPnt2.X())
					/ rVec2.X();
		}
		else
		{
			if(rVec2.Y() > rVec2.Z())
			{
				// Y
				if(rVec2.Y() != 0.0)
					rFac2 = (rPnt1.Y() + rFac1 * rVec1.Y() - rPnt2.Y())
						/ rVec2.Y();
			}
			else
			{
				// Z
				if(rVec2.Z() != 0.0)
					rFac2 = (rPnt1.Z() + rFac1 * rVec1.Z() - rPnt2.Z())
						/ rVec2.Z();
			}
		}
		return TRUE;
	}
	return FALSE;
}

/*************************************************************************
|*
|* Leeren aller Zwischenspeicher, Vorbereitung auf eine neue Szene
|*
\************************************************************************/

void Base3DPrinter::EmptyBuckets()
{
	// Punkte wegschmeissen
	aEntityBucket.Erase();

	// Primitive wegschmeissen
	aPrimitiveBucket.Erase();

	// Materialien wegschmeissen und aktuelle holen
	aMaterialBucket.Erase();
	nMaterialFrontIndex = 0;
	bFrontMaterialChanged = FALSE;
	aMaterialBucket.Append(GetMaterialObject(Base3DMaterialFront));
	nMaterialBackIndex = 1;
	bBackMaterialChanged = FALSE;
	aMaterialBucket.Append(GetMaterialObject(Base3DMaterialBack));

	// LocalVars wegschmeissen und Speicher freigeben
	aLocalVarBucket.Empty();

	// BSPTree zuruecksetzen
	pBSPTreeRoot = NULL;
}


