/*************************************************************************
 *
 *  $RCSfile: rmtcont.cxx,v $
 *
 *  $Revision: 1.13 $
 *
 *  last change: $Author: sb $ $Date: 2001/07/27 13:52:39 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _UCPRMT_RMTCONT_HXX_
#include <rmtcont.hxx>
#endif

#include <algorithm>
#include <list>
#include <memory>
#include <new>
#include <vector>

#ifndef _COM_SUN_STAR_BEANS_PROPERTYVALUE_HPP_
#include <com/sun/star/beans/PropertyValue.hpp>
#endif
#ifndef _COM_SUN_STAR_BEANS_XPROPERTYSETINFO_HPP_
#include <com/sun/star/beans/XPropertySetInfo.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_DISPOSEDEXCEPTION_HPP_
#include <com/sun/star/lang/DisposedException.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_XMULTISERVICEFACTORY_HPP_
#include <com/sun/star/lang/XMultiServiceFactory.hpp>
#endif
#ifndef _COM_SUN_STAR_REFLECTION_XPROXYFACTORY_HPP_
#include <com/sun/star/reflection/XProxyFactory.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_COMMANDABORTEDEXCEPTION_HPP_
#include <com/sun/star/ucb/CommandAbortedException.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_COMMANDFAILEDEXCEPTION_HPP_
#include <com/sun/star/ucb/CommandFailedException.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_INTERACTIVEAUGMENTEDIOEXCEPTION_HPP_
#include <com/sun/star/ucb/InteractiveAugmentedIOException.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_INTERACTIVEBADTRANSFERURLEXCEPTION_HPP_
#include <com/sun/star/ucb/InteractiveBadTransferURLException.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_OPENCOMMANDARGUMENT_HPP_
#include <com/sun/star/ucb/OpenCommandArgument.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_SEARCHCOMMANDARGUMENT_HPP_
#include <com/sun/star/ucb/SearchCommandArgument.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_SERVICENOTFOUNDEXCEPTION_HPP_
#include <com/sun/star/ucb/ServiceNotFoundException.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_TRANSFERINFO_HPP_
#include <com/sun/star/ucb/TransferInfo.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_XDYNAMICRESULTSET_HPP_
#include <com/sun/star/ucb/XDynamicResultSet.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_XCACHEDDYNAMICRESULTSETFACTORY_HPP_
#include <com/sun/star/ucb/XCachedDynamicResultSetFactory.hpp>
#endif
#ifndef _COM_SUN_STAR_UCB_XCOMMANDENVIRONMENT_HPP_
#include <com/sun/star/ucb/XCommandEnvironment.hpp>
#endif
#ifndef _COM_SUN_STAR_UNO_REFERENCE_HXX_
#include <com/sun/star/uno/Reference.hxx>
#endif
#ifndef _COM_SUN_STAR_UNO_RUNTIMEEXCEPTION_HPP_
#include <com/sun/star/uno/RuntimeException.hpp>
#endif
#ifndef _CPPUHELPER_INTERFACECONTAINER_H_ 
#include <cppuhelper/interfacecontainer.h>
#endif
#ifndef _OSL_DIAGNOSE_H_
#include <osl/diagonse.h>
#endif
#ifndef _RTL_USTRING_H_
#include <rtl/ustring.h>
#endif
#ifndef _RTL_USTRING_HXX_
#include <rtl/ustring.hxx>
#endif

#ifndef INCLUDED_UCB_RAPCOMMANDENVIRONMENT_HXX
#include <rapcommandenvironment.hxx>
#endif
#ifndef _UCPRMT_RMTCONN_HXX_
#include <rmtconn.hxx>
#endif
#ifndef _UCPRMT_RMTIMAP_HXX_
#include <rmtimap.hxx>
#endif
#ifndef _UCPRMT_RMTPCLST_HXX_
#include <rmtpclst.hxx>
#endif

using namespace com::sun;
using namespace com::sun::star;
using namespace ucprmt;

//============================================================================
//
//  Content
//
//============================================================================

class Content::ListenerAction
{
public:
    virtual inline ~ListenerAction() {}

    virtual void process() = 0;
};

//============================================================================
namespace {

class AddEventListenerAction: public Content::ListenerAction
{
public:
    inline
    AddEventListenerAction(
        uno::Reference< lang::XComponent > const & rComponent,
        uno::Reference< lang::XEventListener > const & rListener):
        m_xComponent(rComponent), m_xListener(rListener) {}

    virtual void process();

private:
    uno::Reference< lang::XComponent > m_xComponent;
    uno::Reference< lang::XEventListener > m_xListener;
};

}

void AddEventListenerAction::process()
{
    m_xComponent->addEventListener(m_xListener);
}

//============================================================================
namespace {

class RemoveEventListenerAction: public Content::ListenerAction
{
public:
    inline
    RemoveEventListenerAction(
        uno::Reference< lang::XComponent > const & rComponent,
        uno::Reference< lang::XEventListener > const & rListener):
        m_xComponent(rComponent), m_xListener(rListener) {}

    virtual void process();

private:
    uno::Reference< lang::XComponent > m_xComponent;
    uno::Reference< lang::XEventListener > m_xListener;
};

}

void RemoveEventListenerAction::process()
{
    m_xComponent->removeEventListener(m_xListener);
}

//============================================================================
namespace {

class AddContentEventListenerAction: public Content::ListenerAction
{
public:
    inline
    AddContentEventListenerAction(
        uno::Reference< star::ucb::XContent > const & rContent,
        uno::Reference< star::ucb::XContentEventListener > const & rListener):
        m_xContent(rContent), m_xListener(rListener) {}

   virtual void process();

private:
    uno::Reference< star::ucb::XContent > m_xContent;
    uno::Reference< star::ucb::XContentEventListener > m_xListener;
};

}

void AddContentEventListenerAction::process()
{
    m_xContent->addContentEventListener(m_xListener);
}

//============================================================================
namespace {

class RemoveContentEventListenerAction: public Content::ListenerAction
{
public:
    inline
    RemoveContentEventListenerAction(
        uno::Reference< star::ucb::XContent > const & rContent,
        uno::Reference< star::ucb::XContentEventListener > const & rListener):
        m_xContent(rContent), m_xListener(rListener) {}

    virtual void process();

private:
    uno::Reference< star::ucb::XContent > m_xContent;
    uno::Reference< star::ucb::XContentEventListener > m_xListener;
};

}

void RemoveContentEventListenerAction::process()
{
    m_xContent->removeContentEventListener(m_xListener);
}

//============================================================================
namespace {

class AddPropertiesChangeListenerAction: public Content::ListenerAction
{
public:
    inline
    AddPropertiesChangeListenerAction(
        uno::Reference< beans::XPropertiesChangeNotifier > const & rNotifier,
        uno::Reference< beans::XPropertiesChangeListener > const & rListener,
        uno::Sequence< rtl::OUString > const & rPropertyNames);

    virtual void process();

private:
    uno::Reference< beans::XPropertiesChangeNotifier > m_xNotifier;
    uno::Reference< beans::XPropertiesChangeListener > m_xListener;
    uno::Sequence< rtl::OUString > m_aPropertyNames;
};

inline
AddPropertiesChangeListenerAction::AddPropertiesChangeListenerAction(
    uno::Reference< beans::XPropertiesChangeNotifier > const & rNotifier,
    uno::Reference< beans::XPropertiesChangeListener > const & rListener,
    uno::Sequence< rtl::OUString > const & rPropertyNames):
    m_xNotifier(rNotifier),
    m_xListener(rListener),
    m_aPropertyNames(rPropertyNames)
{}

}

void AddPropertiesChangeListenerAction::process()
{
    m_xNotifier->addPropertiesChangeListener(m_aPropertyNames, m_xListener);
}

//============================================================================
namespace {

class RemovePropertiesChangeListenerAction: public Content::ListenerAction
{
public:
    inline
    RemovePropertiesChangeListenerAction(
        uno::Reference< beans::XPropertiesChangeNotifier > const & rNotifier,
        uno::Reference< beans::XPropertiesChangeListener > const & rListener,
        uno::Sequence< rtl::OUString > const & rPropertyNames);

    virtual void process();

private:
    uno::Reference< beans::XPropertiesChangeNotifier > m_xNotifier;
    uno::Reference< beans::XPropertiesChangeListener > m_xListener;
    uno::Sequence< rtl::OUString > m_aPropertyNames;
};

inline
RemovePropertiesChangeListenerAction::RemovePropertiesChangeListenerAction(
    uno::Reference< beans::XPropertiesChangeNotifier > const & rNotifier,
    uno::Reference< beans::XPropertiesChangeListener > const & rListener,
    uno::Sequence< rtl::OUString > const & rPropertyNames):
    m_xNotifier(rNotifier),
    m_xListener(rListener),
    m_aPropertyNames(rPropertyNames)
{}

}

void RemovePropertiesChangeListenerAction::process()
{
    m_xNotifier->removePropertiesChangeListener(m_aPropertyNames,
                                                m_xListener);
}

//============================================================================
namespace {

class AddPropertySetInfoChangeListenerAction: public Content::ListenerAction
{
public:
    inline
    AddPropertySetInfoChangeListenerAction(
        uno::Reference< beans::XPropertySetInfoChangeNotifier > const &
            rNotifier,
        uno::Reference< beans::XPropertySetInfoChangeListener > const &
            rListener):
        m_xNotifier(rNotifier), m_xListener(rListener) {}

    virtual void process();

private:
    uno::Reference< beans::XPropertySetInfoChangeNotifier > m_xNotifier;
    uno::Reference< beans::XPropertySetInfoChangeListener > m_xListener;
};

}

void AddPropertySetInfoChangeListenerAction::process()
{
    m_xNotifier->addPropertySetInfoChangeListener(m_xListener);
}

//============================================================================
namespace {

class RemovePropertySetInfoChangeListenerAction:
    public Content::ListenerAction
{
public:
    inline
    RemovePropertySetInfoChangeListenerAction(
        uno::Reference< beans::XPropertySetInfoChangeNotifier > const &
            rNotifier,
        uno::Reference< beans::XPropertySetInfoChangeListener > const &
            rListener):
        m_xNotifier(rNotifier), m_xListener(rListener) {}

    virtual void process();

private:
    uno::Reference< beans::XPropertySetInfoChangeNotifier > m_xNotifier;
    uno::Reference< beans::XPropertySetInfoChangeListener > m_xListener;
};

}

void RemovePropertySetInfoChangeListenerAction::process()
{
    m_xNotifier->removePropertySetInfoChangeListener(m_xListener);
}

//============================================================================
namespace {

class AddCommandInfoChangeListenerAction: public Content::ListenerAction
{
public:
    inline
    AddCommandInfoChangeListenerAction(
        uno::Reference< star::ucb::XCommandInfoChangeNotifier > const &
            rNotifier,
        uno::Reference< star::ucb::XCommandInfoChangeListener > const &
            rListener):
        m_xNotifier(rNotifier), m_xListener(rListener) {}

    virtual void process();

private:
    uno::Reference< star::ucb::XCommandInfoChangeNotifier > m_xNotifier;
    uno::Reference< star::ucb::XCommandInfoChangeListener > m_xListener;
};

}

void AddCommandInfoChangeListenerAction::process()
{
    m_xNotifier->addCommandInfoChangeListener(m_xListener);
}

//============================================================================
namespace {

class RemoveCommandInfoChangeListenerAction: public Content::ListenerAction
{
public:
    inline
    RemoveCommandInfoChangeListenerAction(
        uno::Reference< star::ucb::XCommandInfoChangeNotifier > const &
            rNotifier,
        uno::Reference< star::ucb::XCommandInfoChangeListener > const &
            rListener):
        m_xNotifier(rNotifier), m_xListener(rListener) {}

    virtual void process();

private:
    uno::Reference< star::ucb::XCommandInfoChangeNotifier > m_xNotifier;
    uno::Reference< star::ucb::XCommandInfoChangeListener > m_xListener;
};

}

void RemoveCommandInfoChangeListenerAction::process()
{
    m_xNotifier->removeCommandInfoChangeListener(m_xListener);
}

//============================================================================
inline bool Content::isDisposed() const
{
    return !m_xProxy.is();
}

//============================================================================
inline void Content::checkDisposed() const
{
    if (isDisposed())
        throw lang::DisposedException();
}

//============================================================================
Content::Content(uno::Reference< reflection::XProxyFactory > const &
                     rProxyFactory,
                 rtl::Reference< Connection > const & rTheConnection,
                 uno::Reference< star::ucb::XContentIdentifier > const &
                     rTheLocalIdentifier,
                 uno::Reference< star::ucb::XContent > const &
                     rTheRemoteContent)
    throw (uno::RuntimeException):
    m_xConnection(rTheConnection),
    m_xLocalIdentifier(rTheLocalIdentifier),
    m_xRemoteContent(rTheRemoteContent),
    m_pEventListeners(0),
    m_pContentEventListeners(0),
    m_pPropertiesChangeListeners(0),
    m_pPropertySetInfoChangeListeners(0),
    m_pCommandInfoChangeListeners(0),
    m_eAvailableComponent(AVAIL_UNKNOWN),
    m_eAvailableCommandProcessor(AVAIL_UNKNOWN),
    m_eAvailablePropertiesChangeNotifier(AVAIL_UNKNOWN),
    m_eAvailablePropertySetInfoChangeNotifier(AVAIL_UNKNOWN),
    m_eAvailableCommandInfoChangeNotifier(AVAIL_UNKNOWN),
    m_eAvailableContentCreator(AVAIL_UNKNOWN),
    m_eAvailableChild(AVAIL_UNKNOWN),
    m_eAvailableUnoTunnel(AVAIL_UNKNOWN),
    m_bQueueInOperation(false)
{
    OSL_ASSERT(rProxyFactory.is() && m_xConnection.is()
               && m_xRemoteContent.is());

    m_xProxy = rProxyFactory->createProxy(m_xRemoteContent);
    if (!m_xProxy.is())
        throw uno::RuntimeException(); // bad createProxy() result
}

//============================================================================
void Content::disposeAndClear()
{
    bool bDispose = false;
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (m_xProxy.is())
        {
            m_xProxy->setDelegator(0);
            m_xProxy = 0;
            bDispose = true;
        }
    }

    if (bDispose)
    {
        cppu::OInterfaceContainerHelper * pListeners = 0;
        {
            osl::MutexGuard aGuard(m_aMutex);
            pListeners = m_pEventListeners;
            m_pEventListeners = 0;
        }
        if (pListeners)
        {
            pListeners->
                disposeAndClear(lang::EventObject(
                                    static_cast< lang::XComponent * >(this)));
            delete pListeners;
        }

        if (m_pContentEventListeners)
            m_pContentEventListeners->
                disposeAndClear(lang::EventObject(
                                    static_cast< lang::XComponent * >(this)));

        PropertiesChangeListeners * pPCListeners;
        {
            osl::MutexGuard aGuard(m_aMutex);
            pPCListeners = m_pPropertiesChangeListeners;
            m_pPropertiesChangeListeners = 0;
        }
        if (pPCListeners)
        {
            pPCListeners->
                dispose(lang::EventObject(static_cast< lang::XComponent * >(
                                              this)));
            delete pPCListeners;
        }

        if (m_pPropertySetInfoChangeListeners)
            m_pPropertySetInfoChangeListeners->
                disposeAndClear(lang::EventObject(
                                    static_cast< lang::XComponent * >(this)));

        if (m_pCommandInfoChangeListeners)
            m_pCommandInfoChangeListeners->
                disposeAndClear(lang::EventObject(
                                    static_cast< lang::XComponent * >(this)));

        m_xRemoteContent = 0;
        m_xRemoteComponent = 0;
        m_xRemoteCommandProcessor = 0;
        m_xRemotePropertiesChangeNotifier = 0;
        m_xRemotePropertySetInfoChangeNotifier = 0;
        m_xRemoteCommandInfoChangeNotifier = 0;
        m_xRemoteContentCreator = 0;
        m_xRemoteChild = 0;
        m_xRemoteUnoTunnel = 0;
    }
}

//============================================================================
bool Content::detectComponent()
{
    if (m_eAvailableComponent == AVAIL_UNKNOWN)
    {
        m_xRemoteComponent
            = uno::Reference< lang::XComponent >(m_xRemoteContent,
                                                 uno::UNO_QUERY);
        m_eAvailableComponent
            = m_xRemoteComponent.is() ? AVAIL_YES : AVAIL_NO;
    }
    return m_eAvailableComponent == AVAIL_YES;
}

//============================================================================
bool Content::detectCommandProcessor()
{
    if (m_eAvailableCommandProcessor == AVAIL_UNKNOWN)
    {
        m_xRemoteCommandProcessor
            = uno::Reference< star::ucb::XCommandProcessor >(m_xRemoteContent,
                                                             uno::UNO_QUERY);
        m_eAvailableCommandProcessor
            = m_xRemoteCommandProcessor.is() ? AVAIL_YES : AVAIL_NO;
    }
    return m_eAvailableCommandProcessor == AVAIL_YES;
}

//============================================================================
bool Content::detectPropertiesChangeNotifier()
{
    if (m_eAvailablePropertiesChangeNotifier == AVAIL_UNKNOWN)
    {
        m_xRemotePropertiesChangeNotifier
            = uno::Reference< beans::XPropertiesChangeNotifier >(
                  m_xRemoteContent, uno::UNO_QUERY);
        m_eAvailablePropertiesChangeNotifier
            = m_xRemotePropertiesChangeNotifier.is() ? AVAIL_YES : AVAIL_NO;
    }
    return m_eAvailablePropertiesChangeNotifier == AVAIL_YES;
}

//============================================================================
bool Content::detectPropertySetInfoChangeNotifier()
{
    if (m_eAvailablePropertySetInfoChangeNotifier == AVAIL_UNKNOWN)
    {
        m_xRemotePropertySetInfoChangeNotifier
            = uno::Reference< beans::XPropertySetInfoChangeNotifier >(
                  m_xRemoteContent, uno::UNO_QUERY);
        m_eAvailablePropertySetInfoChangeNotifier
            = m_xRemotePropertySetInfoChangeNotifier.is() ? AVAIL_YES :
                                                            AVAIL_NO;
    }
    return m_eAvailablePropertySetInfoChangeNotifier == AVAIL_YES;
}

//============================================================================
bool Content::detectCommandInfoChangeNotifier()
{
    if (m_eAvailableCommandInfoChangeNotifier == AVAIL_UNKNOWN)
    {
        m_xRemoteCommandInfoChangeNotifier
            = uno::Reference< star::ucb::XCommandInfoChangeNotifier >(
                  m_xRemoteContent, uno::UNO_QUERY);
        m_eAvailableCommandInfoChangeNotifier
            = m_xRemoteCommandInfoChangeNotifier.is() ? AVAIL_YES : AVAIL_NO;
    }
    return m_eAvailableCommandInfoChangeNotifier == AVAIL_YES;
}

//============================================================================
bool Content::detectContentCreator()
{
    if (m_eAvailableContentCreator == AVAIL_UNKNOWN)
    {
        m_xRemoteContentCreator
            = uno::Reference< star::ucb::XContentCreator >(m_xRemoteContent,
                                                           uno::UNO_QUERY);
        m_eAvailableContentCreator
            = m_xRemoteContentCreator.is() ? AVAIL_YES : AVAIL_NO;
    }
    return m_eAvailableContentCreator == AVAIL_YES;
}

//============================================================================
bool Content::detectChild()
{
    if (m_eAvailableChild == AVAIL_UNKNOWN)
    {
        m_xRemoteChild = uno::Reference< container::XChild >(m_xRemoteContent,
                                                             uno::UNO_QUERY);
        m_eAvailableChild = m_xRemoteChild.is() ? AVAIL_YES : AVAIL_NO;
    }
    return m_eAvailableChild == AVAIL_YES;
}

//============================================================================
bool Content::detectUnoTunnel()
{
    if (m_eAvailableUnoTunnel == AVAIL_UNKNOWN)
    {
        m_xRemoteUnoTunnel
            = uno::Reference< lang::XUnoTunnel >(m_xRemoteContent,
                                                 uno::UNO_QUERY);
        m_eAvailableUnoTunnel
            = m_xRemoteUnoTunnel.is() ? AVAIL_YES : AVAIL_NO;
    }
    return m_eAvailableUnoTunnel == AVAIL_YES;
}

//============================================================================
void Content::addToListenerQueue(std::auto_ptr< ListenerAction > & rAction)
{
    osl::MutexGuard aGuard(m_aQueueMutex);
    m_aListenerQueue.push(rAction.get());
    rAction.release();
}

//============================================================================
void Content::processListenerQueue()
{
    for (bool bFirst = true;; bFirst = false)
    {
        std::auto_ptr< ListenerAction > xAction;
        {
            osl::MutexGuard aGuard(m_aQueueMutex);
            if (!bFirst)
                m_bQueueInOperation = false;
            if (m_bQueueInOperation || m_aListenerQueue.empty())
                break;
            xAction
                = std::auto_ptr< ListenerAction >(m_aListenerQueue.front());
            m_aListenerQueue.pop();
            m_bQueueInOperation = true;
        }
        try
        {
            xAction->process();
        }
        catch (...)
        {
            {
                osl::MutexGuard aGuard(m_aQueueMutex);
                m_bQueueInOperation = false;
            }
            throw;
        }
    }
}

//============================================================================
// static
uno::Reference< star::ucb::XContent >
Content::create(uno::Reference< lang::XMultiServiceFactory > const & rFactory,
                rtl::Reference< Connection > const & rTheConnection,
                uno::Reference< star::ucb::XContentIdentifier > const &
                    rTheLocalIdentifier,
                uno::Reference< star::ucb::XContent > const &
                    rTheRemoteContent)
    throw (uno::RuntimeException)
{
    OSL_ASSERT(rFactory.is());

    if (!rTheRemoteContent.is())
        return 0;

    uno::Reference< reflection::XProxyFactory > xProxyFactory;
    try
    {
        xProxyFactory
            = uno::Reference< reflection::XProxyFactory >(
                  rFactory->
                      createInstance(
                          rtl::OUString::createFromAscii(
                              "com.sun.star.reflection.ProxyFactory")),
                  uno::UNO_QUERY);
    }
    catch (uno::RuntimeException const &) { throw; }
    catch (uno::Exception const &) {}
    if (!xProxyFactory.is())
        throw uno::RuntimeException();
            // no service, or service lacks interface

    Content * pContent = new Content(xProxyFactory, rTheConnection,
                                     rTheLocalIdentifier, rTheRemoteContent);
    uno::Reference< star::ucb::XContent > xTheContent(pContent);
    pContent->m_xProxy->setDelegator(xTheContent);
    return xTheContent;
}

//============================================================================
// virtual
Content::~Content()
{
    if (m_xProxy.is())
    {
        m_xProxy->setDelegator(0);
        m_xProxy = 0;
    }
    delete m_pEventListeners;
    delete m_pContentEventListeners;
    delete m_pPropertiesChangeListeners;
    delete m_pPropertySetInfoChangeListeners;
    delete m_pCommandInfoChangeListeners;
}

//============================================================================
// virtual
uno::Any SAL_CALL Content::queryInterface(uno::Type const & rType)
    throw (uno::RuntimeException)
{
    uno::Any
        aRet(cppu::queryInterface(
                 rType,
                 static_cast< star::ucb::XContent * >(this),
                 static_cast< lang::XUnoTunnel * >(this),
                 static_cast< lang::XEventListener * >(
                     static_cast< star::ucb::XContentEventListener * >(this)),
                 static_cast< star::ucb::XContentEventListener * >(this),
                 static_cast< beans::XPropertiesChangeListener * >(this),
                 static_cast< beans::XPropertySetInfoChangeListener * >(this),
                 static_cast< star::ucb::XCommandInfoChangeListener * >(
                     this)));
    if (!aRet.hasValue())
        if (rType
                == getCppuType(
                       static_cast< uno::Reference< lang::XComponent > * >(
                           0)))
        {
            osl::MutexGuard aGuard(m_aMutex);
            if (detectComponent())
            {
                lang::XComponent * p = this;
                return uno::Any(&p, rType);
            }
            else
                return uno::Any();
        }
        else if (rType
                     == getCppuType(
                            static_cast< uno::Reference<
                                           star::ucb::XCommandProcessor > * >(
                                0)))
        {
            osl::MutexGuard aGuard(m_aMutex);
            if (detectCommandProcessor())
            {
                star::ucb::XCommandProcessor * p = this;
                return uno::Any(&p, rType);
            }
            else
                return uno::Any();
        }
        else if (rType
                     == getCppuType(
                            static_cast<
                                    uno::Reference<
                                       beans::XPropertiesChangeNotifier > * >(
                                0)))
        {
            osl::MutexGuard aGuard(m_aMutex);
            if (detectPropertiesChangeNotifier())
            {
                beans::XPropertiesChangeNotifier * p = this;
                return uno::Any(&p, rType);
            }
            else
                return uno::Any();
        }
        else if (rType
                     == getCppuType(
                            static_cast<
                                    uno::Reference<
                                  beans::XPropertySetInfoChangeNotifier > * >(
                                0)))
        {
            osl::MutexGuard aGuard(m_aMutex);
            if (detectPropertySetInfoChangeNotifier())
            {
                beans::XPropertySetInfoChangeNotifier * p = this;
                return uno::Any(&p, rType);
            }
            else
                return uno::Any();
        }
        else if (rType
                     == getCppuType(
                            static_cast<
                                    uno::Reference<
                                  star::ucb::XCommandInfoChangeNotifier > * >(
                                0)))
        {
            osl::MutexGuard aGuard(m_aMutex);
            if (detectCommandInfoChangeNotifier())
            {
                star::ucb::XCommandInfoChangeNotifier * p = this;
                return uno::Any(&p, rType);
            }
            else
                return uno::Any();
        }
        else if (rType
                     == getCppuType(
                             static_cast< uno::Reference<
                                             star::ucb::XContentCreator > * >(
                                 0)))
        {
            osl::MutexGuard aGuard(m_aMutex);
            if (detectContentCreator())
            {
                star::ucb::XContentCreator * p = this;
                return uno::Any(&p, rType);
            }
            else
                return uno::Any();
        }
        else if (rType
                     == getCppuType(static_cast< uno::Reference<
                                                     container::XChild > * >(
                                        0)))
        {
            osl::MutexGuard aGuard(m_aMutex);
            if (detectChild())
            {
                container::XChild * p = this;
                return uno::Any(&p, rType);
            }
            else
                return uno::Any();
        }
    if (!aRet.hasValue())
        aRet = OWeakObject::queryInterface(rType);
    if (!aRet.hasValue())
        aRet = m_xProxy->queryAggregation(rType);
    return aRet;
}

//============================================================================
// virtual
void SAL_CALL Content::acquire() throw ()
{
    OWeakObject::acquire();
}

//============================================================================
// virtual
void SAL_CALL Content::release() throw ()
{
    OWeakObject::release();
}

//============================================================================
// virtual
uno::Reference< star::ucb::XContentIdentifier > SAL_CALL
Content::getIdentifier() throw (uno::RuntimeException)
{
    return m_xLocalIdentifier;
}

//============================================================================
// virtual
rtl::OUString SAL_CALL Content::getContentType() throw (uno::RuntimeException)
{
    uno::Reference< star::ucb::XContent > xRemote;
    {
        osl::MutexGuard aGuard(m_aMutex);
        checkDisposed();
        xRemote = m_xRemoteContent;
    }
    return xRemote->getContentType();
}

//============================================================================
// virtual
void SAL_CALL
Content::addContentEventListener(
    uno::Reference< star::ucb::XContentEventListener > const & rListener)
    throw (uno::RuntimeException)
{
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (isDisposed())
            return;
        if (!m_pContentEventListeners)
            m_pContentEventListeners
                = new cppu::OInterfaceContainerHelper(m_aMutex);
        if (m_pContentEventListeners->addInterface(rListener) == 1)
        {
            std::auto_ptr< ListenerAction >
                xAction(new AddContentEventListenerAction(m_xRemoteContent,
                                                          this));
            addToListenerQueue(xAction);
        }
    }
    processListenerQueue();
}

//============================================================================
// virtual
void SAL_CALL
Content::removeContentEventListener(
    uno::Reference< star::ucb::XContentEventListener > const & rListener)
    throw (uno::RuntimeException)
{
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (m_pContentEventListeners
            && m_pContentEventListeners->removeInterface(rListener) == 0
            && !isDisposed())
        {
            std::auto_ptr< ListenerAction >
                xAction(new RemoveContentEventListenerAction(m_xRemoteContent,
                                                             this));
            addToListenerQueue(xAction);
        }
    }
    processListenerQueue();
}

//============================================================================
// virtual
void SAL_CALL Content::dispose() throw (uno::RuntimeException)
{
    disposeAndClear();
}

//============================================================================
// virtual
void SAL_CALL
Content::addEventListener(
    uno::Reference< lang::XEventListener > const & rListener)
    throw (uno::RuntimeException)
{
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (isDisposed())
            return;

        detectComponent();
        OSL_ASSERT(m_xRemoteComponent.is());

        if (!m_pEventListeners)
            m_pEventListeners = new cppu::OInterfaceContainerHelper(m_aMutex);
        if (m_pEventListeners->addInterface(rListener) == 1)
        {
            std::auto_ptr< ListenerAction >
                xAction(
                    new AddEventListenerAction(
                            m_xRemoteComponent,
                            static_cast< star::ucb::XContentEventListener * >(
                                this)));
            addToListenerQueue(xAction);
        }
    }
    processListenerQueue();
}

//============================================================================
// virtual
void SAL_CALL Content::removeEventListener(
                  uno::Reference< lang::XEventListener > const & rListener)
    throw (uno::RuntimeException)
{
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (m_pEventListeners
            && m_pEventListeners->removeInterface(rListener) == 0
            && !isDisposed())
        {
            std::auto_ptr< ListenerAction >
                xAction(new RemoveEventListenerAction(
                                m_xRemoteComponent,
                                static_cast<
                                        star::ucb::XContentEventListener * >(
                                    this)));
            addToListenerQueue(xAction);
        }
    }
    processListenerQueue();
}

//============================================================================
// virtual
sal_Int32 SAL_CALL Content::createCommandIdentifier()
    throw (uno::RuntimeException)
{
    uno::Reference< star::ucb::XCommandProcessor > xRemote;
    {
        osl::MutexGuard aGuard(m_aMutex);
        checkDisposed();

        detectCommandProcessor();
        OSL_ASSERT(m_xRemoteCommandProcessor.is());
        xRemote = m_xRemoteCommandProcessor;
    }
    return xRemote->createCommandIdentifier();
}

//============================================================================
// virtual
uno::Any SAL_CALL
Content::execute(star::ucb::Command const & rCommand, sal_Int32 nCommandId,
                 uno::Reference< star::ucb::XCommandEnvironment > const &
                     rEnvironment)
    throw (uno::Exception, star::ucb::CommandAbortedException,
           uno::RuntimeException)
{
    uno::Reference< star::ucb::XCommandProcessor > xRemote;
    {
        osl::MutexGuard aGuard(m_aMutex);
        checkDisposed();

        detectCommandProcessor();
        OSL_ASSERT(m_xRemoteCommandProcessor.is());
        xRemote = m_xRemoteCommandProcessor;
    }

    star::ucb::Command aTheCommand(rCommand);
    enum Type
    {
        TYPE_OPEN,
        TYPE_SEARCH,
        TYPE_TRANSFER,
        TYPE_OTHER
    };
    Type eType
        = aTheCommand.Name.equalsAsciiL(RTL_CONSTASCII_STRINGPARAM(
                                            "open")) ?
              TYPE_OPEN :
          aTheCommand.Name.equalsAsciiL(RTL_CONSTASCII_STRINGPARAM(
                                            "search")) ?
              TYPE_SEARCH :
          aTheCommand.Name.equalsAsciiL(RTL_CONSTASCII_STRINGPARAM(
                                            "transfer")) ?
              TYPE_TRANSFER :
              TYPE_OTHER;

    switch (eType)
    {
    case TYPE_TRANSFER:
        {
            star::ucb::TransferInfo aInfo;
            if (aTheCommand.Argument >>= aInfo)
                if (m_xConnection->mapsToThisConnection(aInfo.SourceURL))
                {
                    aInfo.SourceURL
                        = m_xConnection->mapToRemoteURI(aInfo.SourceURL);
                    aTheCommand.Argument <<= aInfo;
                }
                else
                    throw star::ucb::InteractiveBadTransferURLException();
            break;
        }
    }

    uno::Reference< star::ucb::XCommandEnvironment > xEnvironment;
    if (rEnvironment.is())
        try
        {
            xEnvironment = new CommandEnvironment(rEnvironment,
                                                  m_xConnection);
        }
        catch (std::bad_alloc)
        {
            throw uno::RuntimeException(
                      rtl::OUString(RTL_CONSTASCII_USTRINGPARAM(
                                        "Out of memory")),
                      *this);
        }

    uno::Any aReturn;
    try
    {
        aReturn = xRemote->execute(aTheCommand, nCommandId, xEnvironment);
    }
    catch (star::ucb::InteractiveAugmentedIOException const & rException)
    {
        throw m_xConnection->mapToLocalException(rException);
    }
    catch (star::ucb::CommandFailedException const & rException)
    {
        star::ucb::InteractiveAugmentedIOException aException;
        if (rException.Reason >>= aException)
            throw m_xConnection->mapToLocalException(aException);
        throw;
    }

    switch (eType)
    {
    case TYPE_OPEN:
    case TYPE_SEARCH:
        {
            uno::Reference< star::ucb::XDynamicResultSet > xResultSet;
            if ((aReturn >>= xResultSet) && xResultSet.is())
            {
                uno::Reference< star::ucb::XCachedDynamicResultSetFactory >
                    xFactory(
                        m_xConnection->
                            getFactory()->
                                createInstance(
                                    rtl::OUString::createFromAscii(
                           "com.sun.star.ucb.CachedDynamicResultSetFactory")),
                        uno::UNO_QUERY);
                if (!xFactory.is())
                    throw star::ucb::ServiceNotFoundException();

                uno::Reference< star::ucb::XContentIdentifierMapping >
                    xMapping;
                switch (eType)
                {
                case TYPE_OPEN:
                    {
                        star::ucb::OpenCommandArgument aArg;
                        if (aTheCommand.Argument >>= aArg)
                            xMapping
                                = new ContentIdentifierMapping(
                                          m_xConnection, aArg.Properties);
                        break;
                    }

                case TYPE_SEARCH:
                    {
                        star::ucb::SearchCommandArgument aArg;
                        if (aTheCommand.Argument >>= aArg)
                            xMapping
                                = new ContentIdentifierMapping(
                                          m_xConnection, aArg.Properties);
                        break;
                    }
                }

                uno::Reference< star::ucb::XDynamicResultSet >
                    xCached(xFactory->createCachedDynamicResultSet(0,
                                                                   xMapping));
                if (!xCached.is())
                    throw star::ucb::CommandAbortedException(); //TODO!

                xResultSet->connectToCache(xCached);
                aReturn <<= xCached;
            }
            break;
        }
    }
    return aReturn;
}

//============================================================================
// virtual
void SAL_CALL Content::abort(sal_Int32 nCommandId)
    throw (uno::RuntimeException)
{
    uno::Reference< star::ucb::XCommandProcessor > xRemote;
    {
        osl::MutexGuard aGuard(m_aMutex);
        checkDisposed();

        detectCommandProcessor();
        OSL_ASSERT(m_xRemoteCommandProcessor.is());
        xRemote = m_xRemoteCommandProcessor;
    }
    xRemote->abort(nCommandId);
}

//============================================================================
// virtual
void SAL_CALL
Content::addPropertiesChangeListener(
    uno::Sequence< rtl::OUString > const & rPropertyNames,
    uno::Reference< beans::XPropertiesChangeListener > const & rListener)
    throw (uno::RuntimeException)
{
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (isDisposed())
            return;

        detectPropertiesChangeNotifier();
        OSL_ASSERT(m_xRemotePropertiesChangeNotifier.is());

        if (!m_pPropertiesChangeListeners)
            m_pPropertiesChangeListeners = new PropertiesChangeListeners;

        PropertiesChangeListeners::iterator
            aEnd(m_pPropertiesChangeListeners->end());
        PropertiesChangeListeners::iterator
            aIt(std::find(m_pPropertiesChangeListeners->begin(),
                          aEnd,
                          rListener));
        if (aIt == aEnd)
            // Adding a new listener makes it either an 'all' or a 'some'
            // listener:
            m_pPropertiesChangeListeners->
                push_back(PropertiesChangeListener(
                              rListener,
                              rPropertyNames.getLength() == 0 ?
                                  PropertiesChangeListener::ALL_PROPS :
                                  PropertiesChangeListener::SOME_PROPS,
                              rPropertyNames));
        else
            switch (aIt->getKind())
            {
                case PropertiesChangeListener::ALL_PROPS:
                    // Adding properties to an 'all' listener does nothing:
                    return;

                case PropertiesChangeListener::SOME_PROPS:
                    // Adding properties to a 'some' listener either makes it
                    // an 'all' listener or increases its list of properties:
                    if (rPropertyNames.getLength() == 0)
                        aIt->setKind(PropertiesChangeListener::ALL_PROPS,
                                     uno::Sequence< rtl::OUString >());
                    else
                        aIt->addProperties(rPropertyNames);
                    break;

                case PropertiesChangeListener::ALL_BUT_SOME_PROPS:
                    // Adding properties to an 'all but some' listener either
                    // makes it an 'all' listener or reduces its list of
                    // properties:
                    if (rPropertyNames.getLength() == 0
                        || !aIt->removeProperties(rPropertyNames))
                        aIt->setKind(PropertiesChangeListener::ALL_PROPS,
                                     uno::Sequence< rtl::OUString >());
                    break;
            }

        std::auto_ptr< ListenerAction >
            xAction(new AddPropertiesChangeListenerAction(
                            m_xRemotePropertiesChangeNotifier,
                            this,
                            rPropertyNames));
        addToListenerQueue(xAction);
    }
    processListenerQueue();
}

//============================================================================
// virtual
void SAL_CALL
Content::removePropertiesChangeListener(
    uno::Sequence< rtl::OUString > const & rPropertyNames,
    uno::Reference< beans::XPropertiesChangeListener > const & rListener)
    throw (uno::RuntimeException)
{
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (m_pPropertiesChangeListeners)
        {
            PropertiesChangeListeners::iterator
                aEnd(m_pPropertiesChangeListeners->end());
            PropertiesChangeListeners::iterator
                aIt(std::find(m_pPropertiesChangeListeners->begin(), aEnd,
                              rListener));
            if (aIt != aEnd)
                switch (aIt->getKind())
                {
                case PropertiesChangeListener::ALL_PROPS:
                    // Removing properties from an 'all' listener either
                    // removes it or makes it an 'all but some' listener:
                    if (rPropertyNames.getLength() == 0)
                        m_pPropertiesChangeListeners->erase(aIt);
                    else
                    {
                        aIt->
                            setKind(
                                PropertiesChangeListener::ALL_BUT_SOME_PROPS,
                                rPropertyNames);
                    }
                    break;

                case PropertiesChangeListener::SOME_PROPS:
                    // Removing properties from a 'some' listener either
                    // removes it or reduces its list of properties:
                    if (rPropertyNames.getLength() == 0
                        || !aIt->removeProperties(rPropertyNames))
                        m_pPropertiesChangeListeners->erase(aIt);
                    break;

                case PropertiesChangeListener::ALL_BUT_SOME_PROPS:
                    // Removing properties from an 'all but some' listener
                    // either removes it or increases its list of properties:
                    if (rPropertyNames.getLength() == 0)
                        m_pPropertiesChangeListeners->erase(aIt);
                    else
                        aIt->addProperties(rPropertyNames);
                    break;
                }
        }

        if (m_pPropertiesChangeListeners->empty() && !isDisposed())
        {
            detectPropertiesChangeNotifier();
            OSL_ASSERT(m_xRemotePropertiesChangeNotifier.is());
            std::auto_ptr< ListenerAction >
                xAction(new RemovePropertiesChangeListenerAction(
                                m_xRemotePropertiesChangeNotifier,
                                this,
                                rPropertyNames));
            addToListenerQueue(xAction);
        }
    }
    processListenerQueue();
}

//============================================================================
// virtual
void SAL_CALL
Content::addPropertySetInfoChangeListener(
    uno::Reference< beans::XPropertySetInfoChangeListener > const & rListener)
    throw (uno::RuntimeException)
{
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (isDisposed())
            return;

        detectPropertySetInfoChangeNotifier();
        OSL_ASSERT(m_xRemotePropertySetInfoChangeNotifier.is());

        if (!m_pPropertySetInfoChangeListeners)
            m_pPropertySetInfoChangeListeners
                = new cppu::OInterfaceContainerHelper(m_aMutex);
        if (m_pPropertySetInfoChangeListeners->addInterface(rListener) == 1)
        {
            std::auto_ptr< ListenerAction >
                xAction(new AddPropertySetInfoChangeListenerAction(
                                m_xRemotePropertySetInfoChangeNotifier,
                                this));
            addToListenerQueue(xAction);
        }
    }
    processListenerQueue();
}

//============================================================================
// virtual
void SAL_CALL
Content::removePropertySetInfoChangeListener(
    uno::Reference< beans::XPropertySetInfoChangeListener > const & rListener)
    throw (uno::RuntimeException)
{
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (m_pPropertySetInfoChangeListeners
            && m_pPropertySetInfoChangeListeners->removeInterface(rListener)
                   == 0
            && !isDisposed())
        {
            std::auto_ptr< ListenerAction >
                xAction(new RemovePropertySetInfoChangeListenerAction(
                                m_xRemotePropertySetInfoChangeNotifier,
                                this));
            addToListenerQueue(xAction);
        }
    }
    processListenerQueue();
}

//============================================================================
// virtual
void SAL_CALL
Content::addCommandInfoChangeListener(
    uno::Reference< star::ucb::XCommandInfoChangeListener > const & rListener)
    throw (uno::RuntimeException)
{
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (isDisposed())
            return;

        detectCommandInfoChangeNotifier();
        OSL_ASSERT(m_xRemoteCommandInfoChangeNotifier.is());

        if (!m_pCommandInfoChangeListeners)
            m_pCommandInfoChangeListeners
                = new cppu::OInterfaceContainerHelper(m_aMutex);
        if (m_pCommandInfoChangeListeners->addInterface(rListener) == 1)
        {
            std::auto_ptr< ListenerAction >
                xAction(new AddCommandInfoChangeListenerAction(
                                m_xRemoteCommandInfoChangeNotifier, this));
            addToListenerQueue(xAction);
        }
    }
    processListenerQueue();
}

//============================================================================
// virtual
void SAL_CALL
Content::removeCommandInfoChangeListener(
    uno::Reference< star::ucb::XCommandInfoChangeListener > const & rListener)
    throw (uno::RuntimeException)
{
    {
        osl::MutexGuard aGuard(m_aMutex);
        if (m_pCommandInfoChangeListeners
            && m_pCommandInfoChangeListeners->removeInterface(rListener) == 0
            && !isDisposed())
        {
            std::auto_ptr< ListenerAction >
                xAction(new RemoveCommandInfoChangeListenerAction(
                                m_xRemoteCommandInfoChangeNotifier, this));
            addToListenerQueue(xAction);
        }
    }
    processListenerQueue();
}

//============================================================================
// virtual
uno::Sequence< star::ucb::ContentInfo > SAL_CALL
Content::queryCreatableContentsInfo() throw (uno::RuntimeException)
{
    uno::Reference< star::ucb::XContentCreator > xRemote;
    {
        osl::MutexGuard aGuard(m_aMutex);
        checkDisposed();

        detectContentCreator();
        OSL_ASSERT(m_xRemoteContentCreator.is());
        xRemote = m_xRemoteContentCreator;
    }
    return xRemote->queryCreatableContentsInfo();
}

//============================================================================
// virtual
uno::Reference< star::ucb::XContent > SAL_CALL
Content::createNewContent(star::ucb::ContentInfo const & rInfo)
    throw (uno::RuntimeException)
{
    uno::Reference< star::ucb::XContentCreator > xRemote;
    {
        osl::MutexGuard aGuard(m_aMutex);
        checkDisposed();

        detectContentCreator();
        OSL_ASSERT(m_xRemoteContentCreator.is());
        xRemote = m_xRemoteContentCreator;
    }
    return m_xConnection->mapToLocalContent(xRemote->createNewContent(rInfo));
}

//============================================================================
// virtual
uno::Reference< uno::XInterface > SAL_CALL Content::getParent()
    throw (uno::RuntimeException)
{
    uno::Reference< container::XChild > xRemote;
    {
        osl::MutexGuard aGuard(m_aMutex);
        checkDisposed();

        detectChild();
        OSL_ASSERT(m_xRemoteChild.is());
        xRemote = m_xRemoteChild;
    }

    uno::Reference< uno::XInterface > xParent(xRemote->getParent());
    uno::Reference< star::ucb::XContent >
        xParentContent(xParent, uno::UNO_QUERY);
    if (xParentContent.is())
        return m_xConnection->mapToLocalContent(xParentContent);
    else
        return xParent;
}

//============================================================================
// virtual
void SAL_CALL Content::setParent(uno::Reference< uno::XInterface > const &
                                     rParent)
    throw (lang::NoSupportException, uno::RuntimeException)
{
    uno::Reference< container::XChild > xRemote;
    {
        osl::MutexGuard aGuard(m_aMutex);
        checkDisposed();

        detectChild();
        OSL_ASSERT(m_xRemoteChild.is());
        xRemote = m_xRemoteChild;
    }

    uno::Reference< star::ucb::XContent >
        xParentContent(rParent, uno::UNO_QUERY);
    if (xParentContent.is())
        xRemote->setParent(m_xConnection->mapToRemoteContent(xParentContent));
    else
        xRemote->setParent(rParent);
}

//============================================================================
// virtual
sal_Int64 SAL_CALL Content::getSomething(uno::Sequence< sal_Int8 > const &
                                             rIdentifier)
    throw (uno::RuntimeException)
{
    if (m_xConnection->isUUID(rIdentifier))
        return 1;
    else
    {
        uno::Reference< lang::XUnoTunnel > xRemote;
        {
            osl::MutexGuard aGuard(m_aMutex);
            if (!isDisposed() && detectUnoTunnel())
                xRemote = m_xRemoteUnoTunnel;
        }
        return xRemote.is() ? xRemote->getSomething(rIdentifier) : 0;
    }
}

//============================================================================
// virtual
void SAL_CALL Content::disposing(lang::EventObject const & rSource)
    throw (uno::RuntimeException)
{
    bool bDispose;
    {
        osl::MutexGuard aGuard(m_aMutex);
        bDispose = detectComponent() && rSource.Source == m_xRemoteComponent;
    }

    if (bDispose)
        disposeAndClear();
}

//============================================================================
// virtual
void SAL_CALL Content::contentEvent(star::ucb::ContentEvent const & rEvt)
    throw (uno::RuntimeException)
{
    osl::ClearableMutexGuard aGuard(m_aMutex);
    if (m_pContentEventListeners && !isDisposed())
    {
        cppu::OInterfaceIteratorHelper aIt(*m_pContentEventListeners);

        star::ucb::ContentEvent aLocalEvt(rEvt);
        if (rEvt.Source == m_xRemoteContent)
        {
            aLocalEvt.Source = static_cast< cppu::OWeakObject * >(this);
            aLocalEvt.Content
                = m_xConnection->mapToLocalContent(rEvt.Content);
            aLocalEvt.Id = m_xConnection->mapToLocalIdentifier(rEvt.Id);
        }

        aGuard.clear();

        while (uno::XInterface * p = aIt.next())
        {
            uno::Reference< star::ucb::XContentEventListener >
                xListener(p, uno::UNO_QUERY);
            if (xListener.is())
                xListener->contentEvent(aLocalEvt);
        }
    }
}

//============================================================================
// virtual
void SAL_CALL
Content::propertiesChange(uno::Sequence< beans::PropertyChangeEvent > const &
                              rEvt)
    throw (uno::RuntimeException)
{
    osl::ClearableMutexGuard aGuard(m_aMutex);
    if (m_pPropertiesChangeListeners)
    {
        typedef
            std::list< uno::Reference< beans::XPropertiesChangeListener > >
            Listeners;
        Listeners aListeners;
        PropertiesChangeListeners::const_iterator
            aEnd(m_pPropertiesChangeListeners->end());
        for (PropertiesChangeListeners::const_iterator
                 aIt(m_pPropertiesChangeListeners->begin());
             aIt != aEnd; ++aIt)
            switch (aIt->getKind())
            {
                case PropertiesChangeListener::ALL_PROPS:
                    // Add all of the 'all' listeners:
                    aListeners.push_back(aIt->getListener());
                    break;

                case PropertiesChangeListener::SOME_PROPS:
                    // Add those 'some' listeners whose sets of properties
                    // intersect with the set of event properties:
                    if (aIt->intersectsWith(rEvt))
                        aListeners.push_back(aIt->getListener());
                    break;

                case PropertiesChangeListener::ALL_BUT_SOME_PROPS:
                    // Add those 'all but some' listeners whose sets of
                    // properties are not subsets of the set of event
                    // properties:
                    if (!aIt->subsetOf(rEvt))
                        aListeners.push_back(aIt->getListener());
                    break;
            }

        uno::Sequence< beans::PropertyChangeEvent > aLocalEvt(rEvt);
        if (detectPropertiesChangeNotifier())
        {
            beans::PropertyChangeEvent * p = aLocalEvt.getArray();
            beans::PropertyChangeEvent * pEnd = p + aLocalEvt.getLength();
            for (; p != pEnd; ++p)
                if (p->Source == m_xRemotePropertiesChangeNotifier)
                    p->Source = static_cast< cppu::OWeakObject * >(this);
        }

        aGuard.clear();

        {
            Listeners::const_iterator aEnd(aListeners.end());
            for (Listeners::const_iterator aIt(aListeners.begin());
                 aIt != aEnd; ++aIt)
                (*aIt)->propertiesChange(aLocalEvt);
        }
    }
}

//============================================================================
// virtual
void SAL_CALL Content::propertySetInfoChange(
                  beans::PropertySetInfoChangeEvent const & rEvt)
    throw (uno::RuntimeException)
{
    osl::ClearableMutexGuard aGuard(m_aMutex);
    if (m_pPropertySetInfoChangeListeners && !isDisposed())
    {
        cppu::OInterfaceIteratorHelper
            aIt(*m_pPropertySetInfoChangeListeners);

        beans::PropertySetInfoChangeEvent aLocalEvt(rEvt);
        if (detectPropertySetInfoChangeNotifier()
            && rEvt.Source == m_xRemotePropertySetInfoChangeNotifier)
            aLocalEvt.Source = static_cast< cppu::OWeakObject * >(this);

        aGuard.clear();

        while (uno::XInterface * p = aIt.next())
        {
            uno::Reference< beans::XPropertySetInfoChangeListener >
                xListener(p, uno::UNO_QUERY);
            if (xListener.is())
                xListener->propertySetInfoChange(aLocalEvt);
        }
    }
}

//============================================================================
// virtual
void SAL_CALL
Content::commandInfoChange(star::ucb::CommandInfoChangeEvent const & rEvt)
    throw (uno::RuntimeException)
{
    osl::ClearableMutexGuard aGuard(m_aMutex);
    if (m_pCommandInfoChangeListeners && !isDisposed())
    {
        cppu::OInterfaceIteratorHelper
            aIt(*m_pCommandInfoChangeListeners);

        star::ucb::CommandInfoChangeEvent aLocalEvt(rEvt);
        if (detectCommandInfoChangeNotifier()
            && rEvt.Source == m_xRemoteCommandInfoChangeNotifier)
            aLocalEvt.Source = static_cast< cppu::OWeakObject * >(this);

        aGuard.clear();

        while (uno::XInterface * p = aIt.next())
        {
            uno::Reference< star::ucb::XCommandInfoChangeListener >
                xListener(p, uno::UNO_QUERY);
            if (xListener.is())
                xListener->commandInfoChange(aLocalEvt);
        }
    }
}
