/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: PaneManager.cxx,v $
 *
 *  $Revision: 1.26 $
 *
 *  last change: $Author: kz $ $Date: 2006/02/01 12:52:07 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "PaneManager.hxx"

#include "ViewShellBase.hxx"
#include "ViewShellHint.hxx"
#ifndef SD_DRAW_VIEW_SHELL_HXX
#include "DrawViewShell.hxx"
#endif
#ifndef SD_OUTLINE_VIEW_SHELL_HXX
#include "OutlineViewShell.hxx"
#endif
#include "Window.hxx"
#include "DrawController.hxx"
#include "UpdateLockManager.hxx"

#include "sdresid.hxx"
#include "app.hrc"
#include "strings.hrc"
#ifndef _SFXREQUEST_HXX
#include <sfx2/request.hxx>
#endif
#ifndef _SFXDISPATCH_HXX
#include <sfx2/dispatch.hxx>
#endif
#ifndef SD_DRAW_VIEW_HXX
#include "drawview.hxx"
#endif
#include <sfx2/viewfrm.hxx>
#ifndef _DRAWDOC_HXX
#include "drawdoc.hxx"
#endif
#ifndef SD_PANE_CHILD_WINDOWS_HXX
#include "PaneChildWindows.hxx"
#endif
#include "PaneDockingWindow.hrc"
#ifndef SD_NOTES_CHILD_WINDOW_HXX
#include "NotesChildWindow.hxx"
#endif
#include "FrameView.hxx"
#ifndef SD_VIEW_SHELL_MANAGER_HXX
#include "ViewShellManager.hxx"
#endif
#include "MasterPageObserver.hxx"
#ifndef SD_DRAW_DOC_SHELL_HXX
#include "DrawDocShell.hxx"
#endif
#ifndef _SFX_WHITER_HXX
#include <svtools/whiter.hxx>
#endif

#include <vector>

/** The implementation of the PaneManager is split into the following
    classes:
    namsepace sd {
    PaneManager
        The facade of the pane manager.
    PaneManager::Implementation
        Controller of the pane manager.
    PaneManagerEvent
        Events of this type are broadcasted by the EventBroadcaster
    }
    namespace {
    PaneWindow, PaneFrameWindow, PaneChildWindow
        Wrappers around windows that represent the panes.
    EventBroadcaster
        Management of event listeners
    PaneDescriptor
        Management of windows and view shells that are associated to one
        pane.
    PaneConfiguration
        The pane configuration stores the states of the window visibility and
        the types of view shell to be used for each pane.  When these differ
        from the current values (as stored in the PaneDescriptors) then it
        is the task of the PaneManager::Implementation to bring the
        PaneDescriptors up to date.

    }
*/

using namespace sd;

/** Define this variable to destroy unused pane windows.  When undefined
    unused pane windows are hidden.
*/
#if 0
#define DESTROY_UNUSED_PANES
#else
#undef DESTROY_UNUSED_PANES
#endif

using namespace ::com::sun::star::uno;
using namespace ::com::sun::star;


/** Define a simple comparation between Link objects that is used by the
    EventListenerContainer.  The order is not so important.  This function
    has to be in the global namespace.
*/
bool operator< (const Link& rLinkA, const Link& rLinkB)
{
    return &rLinkA < &rLinkB;
}

namespace {

/** Base class for abstracted handling of windows that represent the panes
    in which the view shells are displayed.
*/
class PaneWindow
{
public:
    PaneWindow (void);
    virtual void Show (void) = 0;
    virtual void Hide (void) = 0;
    ::Window* GetWindow (void);
    virtual void SetWindow (::Window* pWindow);
    bool IsVisible (void);

protected:
    ::Window* mpWindow;
};




/** Wrapper around the frame window that usually represents the center pane.
*/
class PaneFrameWindow : public PaneWindow
{
public:
    PaneFrameWindow (ViewShellBase& rBase);
    virtual void Show (void);
    virtual void Hide (void);
};



/** Wrapper around the child windows that usually represent the side panes.
*/
class PaneChildWindow : public PaneWindow
{
public:
    PaneChildWindow (ViewShellBase& rBase, USHORT nChildWindowId);
    virtual void Show (void);
    virtual void Hide (void);
    virtual void SetWindow (::Window* pWindow);

private:
    ViewShellBase& mrBase;
    USHORT mnChildWindowId;
    bool mbInitialized;

    void Initialize (void);
};




//===== EventBroadcaster ======================================================

/** The event broadcaster manages the list of listeners that are to informed
    of events regarding the PaneManager.
*/
class EventBroadcaster
{
public:
    EventBroadcaster (ViewShellBase& rBase);

    /** Call the registered event listeners and inform them about an event
        described by the given arguments.  Depending on the type of event
        the event will take place or has already taken place.
        @param eEventId
            Type of the event.
        @param ePane
            The pane in which the event took/will take place.
        @param pShell
            When eEventId is EID_VIEW_SHELL_ADDED then this argument is
            expected to hold a pointer to the added shell.
    */
    void CallEventListeners (
        ViewShellBase& rBase,
        PaneManagerEvent::EventId eEventId,
        PaneManager::PaneType ePane,
        ViewShell* pShell);

    void AddEventListener (const Link& rEventListener);
    void RemoveEventListener (const Link& rEventListener);

private:
    ViewShellBase& mrBase;
    typedef ::std::vector<Link> EventListenerContainer;
    ::std::auto_ptr<EventListenerContainer> mpEventListeners;
};




/** A pane descriptor remembers for a pane the window that represents the
    pane and the view shell that is displayed in the pane/window.
    It manages the creation/switching of view shells.
*/
class PaneDescriptor
{
public:
    PaneDescriptor (
        PaneManager::PaneType mePane,
        ViewShellBase& rBase, 
        EventBroadcaster& rBroadcaster);
    PaneDescriptor (
        PaneManager::PaneType mePane,
        ViewShellBase& rBase, 
        EventBroadcaster& rBroadcaster, 
        USHORT nChildWindowId);
    ViewShell::ShellType GetViewShellType (void) const;
    bool GetWindowVisibility (void) const;

    void SetViewShell (ViewShell* pViewShell);
    ViewShell* GetViewShell (void) const;

    ::Window* GetWindow (void);
    void SetWindow (::Window* pWindow);

    void HideWindow (void);
    void ShowWindow (void);

    /** Replace the view shell in the specified pane by one with the
        requested type.
        @param eType
            The requested type of the new view shell.
        @return
            Returns whether the current shell has been replaced by one of
            the requested type.  Reasons for returning <FALSE/>, i.e. that
            the current shell is not replaced, are that the the current
            shell is already of the requested type or that switching the
            view shell is not possible at the moment.
    */
    bool SetViewShellType (ViewShell::ShellType eType, bool bForce = false);

private:
    PaneManager::PaneType mePane;
    ViewShellBase& mrBase;
    EventBroadcaster& mrBroadcaster;
    ViewShell* mpViewShell;
    ::std::auto_ptr<PaneWindow> mpWindow;

    /** This window is displayed in the center pane in the usually short
        times when one view shell is replaced by another.
    */
    ::std::auto_ptr< ::Window> mpIntermissionWindow;

    void ShutDownShell (void);
    void SetupShell (void);
};




/** A pane configuration describes the desired state of all panes.  There is
    one pane configuration for every type of view shell that can be
    displayed in the center pane.  So when the view shell type of the center
    pane changes the pane configuration for that type is made the current
    one.
*/
class PaneConfiguration
{
public:
    PaneConfiguration (void);
    PaneConfiguration (
        ViewShell::ShellType eLeftShellTye,
        ViewShell::ShellType eCenterShellTye,
        ViewShell::ShellType eRightShellTye);
    PaneConfiguration (const PaneConfiguration& rConfiguration);

    PaneConfiguration& operator= (const PaneConfiguration& rConfiguration);

    void SetViewShellType (PaneManager::PaneType ePane, ViewShell::ShellType eShellType);
    ViewShell::ShellType GetViewShellType (PaneManager::PaneType ePane) const;

    /** Set the desired visibility of the specified pane.
        @return
            Returns whether an update of the UI is necessart to reflect the
            new value of the configuration.  When the given value is the
            same as the current one then <FALSE/> is returned to indicate
            that nothing has to be done.
    */
    bool SetWindowVisibility (PaneManager::PaneType ePane, bool bVisibility);
    bool GetWindowVisibility (PaneManager::PaneType ePane) const;

private:
    class Descriptor {public:
        Descriptor(ViewShell::ShellType eViewShellType ,bool bWindowVisibility)
            : meViewShellType(eViewShellType),mbWindowVisibility(bWindowVisibility)
        {}
        Descriptor(const Descriptor& rDescriptor)
            : meViewShellType(rDescriptor.meViewShellType),
              mbWindowVisibility(rDescriptor.mbWindowVisibility)
        {}
        Descriptor& operator= (const Descriptor& rDescriptor)
        {
            meViewShellType = rDescriptor.meViewShellType;
            mbWindowVisibility = rDescriptor.mbWindowVisibility;
            return *this;
        }
        ViewShell::ShellType meViewShellType;
        bool mbWindowVisibility;
    };
    Descriptor maLeftPane;
    Descriptor maCenterPane;
    Descriptor maRightPane;

    const Descriptor& GetDescriptor (PaneManager::PaneType ePane) const;
    Descriptor& GetDescriptor (PaneManager::PaneType ePane);
};
typedef ::std::map<ViewShell::ShellType,PaneConfiguration> ConfigurationList;


class ReentranceGuard
{
public:
    ReentranceGuard(bool& rFlag):mrFlag(rFlag) {mrFlag = true;}
    ~ReentranceGuard(void) {mrFlag = false;}
private:
    bool& mrFlag;
};


} // end of anonymous namespace

namespace sd {



//===== PaneManager::Implementation ===========================================

/** Implementation class of the PaneManager.  For the documentation of most
    methods and members see PaneManager class.

    This class may look confusing at first sight.  Well, probably it is
    confusing.  The reason for this is that it has several responsibilities
    which are to a certain degree intertwined:
    1. Management of pane configuration.
    2. Management of individual panes.
    3. Synchronous and asynchronous shell switching.
    4. Visibility of regular windows and child windows.
    5. Management of slots (process slot calls and return slot stati) that
    are related to pane windows and shells.

    If time permits these responsibilities may be separated into individual
    classes in the future.

    In here we have to handle the case that a pane type is not available.
    At the current time this is the case with the task pane that must not be
    displayed when the document is read only.  To handle this properly there
    are the two methods GetRequestedShellType() and
    GetRequestedWindowVisibility() which have to be called instead of
    accessing the current pane configuration directly.
*/
class PaneManager::Implementation
{
public:
    Implementation (
        ViewShellBase& rBase, 
        ViewShell::ShellType eInitialConfiguration);
    ~Implementation (void);

    ViewShell* CreateViewShell (
        PaneType ePane, 
        ViewShell::ShellType eShellType);

    /** Return the descriptor for the specified pane.
    */
    const PaneDescriptor* GetPaneDescriptor (PaneType ePane) const;
    PaneDescriptor* GetPaneDescriptor (PaneType ePane);

    void Shutdown (bool bForgetOnly);

    ViewShellBase& GetViewShellBase (void) const;
    ViewShell::ShellType GetCurrentViewShellType (PaneType ePane) const;
    ViewShell* GetViewShell (PaneType ePane) const;
    PaneManager::PaneType GetPaneOfViewShell (ViewShell* pViewShell) const;
    void SetWindow (
        PaneType ePane, 
        ::Window* pWindow,
        CallMode eCallMode);
    ::Window* GetWindow (PaneType ePane) const;
    String GetWindowTitle (PaneType ePane) const;
    ResId GetDockingWindowTitle (PaneType ePane) const;
    void HandleModeChangeSlot (ULONG nSlotId, SfxRequest& rRequest);

    /** Return the type of the view shell that the current pane
        configuration wants to have displayed in the specified pane.  The
        value of the current configuration but is adapted in this method
        when the specified pane is not available.
    */
    ViewShell::ShellType GetRequestedViewShellType (PaneType ePane) const;
    /** Return the visibility of the specified pane as requested by the
        current configuration.  The value of the current configuration but
        is adapted in this method when the specified pane is not available.
    */
    bool GetRequestedWindowVisibility (PaneType ePane) const;

    bool SwitchConfiguration (
        ViewShell::ShellType eType);

    bool SwitchViewShell (
        PaneType ePane,
        ViewShell::ShellType eType);

    void RequestUpdateConfiguration (CallMode eCallMode);

    PaneConfiguration& GetPaneConfiguration (
        ViewShell::ShellType eCenterPaneShellType);

    /** Return the configuration that currently is in use.  This
        configuration may not yet be visible when the last request to change
        it has been made asynchronously and the update has not yet been
        made.
        @return
            The returned pointer is NULL only in rare cases.  When the pane
            manager has not yet been properly initialized, it has been shut
            down, or the internal pointer has been incorrectly set.
    */
    PaneConfiguration* GetCurrentConfiguration (void) const;

    /** The second part of an asynchronous shell change.  This method is
        called as response to an SID_SWITCH_SHELL slot call (these slot
        calls are received by ViewShellBase and forwarded here.)
    */
    void ExecuteModeChange (SfxRequest& rRequest);

    /** Execute slot calls delegated to the pane manager by a ViewShellBase
        object. Currently we process only slot calls that a) change the
        visibility of the pane windows or b) switch shells in panes.
    */
    void ExecuteSlot (SfxRequest& rRequest);
    void GetSlotState (SfxItemSet& rSet);

    EventBroadcaster& GetEventBroadcaster (void);

    ::std::auto_ptr<PaneManagerState> GetState (void) const;
    void SetState (const PaneManagerState& rState);

private:
    ViewShellBase& mrBase;

    /** Remember in this flag whether a PaneManager object has been
        initialized.  Initilization is done in InitPanes() when the
        ViewShellBase is on the shell stack and can process slot calls.
    */
    bool mbIsInitialized;

    EventBroadcaster maEventBroadcaster;

    ConfigurationList maConfigurationList;

    ViewShell::ShellType meDesiredPaneConfiguration;
    PaneConfiguration* mpCurrentConfiguration;

    PaneDescriptor maLeftPane;
    PaneDescriptor maCenterPane;
    PaneDescriptor maRightPane;

    /** This flag is true while UpdateConfiguration is being executed and
        serves to prevent recursive execution.
    */
    bool mbIsConfigurationUpdateRunning;

    /** Set the visibility state of the specified child window.  When the given
        request has an argument then set the visibility accordingly.  Otherwise
        toggle the visiblity.
    */
    void SetChildWindowVisibility(
        USHORT nSlotId,
        USHORT nChildWindowId,
        SfxRequest& rRequest,
        PaneType ePane);
    /** Show or hide the specified pane according to the given flag.  For
        the side panes we can show or hide the child windows.  For the
        center pane we can hide or show that window.
    */
    void SetupPaneVisibility (
        PaneType ePane,
        bool bShow);

    /** Return whether the specified pane is available for display.
    */
    bool IsPaneAvailable (PaneType ePane);

    bool IsUpToDate (PaneType ePane);
    bool IsUpToDate (void);


    /** Call this method to make the current configuration reflect the
        desired values.
    */
    void UpdateConfiguration (void);

    /** Set the configuration of shell types that are associated with the
        type of view shell of the center pane.
    */
    void SetConfiguration (
        ViewShell::ShellType eCenterShellType,
        bool bLeftPaneVisible,
        bool bRightPaneVisible);
};


/** Implementation class of PaneManagerState whose internals are known only
    to PaneManager, not to PaneManagerState.
*/
class PaneManager::StateImplementation
{
public:
    ConfigurationList maConfigurationList;
    ViewShell::ShellType meCurrentConfiguration;

    StateImplementation(
        const ConfigurationList& rConfigurationList,
        ViewShell::ShellType eCurrentConfiguration)
        : maConfigurationList(rConfigurationList),
          meCurrentConfiguration(eCurrentConfiguration)
    {}
    StateImplementation (const StateImplementation& rState)
        : maConfigurationList(rState.maConfigurationList),
          meCurrentConfiguration(rState.meCurrentConfiguration)
    {}
};

} // end of namespace sd




//===== PaneManager ===========================================================

namespace sd {

PaneManager::PaneManager (
    ViewShellBase& rBase,
    ViewShell::ShellType eType)
    : mpImpl (new Implementation (rBase, eType)),
      mbValid (true)
{
}




PaneManager::~PaneManager (void)
{
    Shutdown ();
}




void PaneManager::LateInit (void)
{
    // Try to init the panes for which the windows are already present.
    // This usually includes the center pane but not the panes inside
    // dockable child windows.
    if (mbValid)
        mpImpl->RequestUpdateConfiguration(PaneManager::CM_SYNCHRONOUS);
}




void PaneManager::InitPanes (void)
{
    if (mbValid)
        mpImpl->RequestUpdateConfiguration(PaneManager::CM_ASYNCHRONOUS);
}




void PaneManager::Shutdown (bool bForgetOnly)
{
    if (mbValid)
    {
        mpImpl->Shutdown(bForgetOnly);
        mbValid = false;
    }
}




ViewShell::ShellType PaneManager::GetViewShellType (PaneType ePane) const
{
    if (mbValid)
        return mpImpl->GetCurrentViewShellType(ePane);
    else
        return ViewShell::ST_NONE;
}




ViewShell* PaneManager::GetViewShell (PaneType ePane) const
{
    if (mbValid)
        return mpImpl->GetViewShell(ePane);
    else
        return NULL;
}




PaneManager::PaneType PaneManager::GetPaneOfViewShell (
    ViewShell* pViewShell) const
{
    if (mbValid)
        return mpImpl->GetPaneOfViewShell (pViewShell);
    else
        return PT_NONE;
}




void PaneManager::SetWindow (
    PaneType ePane, 
    ::Window* pWindow, 
    CallMode eCallMode)
{
    if (mbValid)
        mpImpl->SetWindow (ePane, pWindow, eCallMode);
}




::Window* PaneManager::GetWindow (PaneType ePane) const
{
    if (mbValid)
        return mpImpl->GetWindow (ePane);
    else
        return NULL;
}




String PaneManager::GetWindowTitle (PaneType ePane) const
{
    if (mbValid)
        return mpImpl->GetWindowTitle (ePane);
    else
        return String();
}




ResId PaneManager::GetDockingWindowTitle (PaneType ePane) const
{
    if (mbValid)
        return mpImpl->GetDockingWindowTitle (ePane);
    else
        throw ::com::sun::star::uno::RuntimeException(
            ::rtl::OUString::createFromAscii(
                "PaneManager::GetDockingWindowTitle() called, but PaneManager is invalid"),
            NULL);
}




bool PaneManager::RequestWindowVisibilityChange (
    PaneType ePane, 
    bool bVisible,
    CallMode eCallMode)
{
    bool bUpdateNecessary (false);
    if (mbValid)
    {
        bUpdateNecessary = mpImpl->GetCurrentConfiguration()->SetWindowVisibility(
            ePane,
            bVisible);
        if (bUpdateNecessary)
            mpImpl->RequestUpdateConfiguration(eCallMode);
    }
    return bUpdateNecessary;
}




void PaneManager::HandleModeChangeSlot (ULONG nSlotId, SfxRequest& rRequest)
{
    if (mbValid)
        mpImpl->HandleModeChangeSlot (nSlotId, rRequest);
}




bool PaneManager::RequestMainViewShellChange (
    ViewShell::ShellType eType,
    CallMode eCallMode)
{
    bool bUpdateNecessary (false);
    if (mbValid)
    {
        bUpdateNecessary = mpImpl->SwitchConfiguration(eType);
        if (bUpdateNecessary)
            mpImpl->RequestUpdateConfiguration(eCallMode);
    }

    return bUpdateNecessary;
}




bool PaneManager::RequestViewShellChange(
    PaneType ePane, 
    ViewShell::ShellType eType,
    CallMode eCallMode)
{
    bool bUpdateNecessary (false);
    if (mbValid)
    {
        bUpdateNecessary = mpImpl->SwitchViewShell(ePane,eType);
        bUpdateNecessary |= mpImpl->GetCurrentConfiguration()->SetWindowVisibility(
            ePane,
            eType!=ViewShell::ST_NONE);
        if (bUpdateNecessary)
            mpImpl->RequestUpdateConfiguration(eCallMode);
    }
    return bUpdateNecessary;
}




void PaneManager::ExecuteModeChange (SfxRequest& rRequest)
{
    if (mbValid)
        mpImpl->ExecuteModeChange (rRequest);
}




void PaneManager::ExecuteSlot (SfxRequest& rRequest)
{
    if (mbValid)
        mpImpl->ExecuteSlot (rRequest);
}




void PaneManager::GetSlotState (SfxItemSet& rSet)
{
    if (mbValid)
        mpImpl->GetSlotState (rSet);
}



void PaneManager::AddEventListener (const Link& rEventListener)
{
    if (mbValid)
        mpImpl->GetEventBroadcaster().AddEventListener (rEventListener);
}




void PaneManager::RemoveEventListener (const Link& rEventListener)
{
    if (mbValid)
        mpImpl->GetEventBroadcaster().RemoveEventListener (rEventListener);
}




::std::auto_ptr<PaneManagerState> PaneManager::GetState (void) const
{
    if (mbValid)
        return mpImpl->GetState();
    else
        return ::std::auto_ptr<PaneManagerState>(NULL);
}




void PaneManager::SetState (const PaneManagerState& rState)
{
    if (mbValid)
        mpImpl->SetState(rState);
}




::std::vector<PaneManager::PaneType> PaneManager::GetPaneList (void) const
{
    ::std::vector<PaneType> aList;
    aList.push_back(PT_CENTER);
    aList.push_back(PT_LEFT);
    aList.push_back(PT_RIGHT);
    return aList;
}



} // end of namespace sd


namespace {

//===== PaneWindow ============================================================

PaneWindow::PaneWindow (void)
    : mpWindow(NULL)
{
}




::Window* PaneWindow::GetWindow (void)
{
    return mpWindow;
}




void PaneWindow::SetWindow (::Window* pWindow)
{
    mpWindow = pWindow;
}




bool PaneWindow::IsVisible (void)
{
    if (mpWindow != NULL)
        return mpWindow->IsReallyVisible();
    else
        return false;
}




//===== PaneFrameWindow =======================================================

PaneFrameWindow::PaneFrameWindow (ViewShellBase& rBase)
{
    mpWindow = &rBase.GetFrame()->GetWindow();
}




void PaneFrameWindow::Show (void)
{
    if (mpWindow != NULL)
        mpWindow->Show();
}




void PaneFrameWindow::Hide (void)
{
    if (mpWindow != NULL)
        mpWindow->Hide();
}




//===== PaneChildWindow =======================================================

PaneChildWindow::PaneChildWindow (ViewShellBase& rBase, USHORT nChildWindowId)
    : mrBase(rBase),
      mnChildWindowId(nChildWindowId),
      mbInitialized(false)
{
	mrBase.GetViewFrame()->SetChildWindow(mnChildWindowId, FALSE);
    Initialize();
}




void PaneChildWindow::Initialize (void)
{
    if ( ! mbInitialized
        && mrBase.GetViewFrame()->HasChildWindow(mnChildWindowId))
    {
        SfxChildWindow* pChildWindow = mrBase.GetViewFrame()->GetChildWindow(mnChildWindowId);
        if (pChildWindow != NULL)
        {
            pChildWindow->SetHideNotDelete(sal_True);
            pChildWindow->SetHideAtToggle(sal_True);

            mpWindow = pChildWindow->GetWindow();

            mbInitialized = true;
        }
    }
}




void PaneChildWindow::SetWindow (::Window* pWindow)
{
    PaneWindow::SetWindow(pWindow);
    if (pWindow != NULL)
    {
        mbInitialized = false;
        Initialize();
    }
}




void PaneChildWindow::Show (void)
{
    if ( ! mbInitialized)
        Initialize();

        // Make the child window visible.
#ifdef DESTROY_UNUSED_PANES
    mrBase.GetViewFrame()->SetChildWindow(mnChildWindowId, TRUE);
#else
    if (mrBase.GetViewFrame()->KnowsChildWindow(mnChildWindowId))
        mrBase.GetViewFrame()->SetChildWindow(mnChildWindowId, TRUE);
    //    mrBase.GetViewFrame()->ShowChildWindow(mnChildWindowId, TRUE);
#endif

    // Get the actual window pointer.
    SfxChildWindow* pChildWindow = mrBase.GetViewFrame()->GetChildWindow(mnChildWindowId);
    if (pChildWindow != NULL)
        mpWindow = pChildWindow->GetWindow();

    if (mpWindow != NULL)
        mpWindow->Show();
}




void PaneChildWindow::Hide (void)
{
    if ( ! mbInitialized)
        Initialize();

    if (mpWindow != NULL)
        mpWindow->Hide();

    // Hide the child window.
#ifdef DESTROY_UNUSED_PANES
    mrBase.GetViewFrame()->SetChildWindow(mnChildWindowId, FALSE);
#else
    if (mrBase.GetViewFrame()->KnowsChildWindow(mnChildWindowId))
        mrBase.GetViewFrame()->SetChildWindow(mnChildWindowId, FALSE);
    //    mrBase.GetViewFrame()->ShowChildWindow(mnChildWindowId, FALSE);
#endif

    // Get the actual window pointer.
    SfxChildWindow* pChildWindow = mrBase.GetViewFrame()->GetChildWindow(mnChildWindowId);
    if (pChildWindow != NULL)
        mpWindow = pChildWindow->GetWindow();
}




//===== EventBroadcaster ======================================================

EventBroadcaster::EventBroadcaster (ViewShellBase& rBase)
    : mrBase(rBase),
      mpEventListeners (new EventListenerContainer())
{
}



void EventBroadcaster::CallEventListeners (
    ViewShellBase& rBase,
    PaneManagerEvent::EventId eEventId,
    PaneManager::PaneType ePane,
    ViewShell* pShell)
{
    PaneManagerEvent aEvent (rBase, eEventId, ePane, pShell);
    // Create a copy of the current list of event listeners and iterate over
    // that because as reaction to some listener calls the list of listeners
    // may change and thus invalidate the iterator.
    EventListenerContainer aListenersCopy (*mpEventListeners);
    EventListenerContainer::iterator aLink (aListenersCopy.begin());
    EventListenerContainer::iterator aEnd (aListenersCopy.end());
    while (aLink!=aEnd)
    {
        aLink->Call (&aEvent);
        ++aLink;
    }
}




void EventBroadcaster::AddEventListener (
    const Link& rEventListener)
{
    if (::std::find (
        mpEventListeners->begin(), 
        mpEventListeners->end(),
        rEventListener) == mpEventListeners->end())
    {
        mpEventListeners->push_back (rEventListener);
    }
}




void EventBroadcaster::RemoveEventListener (
    const Link& rEventListener)
{
    mpEventListeners->erase (
        ::std::find (
            mpEventListeners->begin(),
            mpEventListeners->end(),
            rEventListener));
}




//===== PaneDescriptor ========================================================

PaneDescriptor::PaneDescriptor (
    PaneManager::PaneType ePane,
    ViewShellBase& rBase,
    EventBroadcaster& rBroadcaster)
    : mePane(ePane),
      mrBase(rBase),
      mrBroadcaster(rBroadcaster),
      mpViewShell(NULL),
      mpWindow(new PaneFrameWindow(rBase)),
      mpIntermissionWindow(NULL)
{
    if (mePane == PaneManager::PT_CENTER)
    {
        mpIntermissionWindow.reset (new ::Window(mpWindow->GetWindow()));
        mpIntermissionWindow->SetBackground(Wallpaper(COL_RED));
    }
}




PaneDescriptor::PaneDescriptor (
    PaneManager::PaneType ePane,
    ViewShellBase& rBase, 
    EventBroadcaster& rBroadcaster,
    USHORT nChildWindowId)
    : mePane(ePane),
      mrBase(rBase),
      mrBroadcaster(rBroadcaster),
      mpViewShell(NULL),
      mpWindow(new PaneChildWindow(rBase, nChildWindowId)),
      mpIntermissionWindow(NULL)
{
}




ViewShell::ShellType PaneDescriptor::GetViewShellType (void) const
{
    if (mpViewShell == NULL)
        return ViewShell::ST_NONE;
    else
        return mpViewShell->GetShellType();
}




bool PaneDescriptor::GetWindowVisibility (void) const
{
    if (mpWindow.get() == NULL)
        return false;
    else
        return mpWindow->IsVisible();
}




void PaneDescriptor::SetViewShell (ViewShell* pViewShell)
{
    mpViewShell = pViewShell;
}




ViewShell* PaneDescriptor::GetViewShell (void) const
{
    return mpViewShell;
}




::Window* PaneDescriptor::GetWindow (void)
{
    return mpWindow->GetWindow();
}




void PaneDescriptor::SetWindow (::Window* pWindow)
{
    SetViewShellType (ViewShell::ST_NONE);
    mpWindow->SetWindow(pWindow);
}




void PaneDescriptor::HideWindow (void)
{
    mpWindow->Hide();
}



void PaneDescriptor::ShowWindow (void)
{
    mpWindow->Show();
}




/** This method performs several tasks:
    1.) Connect the new main sub shell to the document shell.
    2.) Stack the new main sub shell and (implicitly) its relevant
    objects bars onto the sub shell stack.
    3.) Create and initialize the new main sub shell.
*/
bool PaneDescriptor::SetViewShellType (ViewShell::ShellType eShellType, bool bForceChange)
{
    if (bForceChange
        || (mpViewShell==NULL && eShellType!=ViewShell::ST_NONE)
        || (mpViewShell!=NULL && mpViewShell->GetShellType()!=eShellType))
    {
        // 1. Shut down the current shell in the pane.

        // Tell listeners that the view shell is about to be removed.
        mrBroadcaster.CallEventListeners (
            mrBase,
            PaneManagerEvent::EID_VIEW_SHELL_REMOVED, 
            mePane, 
            mpViewShell);

        // Remember the frame view of the current main view shell.
        FrameView* pFrameView = NULL;
        if (mpViewShell != NULL)
            pFrameView = mpViewShell->GetFrameView();
        if (pFrameView != NULL)
            // Connect to it to prevent its deletion when the ViewShell is
            // destroyed.
            pFrameView->Connect();

        // Lock the view shell manager so that the view shells can safely be
        // destroyed and created.
        ViewShellManager::UpdateLock aLock (mrBase.GetViewShellManager());

        ShutDownShell();


        // 2. Create the new view shell.

        // Switch to and activate the new current main view shell.
        mpViewShell = mrBase.GetViewShellManager().ActivateViewShell (
            eShellType,
            mpWindow->GetWindow(),
            pFrameView);


        // 3. Setup the new view shell in the pane.

        SetupShell();

        // We can no safely disconnect from the frame view without
        // accidentally deleting it.
        if (pFrameView != NULL)
            pFrameView->Disconnect();

        mrBroadcaster.CallEventListeners (
            mrBase,
            PaneManagerEvent::EID_VIEW_SHELL_ADDED, 
            mePane, 
            mpViewShell);

        // Tell the caller that the view shell in this pane has successfully
        // been replaced.
        return true;
    }
    else
        return false;
}




void PaneDescriptor::ShutDownShell (void)
{
    // Lock the view shell manager so that the view shells can safely be
    // destroyed and created.
    ViewShellManager::UpdateLock aLock (mrBase.GetViewShellManager());

    ViewShell* pCurrentViewShell = mpViewShell;

    if (pCurrentViewShell != NULL)
    {
        // Turn off effects.
        if (pCurrentViewShell->ISA(DrawViewShell))
        {
            if (pCurrentViewShell->GetSlideShow() != NULL)
            {
                // Turn off effects.
                static_cast<DrawView*>(pCurrentViewShell->GetDrawView())->SetAnimationMode(FALSE);
            }
        }
        else if (pCurrentViewShell->ISA(OutlineViewShell))
        {
            // #116350# update model from outliner (as for SID_SAVEDOC etc.)
            pCurrentViewShell->PrepareClose();
        }

        if (mePane == PaneManager::PT_CENTER)
        {
            // We are switching from one shell to another.  Prevent the
            // controller from destroying the frame when it is disposed.
            DrawController* pController = pCurrentViewShell->GetController();
            if (pController != NULL)
                pController->FrameIsReleasedWithController (sal_False);
        }

        // Shut down the current view shell.
        pCurrentViewShell->Shutdown ();
        mrBase.GetDocShell()->Disconnect(pCurrentViewShell);
        mrBase.GetViewShellManager().DeactivateViewShell(pCurrentViewShell);

        // There has to be a window in the SfxViewShell at all times.
        // Therefore we create a simple one when the view shell of the
        // center pane is destroyed.
        if (mePane == PaneManager::PT_CENTER)
        {
            mpIntermissionWindow->Show();
            mrBase.SetWindow (mpIntermissionWindow.get());
        }
        mpViewShell = NULL;
    }
    else
    {
        if (mrBase.GetController().is() && mePane==PaneManager::PT_CENTER)
        {
            SfxBaseController* pController = static_cast<SfxBaseController*>(
                mrBase.GetController().get());
            if (pController != NULL)
                pController->FrameIsReleasedWithController (sal_False);
        }
    }
}




void PaneDescriptor::SetupShell (void)
{
    if (mpViewShell != NULL)
    {
        // When the intermission window is shown in the center pane then
        // remove it.
        if (mePane == PaneManager::PT_CENTER)
            mpIntermissionWindow->Hide();

        mpViewShell->SetIsMainViewShell(mePane==PaneManager::PT_CENTER);
        mpViewShell->Init ();

        if (mePane == PaneManager::PT_CENTER)
        {
            mrBase.GetDocShell()->Connect (mpViewShell);

            // Now that the controller has been set at the frame we can
            // switch back on the feature that the frame is destroyed when
            // the controller is disposed.  This has to be done when the
            // document window is closed.
            DrawController* pController = mpViewShell->GetController();
            if (pController != NULL)
                pController->FrameIsReleasedWithController (sal_True);
        }

        // During the creation of the new sub-shell resize requests were
        // not forwarded to it because it was not yet registered.
        // Therefore, we have to request a resize now.
        if (mePane == PaneManager::PT_CENTER)
        {
            mpViewShell->UIFeatureChanged();
            if (mrBase.GetDocShell()->IsInPlaceActive())
                mrBase.GetViewFrame()->Resize(TRUE);
        }
        else
            GetWindow()->Resize();
    }
}




//===== PaneConfiguration ======================================================

PaneConfiguration::PaneConfiguration (void)
    : maLeftPane(ViewShell::ST_NONE, false),
      maCenterPane(ViewShell::ST_NONE, false),
      maRightPane(ViewShell::ST_NONE, false)
{
}




PaneConfiguration::PaneConfiguration (
    ViewShell::ShellType eLeftShellType,
    ViewShell::ShellType eCenterShellType,
   ViewShell::ShellType eRightShellType)
    : maLeftPane(eLeftShellType, eLeftShellType!=ViewShell::ST_NONE),
      maCenterPane(eCenterShellType, eCenterShellType!=ViewShell::ST_NONE),
      maRightPane(eRightShellType, eRightShellType!=ViewShell::ST_NONE)
{
}




PaneConfiguration::PaneConfiguration (const PaneConfiguration& rConfiguration)
    : maLeftPane(rConfiguration.maLeftPane),
      maCenterPane(rConfiguration.maCenterPane),
      maRightPane(rConfiguration.maRightPane)
{
}




PaneConfiguration& PaneConfiguration::operator= (const PaneConfiguration& rConfiguration)
{
    maLeftPane = rConfiguration.maLeftPane;
    maCenterPane = rConfiguration.maCenterPane;
    maRightPane = rConfiguration.maRightPane;

    return *this;
}




void PaneConfiguration::SetViewShellType (
    PaneManager::PaneType ePane,
    ViewShell::ShellType eViewShellType)
{
    Descriptor& rDescriptor (GetDescriptor(ePane));
    rDescriptor.meViewShellType = eViewShellType;
}




ViewShell::ShellType PaneConfiguration::GetViewShellType (
    PaneManager::PaneType ePane) const
{
    const Descriptor& rDescriptor (GetDescriptor(ePane));
    return rDescriptor.meViewShellType;
}




bool PaneConfiguration::SetWindowVisibility (
    PaneManager::PaneType ePane,
    bool bVisibility)
{
    bool bUpdateNecessary (false);
    Descriptor& rDescriptor (GetDescriptor(ePane));
    bUpdateNecessary = (rDescriptor.mbWindowVisibility != bVisibility);
    rDescriptor.mbWindowVisibility = bVisibility;
    return bUpdateNecessary;
}




bool PaneConfiguration::GetWindowVisibility (
    PaneManager::PaneType ePane) const
{
    const Descriptor& rDescriptor (GetDescriptor(ePane));
    return rDescriptor.mbWindowVisibility;
}




const PaneConfiguration::Descriptor& PaneConfiguration::GetDescriptor (
    PaneManager::PaneType ePane) const
{
    return const_cast<PaneConfiguration*>(this)->GetDescriptor(ePane);
}




PaneConfiguration::Descriptor& PaneConfiguration::GetDescriptor (
    PaneManager::PaneType ePane)
{
    switch (ePane)
    {
        case PaneManager::PT_LEFT:
            return maLeftPane;

        case PaneManager::PT_CENTER:
        case PaneManager::PT_MAIN:
            return maCenterPane;

        case PaneManager::PT_RIGHT:
            return maRightPane;

        default:
            OSL_TRACE(false, "PaneConfiguration::GetDescriptor() called for invalid pane id");
            return maCenterPane;
    }
}



} // end of anonymous namespace




namespace sd {

//===== PaneManagerEvent ======================================================

PaneManagerEvent::PaneManagerEvent (
    const ViewShellBase& rBase,
    EventId eEventId,
    PaneManager::PaneType ePane,
    ViewShell* pShell)
    : mrBase(rBase),
      meEventId(eEventId),
      mePane(ePane),
      mpShell(pShell)
{
}




//===== PaneManager::Implementation ===========================================

PaneManager::Implementation::Implementation (
    ViewShellBase& rBase,
    ViewShell::ShellType eInitialConfiguration)
    : mrBase(rBase),
      mbIsInitialized (false),
      maEventBroadcaster(rBase),
      meDesiredPaneConfiguration(eInitialConfiguration),
      mpCurrentConfiguration(NULL),
      maLeftPane(PT_LEFT, rBase, maEventBroadcaster, ::sd::LeftPaneChildWindow::GetChildWindowId()),
      maCenterPane(PT_CENTER, rBase, maEventBroadcaster),
      maRightPane(PT_RIGHT, rBase, maEventBroadcaster, ::sd::RightPaneChildWindow::GetChildWindowId()),
      mbIsConfigurationUpdateRunning(false)
{
    // The fall-back configuration.  Used for shutting down the PaneManager.
    SetConfiguration (ViewShell::ST_NONE, false, false);

    // Several configurations for Impress.
    SetConfiguration(ViewShell::ST_IMPRESS, true, true);
    SetConfiguration(ViewShell::ST_OUTLINE, true, false);
    SetConfiguration(ViewShell::ST_NOTES, true, true);
    SetConfiguration(ViewShell::ST_HANDOUT, false, true);
    SetConfiguration(ViewShell::ST_SLIDE_SORTER, false, true);

    // Draw has only one configuration.
    SetConfiguration(ViewShell::ST_DRAW, true, false);

    // The full screen presentation.
    SetConfiguration(ViewShell::ST_PRESENTATION, false, false);

    mpCurrentConfiguration = &GetPaneConfiguration(eInitialConfiguration);

    MasterPageObserver::Instance().RegisterDocument (*mrBase.GetDocShell()->GetDoc());
}




void PaneManager::Implementation::SetConfiguration (
    ViewShell::ShellType eCenterShellType,
    bool bLeftPaneVisible,
    bool bRightPaneVisible)
{
    // Set the type of the view shells int the left and right panes to
    // values that avoid construction and destruction of their view shells
    // when possible.  As a result ST_NONE is used only for the full screen
    // presentation and the configuration used for shutting down the pane
    // manager.  For all other configurations the slide sorter and task pane
    // are used respectively, regardless of whether they are visible or not.
    ViewShell::ShellType eLeftShellType;
    ViewShell::ShellType eRightShellType;
    switch (eCenterShellType)
    {
        case ViewShell::ST_NONE:
        case ViewShell::ST_PRESENTATION:
            eLeftShellType = ViewShell::ST_NONE;
            eRightShellType = ViewShell::ST_NONE;
            break;

        default:
            eLeftShellType = ViewShell::ST_SLIDE_SORTER;
            eRightShellType = ViewShell::ST_TASK_PANE;
            break;
    }

    maConfigurationList[eCenterShellType] = PaneConfiguration(
        eLeftShellType,
        eCenterShellType,
        eRightShellType);
    maConfigurationList[eCenterShellType].SetWindowVisibility(PT_LEFT, bLeftPaneVisible);
    maConfigurationList[eCenterShellType].SetWindowVisibility(PT_RIGHT, bRightPaneVisible);
}




PaneManager::Implementation::~Implementation (void)
{
    MasterPageObserver::Instance().UnregisterDocument (
        *mrBase.GetDocShell()->GetDoc());
}




void PaneManager::Implementation::Shutdown (bool bForgetOnly)
{
    SwitchConfiguration(ViewShell::ST_NONE);

    if (bForgetOnly)
    {
        // The shells are shut down at another place, presumably the
        // ViewShellManager::Shutdown() method.  Simply reset our internal
        // structurs to forget about shells and windows.

        PaneDescriptor& rLeftPane (*GetPaneDescriptor(PT_LEFT));
        rLeftPane.SetViewShell(NULL);
        rLeftPane.SetWindow(NULL);

        PaneDescriptor& rCenterPane (*GetPaneDescriptor(PT_CENTER));
        rCenterPane.SetViewShell(NULL);
        rCenterPane.SetWindow(NULL);

        PaneDescriptor& rRightPane (*GetPaneDescriptor(PT_RIGHT));
        rRightPane.SetViewShell(NULL);
        rRightPane.SetWindow(NULL);
    }
    else
    {
        // Use the regular way to shut down the shells and panes.
        RequestUpdateConfiguration (PaneManager::CM_SYNCHRONOUS);
    }

    maEventBroadcaster.CallEventListeners (
        mrBase,
        PaneManagerEvent::EID_PANE_MANAGER_DYING, 
        PT_NONE, 
        NULL);
}




const PaneDescriptor* PaneManager::Implementation::GetPaneDescriptor (PaneType ePane) const
{
    return const_cast<PaneManager::Implementation*>(this)->GetPaneDescriptor(ePane);
}




PaneDescriptor* PaneManager::Implementation::GetPaneDescriptor (PaneType ePane)
{
    switch (ePane)
    {
        case PT_LEFT:
            return &maLeftPane;

        case PT_CENTER:
        case PT_MAIN:
            return &maCenterPane;

        case PT_RIGHT:
            return &maRightPane;

        default:
            return NULL;
    }
}




ViewShell::ShellType PaneManager::Implementation::GetCurrentViewShellType (
    PaneType ePane) const
{
    ViewShell::ShellType eType (ViewShell::ST_NONE);

    const PaneDescriptor* pPane = GetPaneDescriptor (ePane);
    if (pPane != NULL)
        eType = pPane->GetViewShellType();

    return eType;
}




ViewShell* PaneManager::Implementation::GetViewShell (PaneType ePane) const
{
    ViewShell* pShell = NULL;

    const PaneDescriptor* pPane = GetPaneDescriptor (ePane);
    if (pPane != NULL)
        pShell = pPane->GetViewShell();

    return pShell;
}




ViewShell::ShellType PaneManager::Implementation::GetRequestedViewShellType (
    PaneType ePane) const
{
    ViewShell::ShellType eRequestedType (mpCurrentConfiguration->GetViewShellType(ePane));
    ViewShell::ShellType eRequestedMainType (mpCurrentConfiguration->GetViewShellType(PT_CENTER));

    // The task pane is not shown when either the document is read only or
    // when it is a draw document.
    if (eRequestedType == ViewShell::ST_TASK_PANE
        && (mrBase.GetDocShell()->IsReadOnly() || eRequestedMainType == ViewShell::ST_DRAW))
    {
        eRequestedType = ViewShell::ST_NONE;
    }

    return eRequestedType;
}




bool PaneManager::Implementation::GetRequestedWindowVisibility (PaneType ePane) const
{
    ViewShell::ShellType eRequestedType (mpCurrentConfiguration->GetViewShellType(ePane));
    ViewShell::ShellType eRequestedMainType (mpCurrentConfiguration->GetViewShellType(PT_CENTER));
    bool bRequestedVisibility (mpCurrentConfiguration->GetWindowVisibility(ePane));

    // The task pane is not shown when either the document is read only or
    // when it is a draw document.
    if (eRequestedType == ViewShell::ST_TASK_PANE
        && (mrBase.GetDocShell()->IsReadOnly() || eRequestedMainType == ViewShell::ST_DRAW))
    {
        bRequestedVisibility = false;
    }
    return bRequestedVisibility;
}




PaneManager::PaneType PaneManager::Implementation::GetPaneOfViewShell (
    ViewShell* pViewShell) const
{
    if (pViewShell == GetViewShell(PT_CENTER))
        return PT_CENTER;
    else if (pViewShell == GetViewShell(PT_LEFT))
        return PT_LEFT;
    else if (pViewShell == GetViewShell(PT_RIGHT))
        return PT_RIGHT;
    else
        return PT_NONE;
}




void PaneManager::Implementation::SetWindow (
    PaneType ePane, 
    ::Window* pWindow,
    CallMode eCallMode)
{
    PaneDescriptor* pPane = GetPaneDescriptor(ePane);
    if (pPane != NULL)
    {
        pPane->SetWindow(pWindow);
        RequestUpdateConfiguration(CM_ASYNCHRONOUS);
    }
}




::Window* PaneManager::Implementation::GetWindow (PaneType ePane) const
{
    ::Window* pWindow = NULL;

    const PaneDescriptor* pPane = GetPaneDescriptor (ePane);
    if (pPane != NULL)
        pWindow = const_cast<PaneDescriptor*>(pPane)->GetWindow();

    return pWindow;
}




String PaneManager::Implementation::GetWindowTitle (PaneType ePane) const
{
    String sTitle;

    switch (ePane)
    {
        case PT_LEFT:
            // The title of the left pane depends on the application.  We
            // simply interpret the type of the view shell in the center
            // pane: ST_DRAW indicates DRAW, all other indicate Impress.
            if (GetCurrentViewShellType(PT_CENTER) == ViewShell::ST_DRAW)
                sTitle = SdResId(STR_LEFT_PANE_DRAW_TITLE);
            else
                sTitle = SdResId(STR_LEFT_PANE_IMPRESS_TITLE);
            break;

        case PT_CENTER:
        case PT_MAIN:
            // The center pane has no visible title bar so we do not bother
            // to create a title.
            break;

        case PT_RIGHT:
            // The title of the right pane is always the same.
            sTitle = SdResId(STR_RIGHT_PANE_TITLE);
            break;
    }

    return sTitle;
}




ResId PaneManager::Implementation::GetDockingWindowTitle (PaneType ePane) const
{
    switch (ePane)
    {
        case PT_LEFT:

            
            // The title of the left pane depends on the application.  We
            // simply interpret the type of the view shell in the center
            // pane: ST_DRAW indicates DRAW, all other indicate Impress.
            if (GetCurrentViewShellType(PT_CENTER) == ViewShell::ST_DRAW)
                return SdResId(FLT_LEFT_PANE_DRAW_DOCKING_WINDOW);
            else
                return SdResId(FLT_LEFT_PANE_IMPRESS_DOCKING_WINDOW);
            break;

        case PT_CENTER:
        case PT_MAIN:
        default:
            // The center pane has no visible title bar so we do not bother
            // to create a title.
            throw ::com::sun::star::uno::RuntimeException(
                ::rtl::OUString::createFromAscii(
                    "PaneManager::GetDockingWindowTitle() called for center pane, which has no title"),
            NULL);
            break;

        case PT_RIGHT:
            // The docking window title of the right pane is always the same.
            return SdResId(FLT_RIGHT_PANE_DOCKING_WINDOW);
            break;
    }
}




void PaneManager::Implementation::HandleModeChangeSlot (
    ULONG nSlotId, 
    SfxRequest& rRequest)
{
    BOOL bIsActive = TRUE;
    switch (nSlotId)
    {
        case SID_DRAWINGMODE:
        case SID_NOTESMODE:
		case SID_HANDOUTMODE:
		case SID_DIAMODE:
		case SID_OUTLINEMODE:
        {
			const SfxItemSet* pRequestArguments = rRequest.GetArgs();
			if (pRequestArguments)
            {
                SFX_REQUEST_ARG (rRequest, 
                    pIsActive, 
                    SfxBoolItem, 
                    (USHORT)nSlotId,
                    FALSE);
				bIsActive = pIsActive->GetValue ();
			}
        }
        break;
    }

    FrameView* pFrameView = NULL;
    ViewShell* pCenterViewShell = GetViewShell(PT_CENTER);
    if (pCenterViewShell != NULL)
        pFrameView = pCenterViewShell->GetFrameView();
    ViewShell::ShellType eRequestedShellType = ViewShell::ST_NONE;
    if (bIsActive)
    {
        switch (nSlotId)
        {
            case SID_NORMAL_MULTI_PANE_GUI:
            case SID_DRAWINGMODE:   
                eRequestedShellType = ViewShell::ST_IMPRESS;
                pFrameView->SetViewShEditMode(EM_PAGE, PK_STANDARD);
                break;

            case SID_NOTESMODE:
                eRequestedShellType = ViewShell::ST_NOTES;
                pFrameView->SetViewShEditMode(EM_PAGE, PK_NOTES);
                break;

            case SID_HANDOUTMODE:
                eRequestedShellType = ViewShell::ST_HANDOUT;
                pFrameView->SetViewShEditMode(EM_PAGE, PK_HANDOUT);
                break;

            case SID_SLIDE_SORTER_MULTI_PANE_GUI:
            case SID_DIAMODE:
                eRequestedShellType = ViewShell::ST_SLIDE_SORTER;
                break;

            case SID_OUTLINEMODE:
                eRequestedShellType = ViewShell::ST_OUTLINE;
                break;
        }
    }
    if (GetCurrentViewShellType(PT_CENTER) == eRequestedShellType)
    {
        // We do not have to switch the view shell but maybe the edit mode
        // has changed.
        if (pCenterViewShell->ISA(DrawViewShell))
        {
            pCenterViewShell->Broadcast (
                ViewShellHint(ViewShellHint::HINT_CHANGE_EDIT_MODE_START));

            DrawViewShell* pDrawViewShell 
                = PTR_CAST(DrawViewShell,pCenterViewShell);
            pDrawViewShell->ChangeEditMode (
                EM_PAGE, pDrawViewShell->IsLayerModeActive());

            pCenterViewShell->Broadcast (
                ViewShellHint(ViewShellHint::HINT_CHANGE_EDIT_MODE_END));
        }
    }
    else
    {
        SwitchConfiguration(eRequestedShellType);
        RequestUpdateConfiguration(CM_ASYNCHRONOUS);
    }
}




bool PaneManager::Implementation::SwitchConfiguration (
    ViewShell::ShellType eType)
{
    meDesiredPaneConfiguration = eType;
    return ! IsUpToDate();
}




bool PaneManager::Implementation::SwitchViewShell (
    PaneType ePane,
    ViewShell::ShellType eType)
{
    OSL_ASSERT(ePane != PT_MAIN);
    mpCurrentConfiguration->SetViewShellType(ePane, eType);
    return ! IsUpToDate(ePane);
}




void PaneManager::Implementation::ExecuteModeChange (SfxRequest& rRequest)
{
    UpdateConfiguration();
}




void PaneManager::Implementation::ExecuteSlot (SfxRequest& rRequest)
{
	USHORT nSlotId = rRequest.GetSlot();
	switch (nSlotId)
    {
        case SID_LEFT_PANE_IMPRESS:
        case SID_LEFT_PANE_DRAW:
            SetChildWindowVisibility (
                nSlotId,
                ::sd::LeftPaneChildWindow::GetChildWindowId(),
                rRequest,
                PT_LEFT);
			rRequest.Ignore ();
            break;

        case SID_RIGHT_PANE:
            SetChildWindowVisibility (
                nSlotId,
                ::sd::RightPaneChildWindow::GetChildWindowId(),
                rRequest,
                PT_RIGHT);
			rRequest.Ignore ();
            break;

        case SID_NORMAL_MULTI_PANE_GUI:
        case SID_SLIDE_SORTER_MULTI_PANE_GUI:
        case SID_DRAWINGMODE:
        case SID_DIAMODE:
        case SID_OUTLINEMODE:
        case SID_NOTESMODE:
        case SID_HANDOUTMODE:
            HandleModeChangeSlot (nSlotId, rRequest);
            break;

        default:
            break;
    }
}




void PaneManager::Implementation::GetSlotState (SfxItemSet& rSet)
{
    SfxViewFrame* pFrame = mrBase.GetViewFrame();

    // Get some frequently used values.
    ViewShell::ShellType eCenterViewShellType (GetCurrentViewShellType(PT_CENTER));
    ViewShell* pCenterViewShell = GetViewShell(PT_CENTER);
    // Determine the master page mode.
    bool bMasterPageMode (false);
    if (pCenterViewShell->ISA(DrawViewShell))
        if (PTR_CAST(DrawViewShell,pCenterViewShell)->GetEditMode()
            == EM_MASTERPAGE)
            bMasterPageMode = true;

    SfxWhichIter aSetIterator (rSet);
    sal_uInt16 nItemId (aSetIterator.FirstWhich());
    while (nItemId > 0)
    {
        switch (nItemId)
        {
            case SID_LEFT_PANE_IMPRESS:
            case SID_LEFT_PANE_DRAW:
            {
                ::Window* pWindow = GetWindow(PT_LEFT);
                BOOL bVisible (pWindow!=NULL && pWindow->IsVisible());
                rSet.Put(SfxBoolItem(nItemId, bVisible));
            }
            break;

            case SID_RIGHT_PANE:
            {
                ::Window* pWindow = GetWindow(PT_RIGHT);
                BOOL bVisible (pWindow!=NULL && pWindow->IsVisible());
                rSet.Put(SfxBoolItem(SID_RIGHT_PANE, bVisible));
            }
            break;
            
            case SID_NOTES_WINDOW:
            {
                USHORT nId = ::sd::notes::NotesChildWindow::GetChildWindowId();
                rSet.Put (SfxBoolItem(SID_NOTES_WINDOW, pFrame->HasChildWindow(nId)));
            }
            break;

            case SID_NORMAL_MULTI_PANE_GUI:
            {
                bool bValue (eCenterViewShellType==ViewShell::ST_IMPRESS
                    && ! bMasterPageMode);
                rSet.Put (SfxBoolItem(SID_NORMAL_MULTI_PANE_GUI, bValue ? TRUE:FALSE));
            }
            break;

            case SID_SLIDE_SORTER_MULTI_PANE_GUI:
            {
                bool bValue (eCenterViewShellType==ViewShell::ST_SLIDE_SORTER);
                rSet.Put (SfxBoolItem(SID_SLIDE_SORTER_MULTI_PANE_GUI, 
                        bValue ? TRUE:FALSE));
            }
            break;

            case SID_DIAMODE:
            {
                bool bValue (eCenterViewShellType==ViewShell::ST_SLIDE_SORTER);
                rSet.Put (SfxBoolItem(SID_DIAMODE, bValue ? TRUE:FALSE));
            }
            break;

            case SID_OUTLINEMODE:
            {
                bool bValue (eCenterViewShellType==ViewShell::ST_OUTLINE);
                rSet.Put (SfxBoolItem(SID_OUTLINEMODE, bValue ? TRUE:FALSE));
            }
            break;

            case SID_HANDOUTMODE:
            {
                // There is only the master page mode for the handout view
                // so ignore the master page flag.
                bool bValue (eCenterViewShellType==ViewShell::ST_HANDOUT);
                rSet.Put (SfxBoolItem(SID_HANDOUTMODE, bValue ? TRUE:FALSE));
            }
            break;

            case SID_NOTESMODE:
            {
                bool bValue (eCenterViewShellType==ViewShell::ST_NOTES
                    && ! bMasterPageMode);
                rSet.Put (SfxBoolItem(SID_NOTESMODE, bValue ? TRUE:FALSE));
            }
            break;

            default:
                // Ignore all other items.  They are not meant to be handled
                // by us.
                break;
        }

        nItemId = aSetIterator.NextWhich();
    }
}




EventBroadcaster& PaneManager::Implementation::GetEventBroadcaster (void)
{
    return maEventBroadcaster;
}




void PaneManager::Implementation::SetChildWindowVisibility(
    USHORT nSlotId,
    USHORT nChildWindowId,
    SfxRequest& rRequest,
    PaneType ePane)
{
    // Determine the new visibility state.
    SfxViewFrame* pFrame = mrBase.GetViewFrame();
    const SfxItemSet* pArguments = rRequest.GetArgs();
    BOOL bShowChildWindow;
    if (pArguments != NULL)
        bShowChildWindow = static_cast<const SfxBoolItem&>(
            pArguments->Get(nSlotId)).GetValue();
    else
    {
        ::Window* pWindow = GetWindow(ePane);
        bShowChildWindow = ! (pWindow!=NULL && pWindow->IsVisible());
    }
    
    // Set the desired visibility state at the current configuration and
    // update it accordingly.
    mpCurrentConfiguration->SetWindowVisibility(ePane, bShowChildWindow);
    if ( ! IsUpToDate(ePane))
    {
        UpdateConfiguration();
        pFrame->GetBindings().Invalidate(nSlotId);
    }
}




void PaneManager::Implementation::RequestUpdateConfiguration (CallMode eCallMode)
{
    if ( ! IsUpToDate())
    {
        bool bSynchronous (eCallMode==CM_SYNCHRONOUS);
        if (mbIsConfigurationUpdateRunning)
            bSynchronous = false;
        
        if (bSynchronous)
            UpdateConfiguration();
        else
            if (mrBase.IsActive())
            {
                // The SfxDispatcher of the SfxViewFrame should be always
                // present and valid.  However, stacktrace issue #127342#
                // indicates otherwise, so check the pointers.
                SfxViewFrame* pViewFrame = mrBase.GetViewFrame();
                if (pViewFrame != NULL)
                {
                    SfxDispatcher* pDispatcher = pViewFrame->GetDispatcher();
                    if (pDispatcher != NULL)
                        pDispatcher->Execute (
                            SID_SWITCH_SHELL,
                            SFX_CALLMODE_ASYNCHRON | SFX_CALLMODE_RECORD);
                }
            }
    }
}




void PaneManager::Implementation::UpdateConfiguration (void)
{
    ReentranceGuard aGuard (mbIsConfigurationUpdateRunning);
    
    // AutoLayouts have to be ready.
    mrBase.GetDocument()->StopWorkStartupDelay();

    // First make the desired configuration the current one.
    if (meDesiredPaneConfiguration != GetCurrentViewShellType(PT_CENTER))
    {
        PaneConfiguration* pConfiguration = &GetPaneConfiguration(meDesiredPaneConfiguration);
        if (pConfiguration != NULL)
            mpCurrentConfiguration = pConfiguration;
    }

    // Now update pane visibility and the view shells.
    PaneDescriptor& rLeftPane (*GetPaneDescriptor(PT_LEFT));
    bool bShowLeftPane (GetRequestedWindowVisibility(PT_LEFT));
    ViewShell::ShellType eLeftType (GetRequestedViewShellType(PT_LEFT));
    bool bLeftUpToDate (IsUpToDate(PT_LEFT));

    PaneDescriptor& rCenterPane (*GetPaneDescriptor(PT_CENTER));
    bool bShowCenterPane (GetRequestedWindowVisibility(PT_CENTER));
    ViewShell::ShellType eCenterType (GetRequestedViewShellType(PT_CENTER));
    bool bCenterUpToDate (IsUpToDate(PT_CENTER));

    PaneDescriptor& rRightPane (*GetPaneDescriptor(PT_RIGHT));
    bool bShowRightPane (GetRequestedWindowVisibility(PT_RIGHT));
    ViewShell::ShellType eRightType (GetRequestedViewShellType(PT_RIGHT));
    bool bRightUpToDate (IsUpToDate(PT_RIGHT));

    // Prevent the shell stack from being updated for each pane
    // individually.  That is done when all changes are made.
    if ( ! (bCenterUpToDate && bLeftUpToDate && bRightUpToDate))
    {
        ViewShellManager::UpdateLock aLock (mrBase.GetViewShellManager());
        bool bIsAnyShellReplaced (false);
        // Lock UI updates while we are switching the views except for the
        // first time after creation.  Outherwise this leads to problems
        // after reload (missing resizes for the side panes).
        if (mbIsInitialized)
            mrBase.GetUpdateLockManager().Lock();

        // First update the center pane.
        if ( ! bCenterUpToDate)
        {
            bool bIsShellReplaced = rCenterPane.SetViewShellType(eCenterType);
            // When the view shell in the center pane has changed we have to
            // update the controller of the ViewShellBase.  We do not do
            // this when the ViewShellBase is just being initialized.  In
            // that case the controller is set automatically.  Interfering
            // with that would be dangerous.
            ViewShell* pShell = rCenterPane.GetViewShell();
            if (mbIsInitialized 
                && pShell!=NULL
                && bIsShellReplaced)
            {
                mrBase.UpdateController(pShell->GetController());
            }
            bIsAnyShellReplaced = bIsShellReplaced;
        }

        // Now update the side panes.  Note that the visibility of the
        // windows may change.  A window is shown before or hidden after the
        // shell type is changed so that either a new shell can access the
        // new window or the current shell can access the current window.
        // The shell type is switched only when the pane is visible
        // afterwards.  This avoids the constant construction and
        // destruction of shells just because their panes are temporarily
        // hidden.

        if ( ! bLeftUpToDate)
        {
            if (bShowLeftPane)
                SetupPaneVisibility(PT_LEFT, true);
            if (bShowLeftPane)
                bIsAnyShellReplaced |= rLeftPane.SetViewShellType(eLeftType);
            if ( ! bShowLeftPane)
                SetupPaneVisibility(PT_LEFT, false);
        }

        if ( ! bRightUpToDate)
        {
            if (bShowRightPane)
                SetupPaneVisibility(PT_RIGHT, true);
            if (bShowRightPane)
                bIsAnyShellReplaced |= rRightPane.SetViewShellType(eRightType);
            if ( ! bShowRightPane)
                SetupPaneVisibility(PT_RIGHT, false);
        }

        // Make the center pane the active one.
        mrBase.GetViewShellManager().MoveToTop (GetViewShell(PT_CENTER));

        // Unlock the update lock here when only the visibility of windows but
        // not the view shells displayed in them have changed.  Otherwise
        // the UpdateLockManager takes care of unlocking at the right time.
        if ( ! bIsAnyShellReplaced && mrBase.GetUpdateLockManager().IsLocked())
            mrBase.GetUpdateLockManager().Unlock();

        mbIsInitialized = true;
    }
}




PaneConfiguration& PaneManager::Implementation::GetPaneConfiguration (
    ViewShell::ShellType eCenterPaneShellType)
{
    ConfigurationList::iterator iConfiguration (maConfigurationList.find(eCenterPaneShellType));
    if (iConfiguration == maConfigurationList.end())
        iConfiguration = maConfigurationList.find(ViewShell::ST_NONE);
    return iConfiguration->second;
}




PaneConfiguration* PaneManager::Implementation::GetCurrentConfiguration (void) const
{
    return mpCurrentConfiguration;
}




void PaneManager::Implementation::SetupPaneVisibility (PaneType ePane, bool bShow)
{
    bool bValid (mrBase.IsActive());
    PaneDescriptor* pPane = GetPaneDescriptor(ePane);
    if (bValid && pPane != NULL)
        if (bShow)
            pPane->ShowWindow();
        else
            pPane->HideWindow();
}




bool PaneManager::Implementation::IsUpToDate (PaneType ePane)
{
    bool bIsUpToDate (true);
    PaneDescriptor* pDescriptor (GetPaneDescriptor(ePane));
    if (pDescriptor != NULL)
    {
        ViewShell::ShellType eCurrentType (pDescriptor->GetViewShellType());
        ViewShell::ShellType eRequestedType (GetRequestedViewShellType(ePane));
        bool bCurrentVisibility (pDescriptor->GetWindowVisibility());
        bool bRequestedVisibility (GetRequestedWindowVisibility(ePane));
        bIsUpToDate = 
            eCurrentType == eRequestedType
            && bCurrentVisibility == bRequestedVisibility;
    }

    return bIsUpToDate;
}




bool PaneManager::Implementation::IsUpToDate (void)
{
    bool bIsUpToDate (true);

    if (meDesiredPaneConfiguration != GetCurrentViewShellType(PT_CENTER))
        bIsUpToDate = false;
    else
        bIsUpToDate = IsUpToDate(PT_LEFT) && IsUpToDate(PT_CENTER) && IsUpToDate(PT_RIGHT);

    return bIsUpToDate;
}




::std::auto_ptr<PaneManagerState> PaneManager::Implementation::GetState (void) const
{
    ::std::auto_ptr<PaneManagerState> pState (new PaneManagerState());
    pState->SetImplementation(::std::auto_ptr<PaneManager::StateImplementation>(
        new PaneManager::StateImplementation(
            maConfigurationList,
            meDesiredPaneConfiguration)));

    return pState;
}




void PaneManager::Implementation::SetState (const PaneManagerState& rState)
{
    const PaneManager::StateImplementation& rStateImplementation (rState.GetImplementation());
    maConfigurationList = rStateImplementation.maConfigurationList;
    meDesiredPaneConfiguration = rStateImplementation.meCurrentConfiguration;

    mpCurrentConfiguration = &GetPaneConfiguration(meDesiredPaneConfiguration);

    RequestUpdateConfiguration(CM_ASYNCHRONOUS);
}




//===== PaneManagerState ======================================================

PaneManagerState::PaneManagerState (void)
    : mpStateImpl(NULL)
{
}




PaneManagerState::PaneManagerState (const PaneManagerState& rPaneManagerState)
    : mpStateImpl(new PaneManager::StateImplementation(*rPaneManagerState.mpStateImpl))
{
}




PaneManagerState::~PaneManagerState (void)
{
}




PaneManagerState& PaneManagerState::operator= (const PaneManagerState& rPaneManagerState)
{
    mpStateImpl.reset(new PaneManager::StateImplementation(*rPaneManagerState.mpStateImpl));
    return *this;
}




void PaneManagerState::SetImplementation (const PaneManager::StateImplementation& rImplementation)
{
    mpStateImpl.reset(new PaneManager::StateImplementation(rImplementation));
}




void PaneManagerState::SetImplementation (
    ::std::auto_ptr<PaneManager::StateImplementation> pImplementation)
{
    mpStateImpl = pImplementation;
}




const PaneManager::StateImplementation PaneManagerState::GetImplementation (void) const
{
    return *mpStateImpl;
}

} // end of namespace sd
