/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: ViewShellCache.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: hr $ $Date: 2005/09/23 11:33:39 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "ViewShellCache.hxx"

#include "ViewShell.hxx"
#include "ViewShellManager.hxx"

#include <osl/mutex.hxx>
#include <vector>
#include <hash_map>

namespace sd
{

/** Define this symbol to keep task pane view shells in the cache
    (experimental.)  Undefine it to remove those together will all other
    kinds of view shells from the cache.
*/
//#define KEEP_TASK_PANE_IN_CACHE


class ViewShellCache::Implementation
{
public:
    class WindowHash {public:
        size_t operator()(const ::Window* p) const { return (size_t)p; } 
    };
    class Descriptor{public:
        Descriptor(ViewShell*pShell=NULL):mpViewShell(pShell),mbUsed(true){}
        void SetUsed (bool bUsed) {mbUsed=bUsed;}
        ViewShell* mpViewShell;
        bool mbUsed;
    };
    typedef ::std::hash_map<ShellId, Descriptor> CachePerWindow;
    typedef ::std::hash_map< ::Window*, CachePerWindow, WindowHash>
        ViewShellCacheList;


    /** The view shell manager is used as a factory for the creation of new
        view shells.
    */
    ViewShellManager& mrManager;

    Implementation (ViewShellManager& rManager);

    ViewShell* GetViewShell (
        ::Window* pWindow,
        ShellId nId);
    void PutViewShell (
        ::Window* pWindow,
        ShellId nId,
        ViewShell* pViewShell);
    void ReleaseViewShell (
        ::Window* pWindow,
        ViewShell* pViewShell);

private:
    ::osl::Mutex maMutex;
    ViewShellCacheList maViewShellCacheList;

    /** Print the content of the cache via OSL_TRACE().
    */
    void PrintContent (const char* pMessage);
};



//===== ViewShellCache ========================================================

ViewShellCache::ViewShellCache (ViewShellManager& rManager)
    : mpImpl (new Implementation (rManager))
{
}




ViewShellCache::~ViewShellCache (void)
{
    Shutdown();
}




void ViewShellCache::Shutdown (void)
{
    if (mpImpl.get() != NULL)
        mpImpl.reset();
}




ViewShell* ViewShellCache::GetViewShell (
    ShellId nId,
    ::Window* pParentWindow,
    FrameView* pFrameView)
{
    ViewShell* pViewShell = NULL;
    // Ignore the call when no parent window is given.  In that case a view
    // shell can not be created.
    if (mpImpl.get() != NULL
        && pParentWindow != NULL)
    {
        pViewShell = mpImpl->GetViewShell (pParentWindow, nId);
        if (pViewShell == NULL)
        {
            pViewShell = mpImpl->mrManager.CreateViewShell (
                nId, 
                pParentWindow, 
                pFrameView);

            // The view shell is NULL for certain ids.
            if (pViewShell != NULL)
                mpImpl->PutViewShell (pParentWindow, nId, pViewShell);
        }
    }
    return pViewShell;
}




void ViewShellCache::ReleaseViewShell (ViewShell* pViewShell)
{
    if (mpImpl.get() != NULL)
        mpImpl->ReleaseViewShell (
            pViewShell->GetParentWindow(),
            pViewShell);
}




//===== ViewShellCache::Implementation ========================================

ViewShellCache::Implementation::Implementation (ViewShellManager& rManager)
    : mrManager (rManager)
{
}




ViewShell* ViewShellCache::Implementation::GetViewShell (
    ::Window* pWindow,
    ShellId nId)
{
    ::osl::MutexGuard aGuard (maMutex);
    CachePerWindow& rCache (maViewShellCacheList[pWindow]);
    CachePerWindow::iterator iEntry (rCache.find(nId));
    if (iEntry != rCache.end())
        return iEntry->second.mpViewShell;
    else
        return NULL;
}




void ViewShellCache::Implementation::PutViewShell (
    ::Window* pWindow,
    ShellId nId,
    ViewShell* pViewShell)
{
    ::osl::MutexGuard aGuard (maMutex);
    CachePerWindow& rCache (maViewShellCacheList[pWindow]);
    rCache[nId] = Descriptor(pViewShell);
}




void ViewShellCache::Implementation::ReleaseViewShell (
    ::Window* pWindow,
    ViewShell* pViewShell)
{
    ::osl::MutexGuard aGuard (maMutex);
    CachePerWindow& rCache (maViewShellCacheList[pWindow]);
    CachePerWindow::iterator iDescriptor;
    for (iDescriptor=rCache.begin(); iDescriptor!=rCache.end(); ++iDescriptor)
    {
        if (iDescriptor->second.mpViewShell == pViewShell)
        {
            iDescriptor->second.SetUsed(false);
            iDescriptor->second.mpViewShell = NULL;
            // At the moment the cache is not really a cache.  The reuse of
            // a view shell does not yet work, so we have to delete the view
            // shell instead of leaving it in the cache.
#ifdef KEEP_TASK_PANE_IN_CACHE
            if (pViewShell->GetShellType() != ViewShell::ST_TASK_PANE)
#endif
            {
                rCache.erase (iDescriptor);
                delete pViewShell;
            }
            break;
        }
    }
    PrintContent ("ReleaseViewShell()");
}




void ViewShellCache::Implementation::PrintContent (const char* pMessage)
{
    OSL_TRACE("ViewShellCache: cache content: %s", pMessage);
    ::osl::MutexGuard aGuard (maMutex);
    ViewShellCacheList::const_iterator iCache;
    for (iCache=maViewShellCacheList.begin(); 
         iCache!=maViewShellCacheList.end();
         ++iCache)
    {
        OSL_TRACE ("    cache for window %p", iCache->first);
        CachePerWindow::const_iterator iEntry;
        for (iEntry=iCache->second.begin();
             iEntry!=iCache->second.end();
             ++iEntry)
        {
            OSL_TRACE("        %d -> %p", 
                iEntry->first,
                iEntry->second.mpViewShell);
        }
    }
    OSL_TRACE("ViewShellCache: ---------------");
}



} // end of namespace sd
