/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: HelpCompiler.java,v $
 *
 *  $Revision: 1.12.4.1 $
 *
 *  last change: $Author: rt $ $Date: 2006/02/10 12:39:08 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

package com.sun.star.help;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.InputSource;

import com.sun.xml.tree.TreeWalker;
import com.sun.xml.tree.XmlDocument;

public class HelpCompiler {

	private static final String makeRelPrefix = ".." + File.separator;
	private static HelpURLStreamHandlerFactory urlHandler;
	public static void initURLHandler() {
		try {
			// Determine the urlfactory ...
			String urlmode = HelpDatabases.getURLMode();
			urlHandler = new HelpURLStreamHandlerFactory(urlmode);
			URL.setURLStreamHandlerFactory(urlHandler);
		} catch (Exception e) {
			e.printStackTrace();
			System.exit(1);
		}
	}

	private Hashtable streamTable;
	private String inputFile, src, module, lang;
	private String resEmbStylesheet;

	public HelpCompiler(
		Hashtable streamTable,
		String inputFile,
		String src,
		String resEmbStylesheet,
		String module,
		String lang) {
		this.streamTable = streamTable;
		this.inputFile = inputFile;
		this.src = src;
		this.resEmbStylesheet = resEmbStylesheet;
		this.module = module;
		this.lang = lang;

		// ... and clear the returned document
		HelpURLStreamHandlerFactory.setMode(null);

	}

	private Object[] switchFind(Node node) {
		HashSet hs = new HashSet();
		Node next;
		TreeWalker tw = new TreeWalker(node);
		while ((next = tw.getNextElement("switchinline")) != null) {
			Element el = (Element) next;
			if (!el.getAttribute("select").equals("appl"))
				continue;

			NodeList nl = el.getChildNodes();
			for (int i = 0; i < nl.getLength(); ++i) {
				if (nl.item(i).getNodeName().equals("caseinline")) {
					String appl = ((Element) nl.item(i)).getAttribute("select");
					hs.add(appl);
				} else if (nl.item(i).getNodeName().equals("defaultinline")) {
					hs.add("DEFAULT");
				}
			}
		}

		hs.add("DEFAULT");
		return hs.toArray();
	}

	// returns a node representing the whole stuff compiled for the current
	// application.
	private Node clone(Node node, String appl) {
		Node parent = node.cloneNode(false);
		NodeList nl = node.getChildNodes();
		for (int i = 0; i < nl.getLength(); ++i) {
			Node n = nl.item(i);
			if ((n.getNodeName().equals("switchinline")
				|| n.getNodeName().equals("switch"))
				&& ((Element) n).getAttribute("select").equals("appl")) {
				NodeList cl = n.getChildNodes();
				if (appl.equals("DEFAULT")) {
					for (int j = 0; j < cl.getLength(); ++j) {
						Node caseNode = cl.item(j);
						if (caseNode.getNodeName().equals("defaultinline")) {
							NodeList cnl = caseNode.getChildNodes();
							for (int k = 0; k < cnl.getLength(); ++k)
								parent.appendChild(
									clone(
										caseNode.getChildNodes().item(k),
										appl));
							break;
						}
					}
				} else {
					for (int j = 0; j < cl.getLength(); ++j) {
						Node caseNode = cl.item(j);
						if (caseNode.getNodeName().equals("caseinline")
							&& ((Element) caseNode).getAttribute("select").equals(
								appl)) {
							NodeList cnl = caseNode.getChildNodes();
							for (int k = 0; k < cnl.getLength(); ++k)
								parent.appendChild(
									clone(
										caseNode.getChildNodes().item(k),
										appl));
							break;
						}
					}
				}
			} else
				parent.appendChild(clone(nl.item(i), appl));
		}
		return parent;
	}

	public boolean compile() throws UnsupportedEncodingException {
		// we now have the jaroutputstream, which will contain the document.
		// now determine the document as a dom tree in variable docResolved
		File inputFil = new File(inputFile);
        
        // HelpCompiler.getSourceDocument() takes a string that resembles an URL. 
        // Using inputFil.toURL() is way to expensive here, 
        // because it checks the filesystem to find out if inputFil
        // might be a directory. This takes a lot of time if working from
        // Windows via NFS on remote volumes. We "know" that inputfil is
        // a file anyway.
        String path;
        if ( File.separatorChar == '\\' ) {
            path = "file:/";
            path += inputFil.getAbsolutePath();
            path = path.replace('\\','/');
        }
        else {
            path = "file:" + inputFil.getAbsolutePath();
        }
		byte[] embResolved = getSourceDocument(path);

		// now add path to the document
		// resolve the dom
		if (embResolved == null) {
            String sourcePath;
    		try {
	    		sourcePath = inputFil.getCanonicalPath();
    		} catch (IOException e3) {
    			sourcePath = inputFil.getAbsolutePath();
	    	}
			System.err.println("ERROR: file not existing: " + sourcePath);
			System.exit(1);
		}

		ByteArrayInputStream inByte = new ByteArrayInputStream(embResolved);
		InputStreamReader inread;
		try {
			inread = new InputStreamReader(inByte, "UTF8");
		} catch (UnsupportedEncodingException e) {
			System.err.println(
				"ERROR: unsupported Encoding '"
					+ inputFile
					+ "': "
					+ e.getMessage());
			return false;
		}

		InputSource inputSource = new InputSource(inread);
		inputSource.setEncoding("UTF8");
		Document docResolvedOrg = null;
		try {
			docResolvedOrg = XmlDocument.createXmlDocument(inputSource, false);
		} catch (Exception e) {
			System.err.println(
				"ERROR: XmlDocument.createXmlDocument() failed for '"
					+ inputFile
					+ "': "
					+ e.getMessage());
			return false;
		}

		// now find all applications for which one has to compile
		String documentId = null;
		String fileName = null;
		String title = null;
		// returns all applications for which one has to compile
		Object[] applications = switchFind(docResolvedOrg);
        
        for (int i = 0; i < applications.length; ++i) {
			String appl = (String) applications[i];
            // returns a clone of the document with swich-cases resolved
			Element docResolved =
				(Element) clone(docResolvedOrg.getDocumentElement(), appl);
			// now determine the id of the document, which is part of the
			// bookmark - tag (HID)
			Node test;
			TreeWalker treewalker = new TreeWalker(docResolved);
			// a number to determine the anchor of the whole stuff
			HashSet hidlist = new HashSet();
			HashSet extendedHelpText = new HashSet();
			Hashtable keywords = new Hashtable();
			Hashtable helptexts = new Hashtable();
            
			while ((test = treewalker.getNext()) != null) {
				if (fileName == null
					&& test.getNodeName().equals("filename")) {
					NodeList list = test.getChildNodes();
					Node node = list.item(0);
					if (node.getNodeType() == Node.TEXT_NODE)
						fileName = ((Text) node).getData();
				} else if (
					title == null && test.getNodeName().equals("title")) {
                    title = dump(test);
                    if(title == null || title.length() == 0)
                        title = "<notitle>";
				} else if (test.getNodeName().equals("bookmark")) {
					Element el = (Element) test;
					String branch = el.getAttribute("branch");
					String hid = null;
					String anchor = el.getAttribute("id");
					if (branch.startsWith("hid")) {
						int index = branch.indexOf('/');
						if (index != -1) {
							hid = branch.substring(1 + index);
							// one shall serve as a documentId
							if (documentId == null)
								documentId = hid;
							extendedHelpText.add(hid);
							hidlist.add(
								anchor == null
									|| anchor.length() == 0
										? hid
										: hid + "#" + anchor);
						} else
							continue;
					} else if (branch.equals("index")) {
						LinkedList ll = new LinkedList();

						NodeList list = el.getChildNodes();
						for (int j = 0; j < list.getLength(); ++j) {
							Node nd = list.item(j);
							if (!nd.getNodeName().equals("bookmark_value"))
								continue;
							String embedded =
								((Element) nd).getAttribute("embedded");
							boolean isEmbedded =
								embedded != null
									&& "true".equals(embedded.toLowerCase());
							if (isEmbedded)
								continue;

                            
							String keyword =
				                                dump(nd);
                            int keywordSem = keyword.indexOf(';');
                            if(keywordSem != -1) {
                                String tmppre =
                                    keyword.substring(0,keywordSem).trim();
                                String tmppos =
                                    keyword.substring(1+keywordSem).trim();
                                keyword = tmppre + ";" + tmppos;
                            }
							ll.add(keyword);
						}
						if (!ll.isEmpty()) {
                            keywords.put(anchor, ll);
                        }
					} else if (branch.equals("contents")) {
						// currently not used
					}
				} else if (test.getNodeName().equals("ahelp")) {
					String text = dump(test).trim();
                    String name = null; //((Element) test).getAttribute("hid");
					// helptexts.put(name, text);
					Iterator iter = extendedHelpText.iterator();
					while (iter.hasNext()) {
						name = (String) iter.next();
						helptexts.put(name, text);
					}

					if (!extendedHelpText.isEmpty())
						extendedHelpText = new HashSet();
				}
			} // now save the info

			addEntryToJarFile(
				appl,
				"text",
				docResolved.toString().getBytes("UTF8"));
			addEntryToJarFile(appl, "hidlist", hidlist);
			addEntryToJarFile(appl, "helptexts", helptexts);
			addEntryToJarFile(appl, "keywords", keywords);
		} // end iteration over all applications

		try {
			addEntryToJarFile(
				"document",
				"id",
				documentId != null
					? documentId.getBytes("UTF8")
					: "".getBytes("UTF8"));
			addEntryToJarFile(
				"document",
				"path",
				fileName != null
					? fileName.getBytes("UTF8")
					: "".getBytes("UTF8"));
			addEntryToJarFile(
				"document",
				"title",
				title != null ? title.getBytes("UTF8") : "".getBytes("UTF8"));
			String actMod = module;
			if (fileName != null && fileName.length() != 0) {
				if (fileName.startsWith("/text/")) {
					actMod = fileName.substring("/text/".length());
					actMod = actMod.substring(0, actMod.indexOf('/'));
				}
			}
			addEntryToJarFile("document", "module", actMod.getBytes("UTF8"));
		} catch (IOException e1) {
			System.err.println(
				"ERROR: IOException in compile(): '" + e1.getMessage());
			return false;
		}
		return true;
	}


	/**
	 * Returns the embedding resolved document
	 */

	static HelpIndexer.ParseStuff stuff = null;

	private byte[] getSourceDocument(String filePath) { // initialize ...
		if (stuff == null) {
			stuff = new HelpIndexer.ParseStuff(resEmbStylesheet);
			// Setting the parameters
			stuff.setParameter("Language", lang);
			try {
				stuff.setParameter(
					"fsroot",
					(new File(src)).toURL().toString());
			} catch (MalformedURLException e) {
				System.err.println(
					"ERROR: malformed URL '" + src + "': " + e.getMessage());
				System.exit(1);
			}
		} // ... and parse
		return stuff.parse(filePath);
	}

	private String dump(Node node) {
		String app = new String();
		if (node.hasChildNodes()) {
			NodeList list = node.getChildNodes();
			for (int i = 0; i < list.getLength(); ++i) {
                app += dump(list.item(i));
                // System.out.println(list.item(i).toString());
            }
		}
		if (node.getNodeType() == Node.TEXT_NODE) {
			app += ((Text) node).getData();
            // System.out.println(app);
		}
		return app;
	}

	private void addEntryToJarFile(
		String appl,
		String entryName,
		Object keywords) {
		if (keywords == null)
			return;

		streamTable.put(
			(appl + "/" + entryName).toLowerCase().trim(),
			keywords);
	}

	private void addEntryToJarFile(
		String prefix,
		String entryName,
		byte[] bytesToAdd) {
		if (bytesToAdd == null)
			return;

		streamTable.put(
			(prefix + "/" + entryName).toLowerCase().trim(),
			bytesToAdd);
	}
}
