/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: calculat.cxx,v $
 *
 *  $Revision: 1.10 $
 *
 *  last change: $Author: ihi $ $Date: 2006/11/14 14:53:20 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

// MARKER(update_precomp.py): autogen include statement, do not remove
#include "precompiled_sch.hxx"

// header for Point, Rectangle
#ifndef _SV_GEN_HXX
#include <tools/gen.hxx>
#endif
#ifndef _TL_POLY_HXX
#include <tools/poly.hxx>
#endif
// header for DBG_ASSERT
#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif
// header for Line
#ifndef _LINE_HXX
#include <tools/line.hxx>
#endif
#ifndef INCLUDED_RTL_MATH_HXX
#include <rtl/math.hxx>
#endif
// for performance measurement
#ifndef	_RTL_LOGFILE_HXX_
#include <rtl/logfile.hxx>
#endif

// Note: Enable the following to skip points in the resulting spline
// poly-polygon, if they have equal x-values rather than identical points.
// Unitl now, I think there are situations where the output might differ, if you
// do so, so it's not enabled by default.

// #define SPLINE_OPTIMIZE_POINTS

#include "calculat.hxx"

#include <algorithm>
#include <functional>

#ifndef _BGFX_POLYGON_B2DPOLYGON_HXX
#include <basegfx/polygon/b2dpolygon.hxx>
#endif

#ifndef _BGFX_RANGE_B2DRANGE_HXX
#include <basegfx/range/b2drange.hxx>
#endif

#ifndef _BGFX_POLYPOLYGON_B2DPOLYGONTOOLS_HXX
#include <basegfx/polygon/b2dpolypolygontools.hxx>
#endif

#ifndef _BGFX_POLYGON_B2DPOLYGONTOOLS_HXX
#include <basegfx/polygon/b2dpolygontools.hxx>
#endif

using namespace ::std;

::basegfx::B2DPolyPolygon SchCalculationHelper::IntersectPolyPolygonWithRectangle(
    const ::basegfx::B2DPolyPolygon& rPolyPolygon,
    const ::basegfx::B2DRange& rRectangle)
{
	RTL_LOGFILE_CONTEXT_AUTHOR( context, "sch", "bm93744", "SchCalculationHelper::IntersectPolyPolygonWithRectangle");
	::basegfx::B2DPolyPolygon aRetval;
    const sal_uInt32 nCount(rPolyPolygon.count());

    for( sal_uInt32 i(0L); i < nCount; ++i )
    {
		aRetval.append(
			IntersectPolygonWithRectangle(
				rPolyPolygon.getB2DPolygon(i),
				rRectangle));
    }
 
	OSL_TRACE( "IntersectPolyPolygonWithRectangle: result has %d polygons", aRetval.count() );
	return aRetval;
}
    
::basegfx::B2DPolyPolygon SchCalculationHelper::IntersectPolygonWithRectangle( 
	const ::basegfx::B2DPolygon& rPolygon, 
	const ::basegfx::B2DRange& rRectangle)
{
	RTL_LOGFILE_CONTEXT_AUTHOR( context, "sch", "bm93744", "SchCalculationHelper::IntersectPolygonWithRectangle");
	::basegfx::B2DPolyPolygon aRetval;
	const ::basegfx::B2DRange aPolygonRange(::basegfx::tools::getRange(rPolygon));

	if(rRectangle.isInside(aPolygonRange))
	{
        OSL_TRACE( "IntersectPolygonWithRectangle: result has %d polygons", aRetval.count() );
		aRetval.append(rPolygon);
	}
	else
	{
		const sal_uInt32 nCount(rPolygon.count());
		::basegfx::B2DPoint aFrom;
		::basegfx::B2DPoint aTo;

		// set last point to a position outside the rectangle, such that the first
		// time clip2d returns true, the comparison to last will always yield false
		::basegfx::B2DPoint aLast(rRectangle.getMinimum() - ::basegfx::B2DPoint(-1.0, -1.0));
		::basegfx::B2DPolygon aNewSnippet;

		for(sal_uInt32 i(1L); i < nCount; i++)
		{
			aFrom = rPolygon.getB2DPoint(i - 1L);
			aTo = rPolygon.getB2DPoint(i);

			if(clip2d(aFrom, aTo, rRectangle))
			{
				// compose a Polygon of as many consecutive points as possible
				if(aFrom == aLast)
				{
					if(aTo != aFrom)
					{
						aNewSnippet.append(aTo);
					}
				}
				else
				{
					// create a Polygon and put it into the PolyPolygon
					if(aNewSnippet.count())
					{
						aRetval.append(aNewSnippet);
						aNewSnippet.clear();
					}

					// start new sequence
					aNewSnippet.append(aFrom);

					if(aTo != aFrom)
					{
						aNewSnippet.append(aTo);
					}
				}

				aLast = aTo;
			}
		}

		if(aNewSnippet.count())
		{
			aRetval.append(aNewSnippet);
		}

		OSL_TRACE( "IntersectPolygonWithRectangle: result has %d polygons", aRetval.count() );
	}

	return aRetval;
}

sal_Bool SchCalculationHelper::clip2d(
	::basegfx::B2DPoint& rPointA, 
	::basegfx::B2DPoint& rPointB, 
	const ::basegfx::B2DRange& rRectangle)
{
	//	Direction vector of the line.
	::basegfx::B2DVector aD(rPointB - rPointA);

	if(rPointA.equal(rPointB) && rRectangle.isInside(rPointA))
	{
		//	Degenerate case of a zero length line.
		return sal_True;
	}
	else
	{
		//	Values of the line parameter where the line enters resp. leaves the rectangle.
		double fTE(0.0);
		double fTL(1.0);
				
		//	Test wether at least a part lies in the four half-planes with respect to 
		//	the rectangles four edges.
		if (CLIPt(aD.getX(), rRectangle.getMinX() - rPointA.getX(), fTE, fTL))
		{
			if (CLIPt(-aD.getX(), rPointA.getX() - rRectangle.getMaxX(), fTE, fTL))
			{
				if (CLIPt(aD.getY(), rRectangle.getMinY() - rPointA.getY(), fTE, fTL))
				{
					if (CLIPt(-aD.getY(), rPointA.getY() - rRectangle.getMaxY(), fTE, fTL))
					{
						//	At least a part is visible.
						if (fTL < 1.0)
						{
							//	Compute the new end point.
							rPointB.setX(rPointA.getX() + (fTL * aD.getX()));
							rPointB.setY(rPointA.getY() + (fTL * aD.getY()));
						}

						if (fTE > 0.0)
						{
							//	Compute the new starting point.
							rPointA.setX(rPointA.getX() + (fTE * aD.getX()));
							rPointA.setY(rPointA.getY() + (fTE * aD.getY()));
						}

						return sal_True;
					}
				}
			}
		}
					
		//	Line is not visible.
		return sal_False;
	}
}

sal_Bool SchCalculationHelper::CLIPt(double fDenom, double fNum, double& fTE, double& fTL)
{
	double	fT;
	
	if(fDenom > 0.0)			//	Intersection enters: PE
	{
		fT = fNum / fDenom;		//	Parametric value at the intersection.
		if (fT > fTL)			//	fTE and fTL crossover
			return sal_False;	//	  therefore reject the line.
		else if (fT > fTE)		//	A new fTE has been found.
			fTE = fT;
	}
	else if(fDenom < 0.0)		//	Intersection leaves: PL
	{
		fT = fNum / fDenom;		//	Parametric Value at the intersection.
		if (fT < fTE)			//	fTE and fTL crossover
			return sal_False;	//	  therefore reject the line.
		else if (fT < fTL)		//	A new fTL has been found.
			fTL = fT;
	}
	else if(fNum > 0.0)
	{
		return sal_False;			//	Line lies on the outside of the edge.
	}
	
	return sal_True;
}


// --------------------------------------------------------------------------------

// Calculation of Splines

namespace
{

class lcl_SplineCalculation
{
public:
    typedef pair< double, double >   tPointType;
    typedef vector< tPointType >     tPointVecType;

    /** @descr creates an object that calculates cublic splines on construction

        @param rPoints  the points for which splines shall be calculated
        @param fY1FirstDerivation the resulting spline should have the first
               derivation equal to this value at the x-value of the first point
               of rPoints.  If fY1FirstDerivation is set to infinity, a natural
               spline is calculated.
        @param fYnFirstDerivation the resulting spline should have the first
               derivation equal to this value at the x-value of the last point
               of rPoints
     */
    lcl_SplineCalculation( const tPointVecType & rPoints,
                           double fY1FirstDerivation,
                           double fYnFirstDerivation );

    /** @descr this function corresponds to the function splint in [1].

        [1] Numerical Recipies in C, 2nd edition
            William H. Press, et al.,
            Section 3.3, page 116
    */
    double GetInterpolatedValue( double x );

private:
    /// a copy of the points given in the CTOR
    tPointVecType            m_aPoints;

    /// the result of the Calculate() method
    vector< double >         m_aSecDerivY;

    double m_fYp1;
    double m_fYpN;

    // these values are cached for performance reasons
    tPointVecType::size_type m_nKLow;
    tPointVecType::size_type m_nKHigh;
    double m_fLastInterpolatedValue;

    /** @descr this function corresponds to the function spline in [1].

        [1] Numerical Recipies in C, 2nd edition
            William H. Press, et al.,
            Section 3.3, page 115
    */
    void Calculate();
};

template< typename T, typename U >
    struct lcl_LessFirstOfPair : binary_function< pair< T, U >, pair< T, U >, bool >
{
    inline bool operator() ( const pair< T, U > & rOne, const pair< T, U > & rOther )
    {
        return ( rOne.first < rOther.first );
    }
};

template< typename T >
    struct lcl_EqualsFirstDoubleOfPair : binary_function< pair< double, T >, pair< double, T >, bool >
{
    inline bool operator() ( const pair< double, T > & rOne, const pair< double, T > & rOther )
    {
        return ( ::rtl::math::approxEqual( rOne.first, rOther.first ) );
    }
};

// assume Point given in 100th of a mm
//struct lcl_EqualsPointWithDPI : binary_function< Point, Point, bool >
//{
//    lcl_EqualsPointWithDPI( long nDPIX, long nDPIY ) :
//            m_fFactX( static_cast< double >( nDPIX ) / 2540.0 ),
//            m_fFactY( static_cast< double >( nDPIY ) / 2540.0 )
//    {}
//    
//    inline bool operator() ( const Point & p1, const Point & p2 )
//    {
//        bool bXEqual = false;
//        bool bYEqual = false;
//        if( m_fFactX > 0 )
//        {
//            bXEqual =
//                ( static_cast< long >( static_cast< double >( p1.getX()) * m_fFactX ) ==
//                  static_cast< long >( static_cast< double >( p2.getX()) * m_fFactX ) );
//        }
//        else
//        {
//            bXEqual = ( p1.getX() == p2.getX() );
//        }
//
//        if( bXEqual )
//        {
//            if( m_fFactY > 0 )
//            {
//                bYEqual =
//                    ( static_cast< long >( static_cast< double >( p1.getY()) * m_fFactY ) ==
//                      static_cast< long >( static_cast< double >( p2.getY()) * m_fFactY ) );
//            }
//            else
//            {
//                bYEqual = ( p1.getY() == p2.getY() );
//            }
//        }
//
//        return bXEqual && bYEqual;
//    }
//
//private:
//    double m_fFactX;
//    double m_fFactY;
//};

lcl_SplineCalculation::lcl_SplineCalculation(
    const tPointVecType & rPoints,
    double fY1FirstDerivation,
    double fYnFirstDerivation )
        : m_aPoints( rPoints ),
          m_fYp1( fY1FirstDerivation ),
          m_fYpN( fYnFirstDerivation ),
          m_nKLow( 0 ),
          m_nKHigh( rPoints.size() - 1 )
{
    ::rtl::math::setInf( &m_fLastInterpolatedValue, sal_False );

    sort( m_aPoints.begin(), m_aPoints.end(),
          lcl_LessFirstOfPair< double, double >() );

    // #108301# remove points that have equal x-values
    m_aPoints.erase( unique( m_aPoints.begin(), m_aPoints.end(),
                             lcl_EqualsFirstDoubleOfPair< double >() ),
                     m_aPoints.end() );

    Calculate();
}

void lcl_SplineCalculation::Calculate()
{
    // n is the last valid index to m_aPoints
    const tPointVecType::size_type n = m_aPoints.size() - 1;
    if( n < 1 )
        return;

    vector< double > u( n );
    m_aSecDerivY.resize( n + 1, 0.0 );

    if( ::rtl::math::isInf( m_fYp1 ) )
    {
        // natural spline
        m_aSecDerivY[ 0 ] = 0.0;
        u[ 0 ] = 0.0;
    }
    else
    {
        m_aSecDerivY[ 0 ] = -0.5;
        double xDiff = ( m_aPoints[ 1 ].first - m_aPoints[ 0 ].first );
        u[ 0 ] = ( 3.0 / xDiff ) *
            ((( m_aPoints[ 1 ].second - m_aPoints[ 0 ].second ) / xDiff ) - m_fYp1 );
    }

    for( tPointVecType::size_type i = 1; i < n; ++i )
    {
        pair< double, double >
            p_i = m_aPoints[ i ],
            p_im1 = m_aPoints[ i - 1 ],
            p_ip1 = m_aPoints[ i + 1 ];
        
        double sig = ( p_i.first - p_im1.first ) /
            ( p_ip1.first - p_im1.first );
        double p = sig * m_aSecDerivY[ i - 1 ] + 2.0;

        m_aSecDerivY[ i ] = ( sig - 1.0 ) / p;
        u[ i ] =
            ( ( p_ip1.second - p_i.second ) /
              ( p_ip1.first - p_i.first ) ) -
            ( ( p_i.second - p_im1.second ) /
              ( p_i.first - p_im1.first ) );
        u[ i ] =
            ( 6.0 * u[ i ] / ( p_ip1.first - p_im1.first )
              - sig * u[ i - 1 ] ) / p;
    }

    // initialize to values for natural splines (used for m_fYpN equal to
    // infinity)
    double qn = 0.0;
    double un = 0.0;

    if( ! ::rtl::math::isInf( m_fYpN ) )
    {
        qn = 0.5;
        double xDiff = ( m_aPoints[ n ].first - m_aPoints[ n - 1 ].first );
        un = ( 3.0 / xDiff ) *
            ( m_fYpN - ( m_aPoints[ n ].second - m_aPoints[ n - 1 ].second ) / xDiff );
    }

    m_aSecDerivY[ n ] = ( un - qn * u[ n - 1 ] ) * ( qn * m_aSecDerivY[ n - 1 ] + 1.0 );

    // note: the algorithm in [1] iterates from n-1 to 0, but as size_type
    // may be (usuall is) an unsigned type, we can not write k >= 0, as this
    // is always true.
    for( tPointVecType::size_type k = n; k > 0; --k )
    {
        ( m_aSecDerivY[ k - 1 ] *= m_aSecDerivY[ k ] ) += u[ k - 1 ];
    }
}

double lcl_SplineCalculation::GetInterpolatedValue( double x )
{
    DBG_ASSERT( m_aPoints.size() < 1 ||
                ( ( m_aPoints[ 0 ].first <= x ) &&
                  ( x <= m_aPoints[ m_aPoints.size() - 1 ].first ) ),
                "Trying to extrapolate" );

    const tPointVecType::size_type n = m_aPoints.size() - 1;
    if( n < 1 )
    {
        double fNan;
        ::rtl::math::setNan( & fNan );
        return fNan;
    }

    if( x < m_fLastInterpolatedValue )
    {
        m_nKLow = 0;
        m_nKHigh = n;

        // calculate m_nKLow and m_nKHigh
        // first initialization is done in CTOR
        while( m_nKHigh - m_nKLow > 1 )
        {
            tPointVecType::size_type k = ( m_nKHigh + m_nKLow ) / 2;
            if( m_aPoints[ k ].first > x )
                m_nKHigh = k;
            else
                m_nKLow = k;
        }
    }
    else
    {
        while( ( m_aPoints[ m_nKHigh ].first < x ) &&
               ( m_nKHigh <= n ) )
        {
            ++m_nKHigh;
            ++m_nKLow;
        }
        DBG_ASSERT( m_nKHigh <= n, "Out of Bounds" );
    }
    m_fLastInterpolatedValue = x;

    double h = m_aPoints[ m_nKHigh ].first - m_aPoints[ m_nKLow ].first;
    DBG_ASSERT( h != 0, "Bad input to GetInterpolatedValue()" );

    double a = ( m_aPoints[ m_nKHigh ].first - x ) / h;
    double b = ( x - m_aPoints[ m_nKLow ].first  ) / h;

    return ( a * m_aPoints[ m_nKLow ].second +
             b * m_aPoints[ m_nKHigh ].second +
             (( a*a*a - a ) * m_aSecDerivY[ m_nKLow ] +
              ( b*b*b - b ) * m_aSecDerivY[ m_nKHigh ] ) *
             ( h*h ) / 6.0 );
}

typedef lcl_SplineCalculation::tPointVecType::size_type lcl_tSizeType;

} //  anonymous namespace

// ----------------------------------------

::basegfx::B2DPolygon SchCalculationHelper::CalculateCubicSplines(
    const vector< pair< double, double > > & rPoints,
    sal_Int32 nGranularity)
{
	RTL_LOGFILE_CONTEXT_AUTHOR( context, "sch", "bm93744", "SchCalculationHelper::CalculateCubicSplines");
    DBG_ASSERT( nGranularity != 0, "Zero-Granularity is invalid" );
	::basegfx::B2DPolygon aRetval;

    // calculate second derivates
    double fInfty;
    ::rtl::math::setInf( &fInfty, sal_False );
    lcl_SplineCalculation aSpline( rPoints, fInfty, fInfty );

    // fill result polygon with calculated values
    const lcl_tSizeType nLastIndex = rPoints.size() - 1;

    // calculate all necessary points on spline curve
	::basegfx::B2DPoint aLast( -1.0, -1.0 );

    for( sal_uInt32 i(0L); i < nLastIndex; ++i )
    {
        double fBaseX = rPoints[ i ].first;
        double fInc = ( rPoints[ i + 1 ].first - fBaseX ) /
            static_cast< double >( nGranularity );

        for( sal_Int32 j = 0; j < nGranularity; ++j )
        {
            double x = fBaseX + ( fInc * static_cast< double >( j ) );
            const ::basegfx::B2DPoint aPoint( x , aSpline.GetInterpolatedValue( x ));

			if( aPoint != aLast )
            {
                aRetval.append( aPoint );
                aLast = aPoint;
            }
        }
    }
  
	// calculate last point
    const ::basegfx::B2DPoint aPoint( rPoints[ nLastIndex ].first , aSpline.GetInterpolatedValue( rPoints[ nLastIndex ].first ));
    if( aPoint != aLast )
		aRetval.append( aPoint );

	// reduce points
	aRetval.removeDoublePoints();

	return aRetval;
}

// eof
