/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: Properties.hxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: hbrinkm $ $Date: 2006/11/01 09:14:27 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

/*  Copyright 2005 Sun Microsystems, Inc. */

#ifndef INCLUDED_PROPERTIES_HXX
#define INCLUDED_PROPERTIES_HXX

#include <WriterFilterDllApi.hxx>

#ifndef INCLUDED_QNAME_HXX
#include <odiapi/qname/QName.hxx>
#endif

#include "Iterator.hxx"

#include <boost/shared_ptr.hpp>
#include <string>

namespace util { class Logger; }

namespace odiapi { namespace props {

    class PropertyBag;
    typedef boost::shared_ptr<PropertyBag> PropertyBag_Pointer_t;
    class PropertyPoolHandle;
    typedef boost::shared_ptr<PropertyPoolHandle> PropertyPoolHandle_Pointer_t;

    /** Properties are either simple id/value pairs or composite 
        properties (properties containing an undefined number of
        child properties).
    */
    class WRITERFILTER_DLLPUBLIC Property 
    {
    public:
        typedef boost::shared_ptr<Property> Pointer_t;

        /** Get the id of a property instance

        @return the property id 
        */
        virtual writerfilter::QName_t getId() const = 0;

        /** Return the integer value of a property.

        @attention Only StringProperty instances implement this method. 
        
        @return the integer value

	    @throws std::logic_error when this method is unsupported by the runtime instance
        */
        virtual int getIntValue() const = 0;
    
        /** Return the string value of a property.
        
		@attention Every derived class has to implement this method
        
        @return the string value 
        */
        virtual std::string getStringValue() const = 0;

        /** Get child property at position
        
        @attention Only composite property instances implement this method

        @return the requested property or an empty property

		@throws std::logic_error when this method is unsupported by the runtime instance
        */
        virtual Pointer_t getChild(int pos) const = 0;
	
        /** Get child by identifier
        
        @attention Only composite property instances implement this method. 
        
        @return the requested property or an empty property

		@throws std::logic_error when this method is unsupported by the runtime instance
        */
        virtual Pointer_t findChild(writerfilter::QName_t id) const = 0;

        /** Get the property bag hosted by this property
            @attention Only composite property instances support this method. 

            @return the property bag

            @throw std::logic_error when this method is unsupported by the runtime instance
        */
        virtual PropertyBag_Pointer_t getPropertyBag() const = 0;

	/** 
        @return true, iff the property has child properties

	 */
	virtual bool hasPropertyBag() const = 0;


        /** Comparison of two properties

    @return true if 'this' is less than 'other' false otherwise
        */
        virtual bool less(const Property& other) const = 0;

        /**
         */
        virtual void resolve(PropertyPoolHandle_Pointer_t realReference) const = 0;

        /**
         */
//        virtual bool hasFlag(int flag) const = 0;

		
		/** Destructor
         */
        virtual ~Property();
    };

    WRITERFILTER_DLLPUBLIC bool operator<(const Property::Pointer_t& first, const Property::Pointer_t& second);
    WRITERFILTER_DLLPUBLIC bool operator==(const Property::Pointer_t& first, const Property::Pointer_t& second);
    WRITERFILTER_DLLPUBLIC bool operator!=(const Property::Pointer_t& first, const Property::Pointer_t& second);

    /** A container for properties. May be used as either array 
        or structure but not as both at the same time.

        @attention Not thread-safe
    */
    class WRITERFILTER_DLLPUBLIC PropertyBag
    {
    public:
        /** The current number of elements

        @return the number of elements
        */
        virtual size_t size() const = 0;
	
        /** Remove all elements.
        
        Also resets the identity of the respective property bag instance
        (array or structure). After calling this method the client is again
		free to use the PropertyBag as array or structure.
        */
        virtual void clear() = 0;
	
        /** Create a copy of this property bag and its content.
         */
        virtual PropertyBag_Pointer_t copy() const = 0;

        /** Insert a property.
        
        The PropertyBag is used as structure after calling this method.

        @param property
        the property to insert

        @precond not used as array already

        @postcond must not be used as array after calling this method and before 
        calling clear.
        */
        virtual void insert(Property::Pointer_t property) = 0;

        /** Find a property by identifier. 

        @param id
        the identifier

        @returns the property or an empty property if search failed

        @precond PropertyBag is being used as structure
        */
        virtual Property::Pointer_t find(writerfilter::QName_t id) const = 0;

        /** Insert a property.
        
        The PropertyBag is used as array after calling this method.

        @param pos 
        the position at which to insert the property

        @param property
        the property
        */
        virtual void insert(int pos, Property::Pointer_t property) = 0;
      
        /** Find a property by position.
        
        @param pos
        the position

        @returns the property or an empty property if search failed
		
        @precond PropertyBag is being used as array.
        */
        virtual Property::Pointer_t get(int pos) const = 0;

        /** Sort the properties in this bag. 
         */
        virtual void sort() = 0;

        /** Get an iterator to travers properties in this PropertyBag

        @returns an iterator 
        */
        virtual Iterator<Property::Pointer_t>::Pointer_t createIterator() = 0;

        /** Destructor
         */
        virtual ~PropertyBag();
    };

    /** A weak reference to an instances of a arbitrary class. 
        A weak reference refers to an instance of a class but 
        is not the owner. If the refered to instance will be 
        destroyed the weak reference will be informed and must 
        get rid of its reference.
    */
    class WeakReference 
    {
    public:
        /** Get rid of the instance to which this reference
            refers to. The instance to which this WeakRefence 
            refers might no longer be valid after this call.
        */ 
        virtual void disposing() = 0;

        /** Destructor
         */
        virtual ~WeakReference();
    };

    /* An abstract target of weak references 
     */
    class Subject
    {
    public:
        /** Register a WeakReference as listener for disposing events.
        
        @param ref
        the listener
        */
        virtual void registerWeakReference(WeakReference* ref) = 0;
	
        /** Unregister a WeakReferenc
        
        @attention the listener must not call this method during 'disposing' calls

        @param ref
        the listener
        */
        virtual void unregisterWeakReference(WeakReference* ref) = 0;

        /** Destructor
         */
        virtual ~Subject();
    };

  
    /** An abstract reference to a bag of properties in 
        a PropertyPool. 
    */
    class WRITERFILTER_DLLPUBLIC PropertyPoolHandle : public WeakReference
    {
    public:

        /** Get the PropertyBag.
        
        @return the property bag related to this PropertyPoolHandle
        */
        virtual PropertyBag_Pointer_t getPropertyBag() const = 0;
	
        /** Destructor
         */
        virtual ~PropertyPoolHandle();
    };

    /** An abstract container for PropertyBag's
	  
    @attention Not thread-safe
    */
    class WRITERFILTER_DLLPUBLIC PropertyPool
    {
    public:
        typedef boost::shared_ptr<PropertyPool> Pointer_t;

        /** Insert a PropertyBag into the property pool. 

            The pool makes a copy of the provided PropertyBag. 

            @param propertyBag
            the PropertyBag to insert

            @return a PropertyPoolHandle
        */
        virtual PropertyPoolHandle_Pointer_t insert(PropertyBag_Pointer_t propertyBag) = 0;

        /** Free unused memory in the pool
         */
        virtual void garbageCollection() = 0;

        /** Get all PropertyBags in this pool which are still in use.

	    @return An iterator to travers the PropertyBags still in use
		in the PropertyPool.
        */
        virtual Iterator<PropertyBag_Pointer_t>::Pointer_t createIterator() const = 0;

        /** Dump the content of the pool. 

  	    This method is meant for debugging purposes.
        
        @param logger
        an instance inplementing the Logger interface.
        */
        virtual void dump(util::Logger* logger) const = 0;
		
        /** Destructor
         */
        virtual ~PropertyPool();
    };

    /** Comparison operator
     */
    WRITERFILTER_DLLPUBLIC bool operator<(const PropertyPoolHandle_Pointer_t& first, const PropertyPoolHandle_Pointer_t& second);
    WRITERFILTER_DLLPUBLIC bool operator==(const PropertyPoolHandle_Pointer_t& first, const PropertyPoolHandle_Pointer_t& second);
    WRITERFILTER_DLLPUBLIC bool operator!=(const PropertyPoolHandle_Pointer_t& first, const PropertyPoolHandle_Pointer_t& second);

    /** Factory methods
     */
    WRITERFILTER_DLLPUBLIC Property::Pointer_t createIntegerProperty(writerfilter::QName_t id, int value);

    /** Create a positional or dimensional property. 

    @param id
    The identifier of the property.

    @param valueAndUnit
    A value plus measurement unit. Allowed units are 'cm', 'mm', 'inch', 'pt', 'px', 'pc'
    e.g. '1.0cm' or '3.11 inch'. It doesn't matter whether there is a space between 
    the value and the measurment unit or not.

    @throws std::invalid_argument when the provided 'valueAndUnit' doesn't contain one of the
    above mentioned measurement units or the decimal separator is not a '.' see definition
    of the Relax-NG definitions for custom data types in the OASIS OpenDocument specification 
    for details http://docs.oasis-open.org/office/v1.0

    @discussion
    The method 'getIntValue' of the returned property returns the provided 'valueAndUnit'
    converted to twips while 'getStringValue' returns 'valueAndUnit' converted to 'cm' 
    with a space between value and unit.
    */
    WRITERFILTER_DLLPUBLIC Property::Pointer_t createTwipsProperty(writerfilter::QName_t id, const std::string& valueAndUnit);

    /** Create a positional or dimensional property.

    @param id
    The identifier of the property.
  
    @param valueInTwips
    A value in twips.

    @discussion
    The method 'getIntValue' of the returned property returns 'valueInTwips' as provided while
    'getStringValue' returns 'valueInTwips' converted to 'cm' with a space between value and 
    unit.
    */
    WRITERFILTER_DLLPUBLIC Property::Pointer_t createTwipsProperty(writerfilter::QName_t id, int valueInTwips);

    WRITERFILTER_DLLPUBLIC Property::Pointer_t createStringProperty(writerfilter::QName_t id, const std::string& value);
    WRITERFILTER_DLLPUBLIC Property::Pointer_t createCompositeProperty(writerfilter::QName_t id, PropertyPoolHandle_Pointer_t poolHandle);
    WRITERFILTER_DLLPUBLIC Property::Pointer_t createReferenceProperty(writerfilter::QName_t id, writerfilter::QName_t referenceProxy);
    WRITERFILTER_DLLPUBLIC PropertyBag_Pointer_t createPropertyBag();
    WRITERFILTER_DLLPUBLIC PropertyPool::Pointer_t createPropertyPool();

} } // namespace odiapi { namespace props {

#endif // INCLUDED_PROPERTIES_HXX


