/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: folder.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 14:42:23 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
#include <sys/stat.h>
#include <dirent.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>

#define private public
#include <fastfsunx.hxx>
#include <shortcut.hxx>
#undef private

#include <list.hxx>

#include <vos/timer.hxx>
#include <vos/mutex.hxx>

DECLARE_LIST( FolderList, Folder* );

// uncomment the next define to fill in the iconlocation field in getitemidinfo
// this does not make sense at the moment since CHAOS ignores this field
// beware: that operation is not cheap !
//#define ITEMINFO_GET_ICONLOCATION

#define PDATA ((FolderData*)pData)
#define STRPTR() ((String*)PDATA->maPath.pData)

static BOOL bShowExtensions = TRUE;
static String* pDesktopDir = NULL;
static String* pBookmarksDir = NULL;
static FolderList aUpdateFolders;

int		(*pDtIntegrator_GetDesktopEntryCount)( const String& ) = 0;
String	(*pDtIntegrator_GetDesktopEntry)(const String&, int ) = 0;
BOOL	(*pDtIntegrator_GetDesktopEntryInfo)(const String&, FastItemInfo& ) = 0;
String	(*pDtIntegrator_GetDesktopDir)() = 0;
BOOL	(*pDtIntegrator_GetIconLocation)( char*, BOOL, String& ) = 0;
String	(*pDtIntegrator_GetBookmarksDir)() = 0;


class UpdateNotifier : public NAMESPACE_VOS(OTimer)
{
  private:
	Folder*		m_pFolder;
	UINT32		m_nLastChecksum;
  public:
	UpdateNotifier( Folder*, ULONG nMillis );
	virtual ~UpdateNotifier();

	virtual void onShot();
};

struct FolderData
{
	ItemIDPath		maPath;
	FindFlags		mnFlags;
	BOOL			mbIsVirtual;
	BOOL			mbIsDir; // should always be true, this is a paranois setting
	BOOL			mbIsValid;
	int				mnCurrentEntry; // also used by virtual urls
	struct dirent*	mpCurrentEntry;
	DIR*			mpDIR;
	Link			maUpdateLink;
	UpdateNotifier*	mpUpdateNotifier;
	Folder*			mpRealFolder;
	NAMESPACE_VOS(OMutex) aMutex; // to protect critical sections
};

UpdateNotifier::UpdateNotifier( Folder* pFolder, ULONG nMillis ) :
		OTimer( NAMESPACE_VOS(TTimeValue)( nMillis ) ),
		m_pFolder( pFolder )
{
	m_nLastChecksum = m_pFolder->BuildCRC();
}

UpdateNotifier::~UpdateNotifier()
{
}

void UpdateNotifier::onShot()
{
	if( aUpdateFolders.GetPos( m_pFolder ) != LIST_ENTRY_NOTFOUND )
	{
		NAMESPACE_VOS(OGuard) aGuard( ((FolderData*)m_pFolder->pData)->aMutex );
		UINT32 nNewCRC = m_pFolder->BuildCRC();
		if( nNewCRC != m_nLastChecksum )
		{
			m_nLastChecksum = nNewCRC;
			((FolderData*)m_pFolder->pData)->maUpdateLink.Call( m_pFolder );
		}
		start();
	}
}

#define INIT_CURRENT_ENTRY -2

#define FIND_FLAGS_LOCAL_ALL (  FIND_FLAG_INCLUDE_NONFOLDER	| \
								FIND_FLAG_INCLUDE_FOLDER	| \
								FIND_FLAG_INCLUDE_HIDDEN	| \
								FIND_FLAG_INCLUDE_FILES )
static BOOL TestMatch( const String&rPath,
					   struct stat* pStat, FindFlags aFlags,
					   BOOL bAlwaysStat = FALSE)
{
	BOOL bMatch = FALSE;
	if( ( aFlags & FIND_FLAGS_LOCAL_ALL ) == FIND_FLAGS_LOCAL_ALL )
	{
		bMatch = TRUE;
		if( bAlwaysStat && rPath.Len() )
			lstat( (char*)rPath.GetStr(), pStat );
	}
	else
	{
		if( ! lstat( (char*)rPath.GetStr(), pStat ) )
		{
			UINT32 nMode = pStat->st_mode;
			if( S_ISLNK( nMode ) )
			{
				struct stat aStat;
				if( ! stat( (char*)rPath.GetStr(), &aStat ) )
					nMode = aStat.st_mode;
			}
			
			if( aFlags & FIND_FLAG_INCLUDE_NONFOLDER	&&
				! S_ISDIR( nMode ) )
				bMatch = TRUE;
			if( aFlags & FIND_FLAG_INCLUDE_FOLDER		&&
				S_ISDIR( nMode ) )
				bMatch = TRUE;
			if( aFlags & FIND_FLAG_INCLUDE_FILES		&&
				S_ISREG( nMode ) )
				bMatch = TRUE;
			int nTokens = rPath.GetTokenCount( '/' );
			if( rPath.GetToken( nTokens-1, '/' )[0] == '.' )
				bMatch = aFlags & FIND_FLAG_INCLUDE_HIDDEN ?
					bMatch : FALSE;
		}
	}
	return bMatch;
}

void SplitVirtualURL( const String& rIn, String& rVOut, String& rOut )
{
	int nTokens = rIn.GetTokenCount( VURLB[0] );
	if( nTokens < 2 )
	{
		rOut = rIn;
		rVOut = String();
		return;
	}

	rVOut = VURLB;
	rVOut += rIn.GetToken( nTokens-1, VURLB[0] );
	int nPos = rVOut.Search( VURLC[0] );
	rOut = rVOut.Copy( nPos+1 );
	rVOut.Erase( nPos+1 );

	if( rVOut == VIRTUAL_WORKPLACE_URL )
	{
		rVOut = String();
		if( ! rOut.Len() )
			rOut = "/";
	}
	else if( rVOut.Compare( VURLB "linkto", 7 ) == COMPARE_EQUAL )
	{
		// get desktop items pointing to filesystem
		// this has to be done to maintain the virtual URL Hierarchy
		rVOut.Erase( 0, 7 );
		rVOut.Erase( rVOut.Len()-1, 1 );
		while( rVOut.SearchAndReplace( (char)0x7f, '/' ) != STRING_NOTFOUND ) ;
		rOut = rVOut + rOut;
		rVOut = String();
	}
	else if( rVOut == VIRTUAL_DESKTOP_URL )
	{
		if( pDtIntegrator_GetDesktopDir && ! pDesktopDir )
		{
			pDesktopDir = new String( pDtIntegrator_GetDesktopDir() );
			if( ! pDesktopDir->Len() )
			{
				delete pDesktopDir;
				pDesktopDir = NULL;
			}
		}
	
		if( pDesktopDir && rOut.Len() )
		{
			rVOut = String();
			rOut = *pDesktopDir + rOut;
		}
	}
	else if( rVOut == VIRTUAL_BOOKMARKS_URL )
	{
		if( pDtIntegrator_GetBookmarksDir && ! pBookmarksDir )
		{
			pBookmarksDir = new String( pDtIntegrator_GetBookmarksDir() );
			if( ! pBookmarksDir->Len() )
			{
				delete pBookmarksDir;
				pBookmarksDir = NULL;
			}
		}

		if( pBookmarksDir && rOut.Len() )
		{
			rVOut = String();
			rOut = *pBookmarksDir + rOut;
		}
	}
}

Folder* Folder::pRoot = 0;

void Folder::DestroyRoot()
{
	if( pRoot )
	{
		delete pRoot;
		pRoot = 0;
	}
}

Folder& Folder::GetRootFolder()
{
	if( ! pRoot )
	{
		pRoot = new Folder( ItemIDPath( FOLDER_ROOT ) );
		atexit( Folder::DestroyRoot );
	}
	return *pRoot;
}

BOOL Folder::IsAvailable()
{
	return TRUE;
}

void Folder::ShowExtensions( BOOL bShow )
{
	bShowExtensions = bShow;
}

Folder::Folder( const ItemIDPath &rPath, FindFlags aFlags )
{
	struct stat aStat;
	
	pData = new FolderData;
	PDATA->maPath = rPath;
	PDATA->mnFlags = aFlags;
	PDATA->mnCurrentEntry = INIT_CURRENT_ENTRY;
	PDATA->mpCurrentEntry = NULL;
	PDATA->mpDIR = NULL;
	PDATA->mpRealFolder = NULL;
	PDATA->mpUpdateNotifier = NULL;

	String aVirt, aEffUrl;
	SplitVirtualURL( *STRPTR(), aVirt, aEffUrl );
	if( ! aVirt.Len() )
	{
		if( ! stat( ((String*)rPath.pData)->GetStr(), &aStat ) )
		{
			PDATA->mbIsValid = TRUE;
			PDATA->mbIsDir = S_ISDIR( aStat.st_mode ) ? TRUE : FALSE;
		}
		else
			PDATA->mbIsValid = FALSE;
	}
	else
		PDATA->mbIsValid = TRUE;
}

Folder::~Folder()
{
	while( aUpdateFolders.GetPos( this ) != LIST_ENTRY_NOTFOUND )
		aUpdateFolders.Remove( this );
	if( PDATA->mpUpdateNotifier )
	{
		PDATA->mpUpdateNotifier->stop();
		PDATA->mpUpdateNotifier->release();;
	}
	if( pData )
	{
		if( PDATA->mpDIR )
			closedir( PDATA->mpDIR );
		if( PDATA->mpRealFolder )
			delete PDATA->mpRealFolder;
		delete PDATA;
	}
}

BOOL Folder::IsCaseSensitive() const
{
	return TRUE;
}

ItemIDPath Folder::GetItemIDPath() const
{
	return PDATA->maPath;
}

BOOL Folder::GetNextItem( ItemIDPath& rPath )
{
	NAMESPACE_VOS(OGuard) aGuard( PDATA->aMutex );

	String aVirt, aEffUrl;
	SplitVirtualURL( *STRPTR(), aVirt, aEffUrl );
	struct stat aStat;
	
	if( ! aVirt.Len() )
	{
		// im / als erstes {desktop} und {bookmarks} unterbringen
		if( PDATA->mnCurrentEntry < 0 && aEffUrl == "/" )
		{
			if( rPath.pData )
			{
				delete (String*)rPath.pData;
				rPath.pData = NULL;
			}

			if( PDATA->mnCurrentEntry == -2 )
			{
				if(
					( pDtIntegrator_GetDesktopDir && pDtIntegrator_GetDesktopDir().Len() )
					||
					( pDtIntegrator_GetDesktopEntryCount && pDtIntegrator_GetDesktopEntryCount( String() ) > 0 )
					)
				{
					rPath.pData = new String( VIRTUAL_DESKTOP_URL );
				}
				else
					PDATA->mnCurrentEntry++;
			}
			if( PDATA->mnCurrentEntry == -1 )
			{
				if( pDtIntegrator_GetBookmarksDir &&
					pDtIntegrator_GetBookmarksDir().Len() )
				{
					rPath.pData = new String( VIRTUAL_BOOKMARKS_URL );
				}
			}
			PDATA->mnCurrentEntry++;
			if( rPath.pData )
				return TRUE;
		}
		char* pName = NULL;
		if( ! PDATA->mpDIR )
			PDATA->mpDIR = opendir( aEffUrl.GetStr() );
		if( ! PDATA->mpDIR )
			return FALSE;
		
		if( PDATA->mnCurrentEntry < 0 )
			PDATA->mnCurrentEntry = 0;
		
		do
		{
			PDATA->mpCurrentEntry = readdir( PDATA->mpDIR );
			PDATA->mnCurrentEntry++;
			if( ! PDATA->mpCurrentEntry )
			{
				closedir( PDATA->mpDIR );
				PDATA->mpDIR = 0;
				PDATA->mnCurrentEntry = INIT_CURRENT_ENTRY;
				return FALSE;
			}
			pName = PDATA->mpCurrentEntry->d_name;
		} while( ! strcmp( ".", pName ) || ! strcmp( "..", pName ) );

		String aTmpPath( aEffUrl );
		aTmpPath += "/";
		aTmpPath += pName;
		
		if( TestMatch( aTmpPath, &aStat, PDATA->mnFlags ) )
		{
			if( rPath.pData )
				delete (String*)rPath.pData;
			rPath.pData = new String( pName );
			return TRUE;
		}
		return GetNextItem( rPath );
	}
	
	if( aVirt == VIRTUAL_BOOKMARKS_URL && pBookmarksDir && pBookmarksDir->Len() )
	{
		if( PDATA->mnCurrentEntry < 0 )
		{
			PDATA->mpRealFolder = new Folder( ItemIDPath( *pBookmarksDir ) );
			PDATA->mnCurrentEntry = 0;
		}
	}
	else if( aVirt == VIRTUAL_DESKTOP_URL && pDesktopDir && pDesktopDir->Len() )
	{
		if( PDATA->mnCurrentEntry < 0 )
		{
			PDATA->mpRealFolder = new Folder( ItemIDPath( *pDesktopDir ) );
			PDATA->mnCurrentEntry = 0;
		}
	}
	if( PDATA->mpRealFolder )
	{
		if( PDATA->mpRealFolder->GetNextItem( rPath ) )
		{
			PDATA->mnCurrentEntry++;
			return TRUE;
		}
		else
		{
			delete PDATA->mpRealFolder;
			PDATA->mpRealFolder = NULL;
			PDATA->mnCurrentEntry = INIT_CURRENT_ENTRY;
			return FALSE;
		}
	}

	if( aVirt == VIRTUAL_DESKTOP_URL )
	{
		String aDeskFile;
		// Anbindung an DtIntegrator: Einfgen des Desktops
		// Fall 1: DesktopDir existiert
		PDATA->mnCurrentEntry++;
		if( PDATA->mnCurrentEntry < 0 )
			PDATA->mnCurrentEntry = 0;
		// Fall 2: der Desktop ist etwas virtuelles und nur der DtIntegrator
		// kennt ihn
		if( pDtIntegrator_GetDesktopEntryCount		&&
			pDtIntegrator_GetDesktopEntry			)
		{
			aEffUrl.EraseTrailingChars( '/' );
			aDeskFile = pDtIntegrator_GetDesktopEntry( aEffUrl, PDATA->mnCurrentEntry );
			if( ! aDeskFile.Len() )
				PDATA->mnCurrentEntry = INIT_CURRENT_ENTRY;
		}

		String aNext( *STRPTR() );
		aNext.EraseTrailingChars( '/' );
		aNext += "/";
		if( aDeskFile.Len() )
		{
			if( ! stat( aDeskFile.GetStr(), &aStat ) )
			{
				// this seems to be an existing file !
			
				// CHAOS would not understand if the resulting Path had
				// another hierarchy than {Folder}/{Child}
				// so pack an absolute filename in another virtual URL tag
				// that can maintain the virtual hierarchy
				// for this to work replace / by another character
				while( aDeskFile.SearchAndReplace('/', (char)0x7f ) != STRING_NOTFOUND ) ;
				aNext += VURLB "linkto";
				aNext += aDeskFile;
				aNext += VURLC;
			}
			else
				aNext += aDeskFile;
		}
		if( PDATA->mnCurrentEntry < 0 )
			return FALSE;
		
		if( rPath.pData )
			delete (String*)rPath.pData;
		rPath.pData = new String( aNext );
		return TRUE;
	}
	return FALSE;
}

BOOL Folder::RestartEnum( FindFlags aFlags )
{
	PDATA->mnFlags = aFlags;
	PDATA->mnCurrentEntry = INIT_CURRENT_ENTRY;		
	if( PDATA->mpDIR )
	{
		closedir( PDATA->mpDIR );
		PDATA->mpDIR = NULL;
		PDATA->mpCurrentEntry = NULL;
	}
	if( PDATA->mpRealFolder )
	{
		delete PDATA->mpRealFolder;
		PDATA->mpRealFolder = NULL;
	}
	return TRUE;
}

UINT32 Folder::BuildCRC( FindFlags aFlags )
{
	RestartEnum( aFlags );
	UINT32 nCRC = 0;
	struct stat aStat;

	String aVirt, aEffUrl;
	SplitVirtualURL( *STRPTR(), aVirt, aEffUrl );
	if( !aVirt.Len() )
	{
		DIR* pDIR = opendir( aEffUrl.GetStr() );
		if( pDIR )
		{
			dirent* pDirent;
			while( pDirent = readdir( pDIR ) )
			{
				String aTmpPath( aEffUrl );
				aTmpPath += "/";
				aTmpPath += pDirent->d_name;
				if( TestMatch( aTmpPath, &aStat, PDATA->mnFlags, TRUE ) )
					nCRC ^= aStat.st_mtime;
			}
			closedir( pDIR );
		}
	}
	return nCRC;
}

BOOL Folder::GetItemIDInfo( const ItemIDPath& rPath, FastItemInfo& rInfo )
{
	ItemIDPath aTmpPath( PDATA->maPath );
	aTmpPath += rPath;

	String aVirt, aEffUrl;
	SplitVirtualURL( *((String*)aTmpPath.pData), aVirt, aEffUrl );

	if( !aVirt.Len() )
	{
		struct stat aStat;
		if( ! lstat( aEffUrl.GetStr(), &aStat ) )
		{
			int nTokens = aEffUrl.GetTokenCount( '/' );
			uid_t nUID = geteuid();
			gid_t nGID = getegid();
			
			rInfo.aItemKind = ITEM_KIND_FILESYSTEM;
			rInfo.aDisplayName = aEffUrl.GetToken( nTokens-1 , '/' );
			rInfo.aIconLocation = String();
#if defined ITEMINFO_GET_ICONLOCATION
			if( pDtIntegrator_GetIconLocation )
			{
				String aLocation;
				if( pDtIntegrator_GetIconLocation( aEffUrl.GetStr(), FALSE,
												   aLocation ) )
				{
					rInfo.aIconLocation = "LoadIcon:";
					rInfo.aIconLocation += aLocation;
				}
			}
#endif
			rInfo.aAttributes = 0;

			USHORT nPos = 0;
			USHORT nTokenCount = rInfo.aDisplayName.GetTokenCount( '.' );
			String aExtension( rInfo.aDisplayName.GetToken( nTokenCount - 1, '.', nPos ) );
			aExtension.EraseTrailingChars( '/' );
			if( S_ISLNK( aStat.st_mode ) )
			{
				rInfo.aAttributes |= ITEM_FLAG_ISLINK;
				stat( aEffUrl.GetStr(), &aStat );
			}
			if( aExtension == "url" || aExtension == "kdelnk" )
				rInfo.aAttributes |= ITEM_FLAG_ISLINK;

			if( S_ISDIR( aStat.st_mode ) )
				rInfo.aAttributes |=
					ITEM_FLAG_ISFOLDER		|
					ITEM_FLAG_HASSUBFOLDERS	|
					ITEM_FLAG_HASITEMS
					;
			else if( ! bShowExtensions )
			{
				USHORT nPos = rInfo.aDisplayName.SearchCharBackward( "." );
				if( nPos != STRING_NOTFOUND && nPos > 0 )
					rInfo.aDisplayName.Cut( nPos );
			}
			// check COPYABLE, LINKABLE -> read permission
			if( ( nUID == aStat.st_uid && aStat.st_mode & S_IRUSR )		||
				( nGID == aStat.st_gid && aStat.st_mode & S_IRGRP )		||
				aStat.st_mode & S_IROTH )
				rInfo.aAttributes |= ITEM_FLAG_COPYABLE | ITEM_FLAG_LINKABLE;
			// check DISCARDABLE, MOVEABLE, RENAMEABLE -> write permission
			if( ( nUID == aStat.st_uid && aStat.st_mode & S_IWUSR )		||
				( nGID == aStat.st_gid && aStat.st_mode & S_IWGRP )		||
				aStat.st_mode & S_IWOTH )
				rInfo.aAttributes |=
					ITEM_FLAG_MOVABLE		|
					ITEM_FLAG_DISCARDABLE	|
					ITEM_FLAG_RENAMABLE
					;
			
			return TRUE;
		}
	}
	else if( aVirt == VIRTUAL_DESKTOP_URL )
	{
		if( ! aEffUrl.Len() )
		{
			// the desktop itself
			rInfo.aItemKind		= ITEM_KIND_OTHER;
			rInfo.aDisplayName	= "Desktop";
			rInfo.aIconLocation	= "";
			rInfo.aAttributes	= ITEM_FLAG_ISFOLDER | ITEM_FLAG_HASSUBFOLDERS | ITEM_FLAG_HASITEMS;
			return TRUE;
		}
		else if( pDtIntegrator_GetDesktopEntryInfo )
		{
			return pDtIntegrator_GetDesktopEntryInfo( aEffUrl, rInfo );
		}
	}
	else if( aVirt == VIRTUAL_BOOKMARKS_URL )
	{
		if( ! aEffUrl.Len() )
		{
			rInfo.aItemKind		= ITEM_KIND_OTHER;
			rInfo.aDisplayName	= "Bookmarks";
			rInfo.aIconLocation	= "";
			rInfo.aAttributes	= ITEM_FLAG_ISFOLDER | ITEM_FLAG_HASSUBFOLDERS | ITEM_FLAG_HASITEMS;
			return TRUE;
		}
	}

	return FALSE;
}

BOOL Folder::GetFileInfo( const ItemIDPath& rPath, FastFileInfo& rInfo )
{
	ItemIDPath aTmpPath( PDATA->maPath );
	aTmpPath += rPath;

	String aVirt, aEffUrl;
	SplitVirtualURL( *((String*)aTmpPath.pData), aVirt, aEffUrl );

	if( !aVirt.Len() )
	{
		struct stat aStat;
		if( ! stat( aEffUrl.GetStr(), &aStat ) )
		{
			int nTokens = aEffUrl.GetTokenCount('/');
			uid_t nUID = geteuid();
			gid_t nGID = getegid();
			Date aDate( 1,1,1970 );

			rInfo.aAttributes = 0;
			rInfo.aFileSizeHigh = 0;
			rInfo.aAlternateName = String();
			rInfo.aFileName = aEffUrl.GetToken( nTokens-1, '/' );
			rInfo.aCreationTime.MakeDateTimeFromSec( aDate, aStat.st_atime );
			rInfo.aCreationTime.ConvertToLocalTime();
			rInfo.aLastAccessTime.MakeDateTimeFromSec( aDate, aStat.st_ctime );
			rInfo.aLastAccessTime.ConvertToLocalTime();
			rInfo.aLastWriteTime.MakeDateTimeFromSec( aDate, aStat.st_mtime );
			rInfo.aLastWriteTime.ConvertToLocalTime();
			rInfo.aFileSize = aStat.st_size;

			if( S_ISDIR( aStat.st_mode ) )
			{
				rInfo.aFileKind = FILE_KIND_DIRECTORY;
				rInfo.aAttributes |= FILE_FLAG_DIRECTORY;
			}
			else
				rInfo.aFileKind = FILE_KIND_FILE;
		
			// check write permission
			if( ! ( ( nUID == aStat.st_uid && aStat.st_mode & S_IWUSR )		||
					( nGID == aStat.st_gid && aStat.st_mode & S_IWGRP )		||
					aStat.st_mode & S_IWOTH )
				)
				rInfo.aAttributes |= FILE_FLAG_READONLY;
			if( ! S_ISDIR( aStat.st_mode ) )
			{
				if( ( nUID == aStat.st_uid && aStat.st_mode & S_IXUSR )		||
					( nGID == aStat.st_gid && aStat.st_mode & S_IXGRP )		||
					aStat.st_mode & S_IXOTH
					)
					rInfo.aAttributes |= FILE_FLAG_EXECUTABLE;
			}
		
			if( rInfo.aFileName[0] == '.' )
				rInfo.aAttributes |= FILE_FLAG_HIDDEN;
			
			return TRUE;
		}
	}
	return FALSE;
}

BOOL Folder::GetLinkFileInfo( const ItemIDPath& rPath, LinkFileInfo& rInfo )
{
	ItemIDPath aTmpPath( PDATA->maPath );
	aTmpPath += rPath;

	String aVirt, aEffUrl;
	SplitVirtualURL( *((String*)aTmpPath.pData), aVirt, aEffUrl );

	if( !aVirt.Len() )
	{
		struct stat aStat;
		if( ! lstat( aEffUrl.GetStr(), &aStat ) &&
			S_ISLNK( aStat.st_mode )
			)
		{
			int nBytes;
			char pBuf[ 1024 ];
			nBytes = readlink( aEffUrl.GetStr(), pBuf, sizeof( pBuf )-1 );
			if( nBytes != -1 )
			{
				pBuf[ nBytes ] = 0;
				if( *pBuf == '/' )
				{
					rInfo.aTargetPath = String( VIRTUAL_DESKTOP_URL "/" VIRTUAL_WORKPLACE_URL "/" );
					rInfo.aTargetPath += String( pBuf );
				}
				else
				{
					rInfo.aTargetPath = PDATA->maPath;
					rInfo.aTargetPath += String( pBuf );
				}
				rInfo.aTargetURL = rInfo.aTargetPath.GetFileDescription();
				return TRUE;
			}
		}
	}
	return FALSE;
}

Link Folder::InstallChangeNotifier( const Link& rLink )
{
	static char* pEnv = getenv( "SAL_CHANGENOTIFIER_UPDATEINTERVAL" );
	static ULONG nInterval = 1;
	if( nInterval == 1 )
		nInterval = pEnv ? 1000 * (ULONG)String( pEnv ) : 0;
	
	Link aRet = PDATA->maUpdateLink;
	PDATA->maUpdateLink = rLink;

	if( ! PDATA->mpUpdateNotifier && nInterval )
	{
		aUpdateFolders.Insert( this );
		PDATA->mpUpdateNotifier = new UpdateNotifier( this, nInterval );
		PDATA->mpUpdateNotifier->acquire();
		PDATA->mpUpdateNotifier->start();
	}

	return aRet;
}

BOOL Folder::RenameItem( const ItemIDPath& rPath, ItemIDPath& rNewPath, const String& rNewName )
{
	ItemIDPath aFromPath( PDATA->maPath );
	aFromPath += rPath;

	String aVirt, aFromUrl;
	SplitVirtualURL( *((String*)aFromPath.pData), aVirt, aFromUrl );

	if( !aVirt.Len() )
	{
		String aToUrl( aFromUrl );
		// replace display name in aToUrl
		int nTokens = aToUrl.GetTokenCount( '/' );
		String aName( rNewName );
		String aOldName = aFromUrl.GetToken( nTokens-1, '/' );
		int nOldTokens = aOldName.GetTokenCount( '.' );
		String aOldExtension ( aOldName.GetToken( nOldTokens - 1, '.' ));

		if(    (   !bShowExtensions 
				|| (aOldExtension.ICompare("url") == COMPARE_EQUAL) ) 
			&& (   (aOldName[0] != '.' && nOldTokens > 1) 
				|| nOldTokens > 2 )
		  )
		{
			struct stat aStat;
			if( ! stat( aFromUrl.GetStr(), & aStat ) )
			{
				if( ! S_ISDIR( aStat.st_mode ) )
				{
					aName += '.';
					aName += aOldExtension;
				}
			}
			else
				return FALSE;
		}
		
		aToUrl.SetToken( nTokens-1, '/', aName );

		rNewPath = ItemIDPath( rPath );
		nTokens = (*((String*)rNewPath.pData)).GetTokenCount( '/' );
		(*((String*)rNewPath.pData)).SetToken( nTokens-1, '/', aName );

		// #68639#
		// on some nfs connections Linux <-> Solaris rename with
		// equal from and to leads to destruction of file
		if( aFromUrl != aToUrl )
			return rename( aFromUrl.GetStr(), aToUrl.GetStr() ) ? FALSE : TRUE;
		return TRUE;
	}
	return FALSE;
}

BOOL Folder::RenameFile( const ItemIDPath& rPath, ItemIDPath& rNewPath, const String& rNewName )
{
	ItemIDPath aFromPath( PDATA->maPath );
	aFromPath += rPath;

	String aVirt, aFromUrl;
	SplitVirtualURL( *((String*)aFromPath.pData), aVirt, aFromUrl );

	if( !aVirt.Len() )
	{
		String aToUrl( aFromUrl );
		int nTokens = aToUrl.GetTokenCount( '/' );
		aToUrl.SetToken( nTokens-1, '/', rNewName );

		rNewPath = ItemIDPath( rPath );
		nTokens = (*((String*)rNewPath.pData)).GetTokenCount( '/' );
		(*((String*)rNewPath.pData)).SetToken( nTokens-1, '/', rNewName );

		// #68639#
		// on some nfs connections Linux <-> Solaris rename with
		// equal from and to leads to destruction of file
		if( aFromUrl != aToUrl )
			return rename( aFromUrl.GetStr(), aToUrl.GetStr() ) ? FALSE : TRUE;
		return TRUE;
	}
	return FALSE;
}

BOOL Folder::DeleteItem( const ItemIDPath &rPath )
{
	ItemIDPath aPath( PDATA->maPath );
	aPath += rPath;

	String aVirt, aUrl;
	SplitVirtualURL( *((String*)aPath.pData), aVirt, aUrl );

	if( !aVirt.Len() )
		return unlink( aUrl.GetStr() ) ? FALSE : TRUE;
	
	return FALSE;
}

IfcShortcut* Folder::CreateShortcutInstance( const String& crLanguage )
{
	return new UnxShortcut( PDATA->maPath, crLanguage );
}

// functions not relevant for unix

BOOL Folder::IsValid() const
{
	return TRUE;
}

BOOL Folder::GetVolumeInfo( const ItemIDPath & path, VolumeInfo & info )
{
	return FALSE;
}

BOOL Folder::GetNetworkInfo( const ItemIDPath & path, NetworkInfo & info )
{
	return FALSE;
}

BOOL Folder::GetNameSpaceInfo( const ItemIDPath & path, NameSpaceInfo & info )
{
	return FALSE;
}

IfcContextMenu* Folder::GetContextMenu( UINT32 nItems,const ItemIDPath* pPath )
{
	return NULL;
}
