/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: drawshapesubsetting.hxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 21:10:09 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef _SLIDESHOW_DRAWSHAPESUBSETTING_HXX
#define _SLIDESHOW_DRAWSHAPESUBSETTING_HXX

#ifndef BOOST_SHARED_PTR_HPP_INCLUDED
#include <boost/shared_ptr.hpp>
#endif

#include <doctreenode.hxx>


class GDIMetaFile;

namespace presentation
{
    namespace internal
    {
        /** This class encapsulates the subsetting aspects of a
            DrawShape.
         */
        class DrawShapeSubsetting
        {
        public:
            /** Create empty shape subset handling.

            	This method creates a subset handler which contains no
            	subset information. All methods will return default
            	values.

                @param rMtf
                Metafile to retrieve subset info from (must have been
                generated with verbose text comments switched on).                
             */
            DrawShapeSubsetting();

            /** Create new shape subset handling.

            	This method creates a subset handler which initially
            	displays the whole shape.

                @param rMtf
                Metafile to retrieve subset info from (must have been
                generated with verbose text comments switched on).                
             */
            explicit DrawShapeSubsetting( const ::boost::shared_ptr< GDIMetaFile >& rMtf );

            /** Create new shape subset handling.

            	@param rShapeSubset
                The subset this object represents (can be empty, then
                denoting 'represents a whole shape')

                @param rMtf
                Metafile to retrieve subset info from (must have been
                generated with verbose text comments switched on).                
             */
            DrawShapeSubsetting( const DocTreeNode&			 				rShapeSubset,
                                 const ::boost::shared_ptr< GDIMetaFile >&	rMtf );

            /** Change current metafile.

            	Use this method to hand down a new metafile to the
            	subset handler.
             */
            void setMetafile( const ::boost::shared_ptr< GDIMetaFile >& rMtf );


            // Shape subsetting methods
            // ========================================================

            /// Return subset node for this shape
            DocTreeNode 				getSubsetNode		() const;

            /// Return true, if any child subset shapes exist
            bool		 				hasSubsetShapes		() const;

            /// Get subset shape for given node, if any
            AttributableShapeSharedPtr 	getSubsetShape		( const DocTreeNode& rTreeNode ) const;

            /// Add child subset shape (or increase use count, if already existent)
            void 						addSubsetShape		( const AttributableShapeSharedPtr& rShape );

            /** Revoke subset shape

            	This method revokes a subset shape, decrementing the
            	use count for this subset by one. If the use count
            	reaches zero (i.e. when the number of addSubsetShape()
            	matches the number of revokeSubsetShape() calls for
            	the same subset), the subset entry is removed from the
            	internal list, and subsequent getSubsetShape() calls
            	will return the empty pointer for this subset.

                @return true, if the subset shape was physically
                removed from the list (false is returned, when nothing
                was removed, either because only the use count was
                decremented, or there was no such subset found, in the
                first place).
             */
            bool 						revokeSubsetShape	( const AttributableShapeSharedPtr& rShape );


            // Doc tree methods
            // ========================================================

            /// Return overall number of nodes for given type
            sal_Int32 	getNumberOfTreeNodes		( DocTreeNode::NodeType eNodeType ) const;

            /// Return tree node of given index and given type
            DocTreeNode getTreeNode					( sal_Int32				nNodeIndex,
                                                      DocTreeNode::NodeType	eNodeType ) const;

            /// Return number of nodes of given type, below parent node
            sal_Int32 	getNumberOfSubsetTreeNodes	( const DocTreeNode& 	rParentNode,
                                                      DocTreeNode::NodeType eNodeType ) const;

            /// Return tree node of given index and given type, relative to parent node
            DocTreeNode getSubsetTreeNode			( const DocTreeNode& 	rParentNode,
                                                      sal_Int32				nNodeIndex,
                                                      DocTreeNode::NodeType	eNodeType ) const;
            
            // Helper
            // ========================================================

            /** Return a vector of currently active subsets.
                
            	Needed when rendering a shape, this method provides a
            	vector of subsets currently visible (the range as
            	returned by getEffectiveSubset(), minus the parts that
            	are currently hidden, because displayed by child
            	shapes).
             */
            const VectorOfDocTreeNodes& getActiveSubsets() const;

            /** This enum classifies each action index in the
                metafile.

                Of interest are, of course, the places where
                structural shape and/or text elements end. The
                remainder of the action gets classified as 'noop'
             */
            enum IndexClassificator
            {
                CLASS_NOOP,
                CLASS_SHAPE_START,
                CLASS_SHAPE_END,

                CLASS_LINE_END,
                CLASS_PARAGRAPH_END,
                CLASS_SENTENCE_END,
                CLASS_WORD_END,
                CLASS_CHARACTER_CELL_END
            };

            typedef ::std::vector< IndexClassificator > IndexClassificatorVector;

        private:
            // default: disabled copy/assignment
            DrawShapeSubsetting(const DrawShapeSubsetting&);
            DrawShapeSubsetting& operator=( const DrawShapeSubsetting& );

            /** Entry for subset shape

            	This struct contains data for every subset shape
            	generated. Note that for a given start/end action
            	index combination, only one subset instance is
            	generated (and reused for subsequent queries).
             */
            struct SubsetEntry
            {
                AttributableShapeSharedPtr	mpShape;
                sal_Int32					mnStartActionIndex;
                sal_Int32					mnEndActionIndex;

                /// Number of times this subset was queried, and not yet revoked
                int							mnSubsetQueriedCount;

                inline sal_Int32 getHashValue() const
                {
                    // TODO(Q3): That's a hack. We assume that start
                    // index will always be less than 65535 (if this
                    // assumption is violated, hash map performance
                    // will degrade severely)
                    return mnStartActionIndex*SAL_MAX_INT16 + mnEndActionIndex;
                }

                /// The shape set is ordered according to this method
                inline bool operator<(const SubsetEntry& rOther) const
                {
                    return getHashValue() < rOther.getHashValue();
                }
                
            };

            typedef ::std::set< SubsetEntry > 		ShapeSet;

            void ensureInitializedNodeTree() const;
            void updateSubsetBounds( const SubsetEntry& rSubsetEntry );
            void updateSubsets();

            sal_Int32 	implGetNumberOfTreeNodes( const IndexClassificatorVector::const_iterator&	rBegin,
                                                  const IndexClassificatorVector::const_iterator&	rEnd,
                                                  DocTreeNode::NodeType 							eNodeType ) const;
            DocTreeNode implGetTreeNode( const IndexClassificatorVector::const_iterator&	rBegin,
                                         const IndexClassificatorVector::const_iterator&	rEnd,
                                         sal_Int32			 								nNodeIndex,
                                         DocTreeNode::NodeType								eNodeType ) const;
                

            mutable IndexClassificatorVector	maActionClassVector;

            /// Metafile to retrieve subset info from
            ::boost::shared_ptr< GDIMetaFile >	mpMtf;	

            /// Subset of the metafile represented by this object
            const DocTreeNode					maSubset;

            /// the list of subset shapes spawned from this one. 
            ShapeSet							maSubsetShapes;

            /// caches minimal subset index from maSubsetShapes
            sal_Int32							mnMinSubsetActionIndex;

            /// caches maximal subset index from maSubsetShapes
            sal_Int32							mnMaxSubsetActionIndex;

            /** Current number of subsets to render (calculated from
                maSubset and mnMin/MaxSubsetActionIndex).

                Note that this is generally _not_ equivalent to
                maSubset, as it excludes all active subset children!
             */
            mutable VectorOfDocTreeNodes		maCurrentSubsets;
            
            /// Whether the shape's doc tree has been initialized successfully, or not
            mutable bool						mbNodeTreeInitialized;
        };

    }
}

#endif /* _SLIDESHOW_DRAWSHAPESUBSETTING_HXX */
