#!/usr/bin/python

# Copyright (C) 2016 Canonical Ltd.

# This library is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public
# License as published by the Free Software Foundation; either
# version 2.1 of the License, or (at your option) any later version.

# This library is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# Lesser General Public License for more details.

# You should have received a copy of the GNU Lesser General Public
# License along with this library; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

from __future__ import print_function

from optparse import OptionParser
import os
import platform
import psutil
import re
from StringIO import StringIO
import sys

EXTRA_ARGS_RE = r'([^=]+)=(.+)'

def HostArch():
  machine = platform.machine()

  if re.match(r'i.86', machine):
    return "x86"
  elif machine in ["x86_64", "amd64"]:
    if platform.architecture()[0] == "32bit":
      return "x86"
    return "x64"
  elif machine.startswith("arm"):
    return "arm"
  elif machine == "aarch64":
    return "arm64"

  raise Exception("Failed to detect host architecture")

def GetSymbolLevel(enabled, host_arch, is_component_build):
  if not enabled:
    return 0
  if is_component_build:
    # We should be able to cope with maximium debug info in component
    # builds everywhere
    return 2
  if host_arch == "arm" or host_arch == "x86":
    # Reduce debug info in 32-bit native builds
    return 1
  if psutil.virtual_memory().total < 8589934592L:
    # Reduce debug info if we have less than 8GB of RAM
    return 1
  if (psutil.virtual_memory().total + psutil.swap_memory().total < 17179869184L):
    # Reduce debug info if we have less than 16GB of combined RAM and swap
    return 1
  return 2

def GetToolchain(system, toolset, compiler, arch):
  if system != "Linux":
    raise Exception("Invalid system '%s'" % system)

  toolchain = "//oxide/build/config/toolchain/linux:"
  toolchain += "%s_" % toolset

  # XXX: Add clang
  if compiler != "gcc":
    raise Exception("Invalid compiler '%s'" % compiler)

  toolchain += "gcc_%s" % arch

  return toolchain

def GetV8SnapshotArch(host_arch, target_arch):
  if host_arch == target_arch:
    return host_arch

  if host_arch == "x64":
    if target_arch == "x86" or target_arch == "arm":
      return "x86"
    elif target_arch == "arm64":
      return "x64"
  elif host_arch == "x86":
    if target_arch == "arm":
      return "x86"
    elif target_arch == "x64" or target_arch == "arm64":
      return "x64"

  raise Exception("Failed to detect arch for compiling V8 snapshot")

class Options(OptionParser):
  def __init__(self):
    OptionParser.__init__(self)

    self.add_option("--component-build", action="store_true",
                    help="Enable a component build")
    self.add_option("--enable-debug-symbols", action="store_true",
                    help="Whether to enable debug symbols")
    self.add_option("--host-compiler", help="The host compiler name")
    self.add_option("--output-dir",
                    help="The destination directory for build files")
    self.add_option("--target-compiler", help="The target compiler name")
    self.add_option("--target-arch", help="The target architecture")

    self.add_option("-D", action="append", dest="extra_args", type="string",
                    help="Addition build arguments")

class ArgsGnWriter(object):
  def __init__(self):
    self._output_stream = StringIO()
    self._output_stream.write(
"""# THIS FILE IS AUTOMATICALLY GENERATED!
# All manual changes to this file will be lost.
# See "gn.oxide args <out_dir> --list" for available build arguments.
""")

  def WriteBool(self, prop, val):
    self._output_stream.write("%s = %s\n" % (prop, "true" if val else "false"))

  def WriteInt(self, prop, val):
    assert type(val) == int
    self._output_stream.write("%s = %s\n" % (prop, val))

  def WriteString(self, prop, val):
    self._output_stream.write("%s = \"%s\"\n" % (prop, val))

  def WriteStringList(self, prop, *args):
    self._output_stream.write("%s = [ " % prop)
    first_entry = True
    for arg in args:
      if not first_entry:
        self._output_stream.write(", ")
      first_entry = False
      self._output_stream.write("\"%s\"" % arg)
    self._output_stream.write(" ]\n")

  def SaveToFile(self, path):
    with open(path, "w") as fd:
      fd.write(self._output_stream.getvalue())

def WriteStaticArgs(writer):
  writer.WriteBool("is_oxide", True)
  writer.WriteStringList("root_extra_deps", "//oxide:oxide_all")

  # TODO(chrisccoulson): Support clang builds
  writer.WriteBool("is_clang", False)

  # We statically link tcmalloc in to the renderer binary, rather than
  # linking it in base
  writer.WriteString("use_allocator", "none")

  # The shim results in us overriding malloc for any app linked against us
  writer.WriteBool("use_experimental_allocator_shim", False)

  writer.WriteBool("enable_nacl", False)
  writer.WriteBool("is_component_ffmpeg", True)

  # XXX(chrisccoulson): Remove this.
  # There is a strict-overflow warning in third_party/WebKit/Source/wtf/dtoa/bignum.cc
  writer.WriteBool("treat_warnings_as_errors", False)
  writer.WriteBool("fatal_linker_warnings", False)

  # XXX(chrisccoulson): This produces an error:
  #  "ERROR at //build/toolchain/gcc_toolchain.gni:520:30: Clobbering existing value."
  #  use_gold = true is the default anyway
  #writer.WriteBool("use_gold", True)
  writer.WriteBool("linux_use_bundled_binutils", False)
  writer.WriteString("gold_path", "")

  writer.WriteBool("is_desktop_linux", False)
  writer.WriteBool("use_sysroot", False)
  writer.WriteBool("use_ash", False)
  writer.WriteBool("use_aura", True)
  writer.WriteBool("use_cups", False)
  writer.WriteBool("use_gconf", False)
  writer.WriteBool("use_ozone", True)
  writer.WriteString("ozone_platform", "oxide")
  writer.WriteBool("use_pango", True)
  writer.WriteBool("toolkit_views", False)
  writer.WriteBool("enable_basic_printing", False)
  writer.WriteBool("enable_print_preview", False)
  writer.WriteBool("enable_extensions", True)

def WriteConfigurableArgs(writer, options):
  host_arch = HostArch()

  writer.WriteInt("symbol_level",
                  GetSymbolLevel(options.enable_debug_symbols,
                                 host_arch,
                                 options.component_build))

  host_toolchain = GetToolchain(platform.system(),
                                "host",
                                options.host_compiler,
                                host_arch)
  writer.WriteString("host_toolchain", host_toolchain)

  if options.target_arch and options.target_arch != host_arch:
    writer.WriteBool("oxide_enable_cross_toolchains", True)
    writer.WriteString("target_cpu", options.target_arch)
    target_toolchain = GetToolchain(platform.system(),
                                    "cross",
                                    options.target_compiler,
                                    options.target_arch)
    writer.WriteString("custom_toolchain", target_toolchain)

    v8_snapshot_arch = GetV8SnapshotArch(host_arch, options.target_arch)
    v8_snapshot_toolchain = GetToolchain(platform.system(),
                                         "host",
                                         options.host_compiler,
                                         v8_snapshot_arch)
    writer.WriteString("v8_snapshot_toolchain", v8_snapshot_toolchain)
  else:
    writer.WriteString("custom_toolchain", host_toolchain)
    writer.WriteString("v8_snapshot_toolchain", host_toolchain)

  if options.component_build:
    writer.WriteBool("is_component_build", True)

def WriteExtraArgs(writer, args):
  if not args:
    return

  for arg in args:
    m = re.match(EXTRA_ARGS_RE, arg)
    prop = m.groups()[0]
    value = m.groups()[1]
    if value in ("true", "false"):
      writer.WriteBool(prop, True if value == "true" else False)
    elif value.isdigit():
      writer.WriteInt(prop, int(value))
    else:
      writer.WriteString(prop, value)

def WriteGnArgs(options):
  writer = ArgsGnWriter()

  WriteStaticArgs(writer)
  WriteConfigurableArgs(writer, options)
  WriteExtraArgs(writer, options.extra_args)

  output_dir = os.path.abspath(options.output_dir)

  if not os.path.isdir(output_dir):
    os.makedirs(output_dir)

  writer.SaveToFile(os.path.join(output_dir, "args.gn"))

def SanitizeOptions(options):
  if not options.host_compiler:
    raise Exception("No host compiler specified")
  if not options.output_dir:
    raise Exception("No output directory specified")

  if options.target_arch and not options.target_compiler:
    raise Exception("No target compiler specified")

  if options.extra_args:
    for arg in options.extra_args:
      if not re.match(EXTRA_ARGS_RE, arg):
        raise Exception("Invalid extra argument '%s'" % arg)

def main(argv):
  optparser = Options()
  (opts, args) = optparser.parse_args(argv)

  SanitizeOptions(opts)
  WriteGnArgs(opts)

if __name__ == "__main__":
  main(sys.argv[1:])
