/*
 * Copyright 2013 Google Inc.
 *
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */

#ifndef SkPDFResourceDict_DEFINED
#define SkPDFResourceDict_DEFINED

#include "SkPDFTypes.h"
#include "SkTDArray.h"
#include "SkTypes.h"

/** \class SkPDFResourceDict

    A resource dictionary, which maintains the relevant sub-dicts and
    allows generation of a list of referenced SkPDFObjects inserted with
    insertResourceAsRef.
*/
class SkPDFResourceDict : public SkPDFDict {
public:
    SK_DECLARE_INST_COUNT(SkPDFResourceDict)

     enum SkPDFResourceType{
        kExtGState_ResourceType,
        kPattern_ResourceType,
        kXObject_ResourceType,
        kFont_ResourceType,
        // These additional types are defined by the spec, but not
        // currently used by Skia: ColorSpace, Shading, Properties
        kResourceTypeCount
     };

    /** Create a PDF resource dictionary.
     *  The full set of ProcSet entries is automatically created for backwards
     *  compatibility, as recommended by the PDF spec.
     */
    SkPDFResourceDict();

    /** Add the value SkPDFObject as a reference to the resource dictionary
     *  with the give type and key.
     *  The relevant sub-dicts will be automatically generated, and the
     *  resource will be named by concatenating a type-specific prefix and
     *  the input key.
     *  This object will be part of the resource list when requested later.
     *  @param type  The type of resource being entered, like
     *    kPattern_ResourceType or kExtGState_ResourceType.
     *  @param key   The resource key, should be unique within its type.
     *  @param value The resource itself.
     *  @return The value argument is returned.
     */
    SkPDFObject* insertResourceAsReference(SkPDFResourceType type, int key,
                                           SkPDFObject* value);

    /**
     * Returns the name for the resource that will be generated by the resource
     * dict.
     *
     *  @param type  The type of resource being entered, like
     *    kPattern_ResourceType or kExtGState_ResourceType.
     *  @param key   The resource key, should be unique within its type.
     */
    static SkString getResourceName(SkPDFResourceType type, int key);

private:
    /** Add the value to the dictionary with the given key.  Refs value.
     *  The relevant sub-dicts will be automatically generated, and the
     *  resource will be named by concatenating a type-specific prefix and
     *  the input key.
     *  The object will NOT be part of the resource list when requested later.
     *  @param type  The type of resource being entered.
     *  @param key   The resource key, should be unique within its type.
     *  @param value The resource itself.
     *  @return The value argument is returned.
     */
    SkPDFObject* insertResource(SkPDFResourceType type, int key,
                                SkPDFObject* value);

    SkTDArray<SkPDFDict*> fTypes;
    typedef SkPDFDict INHERITED;
};

#endif
