/* ----------------------------------------------------------------------------
 * pbbcmd.c
 * command line control tool for pbbuttonsd
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <getopt.h>
#include <string.h>
#include <malloc.h>
#include <sys/types.h>

#include <pbb.h>

#include "gettext_macros.h"
#include "pbbcmd.h"

#define BUFFERLEN	200

int config_no_answer = 0;

static struct tagtable tagtab[] = {
  { "TAG_SAVECONFIG",        TAG_SAVECONFIG},
  { "TAG_AUTORESCAN",        TAG_AUTORESCAN},
  { "TAG_TIMEFORCMD",        TAG_TIMEFORCMD},
  { "TAG_VERSION",           TAG_VERSION},
  { "TAG_KEYCODE",           TAG_KEYCODE},
  { "TAG_KEYREPEAT",         TAG_KEYREPEAT},
  { "TAG_MODIFIER",          TAG_MODIFIER},
  { "TAG_MOUSEBUTTONS",      TAG_MOUSEBUTTONS},
  { "TAG_MOUSERELX",         TAG_MOUSERELX},
  { "TAG_MOUSERELY",         TAG_MOUSERELY},
  { "TAG_MOUSEWHEEL",        TAG_MOUSEWHEEL},
  { "TAG_SYSINFO",           TAG_SYSINFO},
  { "TAG_REINIT",            TAG_REINIT},
  { "TAG_BACKLIGHTMAX",      TAG_BACKLIGHTMAX},
  { "TAG_BACKLIGHTLEVEL",    TAG_BACKLIGHTLEVEL},
  { "TAG_COVERSTATUS",       TAG_COVERSTATUS},
  { "TAG_TIMEREMAINING",     TAG_TIMEREMAINING},
  { "TAG_POWERSOURCE",       TAG_POWERSOURCE},
  { "TAG_TIMECHANGED",       TAG_TIMECHANGED},
  { "TAG_POWERCHANGED",      TAG_POWERCHANGED},
  { "TAG_REQUESTSLEEP",      TAG_REQUESTSLEEP},
  { "TAG_SLEEPSUPPORTED",    TAG_SLEEPSUPPORTED},
  { "TAG_PREPAREFORSLEEP",   TAG_PREPAREFORSLEEP},
  { "TAG_WAKEUPFROMSLEEP",   TAG_WAKEUPFROMSLEEP},
  { "TAG_PMUDEVICE",         TAG_PMUDEVICE},
  { "TAG_ADBDEVICE",         TAG_ADBDEVICE},
  { "TAG_TPMODEUPKEY",       TAG_TPMODEUPKEY},
  { "TAG_TPMODEUPMOD",       TAG_TPMODEUPMOD},
  { "TAG_TPMODEDOWNKEY",     TAG_TPMODEDOWNKEY},
  { "TAG_TPMODEDOWNMOD",     TAG_TPMODEDOWNMOD},
  { "TAG_TPMODE",            TAG_TPMODE},
  { "TAG_KBDMODE",           TAG_KBDMODE},
  { "TAG_BATLOG",            TAG_BATLOG},
  { "TAG_BATTERYPRESENT",    TAG_BATTERYPRESENT},
  { "TAG_BATCYCLE",          TAG_BATCYCLE},
  { "TAG_AMBIENTLIGHT",      TAG_AMBIENTLIGHT },
  { "TAG_KEYBLIGHTLEVEL",    TAG_KEYBLIGHTLEVEL },
  { "TAG_KEYBLIGHTMAX",      TAG_KEYBLIGHTMAX },
  { "TAG_LCDILLUMUPKEY",     TAG_LCDILLUMUPKEY},
  { "TAG_LCDILLUMUPMOD",     TAG_LCDILLUMUPMOD},
  { "TAG_LCDILLUMDOWNKEY",   TAG_LCDILLUMDOWNKEY},
  { "TAG_LCDILLUMDOWNMOD",   TAG_LCDILLUMDOWNMOD},
  { "TAG_KBDILLUMUPKEY",     TAG_KBDILLUMUPKEY},
  { "TAG_KBDILLUMUPMOD",     TAG_KBDILLUMUPMOD},
  { "TAG_KBDILLUMDOWNKEY",   TAG_KBDILLUMDOWNKEY},
  { "TAG_KBDILLUMDOWNMOD",   TAG_KBDILLUMDOWNMOD},
  { "TAG_KBDILLUMONKEY",     TAG_KBDILLUMONKEY},
  { "TAG_KBDILLUMONMOD",     TAG_KBDILLUMONMOD},
  { "TAG_LCDBRIGHTNESS",     TAG_LCDBRIGHTNESS},
  { "TAG_LCDBRIGHTNESSMAX",  TAG_LCDBRIGHTNESSMAX},
  { "TAG_LCDFADINGSPEED",    TAG_LCDFADINGSPEED},
  { "TAG_LCDAUTOADJUST",     TAG_LCDAUTOADJUST },
  { "TAG_LCDTHRESHOLD",      TAG_LCDTHRESHOLD },
  { "TAG_LCDAUTOADJMINBAT",  TAG_LCDAUTOADJMINBAT },
  { "TAG_LCDAUTOADJMAXBAT",  TAG_LCDAUTOADJMAXBAT },
  { "TAG_LCDAUTOADJMINAC",   TAG_LCDAUTOADJMINAC },
  { "TAG_LCDAUTOADJMAXAC",   TAG_LCDAUTOADJMAXAC },
  { "TAG_KBDBRIGHTNESS",     TAG_KBDBRIGHTNESS},
  { "TAG_KBDONBRIGHTNESS",   TAG_KBDONBRIGHTNESS},
  { "TAG_KBDBRIGHTNESSMAX",  TAG_KBDBRIGHTNESSMAX},
  { "TAG_KBDFADINGSPEED",    TAG_KBDFADINGSPEED},
  { "TAG_KBDTHRESHOLD",      TAG_KBDTHRESHOLD },
  { "TAG_KBDAUTOADJUST",     TAG_KBDAUTOADJUST },
  { "TAG_BRIGHTNESSOP",      TAG_BRIGHTNESSOP},
  { "TAG_FRAMEBUFFERDEVICE", TAG_FRAMEBUFFERDEVICE},
  { "TAG_BLANKFRAMEBUFFER",  TAG_BLANKFRAMEBUFFER},
  { "TAG_DIMFULLYDARK",      TAG_DIMFULLYDARK},
  { "TAG_MIXERDEVICE",       TAG_MIXERDEVICE},
  { "TAG_VOLUMEUPKEY",       TAG_VOLUMEUPKEY},
  { "TAG_VOLUMEUPMOD",       TAG_VOLUMEUPMOD},
  { "TAG_VOLUMEDOWNKEY",     TAG_VOLUMEDOWNKEY},
  { "TAG_VOLUMEDOWNMOD",     TAG_VOLUMEDOWNMOD},
  { "TAG_MUTEKEY",           TAG_MUTEKEY},
  { "TAG_MUTEMOD",           TAG_MUTEMOD},
  { "TAG_MIXERINITDELAY",    TAG_MIXERINITDELAY},
  { "TAG_VOLUME",            TAG_VOLUME },
  { "TAG_MUTE",              TAG_MUTE },
  { "TAG_MIXERCHANNELS",     TAG_MIXERCHANNELS },
  { "TAG_CDROMDEVICE",       TAG_CDROMDEVICE },
  { "TAG_EJECTCDKEY",        TAG_EJECTCDKEY },
  { "TAG_EJECTCDMOD",        TAG_EJECTCDMOD },
  { "TAG_EJECTCDKEYDELAY",   TAG_EJECTCDKEYDELAY },
  { "TAG_EJECTCD",           TAG_EJECTCD },
  { "TAG_SCRIPTPMCS",        TAG_SCRIPTPMCS},
  { "TAG_SLEEPKEY",          TAG_SLEEPKEY },
  { "TAG_SLEEPMOD",          TAG_SLEEPMOD },
  { "TAG_SLEEPKEYDELAY",     TAG_SLEEPKEYDELAY },
  { "TAG_GOTOSLEEP",         TAG_GOTOSLEEP },
  { "TAG_BWLFIRST",          TAG_BWLFIRST },
  { "TAG_BWLSECOND",         TAG_BWLSECOND },
  { "TAG_BWLLAST",           TAG_BWLLAST },
  { "TAG_CURRENTBWL",        TAG_CURRENTBWL },
  { "TAG_CPULOADSLEEPLOCK",  TAG_CPULOADSLEEPLOCK },
  { "TAG_CPULOADMIN",        TAG_CPULOADMIN },
  { "TAG_CPULOADPERIOD",     TAG_CPULOADPERIOD },
  { "TAG_NETLOADSLEEPLOCK",  TAG_NETLOADSLEEPLOCK },
  { "TAG_NETLOADMIN",        TAG_NETLOADMIN },
  { "TAG_NETLOADPERIOD",     TAG_NETLOADPERIOD },
  { "TAG_NETLOADDEV",        TAG_NETLOADDEV },
  { "TAG_POLICY",            TAG_POLICY },
  
  { "TAG_SLEEPKEYSLEEP",     TAG_SLEEPKEYSLEEP },         /* depreciated */
  { "TAG_ONACPOLICY",        TAG_ONACPOLICY },            /* depreciated */
  { "TAG_ONACSLEEP",         TAG_ONACSLEEP },             /* depreciated */
  { "TAG_ONACCOVERSLEEP",    TAG_ONACCOVERSLEEP },        /* depreciated */
  { "TAG_ONACTIMESLEEP",     TAG_ONACTIMESLEEP },         /* depreciated */
  { "TAG_ONACTIMEDIM",       TAG_ONACTIMEDIM },           /* depreciated */
  { "TAG_ONBATTERYPOLICY",   TAG_ONBATTERYPOLICY },       /* depreciated */
  { "TAG_ONBATTERYSLEEP",    TAG_ONBATTERYSLEEP },        /* depreciated */
  { "TAG_ONBATTERYCOVERSLEEP", TAG_ONBATTERYCOVERSLEEP }, /* depreciated */
  { "TAG_ONBATTERYTIMESLEEP", TAG_ONBATTERYTIMESLEEP },   /* depreciated */
  { "TAG_ONBATTERYTIMEDIM",  TAG_ONBATTERYTIMEDIM },      /* depreciated */

  { "TAG_ONAC_POLICY",       TAG_ONAC_POLICY },
  { "TAG_ONAC_TIMERACTION",  TAG_ONAC_TIMERACTION },
  { "TAG_ONAC_COVERACTION",  TAG_ONAC_COVERACTION },
  { "TAG_ONAC_KEYACTION",    TAG_ONAC_KEYACTION },
  { "TAG_ONAC_TIMESUSPEND",  TAG_ONAC_TIMESUSPEND },
  { "TAG_ONAC_TIMEDIM",      TAG_ONAC_TIMEDIM },
  { "TAG_ONBATT_POLICY",     TAG_ONBATT_POLICY },
  { "TAG_ONBATT_TIMERACTION",TAG_ONBATT_TIMERACTION },
  { "TAG_ONBATT_COVERACTION",TAG_ONBATT_COVERACTION },
  { "TAG_ONBATT_KEYACTION",  TAG_ONBATT_KEYACTION },
  { "TAG_ONBATT_TIMESUSPEND",TAG_ONBATT_TIMESUSPEND },
  { "TAG_ONBATT_TIMEDIM",    TAG_ONBATT_TIMEDIM },

  { "TAG_EMERGENCYACTION",   TAG_EMERGENCYACTION },
  { "TAG_HEARTBEATBEEP",     TAG_HEARTBEATBEEP },
  { "TAG_IDENTITY",          TAG_IDENTITY }};

  static struct tagtable errortab[] = {
  { N_("Permission denied"),          E_PERM},
  { N_("Private Tag"),                E_PRIVATE},
  { N_("File doesn't exist"),         E_NOEXIST},
  { N_("File not a charakter device"),E_NOCHAR},
  { N_("File not a block device"),    E_NOBLK},
  { N_("File not a file"),            E_NOFILE},
  { N_("Buffer overflow"),            E_BUFOVL},
  { N_("open error"),                 E_OPEN},
  { N_("format error"),               E_FORMAT},
  { N_("Messageport not available"),  E_MSGPORT},
  { N_("Server already running"),      E_TWICE},
  { N_("Help or version info"),         E_INFO},
  { N_("Server not found"),               E_NOSERVER},
  { N_("Registration failed"),              E_REGISTER},
  { N_("Unsecure script owner"),             E_USER},
  { N_("Script must be write-only by owner"), E_RIGHTS},
  { N_("read-only value"),			E_NOWRITE},
  { N_("write-only value"),	  		E_NOREAD},
  { N_("argument invalid"),	  		 E_INVALID},
  { N_("function not supported"),		  E_NOSUPPORT}};

void
print_usage (char *prgname)
{
	printf(_("%s (version %s) - control client for pbbuttonsd.\n"), prgname, VERSION);
	printf(_("Usage: %s [-%s] <'query'  | 'config'> <tag> [<data>]\n"), prgname, ARG_ALL);
	printf (_("Options:\n"
		"   -%c, --help               display this help and exit\n"
		"   -%c, --version            display version information and exit\n"
		"   -%c, --ignore             ignore config return and error values\n\n"),
		ARG_HELP, ARG_VERSION, ARG_IGNORE);
	print_tags ();
}

void
print_tags ()
{
	int n, z;

	printf (_("Supported tags:\n"));
	printf ("    ");
	for (n=1,z=1; n <= (sizeof(tagtab) / sizeof(struct tagtable)); n++) {
		if (tagtab[n-1].tag & FLG_PRIVATE)
			continue;
		else
			printf ("%-23s", tagtab[n-1].tagname);
		if ((z++ % 3) == 0)
			printf ("\n    ");
	}
	printf ("\n\n    ");
	printf (_("The leading 'TAG_' could be omited.\n"));
}

int
evaluate_args(int argc, char *argv[], struct tagitem *taglist)
{
	struct option const long_options[] = {
		  {"help", no_argument, 0, ARG_HELP},
		  {"version", no_argument, 0, ARG_VERSION},
		  {"ignore", no_argument, 0, ARG_IGNORE},
		  {NULL, 0, NULL, 0}
	};
	char *prgname;
	long tag;
	int c, n;

	if((prgname = strrchr(argv[0],'/')) == NULL)
		prgname = argv[0];
	else prgname++;		/* ignore first slash*/

	while ((c = getopt_long (argc, argv, ARG_ALL, long_options, (int *) 0)) != EOF) {
		switch (c) {
			case ARG_VERSION:
				printf(_("pbbcmd, version %s"), VERSION);
				printf(", (c) 2002 Matthias Grimm\n");
				return CMD_FLAGS;
			case ARG_IGNORE:
				config_no_answer = 1;
				break;
			case ARG_HELP:
			default:
				print_usage (prgname);
				return CMD_FLAGS;
		}
	}

	if (argc - optind >= 2) {
		if ((strcasecmp (argv[optind], "query")) == 0) {
			for (n=optind+1; n < argc; n++) {
				if ((tag = identifytag (argv[n])) != 0) {
					taglist_add (taglist, tag, 0);
				} else {
					printf (_("WARNING: tag %s not supported.\n"), argv[n]);
					print_tags ();
					return 0;
				}
			}
			return CMD_QUERY;
		} else if ((strcasecmp (argv[optind], "config")) == 0) {
			if ((argc - optind - 1) % 2 == 1) {
				printf (_("ERROR: tag/data pairs not complete.\n"));
				return 0;
			} else {
				for (n=optind+1; n < argc; n++) {
					if ((tag = identifytag (argv[n])) != 0) {
						if (tag & FLG_STRING)
							taglist_add (taglist, tag, (long) argv[++n]);
						else
							taglist_add (taglist, tag, (long) atoi (argv[++n]));
					} else {
						printf (_("WARNING: tag %s not supported.\n"), argv[n]);
						print_tags ();
						return 0;
					}
				}
				return CMD_CONFIG;
			}
		}
	}
	print_usage (prgname);
	return 0;
}

int
read_message (char *buffer, int bufferlen)
{
	int timeout = 40;  /* 4 seconds timeout */

	while (timeout > 0) {
		if ((ipc_receive (buffer, bufferlen)) >=0) {
			return 0;  /* ok */
		}
		usleep (100000);   /* wait for 100ms */
		timeout--;
	}
	printf (_("Server didn't send an answer and timed out.\n"));
	return 1; /* fail */
}

long
identifytag (char *name)
{
	int n;

	for (n=(sizeof(tagtab) / sizeof(struct tagtable) - 1); n >= 0; n--)
		if (!strcasecmp (tagtab[n].tagname, name))
			return tagtab[n].tag;
		else if (!strcasecmp ((tagtab[n].tagname) + 4, name))
			return tagtab[n].tag;
	return 0;
}

char *
identifytagname (long tag, char *defname)
{
	int n;

	for (n=(sizeof(tagtab) / sizeof(struct tagtable) - 1); n >= 0; n--)
		if (tagtab[n].tag == tag)
			return tagtab[n].tagname;
	return defname;
}

char *
identifyerror (long error, char *deferror)
{
	int n;

	for (n=(sizeof(errortab) / sizeof(struct tagtable) - 1); n >= 0; n--)
		if (errortab[n].tag == error)
			return errortab[n].tagname;
	return deferror;
}

int main(int argc, char *argv[])
{
	char buffer[BUFFERLEN];
	struct pbbmessage *pbbmsg;
	struct tagitem *taglist, *tags;
	char *tagname, *errname;
	int cmd, rc = 1;

#ifdef ENABLE_NLS
	setlocale(LC_ALL, "");
	bindtextdomain(PACKAGE, PACKAGE_LOCALE_DIR);
	textdomain(PACKAGE);
#endif

	init_libpbb ();

	if ((taglist = (struct tagitem *) malloc (argc * sizeof (struct tagitem))) != NULL) {
		taglist_init (taglist);
		if ((cmd = evaluate_args (argc, argv, taglist)) != 0) {
			if (ipc_init (LIBMODE_CLIENT, 0) == 0) {
				switch (cmd) {
				case CMD_FLAGS:  /* for command line flags -h and -v */
					rc = 0;               /* they are no errors */
					break;
				case CMD_QUERY:
					ipc_send (0, READVALUE, taglist);
					if ((rc = read_message (buffer, BUFFERLEN)) == 0) {
						tags = ((struct pbbmessage *) buffer)->taglist;
						while (tags->tag != TAG_END) {
							if (tags->tag & FLG_ERROR) {
								tagname = identifytagname (tags->tag & TAGMASK, "???");
								errname = identifyerror (tags->data, "???");
								printf (_("Sorry, Couldn't get data for %s: %s."), tagname, _(errname));
								rc = 1;
							} else if (tags->tag & FLG_STRING)
								printf ("%s ", (char *) tags->data);
							else
								printf ("%ld ", tags->data);
							tags++;
						}
						printf ("\n");
					}
					break;
				case CMD_CONFIG:
					ipc_send (0, CHANGEVALUE, taglist);
					if (config_no_answer)
						rc = 0;
					else if ((rc = read_message (buffer, BUFFERLEN)) == 0) {
						pbbmsg = (struct pbbmessage *) buffer;
						if (pbbmsg->action == CHANGEERROR) {
							tags = pbbmsg->taglist;
							while (tags->tag != TAG_END) {
								tagname = identifytagname (tags->tag & TAGMASK, "???");
								errname = identifyerror (tags->data, "???");
								printf (_("Setting of %s failed: %s.\n"), tagname, _(errname));
								tags++;
								rc = 1;
							}
						} else
							printf (_("ERROR: Unexpected answer from server, actioncode %ld.\n"), (long) pbbmsg->action);
					}
					break;
				}
				ipc_exit();
			} else
				printf (_("ERROR: Problems with IPC, maybe server is not running.\n"));
		}
		free (taglist);
	} else
		printf (_("ERROR: Not enough memory for buffer.\n"));
	return rc;
}

void peep_ipc (struct tagitem *taglist) { }
