/* ----------------------------------------------------------------------------
 * support.c
 * funtions usefull for everybody
 *
 * Copyright 2002 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * ----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <fcntl.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <sys/wait.h>

#include <glib.h>
#include <pbb.h>

#include "support.h"
#include "gettext_macros.h"

/* the prototype for getpgid() should be in unistd.h but although it is
 * included above the compiler complains with "implicit declaration for
 * getpgid" if this line is missing. This must be further investigated.
 */
extern pid_t getpgid (pid_t pid);

int
copy_path (char *source, char *dest, int type, int flags)
{
	char *whitespace = NULL;
	int rc = 0, perms = 0600;

	if (type == TYPE_DIR) perms = 0700;

	if (strlen (source) < STDBUFFERLEN) {
		if ((whitespace = strchr (source, ' ')) != NULL)
			*whitespace = 0;

		rc = check_devorfile (source, type);
		if ((rc == 0) && (flags & CPFLG_PBBONLY))
			rc = check_permissions (source, geteuid(), perms, 022);
		if ((rc == E_NOEXIST) && (flags & CPFLG_MAYBEMISSED))
			rc = 0;
		if (type == TYPE_FILE && strlen(source) == 0)
			rc = 0;  /* empty _file_names are acceptable */

		switch (rc) {
		case E_NOEXIST:
			print_msg (PBB_ERR, _("The object '%s' doesn't exist.\n"), source);
			break;
		case E_NOFILE:
			print_msg (PBB_ERR, _("The object '%s' is not a file.\n"), source);
			break;
		case E_NODIR:
			print_msg (PBB_ERR, _("The object '%s' is not a directory.\n"), source);
			break;
		case E_NOCHAR:
			print_msg (PBB_ERR, _("The object '%s' is not a character device.\n"), source);
			break;
		case E_NOBLK:
			print_msg (PBB_ERR, _("The object '%s' is not a block device.\n"), source);
			break;
		case E_USER:
			print_msg (PBB_WARN, _("SECURITY: %s must be owned by the same owner as pbbuttonsd.\n"), source);
			break;
		case E_RIGHTS:
			print_msg (PBB_WARN, _("SECURITY: %s must only be writable by the owner of pbbuttonsd.\n"), source);
			break;
		case E_PERM:
			print_msg (PBB_ERR, _("Insufficient permissions for object %s, at least '0%o' needed.\n"), source, perms);
			break;
		default:
			if (whitespace != NULL) *whitespace = ' ';
			strncpy(dest, source, STDBUFFERLEN);
			break;
		}
	} else {
		print_msg (PBB_ERR, _("Path of '%s' is to long.\n"), source);
		rc = E_BUFOVL;
	}

	if (whitespace != NULL) *whitespace = ' ';
	return rc;
}

#if 0

/* lauching of external programs with glib functions. The program will be
 * executed asynchron but some applications in pbbuttond don't work with
 * asynchron calls. Synchron calls won't neither work because in some
 * cases the IPC system will be blocked. To make this work, a major
 * redesign of pbbuttonsd internal structure would be necessary.
 */

void
cbChildSetup (gpointer data)
{
	/* open own session so that pbbuttonsd can terminate the
	 * script cleanly on error without killing itself
	 */
	setsid();
}

void
cbChildCleanup (gpointer data)
{
	struct ChildData *cd = data;

#ifdef DEBUG
	print_msg (PBB_INFO, "DBG: Child %d cleaned up (ref=%d), exitcode=%d.\n", cd->pid, cd->ref, cd->rc);
#endif
	
	if (g_atomic_int_dec_and_test (&cd->ref)) {
		/* free resources */
		g_spawn_close_pid(cd->pid),
		g_strfreev (cd->argv);
		g_free (cd);
#ifdef DEBUG
		print_msg (PBB_INFO, "DBG: Resources freed.\n");
#endif
	}
}

gboolean
cbChildTimeout (gpointer data) 
{
	struct ChildData *cd = data;
	int pgrp, status;
	gchar *cmd;
	
#ifdef DEBUG
	print_msg (PBB_INFO, "DBG: Child %d, timeout handler called.\n", cd->pid);
#endif

	/* remove child watch */
	g_source_remove (cd->cwatch);
	
	pgrp = getpgid(cd->pid);    /* get childs process group */
	kill (-pgrp, SIGKILL);        /* kill child and his childs */
	waitpid (cd->pid, &status, 0); /* clean up zombie */
	
	cmd = g_strjoinv (" ", cd->argv);
	print_msg (PBB_INFO, _("Script '%s' lauched but killed after %d seconds.\n"), cmd, get_timeforcmd());
	g_free (cmd);
	
	/* remove timeout watch */
	return FALSE;
}

void
cbChildExit (GPid pin, gint status, gpointer data)
{
	struct ChildData *cd = data;
	gchar *cmd;
	
#ifdef DEBUG
	print_msg (PBB_INFO, "DBG: Child %d, exit handler called.\n", cd->pid);
#endif

	/* remove timeout watch */
	g_source_remove (cd->twatch);

	cmd = g_strjoinv (" ", cd->argv);
	if (WIFEXITED(status)) {
		if ((cd->rc = WEXITSTATUS(status)))
			print_msg (PBB_INFO, _("Script '%s' lauched but exitcode is %d.\n"), cmd, cd->rc);
		else
			print_msg (PBB_INFO, _("Script '%s' launched and exited normally.\n"), cmd);
	} else if (WIFSIGNALED(status))
		print_msg (PBB_INFO, _("Script '%s' lauched but exited by signal.\n"), cmd);
	g_free (cmd);
}

//struct ChildData*
int
call_script (char* template, ...)
{
	GError *err = NULL;
	char cmd[MAXCMDLEN+1];
	struct ChildData *cd;
	va_list list;
	int rc = 0;
	
	if (strlen(template) == 0)  /* do nothing, if template is empty */
		return 0;
	
	cd = g_new0 (struct ChildData, 1);
	cd->rc = -1;
	
	va_start(list, template);
	vsnprintf (cmd, MAXCMDLEN, template, list);
	cmd[MAXCMDLEN] = '\0';
	cd->argv = g_strsplit (cmd, " ", 0);
	va_end (list);
	
	if ((rc = check_permissions (cd->argv[0], geteuid(), 0700, 022)) == 0) {
		g_spawn_async ("/", cd->argv, NULL,
			G_SPAWN_DO_NOT_REAP_CHILD|G_SPAWN_STDOUT_TO_DEV_NULL|G_SPAWN_STDERR_TO_DEV_NULL,
			cbChildSetup, (gpointer) cd, &cd->pid, &err);
		if (err == NULL) {
			g_atomic_int_inc (&cd->ref);
			cd->twatch = g_timeout_add_full (G_PRIORITY_DEFAULT,
					get_timeforcmd() * 1000, cbChildTimeout, (gpointer) cd, cbChildCleanup);
			g_atomic_int_inc (&cd->ref);
			cd->cwatch = g_child_watch_add_full (G_PRIORITY_DEFAULT,
					cd->pid, cbChildExit, (gpointer) cd, cbChildCleanup);
//			return cd;
			return rc;
		} else {
			print_msg (PBB_INFO, _("Script '%s' can't be launched - fork() failed.\n"), cmd);
			g_error_free (err);
		}
	} else if (rc == E_NOEXIST) {
		print_msg (PBB_INFO, _("Script '%s' doesn't exist.\n"), cmd);
	} else { /* E_PERM or E_RIGHTS or E_USER */
		print_msg (PBB_INFO, _("Script '%s' skipped because it's not secure.\n"), cmd);
	}
	g_strfreev (cd->argv);
	g_free (cd);
//	return NULL;
	return rc;
}
#endif

int
call_script (char* template, ...)
{
	int rc;
	char script[MAXCMDLEN+1], *ftext, buffer[STDBUFFERLEN];
	va_list list;

	if (strlen(template) == 0)  /* do nothing, if template is empty */
		return 0;

	va_start(list, template);
	vsnprintf (script, MAXCMDLEN, template, list);
	script[MAXCMDLEN] = '\0';
	va_end (list);

	rc = launch_program (script);
	switch (rc) {
	case 0:
		ftext = _("launched and exited normally");
		break;
	case E_NOEXIST:
		ftext =_("doesn't exist");
		break;
	case E_RIGHTS:
	case E_USER:
	case E_PERM:
		ftext =_("skipped because it's not secure");
		break;
	case E_CLDKILL:
		snprintf (buffer, STDBUFFERLEN, _("lauched but killed after %d seconds"), get_timeforcmd());
		ftext = buffer;
		break;
	case E_CLDEXIT:
		ftext =_("lauched but exitcode is not null");
		break;
	case E_CLDFAIL:
		ftext =_("can't be launched - fork() failed");
		break;
	case E_CLDSIG:
		ftext =_("lauched but exited by signal");
		break;
	default:
		ftext =_("failed - unknown error code");
	}
	print_msg (PBB_INFO, _("Script '%s' %s\n"), script, ftext);
	return rc;
}

