/* --------------------------------------------------------------------------
 * driver_backlight_pmu.c
 * This module contains the low level driver to control a LCD backlight via
 * the PMU. The low level driver overloads methods of class backlight.
 *
 * Copyright 2002-2006 Matthias Grimm
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 * -------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/pmu.h>

#include <pbb.h>

#include "gettext_macros.h"
#include "module_pmac.h"
#include "class_backlight.h"
#include "driver_backlight_pmu.h"
#include "support.h"

/** Driver PMU Backlight **/

/* To make this driver work under kernel versions > 2.6.18 the kernel
 * option CONFIG_PMAC_BACKLIGHT_LEGACY must be set. This option includes
 * some old ioctls for /dev/pmu that are needed here. For older kernels
 * it will work out of the box.
 *
 * Every frame buffer driver could register a backlight controller,
 * but only the right one is accepted. Verification is done by checking
 * OpenFirmware's property "backlight-control" in register_backlight_controller().
 * ioctl(base->fd_pmu, PMU_IOC_GET_BACKLIGHT, &val);  returns -ENODEV if no
 * backlight device is registered. This could be used to check if a backlight
 * controller is installed or if we have to call our fallback functions.
 * TODO: This is not implemented yet.
 */

static int pmu_filehandle;

void
driver_backlight_pmu_exit ()
{
	if (pmu_filehandle != -1)
		close (pmu_filehandle);
}

int
pmubl_get_brightness ()
{
	long val = 0;
		
	if ((ioctl(pmu_filehandle, PMU_IOC_GET_BACKLIGHT, &val)) == -1)
		return -1;

	return (int) val;
}

int
pmubl_get_brightness_max ()
{
	return PMUBRIGHTNESSMAX;
}

void
pmubl_set_brightness (int val)
{
	if (val != -1)
		ioctl(pmu_filehandle, PMU_IOC_SET_BACKLIGHT, &val);
}

static struct driver_backlight driver_backlight_pmu = {
	.name               = N_("PMU Backlight Driver"),
	.get_brightness     = pmubl_get_brightness,
	.get_brightness_max = pmubl_get_brightness_max,
	.set_brightness     = pmubl_set_brightness,
	.driver_exit        = driver_backlight_pmu_exit,
};

struct driver_backlight *
driver_backlight_pmu_init (struct tagitem *taglist)
{
	char *devname;
	int pmu_version = 0, fd;
	long DUMMY = 0;

	devname = (char *) tagfind (taglist, TAG_PMUDEVICE, (long) DEFAULT_PMU);
	if ((copy_path (devname, NULL, TYPE_CHARDEV, CPFLG_NONE)) == 0) {
		if ((pmu_filehandle = open(devname, O_RDWR)) < 0) {
			print_msg (PBB_ERR, _("Can't open PMU device %s: %s\n"), devname, strerror(errno));
			return NULL;
		}
	} else return NULL;

	devname = (char *) tagfind (taglist, TAG_ADBDEVICE, (long) DEFAULT_ADB);
	if ((copy_path (devname, NULL, TYPE_CHARDEV, CPFLG_NONE)) == 0) {
		if ((fd = open(devname, O_RDWR)) >= 0) {
			pmu_version = get_pmu_version(fd);
			close (fd);
		} else
			print_msg (PBB_WARN, _("Can't open ADB device %s: %s\n"), devname, strerror(errno));
	};

	if ((ioctl(pmu_filehandle, PMU_IOC_GET_BACKLIGHT, &DUMMY)) == -1)
		return NULL;

	if (pmu_version > OHARE_PMU)
		ioctl(pmu_filehandle, PMU_IOC_GRAB_BACKLIGHT, &DUMMY);

	return &driver_backlight_pmu;
}


