/*
 * pci_res.c
 *
 * Copyright (C) 2006  Sony Computer Entertainment Inc.
 *
 * PS3PF pci resources routine for PCI.
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <linux/kernel.h>
#include <asm/lv1call.h>
#include "pci_res.h"

static void make_first_key(const char *name, uint64_t index, uint64_t *n)
{
	strncpy((char *)n, name, 8);
	*n = (*n >> 32) + index;
}

static void make_key(const char *name, uint64_t index, uint64_t *n)
{
	strncpy((char *)n, name, 8);
	*n += index;
}

int ps3pf_pci_read_platform_id(uint64_t *id)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_PLAT, 0, &n1);
	make_key(PS3PF_BUS_REPOSITORY_ID, 0, &n2);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*id = v1;
	return 0;
}

int ps3pf_pci_read_bus_id(uint64_t bus_index, uint64_t *system_bus_id)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_ID, 0, &n2);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*system_bus_id = v1;
	return 0;
}

int ps3pf_pci_read_bus_type(uint64_t bus_index, uint64_t *bus_type)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_TYPE, 0, &n2);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*bus_type = v1;
	return 0;
}

int ps3pf_pci_read_num_of_dev(uint64_t bus_index, uint64_t *num_of_dev)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_NUM_DEV, 0, &n2);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*num_of_dev = v1;
	return 0;
}

int ps3pf_pci_read_device_id(uint64_t bus_index, uint64_t device_index, 
				uint64_t *device_id)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_DEVICE, device_index, &n2);
	make_key(PS3PF_BUS_REPOSITORY_ID, 0, &n3);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*device_id = v1;
	return 0;
}

int ps3pf_pci_read_device_type(uint64_t bus_index, uint64_t device_index,
				uint64_t *device_type)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_DEVICE, device_index, &n2);
	make_key(PS3PF_BUS_REPOSITORY_TYPE, 0, &n3);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*device_type = v1;
	return 0;
}

int ps3pf_pci_read_pci_bus(uint64_t bus_index, uint64_t device_index, 
				uint64_t *pci_bus_number)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_DEVICE, device_index, &n2);
	make_key(PS3PF_BUS_REPOSITORY_PCI_BUS, 0, &n3);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*pci_bus_number = v1;
	return 0;
}

int ps3pf_pci_read_pci_device(uint64_t bus_index, uint64_t device_index, 
				uint64_t *pci_device_number)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_DEVICE, device_index, &n2);
	make_key(PS3PF_BUS_REPOSITORY_PCI_DEV, 0, &n3);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*pci_device_number = v1;
	return 0;
}

int ps3pf_pci_read_pci_func(uint64_t bus_index, uint64_t device_index,
				uint64_t *pci_func_number)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_DEVICE, device_index, &n2);
	make_key(PS3PF_BUS_REPOSITORY_PCI_FUNC, 0, &n3);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*pci_func_number = v1;
	return 0;
}

int ps3pf_pci_read_interrupt(uint64_t bus_index, uint64_t device_index, 
		uint64_t int_index, uint64_t *int_type, uint64_t *int_data)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_DEVICE, device_index, &n2);
	make_key(PS3PF_BUS_REPOSITORY_INTERRUPT, int_index, &n3);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*int_type = v1;
	*int_data = v2;
	return 0;
}

int ps3pf_pci_read_region_type(uint64_t bus_index, uint64_t device_index, 
			uint64_t region_index, uint64_t *region_type)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_DEVICE, device_index, &n2);
	make_key(PS3PF_BUS_REPOSITORY_REGION, region_index, &n3);
	make_key(PS3PF_BUS_REPOSITORY_TYPE, 0, &n4);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*region_type = v1;
	return 0;
}

int ps3pf_pci_read_region_data(uint64_t bus_index, uint64_t device_index, 
		uint64_t region_index, uint64_t *base_addr, uint64_t *size)
{
	uint64_t n1, n2, n3, n4, v1, v2, status;
	n1 = n2 = n3 = n4 = v1 = v2 = 0;

	make_first_key(PS3PF_BUS_REPOSITORY_BUS, bus_index, &n1);
	make_key(PS3PF_BUS_REPOSITORY_DEVICE, device_index, &n2);
	make_key(PS3PF_BUS_REPOSITORY_REGION, region_index, &n3);
	make_key(PS3PF_BUS_REPOSITORY_DATA, 0, &n4);

	status = lv1_get_repository_node_value(1, n1, n2, n3, n4, &v1, &v2);
	if (status != 0) {
		return status;
	}
	*base_addr = v1;
	*size = v2;
	return 0;
}

