/*
 * arch/powerpc/platforms/ps3pf/pic.c
 *
 * Interrupt controller code for CONFIG_PPC_PS3PF.
 *
 * Author: Frank Rowand frowand@mvista.com or source@mvista.com
 *
 * 2003, 2004 (c) MontaVista Software, Inc. This file is licensed under
 * the terms of the GNU General Public License version 2. This program
 * is licensed "as is" without any warranty of any kind, whether express
 * or implied.
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <asm/processor.h>
#include <linux/irq.h>

#include <asm/lv1call.h>
#include <asm/smp.h>
#include <asm/abs_addr.h>

#ifdef DEBUG
#define DBG(fmt...) udbg_printf(fmt)
#else
#define DBG(fmt...)
#endif

static unsigned long pu_id;
static unsigned long ps3pf_bitmap[2][8];
static unsigned long *ps3pf_irq_status[2];
static unsigned long *ps3pf_irq_mask[2];
static spinlock_t irq_mask_lock = SPIN_LOCK_UNLOCKED;

extern unsigned long ps3pf_max_plug_id;

static void
ps3pf_enable_irq(unsigned int irq)
{
	unsigned long	base_id;
	long		status;

	int		bit;
	int		element;
	unsigned long	flags;

	base_id = (irq / (sizeof(unsigned long) * 8 * 4));
	element = (irq % (sizeof(unsigned long) * 8 * 4)) /
		  (sizeof(unsigned long) * 8);
	bit     =  irq % (sizeof(unsigned long) * 8);

	/* this limit is based on the size of ps3pf_irq_mask[] */
	if (base_id != 0) {
		panic("%s(): irq number too large", __FUNCTION__);
	}

	spin_lock_irqsave(&irq_mask_lock, flags);
	ps3pf_irq_mask[0][element] |= 0x8000000000000000UL>>bit;
	ps3pf_irq_mask[1][element] = ps3pf_irq_mask[0][element];
	status = lv1_did_update_interrupt_mask(pu_id, 0);
	status = lv1_did_update_interrupt_mask(pu_id, 1);
	spin_unlock_irqrestore(&irq_mask_lock, flags);

	if (unlikely(status)) {
		printk(KERN_WARNING "%s(): irq %d lv1_set_interrupt_mask() "
		       "status: %ld\n",
		       __FUNCTION__, irq, status);
	}

	return;
}

static void
ps3pf_disable_irq(unsigned int irq)
{
	unsigned long	base_id;
	long		status;
	int		bit;
	int		element;
	unsigned long	flags;

	base_id = (irq / (sizeof(unsigned long) * 8 * 4));
	element = (irq % (sizeof(unsigned long) * 8 * 4)) /
		  (sizeof(unsigned long) * 8);
	bit     =  irq % (sizeof(unsigned long) * 8);

	/* this limit is based on the size of ps3pf_irq_mask[] */
	if (base_id != 0) {
		panic("%s(): irq number too large", __FUNCTION__);
	}

	spin_lock_irqsave(&irq_mask_lock, flags);
	ps3pf_irq_mask[0][element] &=~(0x8000000000000000UL>>bit);
	ps3pf_irq_mask[1][element] = ps3pf_irq_mask[0][element];
	status = 0;
	spin_unlock_irqrestore(&irq_mask_lock, flags);

	if (unlikely(status)) {
		printk(KERN_WARNING "%s(): irq %d lv1_set_interrupt_mask() "
		       "status: %ld\n",
		       __FUNCTION__, irq, status);
	}

	return;
}

static void
ps3pf_disable_and_ack_irq(unsigned int irq)
{

	ps3pf_disable_irq(irq);

	return;
}

static void
ps3pf_end_irq(unsigned int irq)
{
/* 
  lv1_end_of_interrupt must be called at end_irq. 
  Some lv1 drivers clear irq status in it.
*/
	long		status;
	status = lv1_end_of_interrupt(irq);

	if (unlikely(status)) {
		printk(KERN_WARNING "%s(): irq %d end_of_interrupt() "
		       "status: %ld\n",
		       __FUNCTION__, irq, status);
	}

	if (!(irq_desc[irq].status & (IRQ_DISABLED|IRQ_INPROGRESS))
	    && irq_desc[irq].action) {
		ps3pf_enable_irq(irq);
	}
}

hw_irq_controller ps3pf_IRQ_handler = {
	typename:     "PS3PF irq controller",
	startup:      NULL,
	shutdown:     NULL,
	enable:       ps3pf_enable_irq,
	disable:      ps3pf_disable_irq,
	ack:          ps3pf_disable_and_ack_irq,
	end:          ps3pf_end_irq,
	set_affinity: NULL
};


void __init
ps3pf_init_IRQ(void)
{
	long status;
	int thread_no;
	int		k;

	DBG(" -> ps3pf_init_IRQ\n");

	if (LV1_PLUG_ID_LIMIT > (NR_IRQS - 1)) {
		panic("%s(): LV1_PLUG_ID_LIMIT > (NR_IRQS - 1)", __FUNCTION__);
	}

	for (k = 0; k < NR_IRQS; k++) {
		irq_desc[k].status  = 0;
		irq_desc[k].handler = &ps3pf_IRQ_handler;
		irq_desc[k].action  = NULL;
		irq_desc[k].depth   = 1;
	}

	lv1_get_logical_pu_id(&pu_id);
	for (thread_no = 0; thread_no < 2; thread_no++) {
		ps3pf_irq_status[thread_no] = &ps3pf_bitmap[thread_no][0];
		ps3pf_irq_mask[thread_no] = &ps3pf_bitmap[thread_no][4];

		for (k = 0; k < 4; k++)
			ps3pf_irq_mask[thread_no][k] = 0;

		status = lv1_configure_irq_state_bitmap(pu_id, thread_no,
				virt_to_abs(&ps3pf_bitmap[thread_no]));
		if (unlikely(status)) {
			printk(KERN_WARNING"%s(): configure_irq_state_bitmap: "
			       "status: %ld\n",
			       __FUNCTION__, status);
			panic("Configuring IRQ bitmaps failed");
		}
	}

	DBG(" <- ps3pf_init_IRQ\n");
}

int
ps3pf_get_irq(struct pt_regs *regs)
{
	unsigned long	base_id;
	int             irq_no;
	int		k;
	unsigned long	pending[4];
	int		tmp_irq_no;

	irq_no = -1;

	for (base_id = 0; base_id <= ps3pf_max_plug_id; ) {
		for (k = 0; k < 4; k++) {
			pending[k] = ps3pf_irq_status[smp_processor_id()][k];
			pending[k] &= ps3pf_irq_mask[smp_processor_id()][k];
		}
		for (k = 0; k < 4; k++) {
			tmp_irq_no = 63 - __ffs(pending[k]);
			if (tmp_irq_no != 64) {
				irq_no = base_id + tmp_irq_no;
				break;
			}
			base_id += 64;
		}
		if (irq_no != -1) {
			break;
		}
	}
	return irq_no;
}
