/*

 $Id: _pyosd.c,v 1.1.1.1 2002/06/01 13:43:13 resolve Exp $
  
 pyosd - a wrapper of libxosd which allows the displaying of "on screen display"
         messages.

 Started 7/12/01.
 Copyright (C) 2001, Damien Elmes <resolve@repose.cx>.
 
 This file is licensed under the GPL. Please see the file COPYING
 for more details.

 To compile this file, you will need libxosd. I'm waiting for a response from
 the author about some patches, so for now use the version included with this
 distribution.
 
*/

#include <signal.h>
#include <Python.h>
#include <xosd.h>

// raised if there's an error in the underlying library (such as X not running)
static PyObject *pyosd_error;

// lazy prototypes
static PyObject *pyosd_init(PyObject *self, PyObject *args);
static PyObject *pyosd_deinit(PyObject *self, PyObject *args);
static PyObject *pyosd_display_string(PyObject *self, PyObject *args);
static PyObject *pyosd_display_perc(PyObject *self, PyObject *args);
static PyObject *pyosd_display_slider(PyObject *self, PyObject *args);
static PyObject *pyosd_set_colour(PyObject *self, PyObject *args);
static PyObject *pyosd_set_font(PyObject *self, PyObject *args);
static PyObject *pyosd_set_timeout(PyObject *self, PyObject *args);
static PyObject *pyosd_set_pos(PyObject *self, PyObject *args);
static PyObject *pyosd_set_offset(PyObject *self, PyObject *args);
static PyObject *pyosd_set_shadow_offset(PyObject *self, PyObject *args);

static PyMethodDef pyosd_methods[] = {
    {"init",              pyosd_init,              METH_VARARGS},
    {"deinit",            pyosd_deinit,            METH_VARARGS},
    {"display_string",    pyosd_display_string,    METH_VARARGS},
    {"display_perc",      pyosd_display_perc,      METH_VARARGS},
    {"display_slider",    pyosd_display_slider,    METH_VARARGS},
    {"set_font",          pyosd_set_font,          METH_VARARGS},
    {"set_colour",        pyosd_set_colour,        METH_VARARGS},
    {"set_timeout",       pyosd_set_timeout,       METH_VARARGS},
    {"set_pos",           pyosd_set_pos,           METH_VARARGS},
    {"set_offset",        pyosd_set_offset,        METH_VARARGS},
    {"set_shadow_offset", pyosd_set_shadow_offset, METH_VARARGS},
    {NULL,  NULL}
};

void
init_pyosd(void)
{
  PyObject *self;
  PyObject *dict;

  // create the module and add the functions
  self = Py_InitModule("_pyosd", pyosd_methods);

  // init custom exception
  dict = PyModule_GetDict(self);

  pyosd_error = PyErr_NewException("pyosd.error", NULL, NULL);
  PyDict_SetItemString(dict, "error", pyosd_error);
}

////////////////////////////////////////////////////////////////////////


// check to see that osd is a valid pointer. if it's not, raise an error
// and return return 0
static int assert_osd(xosd *osd, char *error)
{
  if (!osd) {
    PyErr_SetString(pyosd_error, error);
    return 0;
  }

  return 1;
}
    
// pyosd_init(fontdesc, colour, timeout, xosd_pos, offset, shadow)
// initialise the osd interface
static PyObject *
pyosd_init(PyObject *self, PyObject *args)
{
  char *fontdesc, *colour;
  int timeout, offset, shadow, pos, lines;
  xosd_pos osd_pos;
  PyObject *pyc_osd;
  xosd *osd = NULL;

  sigset_t newset;

  if(!PyArg_ParseTuple(args, "ssiiiii", &fontdesc, &colour,
			 &timeout, &pos, &offset, &shadow, &lines))
    return NULL;

  if (pos==0)
    osd_pos = XOSD_top;
  else if (pos==1)
    osd_pos = XOSD_bottom;
  else {
    PyErr_SetString(PyExc_ValueError, "OSD position not in range");
    return NULL;
  }

  // right, everything appeared to be successful so we can continue 
  // and load xosd. xosd uses threads, though, which means we must make
  // python thread safe

  if (osd)
    pyosd_deinit(NULL, NULL);

  // due to an unfortunate interaction with readline, we have to ensure
  // that signals are disabled while calling xosd_init - this stops the
  // threads it spawns from accepting SIGINT, and tripping up python

  sigemptyset(&newset); 
  sigaddset(&newset, SIGINT);

  sigprocmask(SIG_BLOCK, &newset, NULL);

  osd = xosd_init(fontdesc, colour, timeout, osd_pos, offset, shadow, lines);

  // turn SIGINT back on for the main app
  sigprocmask(SIG_UNBLOCK, &newset, NULL);

  if(!osd) {
    // we don't use assert_osd here, because we want to pass back the error
    // from the underlying code
    PyErr_SetString(pyosd_error, xosd_error);
    return NULL;
  }

  // we've now got a osd reference, which we need to package up and return
  // to the surrounding python code

  pyc_osd = PyCObject_FromVoidPtr((void *)osd, NULL);

  return pyc_osd;
}

static PyObject *
pyosd_deinit(PyObject *self, PyObject *args)
{
  PyObject *pyc_osd;
  xosd *osd;

  if(!PyArg_ParseTuple(args, "O", &pyc_osd))
    return NULL;

  osd = (xosd *)PyCObject_AsVoidPtr(pyc_osd);

  if(osd==NULL) {
    PyErr_SetString(pyosd_error, "Already deinitialised");
    return NULL;
  }

  // deinit - the wrapping python should clear out the
  // cobject as well, as it's no longer valid.
  xosd_uninit(osd);
  osd = NULL;

  Py_INCREF(Py_None);
  return Py_None;
}


// pyosd_display_string(line, string)
static PyObject *
pyosd_display_string(PyObject *self, PyObject *args)
{
  int line;
  char *str;
  PyObject *pyc_osd;
  xosd *osd;

  if(!PyArg_ParseTuple(args, "Ois", &pyc_osd, &line, &str))
    return NULL;

  osd = (xosd *)PyCObject_AsVoidPtr(pyc_osd);

  if(!assert_osd(osd, "Run init() first!"))
    return NULL;

  xosd_display(osd, line, XOSD_string, str);

  Py_INCREF(Py_None);
  return Py_None;
}

// FIXME
// pyosd_display_perc(line, percentage)
static PyObject *
pyosd_display_perc(PyObject *self, PyObject *args)
{
  int line;
  int perc;
  PyObject *pyc_osd;
  xosd *osd;  

  if(!PyArg_ParseTuple(args, "Oii", &pyc_osd, &line, &perc))
    return NULL;

  osd = (xosd *)PyCObject_AsVoidPtr(pyc_osd);

  if(!assert_osd(osd, "Run init() first!"))
    return NULL;

  xosd_display(osd, line, XOSD_percentage, perc);

  Py_INCREF(Py_None);
  return Py_None;
}

// pyosd_display_slider(line, slider)
static PyObject *
pyosd_display_slider(PyObject *self, PyObject *args)
{
  int line;
  int slider;
  PyObject *pyc_osd;
  xosd *osd;  

  if(!PyArg_ParseTuple(args, "Oii", &pyc_osd, &line, &slider))
    return NULL;

  osd = (xosd *)PyCObject_AsVoidPtr(pyc_osd);

  if(!assert_osd(osd, "Run init() first!"))
    return NULL;

  xosd_display(osd, line, XOSD_slider, slider);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pyosd_set_font(PyObject *self, PyObject *args)
{
  char *font;
  int res;
  PyObject *pyc_osd;
  xosd *osd;  

  if(!PyArg_ParseTuple(args, "Os", &pyc_osd, &font))
    return NULL;

  osd = (xosd *)PyCObject_AsVoidPtr(pyc_osd);

  if(!assert_osd(osd, "Run init() first!"))
    return NULL;

  res = xosd_set_font(osd, font);

  if(res==-1) {
    PyErr_SetString(pyosd_error, xosd_error);
    return NULL;
  }

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pyosd_set_colour(PyObject *self, PyObject *args)
{
  char *colour;
  PyObject *pyc_osd;
  xosd *osd;  

  if(!PyArg_ParseTuple(args, "Os", &pyc_osd, &colour))
    return NULL;

  osd = (xosd *)PyCObject_AsVoidPtr(pyc_osd);

  if(!assert_osd(osd, "Run init() first!"))
    return NULL;

  xosd_set_colour(osd, colour);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pyosd_set_timeout(PyObject *self, PyObject *args)
{
  int timeout;
  PyObject *pyc_osd;
  xosd *osd;  

  if(!PyArg_ParseTuple(args, "Oi", &pyc_osd, &timeout))
    return NULL;

  osd = (xosd *)PyCObject_AsVoidPtr(pyc_osd);

  if(!assert_osd(osd, "Run init() first!"))
    return NULL;

  xosd_set_timeout(osd, timeout);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pyosd_set_pos(PyObject *self, PyObject *args)
{
  int pos;
  PyObject *pyc_osd;
  xosd *osd;  

  xosd_pos osd_pos;

  if(!PyArg_ParseTuple(args, "Oi", &pyc_osd, &pos))
    return NULL;

  osd = (xosd *)PyCObject_AsVoidPtr(pyc_osd);

  if(!assert_osd(osd, "Run init() first!"))
    return NULL;

  if (pos==0)
    osd_pos = XOSD_top;
  else if (pos==1)
    osd_pos = XOSD_bottom;
  else {
    PyErr_SetString(PyExc_ValueError, "OSD position not in range");
    return NULL;
  }

  xosd_set_pos(osd, pos);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pyosd_set_offset(PyObject *self, PyObject *args)
{
  int offset;
  PyObject *pyc_osd;
  xosd *osd;  

  if(!PyArg_ParseTuple(args, "Oi", &pyc_osd, &offset))
    return NULL;

  osd = (xosd *)PyCObject_AsVoidPtr(pyc_osd);

  if(!assert_osd(osd, "Run init() first!"))
    return NULL;

  xosd_set_vertical_offset(osd, offset);

  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
pyosd_set_shadow_offset(PyObject *self, PyObject *args)
{
  int offset;
  PyObject *pyc_osd;
  xosd *osd;  

  if(!PyArg_ParseTuple(args, "Oi", &pyc_osd, &offset))
    return NULL;

  osd = (xosd *)PyCObject_AsVoidPtr(pyc_osd);

  if(!assert_osd(osd, "Run init() first!"))
    return NULL;

  xosd_set_shadow_offset(osd, offset);

  Py_INCREF(Py_None);
  return Py_None;
}

