// This module implements the QextScintillaLexerRuby class.
//
// Copyright (c) 2005
// 	Riverbank Computing Limited <info@riverbankcomputing.co.uk>
// 
// This file is part of QScintilla.
// 
// This copy of QScintilla is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option) any
// later version.
// 
// QScintilla is supplied in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
// details.
// 
// You should have received a copy of the GNU General Public License along with
// QScintilla; see the file LICENSE.  If not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.


#include <qcolor.h>
#include <qfont.h>
#include <qsettings.h>

#include "qextscintillalexerruby.h"


// The ctor.
QextScintillaLexerRuby::QextScintillaLexerRuby(QObject *parent,
						   const char *name)
	: QextScintillaLexer(parent,name), indent_warn(NoWarning)
{
}


// The dtor.
QextScintillaLexerRuby::~QextScintillaLexerRuby()
{
}


// Returns the language name.
const char *QextScintillaLexerRuby::language() const
{
	return "Ruby";
}


// Returns the lexer name.
const char *QextScintillaLexerRuby::lexer() const
{
	return "ruby";
}


// Return the list of characters that can start a block.
const char *QextScintillaLexerRuby::blockStart(int *style) const
{
	if (style)
		*style = Operator;

	return "";
}


// Return the number of lines to look back when auto-indenting.
int QextScintillaLexerRuby::blockLookback() const
{
	return 1;
}


// Return the style used for braces.
int QextScintillaLexerRuby::braceStyle() const
{
	return Operator;
}


// Returns the foreground colour of the text for a style.
QColor QextScintillaLexerRuby::color(int style) const
{
	switch (style)
	{
	case Default:
		return QColor(0x80,0x80,0x80);

	case Comment:
		return QColor(0x00,0x7f,0x00);

	case Number:
		return QColor(0x00,0x7f,0x7f);

	case DoubleQuotedString:
	case SingleQuotedString:
		return QColor(0x7f,0x00,0x7f);

	case Keyword:
		return QColor(0x00,0x00,0x7f);

	case TripleDoubleQuotedString:
		return QColor(0x7f,0x00,0x00);

	case ClassName:
		return QColor(0x00,0x00,0xff);

	case FunctionMethodName:
		return QColor(0x00,0x7f,0x7f);

	case Operator:
		break;

	case Identifier:
	case CommentBlock:
		return QColor(0x7f,0x7f,0x7f);

	case UnclosedString:
		return QColor(0x00,0x00,0x00);
	}

	return QextScintillaLexer::color(style);
}


// Returns the end-of-line fill for a style.
bool QextScintillaLexerRuby::eolFill(int style) const
{
	return (style == UnclosedString);
}


// Returns the font of the text for a style.
QFont QextScintillaLexerRuby::font(int style) const
{
	QFont f;

	switch (style)
	{
	case Comment:
#if defined(Q_OS_WIN)
		f = QFont("Comic Sans MS",9);
#else
		f = QFont("new century schoolbook",12);
#endif
		break;

	case DoubleQuotedString:
	case SingleQuotedString:
	case UnclosedString:
#if defined(Q_OS_WIN)
		f = QFont("Courier New",10);
#else
		f = QFont("courier",12);
#endif
		break;

	case Keyword:
	case ClassName:
	case FunctionMethodName:
	case Operator:
		f = QextScintillaLexer::font(style);
		f.setBold(TRUE);
		break;

	default:
		f = QextScintillaLexer::font(style);
	}

	return f;
}


// Returns the set of keywords.
const char *QextScintillaLexerRuby::keywords(int set) const
{
	if (set == 0)
		return
			"__FILE__ and def end in or self unless __LINE__ "
			"begin defined? ensure module redo super until BEGIN "
			"break do false next rescue then when END case else "
			"for nil require retry true while alias class elsif "
			"if not return undef yield";

	return 0;
}


// Returns the user name of a style.
QString QextScintillaLexerRuby::description(int style) const
{
	switch (style)
	{
	case Default:
		return tr("Default");

	case Comment:
		return tr("Comment");

	case Number:
		return tr("Number");

	case DoubleQuotedString:
		return tr("Double-quoted string");

	case SingleQuotedString:
		return tr("Single-quoted string");

	case Keyword:
		return tr("Keyword");

	case TripleDoubleQuotedString:
		return tr("Triple double-quoted string");

	case ClassName:
		return tr("Class name");

	case FunctionMethodName:
		return tr("Function or method name");

	case Operator:
		return tr("Operator");

	case Identifier:
		return tr("Identifier");

	case CommentBlock:
		return tr("Comment block");

	case UnclosedString:
		return tr("Unclosed string");
	}

	return QString::null;
}


// Returns the background colour of the text for a style.
QColor QextScintillaLexerRuby::paper(int style) const
{
	if (style == UnclosedString)
		return QColor(0xe0,0xc0,0xe0);

	return QextScintillaLexer::paper(style);
}


// Refresh all properties.
void QextScintillaLexerRuby::refreshProperties()
{
	setTabWhingeProp();
}


// Read properties from the settings.
bool QextScintillaLexerRuby::readProperties(QSettings &qs,const QString &prefix)
{
	int rc = TRUE, num;
	bool ok, flag;

	// Read the indentation warning.
	num = qs.readNumEntry(prefix + "indentwarning",(int)NoWarning,&ok);

	if (ok)
		indent_warn = (IndentationWarning)num;
	else
		rc = FALSE;

	return rc;
}


// Write properties to the settings.
bool QextScintillaLexerRuby::writeProperties(QSettings &qs,const QString &prefix) const
{
	int rc = TRUE;

	// Write the indentation warning.
	if (!qs.writeEntry(prefix + "indentwarning",(int)indent_warn))
		rc = FALSE;

	return rc;
}


// Return the indentation warning.
QextScintillaLexerRuby::IndentationWarning QextScintillaLexerRuby::indentationWarning() const
{
	return indent_warn;
}


// Set the indentation warning.
void QextScintillaLexerRuby::setIndentationWarning(IndentationWarning warn)
{
	indent_warn = warn;

	setTabWhingeProp();
}


// Set the "tab.timmy.whinge.level" property.
void QextScintillaLexerRuby::setTabWhingeProp()
{
	emit propertyChanged("tab.timmy.whinge.level",QString::number(indent_warn));
}
