/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 *
 *
 * This file incorporates work covered by the following copyright and
 * permission notice:
 *
 *  Copyright 2006 Maurizio Monge <maurizio.monge@gmail.com>
 *
 *  BSD License
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 *  IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 *  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 *  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Attention: note that even being named "BSD License" by the
 * copyright holder, the license above is a variation of the license
 * known as BSD License.  This variation is compatible with GNU GPL.
 */

#include <QDebug>
#include <QPixmapCache>
#include <QRegion>
#include <QPainter>
#include <QPaintEvent>

#include "qzionimage.h"
#include "qzionobject_p.h"

static int MAX_CACHE_SIZE = 4096;  // 4 mb

class QZionImagePrivate
{
public:
    QSize size;
    QPixmap pixmap;
};

QZionImage::QZionImage(QZionAbstractCanvas *canvas)
    : QZionObject(canvas), _QZionImage_data(new QZionImagePrivate)
{
    QPixmapCache::setCacheLimit(MAX_CACHE_SIZE);
}

QZionImage::QZionImage(QZionAbstractCanvas *canvas, const QString &filepath)
    : QZionObject(canvas), _QZionImage_data(new QZionImagePrivate)
{
    QPixmapCache::setCacheLimit(MAX_CACHE_SIZE);
    load(filepath);

    QZD(QZionImage);
    setSize(d->pixmap.size());
}

QZionImage::~QZionImage()
{
    delete _QZionImage_data;
}

QPixmap QZionImage::pixmap() const
{
    QZD(QZionImage);
    return d->pixmap;
}

bool QZionImage::load(const QString &filename)
{
    if (_filepath == filename)
        return true;

    _filepath = filename;

    QPixmap result;

    if (QPixmapCache::find(filename, result)) {
        setPixmap(result);
        return true;
    }

    if (!result.load(filename)) {
        clear();
        return false;
    }

    QPixmapCache::insert(filename, result);
    setPixmap(result);

    return true;
}

bool QZionImage::reload()
{
    QZD(QZionImage);

    if (d->pixmap.isNull())
        return true;

    QPixmap result;
    if (!result.load(_filepath)) {
        clear();
        return false;
    }

    QPixmapCache::insert(_filepath, result);
    setPixmap(result);

    return true;
}

void QZionImage::setPixmap(const QPixmap &p)
{
    QZD(QZionImage);

    d->pixmap = p;
    if (visible() && canvas())
        changed();
}

void QZionImage::clear()
{
    setPixmap(QPixmap());
}

QSize QZionImage::size() const
{
    QZD(QZionImage);
    return d->size;
}

void QZionImage::setSize(const QSize &size)
{
    QZD(QZionImage);
    d->size = size;

    QZionObject::setSize(size);
}

void QZionImage::paint(QPainter *p)
{
    QZD(QZionImage);

    if (d->pixmap.isNull())
        return;

    if (d->size == d->pixmap.size()) {
        p->drawPixmap(pos(), d->pixmap);
        return;
    }

    p->translate(pos());
    p->scale((double)d->size.width() / d->pixmap.size().width(),
             (double)d->size.height() / d->pixmap.size().height());
    p->drawPixmap(QPoint(0, 0), d->pixmap);
    p->scale(1, 1);
    p->translate(-pos());
}

QRect QZionImage::rect() const
{
    return QRect(pos(), size());
}

QColor QZionImage::color() const
{
    return QColor(255, 255, 255);
}

void QZionImage::setColor(const QColor &color)
{
    setOpacity(color.alpha());
}

void QZionImage::changed()
{
    QZionObject::changed();
}
