/*
 * Copyright (c) Daniel Silverstone <daniel.silverstone@canonical.com> 2004
 *
 * fileordering.c : Read a list of files on stdin and write it back
 *                  out in order of filesystem/first block
 */

#include <glib.h>
#include <glib/gprintf.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <fcntl.h>
#include <linux/fs.h>

#ifndef BUFFER_SIZE
#define BUFFER_SIZE 4096
#endif

GList *names = NULL;
GHashTable *nametofsys;
GHashTable *nametoblock;

#ifndef FIBMAP
#define FIBMAP BMAP_IOCTL
#endif

void
populate_hashtables()
{
  int fd, block;
  GList *name = names;
  struct stat sb;
    
  nametofsys = g_hash_table_new(g_str_hash, g_str_equal);
  nametoblock = g_hash_table_new(g_str_hash, g_str_equal);
  
  while( name ) {
    gchar *fname = (gchar *)(name->data);
    fd = open(fname, O_RDONLY);
    if( fd >= 0 ) {
      if( fstat( fd, &sb ) == -1 ) {
        close(fd);
        goto failure;
      }
      block = 0;
      if( ioctl( fd, FIBMAP, &block ) == -1 ) {
        close(fd);
        goto failure;
      }
      /* We have a dev_t and a block offset */
      close( fd );
      g_hash_table_insert( nametofsys, fname, GINT_TO_POINTER((int)(sb.st_dev)));
      g_hash_table_insert( nametoblock, fname, GINT_TO_POINTER(block) );
    } else {
      failure:
      g_free( fname );
      name->data = 0;
    }
    name = name->next;
  }
  
  
}


#define COMPRET if( aa < bb ) return -1; if( aa > bb) return 1
gint
compare_by_dev_and_block( gchar *a,
                          gchar *b )
{
  /* Return -1, 0, 1 based on a <=> b */
  /* a and b are file names */
  gpointer aa, bb;
  
  if( a == NULL && b == NULL ) return 0;
  if( a == NULL ) return -1;
  if( b == NULL ) return 1;
  
  aa = g_hash_table_lookup( nametofsys, a );
  bb = g_hash_table_lookup( nametofsys, b );
  COMPRET;
  aa = g_hash_table_lookup( nametoblock, a );
  bb = g_hash_table_lookup( nametoblock, b );
  COMPRET;
  return 0;
}
#undef COMPRET

void
print_filename( gchar *name, GIOChannel *stdout )
{
  if( name ) {
    g_io_channel_write_chars( stdout, name, strlen(name), NULL, NULL );
    g_io_channel_write_chars( stdout, "\n", 1, NULL, NULL );
  }
}

int 
main( int argc, char** argv ) 
{
  gchar *stdin_data, *tmp, **lines, **tmpl, *line;
  gchar buffer[BUFFER_SIZE];
  gsize bread;
  GIOChannel *stdin, *stdout, *stderr;
  GError *err = NULL;
  
  stdin = g_io_channel_unix_new( 0 ); /* If STDIN isn't FD(0) I'm dead */
  stdout = g_io_channel_unix_new( 1 ); /* DITTO STDOUT/1 */
  stderr = g_io_channel_unix_new( 2 ); /* DITTO STDERR/2 */
  
  stdin_data = g_strdup("");
  
  do {
    GIOStatus r = g_io_channel_read_chars( stdin, buffer, BUFFER_SIZE - 1, &bread, &err );
      if( r != G_IO_STATUS_NORMAL && r != G_IO_STATUS_EOF ) {
      g_io_channel_write_chars( stderr, err->message, strlen(err->message), 
                                NULL, NULL );
      return 1;
    }
    if( bread > 0 ) {
      tmp = stdin_data;
      buffer[bread] = 0;
      stdin_data = g_strconcat( stdin_data, buffer, NULL );
      g_free(tmp);
    }
  } while( bread > 0 );
  
  /* stdin_data is now a string of lines */
  
  lines = g_strsplit( stdin_data, "\n", 0 );
  g_free( stdin_data );
  
  tmpl = lines;
  
  /* lines is now a set of filenames, one per line... populate 'names' */
  for( line = lines[0]; line; line = *(++lines) )
    names = g_list_prepend( names, g_strdup( line ) );
  g_strfreev(tmpl);
  
  populate_hashtables();
  
  names = g_list_sort( names, (GCompareFunc)compare_by_dev_and_block );
  
  g_list_foreach( names, (GFunc)print_filename, stdout );
  
  g_io_channel_flush( stdout, NULL );
  g_io_channel_flush( stderr, NULL );
  
}
