/*
 *  arch-tag: Implementation of Rhythmbox tray icon object
 *
 *  Copyright (C) 2003,2004 Colin Walters <walters@redhat.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <gtk/gtk.h>
#include <config.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>

#include "disclosure-widget.h"
#include "rb-tray-icon.h"
#include "rb-stock-icons.h"
#include "rb-debug.h"
#include "eel-gconf-extensions.h"
#include "rb-preferences.h"
#include "rb-shell.h"

static void rb_tray_icon_class_init (RBTrayIconClass *klass);
static void rb_tray_icon_init (RBTrayIcon *shell_player);
static GObject *rb_tray_icon_constructor (GType type, guint n_construct_properties,
					  GObjectConstructParam *construct_properties);
static void rb_tray_icon_finalize (GObject *object);
static void rb_tray_icon_sync_action (RBRemoteProxy *proxy, 
				      gboolean visible, 
				      RBTrayIcon *tray);
static void rb_tray_icon_set_property (GObject *object,
					  guint prop_id,
					  const GValue *value,
					  GParamSpec *pspec);
static void rb_tray_icon_get_property (GObject *object,
					  guint prop_id,
					  GValue *value,
					  GParamSpec *pspec);
static void rb_tray_icon_button_press_event_cb (GtkWidget *ebox, GdkEventButton *event,
						RBTrayIcon *icon);
static void rb_tray_icon_scroll_event_cb (GtkWidget *ebox, GdkEvent *event,
						RBTrayIcon *icon);
static void rb_tray_icon_show_window_changed_cb (GtkAction *action,
						 RBTrayIcon *icon);
static void rb_tray_icon_drop_cb (GtkWidget *widget,
				  GdkDragContext *context,
				  gint x,
				  gint y,
				  GtkSelectionData *data,
				  guint info,
				  guint time,
				  RBTrayIcon *icon);

struct RBTrayIconPrivate
{
	GtkTooltips *tooltips;

	GtkUIManager *ui_manager;
	GtkActionGroup *actiongroup;

	GtkWidget *ebox;

	RBRemoteProxy *proxy;
};

enum
{
	PROP_0,
	PROP_UI_MANAGER,
	PROP_ACTION_GROUP,
	PROP_REMOTE
};

enum
{
	LAST_SIGNAL,
};

static GtkToggleActionEntry rb_tray_icon_toggle_entries [] =
{
	{ "TrayShowWindow", NULL, N_("_Show Window"), NULL,
	  N_("Change the visibility of the main window"),
	  G_CALLBACK (rb_tray_icon_show_window_changed_cb) }
};
static guint rb_tray_icon_n_toggle_entries = G_N_ELEMENTS (rb_tray_icon_toggle_entries);

static const GtkTargetEntry target_uri [] = {{ "text/uri-list", 0, 0 }};

static GObjectClass *parent_class = NULL;

GType
rb_tray_icon_get_type (void)
{
	static GType rb_tray_icon_type = 0;

	if (rb_tray_icon_type == 0)
	{
		static const GTypeInfo our_info =
		{
			sizeof (RBTrayIconClass),
			NULL,
			NULL,
			(GClassInitFunc) rb_tray_icon_class_init,
			NULL,
			NULL,
			sizeof (RBTrayIcon),
			0,
			(GInstanceInitFunc) rb_tray_icon_init
		};

		rb_tray_icon_type = g_type_register_static (EGG_TYPE_TRAY_ICON,
							    "RBTrayIcon",
							    &our_info, 0);
	}

	return rb_tray_icon_type;
}

static void
rb_tray_icon_class_init (RBTrayIconClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = rb_tray_icon_finalize;
	object_class->constructor = rb_tray_icon_constructor;

	object_class->set_property = rb_tray_icon_set_property;
	object_class->get_property = rb_tray_icon_get_property;

	g_object_class_install_property (object_class,
					 PROP_REMOTE,
					 g_param_spec_object ("remote",
							      "RBRemoteProxy",
							      "RBRemoteProxy object",
							      RB_TYPE_REMOTE_PROXY,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (object_class,
					 PROP_UI_MANAGER,
					 g_param_spec_object ("ui-manager",
							      "GtkUIManager",
							      "GtkUIManager object",
							      GTK_TYPE_UI_MANAGER,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (object_class,
					 PROP_ACTION_GROUP,
					 g_param_spec_object ("action-group",
							      "GtkActionGroup",
							      "GtkActionGroup object",
							      GTK_TYPE_ACTION_GROUP,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));
}

static void
rb_tray_icon_init (RBTrayIcon *icon)
{
	GtkWidget *image;

	rb_debug ("setting up tray icon");

	icon->priv = g_new0 (RBTrayIconPrivate, 1);

	icon->priv->tooltips = gtk_tooltips_new ();

	gtk_tooltips_set_tip (icon->priv->tooltips, GTK_WIDGET (icon),
			      _("Not playing"), NULL);

	icon->priv->ebox = gtk_event_box_new ();
	g_signal_connect_object (G_OBJECT (icon->priv->ebox),
				 "button_press_event",
				 G_CALLBACK (rb_tray_icon_button_press_event_cb),
				 icon, 0);
	g_signal_connect_object (G_OBJECT (icon->priv->ebox),
				 "scroll_event",
				 G_CALLBACK (rb_tray_icon_scroll_event_cb),
				 icon, 0);
	gtk_drag_dest_set (icon->priv->ebox, GTK_DEST_DEFAULT_ALL, target_uri, 1, GDK_ACTION_COPY);
	g_signal_connect_object (G_OBJECT (icon->priv->ebox), "drag_data_received",
				 G_CALLBACK (rb_tray_icon_drop_cb), icon, 0);

	image = gtk_image_new_from_stock (RB_STOCK_TRAY_ICON,
					  GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_container_add (GTK_CONTAINER (icon->priv->ebox), image);
	
	gtk_container_add (GTK_CONTAINER (icon), icon->priv->ebox);
	gtk_widget_show_all (GTK_WIDGET (icon->priv->ebox));
}

static GObject *
rb_tray_icon_constructor (GType type, guint n_construct_properties,
			  GObjectConstructParam *construct_properties)
{
	RBTrayIcon *tray;
	RBTrayIconClass *klass;
	GObjectClass *parent_class;  

	klass = RB_TRAY_ICON_CLASS (g_type_class_peek (RB_TYPE_TRAY_ICON));

	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	tray = RB_TRAY_ICON (parent_class->constructor (type, n_construct_properties,
							construct_properties));

	return G_OBJECT (tray);
}

static void
rb_tray_icon_finalize (GObject *object)
{
	RBTrayIcon *tray;

	g_return_if_fail (object != NULL);
	g_return_if_fail (RB_IS_TRAY_ICON (object));

	tray = RB_TRAY_ICON (object);

	g_return_if_fail (tray->priv != NULL);
	
	gtk_object_destroy (GTK_OBJECT (tray->priv->tooltips));

	g_free (tray->priv);

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static void
rb_tray_icon_sync_action (RBRemoteProxy *proxy, gboolean visible, RBTrayIcon *tray)
{
	GtkAction *action;
	if ((tray->priv->actiongroup != NULL) && (tray->priv->proxy != NULL)) {
		action = gtk_action_group_get_action (tray->priv->actiongroup,
						      "TrayShowWindow");
		gtk_toggle_action_set_active (GTK_TOGGLE_ACTION (action),
					      rb_remote_proxy_get_visibility (tray->priv->proxy));
	}
}

static void
rb_tray_icon_set_property (GObject *object,
			   guint prop_id,
			   const GValue *value,
			   GParamSpec *pspec)
{
	RBTrayIcon *tray = RB_TRAY_ICON (object);

	switch (prop_id)
	{
	case PROP_REMOTE:
		tray->priv->proxy = g_value_get_object (value);
		g_signal_connect_object (G_OBJECT (tray->priv->proxy),
					 "visibility_changed",
					 G_CALLBACK (rb_tray_icon_sync_action),
					 tray, 0);
		rb_tray_icon_sync_action (NULL, FALSE, tray);
		break;
	case PROP_UI_MANAGER:
		tray->priv->ui_manager = g_value_get_object (value);
		break;
	case PROP_ACTION_GROUP:
		tray->priv->actiongroup = g_value_get_object (value);
		gtk_action_group_add_toggle_actions (tray->priv->actiongroup,
						     rb_tray_icon_toggle_entries,
						     rb_tray_icon_n_toggle_entries,
						     tray);
		rb_tray_icon_sync_action (NULL, FALSE, tray);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void 
rb_tray_icon_get_property (GObject *object,
			      guint prop_id,
			      GValue *value,
			      GParamSpec *pspec)
{
	RBTrayIcon *tray = RB_TRAY_ICON (object);

	switch (prop_id)
	{
	case PROP_REMOTE:
		g_value_set_object (value, tray->priv->proxy);
		break;
	case PROP_UI_MANAGER:
		g_value_set_object (value, tray->priv->ui_manager);
		break;
	case PROP_ACTION_GROUP:
		g_value_set_object (value, tray->priv->actiongroup);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

RBTrayIcon *
rb_tray_icon_new (GtkUIManager *mgr,
		  GtkActionGroup *group,
		  RBRemoteProxy *remote)
{
	return g_object_new (RB_TYPE_TRAY_ICON,
			     "title", "Rhythmbox tray icon",
			     "ui-manager", mgr,
			     "action-group", group,
			     "remote", remote,
			     NULL);
}

static void
rb_tray_icon_button_press_event_cb (GtkWidget *ebox, GdkEventButton *event,
				    RBTrayIcon *icon)
{
	/* filter out double, triple clicks */
	if (event->type != GDK_BUTTON_PRESS)
		return;

	rb_debug ("tray button press");

	switch (event->button) {
	case 1:
	{
		gboolean visible = rb_remote_proxy_get_visibility (icon->priv->proxy);
		rb_remote_proxy_set_visibility (icon->priv->proxy, visible ? FALSE : TRUE);
		break;
	}

	case 3:
	{
		GtkWidget *popup;
		popup = gtk_ui_manager_get_widget (GTK_UI_MANAGER (icon->priv->ui_manager),
						   "/RhythmboxTrayPopup");
		gtk_menu_set_screen (GTK_MENU (popup), gtk_widget_get_screen (GTK_WIDGET (icon)));
		gtk_menu_popup (GTK_MENU (popup), NULL, NULL,
				NULL, NULL, 2,
				gtk_get_current_event_time ());
	}
	break;
	default:
		break;
	}
}

static void
rb_tray_icon_scroll_event_cb (GtkWidget *ebox, GdkEvent *event,
				    RBTrayIcon *icon)
{
	float volume;
	rb_debug ("tray button scroll");
	volume = eel_gconf_get_float (CONF_STATE_VOLUME);
	switch(event->scroll.direction) {
	case GDK_SCROLL_UP:
		volume += 0.1;
		if (volume > 1.0)
			volume = 1.0;
		break;
	case GDK_SCROLL_DOWN:
		volume -= 0.1;
		if (volume < 0)
			volume = 0;
		break;
	case GDK_SCROLL_LEFT:
	case GDK_SCROLL_RIGHT:
		break;
	}
	
	rb_debug ("got scroll, setting volume to %f", volume);
	eel_gconf_set_float (CONF_STATE_VOLUME, volume);
}

static void
rb_tray_icon_drop_cb (GtkWidget *widget,
		      GdkDragContext *context,
		      gint x,
		      gint y,
		      GtkSelectionData *data,
		      guint info,
		      guint time,
		      RBTrayIcon *icon)
{
	GList *list, *uri_list, *i;
	GtkTargetList *tlist;
	gboolean ret;

	tlist = gtk_target_list_new (target_uri, 1);
	ret = (gtk_drag_dest_find_target (widget, context, tlist) != GDK_NONE);
	gtk_target_list_unref (tlist);

	if (ret == FALSE)
		return;

	list = gnome_vfs_uri_list_parse ((char *) data->data);

	if (list == NULL) {
		gtk_drag_finish (context, FALSE, FALSE, time);
		return;
	}

	uri_list = NULL;

	for (i = list; i != NULL; i = g_list_next (i))
		uri_list = g_list_append (uri_list, gnome_vfs_uri_to_string ((const GnomeVFSURI *) i->data, 0));

	gnome_vfs_uri_list_free (list);

	if (uri_list == NULL) {
		gtk_drag_finish (context, FALSE, FALSE, time);
		return;
	}

	for (i = uri_list; i != NULL; i = i->next) {
		char *uri = i->data;
		if (uri != NULL)
			rb_remote_proxy_load_uri (icon->priv->proxy, uri, FALSE);

		g_free (uri);
	}

	g_list_free (uri_list);

	gtk_drag_finish (context, TRUE, FALSE, time);
}

static void
rb_tray_icon_show_window_changed_cb (GtkAction *action,
				     RBTrayIcon *icon)
{
	rb_debug ("show window clicked");
	rb_remote_proxy_set_visibility (icon->priv->proxy,
					gtk_toggle_action_get_active (GTK_TOGGLE_ACTION (action)));
}

void
rb_tray_icon_set_tooltip (RBTrayIcon *icon, const char *tooltip)
{
	gtk_tooltips_set_tip (icon->priv->tooltips,
			      GTK_WIDGET (icon),
			      tooltip, NULL);
}

