/** @file scim_hangul_imengine.h
 */

/* 
 * Smart Common Input Method
 * 
 * Copyright (c) 2004 James Su <suzhe@tsinghua.org.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_hangul_imengine.h,v 1.2 2005/10/29 09:53:12 hwanjin Exp $
 */

#if !defined (__SCIM_HANGUL_IMENGINE_H)
#define __SCIM_HANGUL_IMENGINE_H

#include "hangul.h"

using namespace scim;

enum HangulInputMode
{
    INPUT_MODE_DIRECT,
    INPUT_MODE_HANGUL,
    INPUT_MODE_HANJA
};

enum HangulOutputMode
{
    OUTPUT_MODE_SYLLABLE = 0,
    OUTPUT_MODE_JAMO     = 1 << 1,
    OUTPUT_MODE_JAMO_EXT = 1 << 2
};

struct HangulCombination 
{
  uint32  key;
  ucs4_t  code;
};

struct HangulFactoryData
{
    const char              *uuid;
    const char              *name;

    HangulKeyboardType       keyboard_type;
};

class HangulFactory : public IMEngineFactoryBase
{
    String                   m_uuid;
    String                   m_name;

    HangulKeyboardType       m_keyboard_type;

    bool                     m_always_use_jamo;

    bool                     m_show_candidate_comment;

    KeyEventList             m_trigger_keys;
    KeyEventList             m_hangul_hanja_keys;

    friend class HangulInstance;

public:
    HangulFactory (const ConfigPointer &config, const HangulFactoryData &data);

    virtual ~HangulFactory ();

    virtual WideString  get_name () const;
    virtual WideString  get_authors () const;
    virtual WideString  get_credits () const;
    virtual WideString  get_help () const;
    virtual String      get_uuid () const;
    virtual String      get_icon_file () const;

    virtual IMEngineInstancePointer create_instance (const String& encoding, int id = -1);
};

class HangulInstance : public IMEngineInstanceBase
{
    HangulFactory       *m_factory;

    CommonLookupTable    m_lookup_table;
    std::vector<String>  m_candidate_comments;

    KeyEvent             m_prev_key;

    HangulInputContext  *m_hic;

    int                  m_index;         /* stack index */
    ucs4_t               m_stack     [12];

    int                  m_lindex;        /* leading consonant index */
    int                  m_vindex;        /* vowel index */
    int                  m_tindex;        /* trailing consonant index */
    ucs4_t               m_choseong  [4];
    ucs4_t               m_jungseong [4];
    ucs4_t               m_jongseong [4];

    int                  m_input_mode;
    int                  m_output_mode;

    bool (HangulInstance::*m_hangul_composer) (const KeyEvent &key);

public:
    HangulInstance (HangulFactory *factory,
                    const String  &encoding,
                    int            id = -1);

    virtual ~HangulInstance ();

    virtual bool process_key_event (const KeyEvent& key);
    virtual void move_preedit_caret (unsigned int pos);
    virtual void select_candidate (unsigned int item);
    virtual void update_lookup_table_page_size (unsigned int page_size);
    virtual void lookup_table_page_up ();
    virtual void lookup_table_page_down ();
    virtual void reset ();
    virtual void focus_in ();
    virtual void focus_out ();
    virtual void trigger_property (const String &property);

private:
    bool is_backspace_key (const KeyEvent &key) const {
        return (key.code == SCIM_KEY_BackSpace);
    }

    bool is_hangul_hanja_key (const KeyEvent &key) const {
        return match_key_event (m_factory->m_hangul_hanja_keys, key);
    }

    /* preedit string */
    WideString get_preedit_string () {
        WideString wstr;
        const wchar_t *wcs = hangul_ic_get_preedit_string(m_hic);
        while (*wcs != L'\0')
            wstr.push_back (*wcs++);
        return wstr;
    }

    WideString get_commit_string () {
        WideString wstr;
        const wchar_t *wcs = hangul_ic_get_commit_string(m_hic);
        while (*wcs != L'\0')
            wstr.push_back (*wcs++);
        return wstr;
    }

    void hangul_update_preedit_string ();

    /* aux string */
    void hangul_update_aux_string ();

    /* candidate functions */
    void   update_candidates ();
    void   delete_candidates ();

    /* candidate keys */
    bool   candidate_key_event (const KeyEvent &key);

    /* match key event */
    bool   match_key_event (const KeyEventList &keys, const KeyEvent &key) const;
};
#endif

/*
vi:ts=4:nowrap:ai:expandtab
*/
