/*  Screem:  formWizard.c
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include <config.h>

#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <libgnomeui/gnome-pixmap.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnome/gnome-i18n.h>

#include <gtk/gtkdialog.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtkcombo.h>
#include <gtk/gtkentry.h>
#include <gtk/gtklabel.h>

#include "screem-plugin.h"
#include "screem-window.h"
#include "screem-editor.h"

#include "fileops.h"

#include "libegg/menu/egg-menu.h"

/* per wizard struct */
typedef struct {
	ScreemWindow *window;
	ScreemEditor *editor;

	GtkWidget *dialog;

	GtkWidget *method;
	GtkWidget *action;
	GtkWidget *enctype;
} FormWizard;

/* keep track of how many wizards we have */
static GList *wizards = NULL;

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 3;

#define NAME "Form Wizard"
#define AUTHOR "David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG NULL

G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );
void form_wizard_display( EggAction *action, gpointer user_data );
void form_wizard_clicked( GtkWidget *widget, gint button, gpointer data );

static GtkWidget* create_dialog( FormWizard *form_wizard );


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	FormWizard *form_wizard;
	const gchar *ui = "\
<Root>\
<menu>\
<submenu name=\"Insert\" verb=\"Insert\">\
<submenu name=\"Wizards\" verb=\"Wizards\">\
<menuitem name=\"FormWizard\" verb=\"FormWizard\"/>\
</submenu>\
</submenu>\
</menu>\
<dockitem name=\"Wizards Toolbar\">\
<toolitem name=\"FormWizard\" verb=\"FormWizard\" />\
</dockitem>\
</Root>";
	static EggActionGroupEntry entries[] = {
		{ "FormWizard", "label",
	  	GTK_STOCK_JUSTIFY_FILL, NULL, "tip",
	  	G_CALLBACK( form_wizard_display ), NULL },
	};
	EggActionGroupEntry *entry;

	gchar *label;
	gchar *tip;

	form_wizard = g_new0( FormWizard, 1 );
	form_wizard->window = SCREEM_WINDOW( window );
	form_wizard->editor = SCREEM_EDITOR( editor );

	label = g_strdup( _( "Form Wizard" ) );
	tip = g_strdup( _( "Create a Form" ) );
	
	entry = entries;
	entry->label = label;
	entry->tooltip = tip;
	entry->user_data = form_wizard;
	
	egg_action_group_add_actions( EGG_ACTION_GROUP( form_wizard->window->action_group ),
					entry, 1 );
	egg_menu_merge_add_ui_from_string( EGG_MENU_MERGE( form_wizard->window->merge ),
						ui, strlen( ui ), NULL );
	g_free( label );
	g_free( tip );

	/* NOTE: make sure that the window being obtained isn't visible */
	form_wizard->dialog = create_dialog( form_wizard );

	wizards = g_list_append( wizards, form_wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	FormWizard *form_wizard;

	for( list = wizards; list; list = list->next ) {
		form_wizard = (FormWizard*)list->data;
		if( form_wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* link_wizard is the one to erase */
	wizards = g_list_remove( wizards, form_wizard );
	g_free( form_wizard );
}

/* End of required section */

void form_wizard_display( EggAction *action, gpointer user_data )
{
	FormWizard *wizard;
	ScreemPage *page;

	wizard = (FormWizard*)user_data;

	page = screem_window_get_document( wizard->window );

	if( ! page )
		return;

	gtk_widget_show_all( wizard->dialog );
	gdk_window_raise( wizard->dialog->window );
}


void form_wizard_clicked( GtkWidget *widget, gint button, gpointer data )
{
	const gchar *method;
	const gchar *action;
	const gchar *enctype;

	GtkCombo *combo;

	gchar *open;
	gchar *close;

	FormWizard *wizard;

	wizard = (FormWizard*)g_object_get_data( G_OBJECT( widget ),
						 "form_wizard" );

	if( button == GTK_RESPONSE_OK || button == GTK_RESPONSE_APPLY ) {
		combo = GTK_COMBO( wizard->method );
		method = gtk_entry_get_text( GTK_ENTRY( combo->entry ) );
		combo = GTK_COMBO( wizard->action );
		action = gtk_entry_get_text( GTK_ENTRY( combo->entry ) );
		combo = GTK_COMBO( wizard->enctype );
		enctype = gtk_entry_get_text( GTK_ENTRY( combo->entry ) );
		if( ! strcmp( "", enctype ) )
			enctype = NULL;

		if( enctype )
			open = g_strdup_printf( "<form method=\"%s\" action=\"%s\" enctype=\"%s\">\n", method, action, enctype );
		else
			open = g_strdup_printf( "<form method=\"%s\" action=\"%s\">\n", method, action );
		
		close = "\n</form>";

		screem_editor_insert_markup( wizard->editor, open, close );

		g_free( open );
	}

	if( button != GTK_RESPONSE_OK ) {
		gtk_widget_hide( widget );
	}
}

static GtkWidget* create_dialog( FormWizard *form_wizard )
{
	GtkWidget *dialog;
	GtkWidget *table;
	GtkWidget *label;
	
	dialog = gtk_dialog_new_with_buttons(_("Insert a Form - Screem"),
					     NULL, 0,
					     GTK_STOCK_CLOSE, 
					     GTK_RESPONSE_CLOSE,
					     GTK_STOCK_APPLY,
					     GTK_RESPONSE_APPLY,
					     GTK_STOCK_OK,
					     GTK_RESPONSE_OK,
					     NULL );

	table = gtk_table_new( 3, 2, FALSE );

	label = gtk_label_new( "method=" );
	gtk_table_attach( GTK_TABLE( table ), label,
			  0, 1, 0, 1, 0, 0, GNOME_PAD, GNOME_PAD );
	label = gtk_label_new( "action=" );
	gtk_table_attach( GTK_TABLE( table ), label,
			  0, 1, 1, 2, 0, 0, GNOME_PAD, GNOME_PAD );
	label = gtk_label_new( "enctype=" );
	gtk_table_attach( GTK_TABLE( table ), label,
			  0, 1, 2, 3, 0, 0, GNOME_PAD, GNOME_PAD );

	/* methods combo */
	{
		const gchar *methods[] = {
			"POST",
			"GET",
			NULL
		};
		GList *method_list = NULL;
		gint i;

		form_wizard->method = gtk_combo_new();
		
		for( i = 0; methods[ i ]; ++ i ) {
			method_list = g_list_append( method_list,
						     (gchar*)methods[ i ] );
		}
		gtk_combo_set_popdown_strings(GTK_COMBO(form_wizard->method),
					      method_list);
		gtk_table_attach( GTK_TABLE( table ), form_wizard->method,
				  1, 2, 0, 1, 
				  GTK_EXPAND | GTK_FILL,
				  GTK_EXPAND | GTK_FILL,
				  GNOME_PAD, GNOME_PAD );
	}
	/* action combo */
	{
		const gchar *actions[] = {
			NULL
		};
		GList *action_list = NULL;
		gint i;

		form_wizard->action = gtk_combo_new();
		
		for( i = 0; actions[ i ]; ++ i ) {
			action_list = g_list_append( action_list,
						     (gchar*)actions[ i ] );
		}
		/*
		  we don't have any actions to set! 
		  gtk_combo_set_popdown_strings(GTK_COMBO(form_wizard->action),
		  action_list);*/
		gtk_table_attach( GTK_TABLE( table ), form_wizard->action,
				  1, 2, 1, 2, 				 
				  GTK_EXPAND | GTK_FILL,
				  GTK_EXPAND | GTK_FILL,
				  GNOME_PAD, GNOME_PAD );

	}
	/* enctype combo */
	{
		const gchar *types[] = {
			"",
			"application/x-www-form-urlencoded",
			"multipart/form-data",
			"image/jpeg",
			"image/gif",
			"image/png",
			"text/plain",
			"text/html",
			"text/xml",
			NULL
		};
		GList *enctype_list = NULL;
		gint i;

		form_wizard->enctype = gtk_combo_new();
		
		for( i = 0; types[ i ]; ++ i ) {
			enctype_list = g_list_append( enctype_list,
						      (gchar*)types[ i ] );
		}
		gtk_combo_set_popdown_strings(GTK_COMBO(form_wizard->enctype),
					      enctype_list);
		gtk_table_attach( GTK_TABLE( table ), form_wizard->enctype,
				  1, 2, 2, 3, 
				  GTK_EXPAND | GTK_FILL,
				  GTK_EXPAND | GTK_FILL,
				  GNOME_PAD, GNOME_PAD );
	}

	gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog )->vbox ), table,
			    TRUE, TRUE, GNOME_PAD );

	g_signal_connect( G_OBJECT( dialog ), "response", 
			  G_CALLBACK( form_wizard_clicked ), NULL );
	
	gtk_window_set_wmclass(GTK_WINDOW( dialog ), "form_wizard", "Screem");

	g_signal_connect( G_OBJECT( dialog ), "delete_event",
			  G_CALLBACK( gtk_widget_hide ), NULL );
	
	g_object_set_data( G_OBJECT( dialog ), "form_wizard", form_wizard );

	return dialog;
}
