/*  Screem:  screem-tagtree.c
 *
 *  the tag tree object
 *
 *  Copyright (C) 2001 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include <glade/glade.h>

#include <gtk/gtktreeview.h>
#include <gtk/gtkscrolledwindow.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtksizegroup.h>
#include <gtk/gtkdialog.h>

#include <gtksourceview/gtksourceview.h>

#include <libgnome/gnome-i18n.h>

#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <libgnomevfs/gnome-vfs-mime-utils.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include <libxml/tree.h>

#include "screem-tagtree.h"

#include "screem-window.h"
#include "screem-window-private.h"
#include "screem-editor.h"
#include "screem-page.h"
#include "screem-dtd.h"

#include "screem-file-browser.h"

#include "fileops.h"
#include "support.h"

#include "libegg/menu/egg-action-group.h"
#include "libegg/menu/egg-toggle-action.h"
#include "libegg/menu/egg-menu-merge.h"

#include "libegg/toolbar/eggtoolbar.h"

#include "eggtreemodelunion.h"

#include "screem-tagfile.h"

#include "screem-top-level-model.h"

static GSList *trees = NULL;
static GSList *treefiles = NULL;
static GtkListStore *filesstore = NULL;
static GMutex *mutex = NULL;

static xmlDocPtr config = NULL;
static gchar *configpath = NULL;

struct ScreemTagTreePrivate {
	GtkWidget *box;

	GtkWidget *view;

	GtkTreeModel *umodel;

	GHashTable *paths;

	ScreemWindow *window;

	GtkWidget *toolbar;

	ScreemDTD *dtd;
	ScreemPage *page;

	guint timeout;

	GSList *dtdcache;
	guint dtdcachelen;

	ScreemTagFile *dtdbranch;

	const gchar *search_pattern;
	gboolean regexp;
	GtkTreePath *search_result;
};

enum {
	ARG_0,
	ARG_WINDOW,
	ARG_PAGE
};
static void screem_tag_tree_add_dtd( ScreemPage *page, GParamSpec *spec,
				     ScreemTagTree *tree );

static void screem_tagtree_sel_change( GtkTreeSelection *selection,
					ScreemTagTree *tree );
static void screem_tag_tree_set_model( ScreemTagTree *tree ); 
static gpointer screem_tag_tree_create( gpointer data );
static void screem_tag_tree_add_file( ScreemTagTree *tree,
					ScreemTagFile *file,
					gboolean prepend );
static void screem_tag_tree_remove_file( ScreemTagTree *tree,
					ScreemTagFile *file );

static void screem_tag_file_active_set( ScreemTagFile *file,
					GParamSpec *spec,
					gpointer data );

static void screem_tag_tree_save_config( void );
static xmlNodePtr screem_tag_tree_get_file_node( const gchar *url );
static gboolean screem_tag_tree_is_active( const gchar *url,
					gchar **name,
					gchar **desc );
static void screem_tag_tree_search( EggAction *action,
				gpointer data );
static gboolean screem_tag_tree_search_func( GtkTreeModel *model,
					GtkTreePath *path,
					GtkTreeIter *iter,
					gpointer data );

static void screem_tag_tree_class_init( ScreemTagTreeClass *klass );
static void screem_tag_tree_init( ScreemTagTree *tag_tree );
static void screem_tag_tree_finalize( GObject *object );
static void screem_tag_tree_set_prop( GObject *object, guint property_id,
				    const GValue *value, GParamSpec *pspec );
static void screem_tag_tree_get_prop( GObject *object, guint property_id,
				      GValue *value, GParamSpec *pspec);
static void screem_tag_tree_size_request( GtkWidget *widget,
                           	  	  GtkRequisition *requisition );
static void screem_tag_tree_size_allocate( GtkWidget *widget,
                              		   GtkAllocation *allocation );

ScreemTagTree *screem_tag_tree_new()
{
	ScreemTagTree *tree;
	GType type;

	type = screem_tag_tree_get_type();
	
	tree = SCREEM_TAG_TREE( g_object_new( type, NULL ) );

	return tree;
}

gboolean screem_tag_tree_load_trees( void )
{
	gchar *dot;

	if( ! mutex ) {
gdk_threads_enter();
		mutex = g_mutex_new();
		filesstore = gtk_list_store_new( MAX_TAG_FILE_COLS,
						G_TYPE_STRING,
						G_TYPE_STRING,
						G_TYPE_STRING,
						G_TYPE_BOOLEAN,
						G_TYPE_POINTER );
		dot = screem_get_dot_dir();
		configpath = g_build_path( G_DIR_SEPARATOR_S,
				dot, "tagtreeconfig.xml",
				NULL );
		g_free( dot );
		config = xmlParseFile( configpath );
		if( ! config ) {
			config = xmlNewDoc( XML_DEFAULT_VERSION );
			xmlDocSetRootElement( config,
					xmlNewDocNode( config, NULL,
						"refconfig", NULL ) );
		}
		gdk_threads_leave();
	}
#if 0
	g_idle_add( screem_tag_tree_create, NULL );
#else
	g_thread_create( screem_tag_tree_create, NULL, FALSE, NULL );
#endif
	
	return FALSE;
}

GtkListStore *screem_tag_tree_get_file_model( void )
{
	return filesstore;
}

gboolean screem_tag_tree_add_uri( const gchar *uri,
				  ScreemTagFileFormat format )
{
	ScreemTagFile *tfile;
	gboolean active;
	gchar *name;
	gchar *desc;
	GtkTreeIter it;
	xmlNodePtr node;
	
	tfile = screem_tag_file_new();

	screem_tag_file_set_uri( tfile, uri );
	g_mutex_lock( mutex );
	active = screem_tag_tree_is_active( uri, &name, &desc );
	if( name ) {
		screem_tag_file_set_name( tfile, name );
	}
	if( desc ) {
		screem_tag_file_set_desc( tfile, desc );
	}

	treefiles = g_slist_prepend( treefiles, tfile );
	g_signal_connect( G_OBJECT( tfile ),
			"notify::active",
			G_CALLBACK( screem_tag_file_active_set ),
			NULL );
gdk_threads_enter();
	gtk_list_store_append( filesstore, &it );
	gtk_list_store_set( filesstore, &it,
			TAG_FILE_NAME_COL, name,
			TAG_FILE_URI_COL, uri,
			TAG_FILE_DESC_COL, desc,
			TAG_FILE_OBJ_COL, tfile,
			-1 );	
	g_mutex_unlock( mutex );
	g_free( name );
	g_free( desc );
	if( active ) {
		screem_tag_file_load( tfile, format );
		name = (gchar*)screem_tag_file_get_name( tfile );
		desc = (gchar*)screem_tag_file_get_desc( tfile );
		gtk_list_store_set( filesstore, &it,
			TAG_FILE_NAME_COL, name,
			TAG_FILE_DESC_COL, desc,
			-1 );
		g_object_set( G_OBJECT( tfile ),
				"active", TRUE, NULL );
		
		g_mutex_lock( mutex );	
		if( ! screem_tag_tree_get_file_node( uri ) ) {
			/* new node being added, add to config */
			node = xmlDocGetRootElement( config );
			node = xmlNewChild( node, NULL, "tagfile", NULL );
			xmlSetProp( node, "src", uri );
			if( name ) {
				xmlSetProp( node, "name", name );
			}
			if( desc ) {
				xmlSetProp( node, "description", desc );
			}
			xmlSetProp( node, "enable", "1" );
			screem_tag_tree_save_config();
		}
		g_mutex_unlock( mutex );
	}

	gdk_threads_leave();
	
	return TRUE;
}

void screem_tag_tree_remove_uri( const gchar *uri )
{
	GSList *tmp;
	ScreemTagFile *tfile;
	const gchar *furi;
	GtkTreeIter *it;
	gboolean active;
	xmlNodePtr node;
	
	g_mutex_lock( mutex );

	for( tmp = treefiles; tmp; tmp = tmp->next ) {
		tfile = SCREEM_TAGFILE( tmp->data );
		furi = screem_tag_file_get_uri( tfile );

		if( gnome_vfs_uris_match( uri, furi ) ) {
			break;
		}
	}
	
	if( tmp ) {
		it = screem_support_find_in_list( filesstore,
				TAG_FILE_URI_COL, furi );
		g_assert( it );

		gtk_list_store_remove( filesstore, it );
		gtk_tree_iter_free( it );

		treefiles = g_slist_remove( treefiles, tfile );

		g_object_get( G_OBJECT( tfile ),
				"active", &active, NULL );
		if( active ) {
			for( tmp = trees; tmp; tmp = tmp->next ) {
				screem_tag_tree_remove_file( tmp->data,
							tfile );
			}
		}
		g_object_unref( tfile );
	}
	if( ( node = screem_tag_tree_get_file_node( uri ) ) ) {
		xmlUnlinkNode( node );
		xmlFreeNode( node );
		screem_tag_tree_save_config();
	}
	g_mutex_unlock( mutex );
}

GSList *screem_tag_tree_autocomplete( const gchar *mime_type,
				const gchar *prefix )
{
	GSList *ret;
	GSList *tmp;
	GSList *tret;
	ScreemTagFile *tfile;
	const gchar *type;
	
	g_return_val_if_fail( mime_type != NULL, NULL );
	g_return_val_if_fail( prefix != NULL, NULL );

	ret = NULL;

	g_mutex_lock( mutex );
	
	for( tmp = treefiles; tmp; tmp = tmp->next ) {
		tfile = SCREEM_TAGFILE( tmp->data );

		type = screem_tag_file_for_type( tfile );
		if( type && ! strcmp( mime_type, type ) ) {

			tret = screem_tag_file_autocomplete( tfile,
					prefix );
			ret = g_slist_concat( ret, tret );
		}
	}
	
	g_mutex_unlock( mutex );
	
	return ret;
}

/* static stuff */

static const gchar *screem_tag_tree_dtd_cache( ScreemTagTree *tree,
					ScreemDTD *dtd )
{
	const gchar *ret;
	GSList *tmp;
	const GSList *elements;
	GString *str;
	gchar *data;

	ret = g_object_get_data( G_OBJECT( dtd ),
				"tagtree_string" );
	if( ! ret ) {
		elements = screem_dtd_get_elements( dtd );
	
		str = g_string_new( "<?xml version=\"1.0\"?>" );
		g_string_append_printf( str, "<ref name=\"%s\" description=\"%s\">",
					_( "Current Doctype" ),
				_( "Elements for the current doctype" ) );
		while( elements ) {
			const gchar *name;
			const gchar *desc;
			ScreemDTDTagExistance close;
			const GSList *attrs;
			const gchar *adesc;
			gchar *escaped;

			name = screem_dtd_element_get_name( (ScreemDTDElement*)elements->data );
			desc = screem_dtd_element_get_description( (ScreemDTDElement*)elements->data );		
			close = screem_dtd_element_close_state( dtd, name );
			attrs = screem_dtd_element_get_attrs( (ScreemDTDElement*)elements->data );
		
			g_string_append_printf(str, 
					"<entry name=\"%s\"><insert open=\"&lt;%s&gt;\"",
					name, name );
			if( close != SCREEM_DTD_MUST_NOT ) {
				g_string_append_printf( str, " close=\"&lt;/%s&gt;\"",
							name );
			}
			g_string_append( str, "/>");
			if( desc ) {
				escaped = g_markup_escape_text( desc, strlen( desc ) );
				g_string_append_printf( str, 
						"<description>%s</description>",
						escaped );
				g_free( escaped );
			}

			while( attrs ) {
				const ScreemDTDAttribute *attr;

				attr = (const ScreemDTDAttribute*)attrs->data;
				adesc = screem_dtd_attribute_get_description( attr );
				if( ! adesc ) {
					adesc = "";
				}
				escaped = g_markup_escape_text( adesc, strlen( adesc ) );
				g_string_append_printf( str,
						"<property name=\"%s\" required=\"%i\" vallist=\"0\" default=\"%s\">%s</property>",
						screem_dtd_attribute_get_name( attr ),
						screem_dtd_attribute_get_required( attr ),
						"",
						escaped
						);
				g_free( escaped );
				attrs = attrs->next;
			}
		
			g_string_append( str, "</entry>" );
	
			elements = elements->next;
		}

		g_string_append( str, "</ref>" );

		tmp = g_slist_append( tree->private->dtdcache,
					dtd );
		tree->private->dtdcache = tmp;
		ret = str->str;
		g_object_set_data( G_OBJECT( dtd ), "tagtree_string",
				   (gpointer)ret );
		
		g_string_free( str, FALSE );
		
		if( ( ++ tree->private->dtdcachelen ) > 4 ) {
			tmp = tree->private->dtdcache;
			tree->private->dtdcache = tmp->next;
			tmp->next = NULL;
			dtd = tmp->data;
			data = g_object_get_data( G_OBJECT( dtd ),
						  "tagtree_string" );
			g_free( data );
			g_object_set_data( G_OBJECT( dtd ), "tagtree_string",
					   NULL );
			g_slist_free( tmp );
			tree->private->dtdcachelen --;
		}
	}
	return ret;
}

static void screem_tag_tree_add_dtd( ScreemPage *page, GParamSpec *spec,
				     ScreemTagTree *tag_tree )
{
	ScreemDTD *dtd;
	const gchar *str;

	if( spec ) {
		tag_tree->private->dtd = screem_page_get_dtd( page );
	}

	dtd = tag_tree->private->dtd;
	str = screem_tag_tree_dtd_cache( tag_tree, dtd );

	screem_tag_file_set_from_string( tag_tree->private->dtdbranch,
					str, 0 );
}

static gboolean screem_tag_tree_click( GtkTreeView *tree_view,
				       GtkTreePath *path,
				       GtkTreeViewColumn *column,
				       gpointer data )
{
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreeSelection *sel;

	ScreemTagTree *tag_tree;

	tag_tree = SCREEM_TAG_TREE( data );

	sel = gtk_tree_view_get_selection( GTK_TREE_VIEW( tree_view ) );

	if( sel ) {
		NodeInfo *info;
		ScreemWindow *window;
		ScreemEditor *editor;

		window = SCREEM_WINDOW( tag_tree->private->window );

		editor = SCREEM_EDITOR( window->details->editor );

		gtk_tree_selection_get_selected( sel, &model, &iter );
		gtk_tree_model_get( model, &iter, FILE_BROWSER_DATA_COL,
					 &info, -1 );

		if( info && ( ! info->text ) && info->file ) {
			GnomeVFSURI *vuri;
			
			vuri = gnome_vfs_uri_new( info->file );
			if( ! vuri ) {
				screem_window_show_error( window, _( "Invalid URI for requested resource" ) );
			} else if( gnome_vfs_uri_is_local( vuri ) ||
				  ! window->details->offline ) {
				GString *data;
				
				data = load_file( info->file );
				/* now set info->text to be the file contents,
			   	this way we don't refetch files we already have */
				info->text = data->str;
				g_string_free( data, FALSE );
			} else {
				screem_window_show_message( window, _( "Remote resources are not available in offline mode" ) );	
			}

			if( vuri ) {	
				gnome_vfs_uri_unref( vuri );
			}
		} else if( info && info->open ) {
			screem_editor_insert_markup( editor, info->open,
						     info->close );
		}
		/* we do info->text here so you can have something like
		 * <insert open="&lt;title&gt;" close="&lt;title&gt;">
		 *      page title
		 * </insert>
		 */
		if( info && info->text ) {
			screem_editor_insert( editor, -1, info->text );
		}
	} else {
		/* hmm, no iterator */
		
	}

	return (sel != NULL);
}


static gboolean screem_tag_tree_press( GtkWidget *widget, 
				       GdkEventButton *event,
				       ScreemTagTree *tag_tree )
{
	
	return FALSE;
}

static void screem_tagtree_sel_change( GtkTreeSelection *selection,
					ScreemTagTree *tree )
{
	ScreemTagTreePrivate *priv;
	ScreemWindow *window;
	gint rows;
	GtkTreeModel *model;
	GtkTreeIter iter;
	NodeInfo *info;
	
	priv = tree->private;
	
	window = SCREEM_WINDOW( priv->window );
	rows = gtk_tree_selection_count_selected_rows( selection );
	if( rows > 0 ) {
		gtk_tree_selection_get_selected( selection, &model, &iter );
		gtk_tree_model_get( model, &iter, FILE_BROWSER_DATA_COL,
					 &info, -1 );
		if( info && info->description ) {
			screem_window_show_message( window, info->description );
		}
	}
	gtk_widget_set_sensitive( priv->toolbar, ( rows > 0 ) );
}

static void screem_tag_tree_set_model( ScreemTagTree *tree )
{
	ScreemTagTreePrivate *priv;
	
	g_return_if_fail( SCREEM_IS_TAG_TREE( tree ) );
	
	priv = tree->private;

	priv->umodel = egg_tree_model_union_new( FILE_BROWSER_MAX_COL,
					   G_TYPE_STRING,
					   GDK_TYPE_PIXBUF,
					   G_TYPE_POINTER,
					   G_TYPE_STRING,
					   G_TYPE_POINTER,
					   G_TYPE_STRING );

	priv->dtdbranch = screem_tag_file_new();

	screem_tag_tree_add_file( tree, priv->dtdbranch, TRUE );
	gtk_tree_view_set_model( GTK_TREE_VIEW( tree->private->view ), 
				 GTK_TREE_MODEL( priv->umodel ) );
}

static gint tagtree_filter( const GnomeVFSFileInfo *info )
{
	const gchar *mime_type;
	gchar *type;
	gint ret;
	
	mime_type = info->mime_type;

	type = screem_get_mime_type_override( info->name, mime_type ); 

	ret = -1;
	if( type ) {
		ret = strcmp( "application/x-screem-tag-tree", 
				type );
		g_free( type );
	}

	return ret;
}
static gboolean screem_tag_tree_add_screem_format( gpointer data )
{
	screem_tag_tree_add_uri( data, SCREEM_TAG_TREE_FORMAT );

	return FALSE;
}

static gpointer screem_tag_tree_create( gpointer data )
{
	GSList *files;
	GSList *tmp;
	
	gint i;
	const gchar *treepaths[] = {
		DATADIR"/screem/tagtrees",
		NULL,
		NULL
	};

	/* load tag trees */
	treepaths[ 1 ] = screem_get_dot_dir();

	for( i = 0; treepaths[ i ]; ++ i ) {
		files = screem_vfs_scandir( treepaths[ i ],
				tagtree_filter, NULL, FALSE );
		for( tmp = files; tmp; tmp = tmp->next ) {
			screem_tag_tree_add_screem_format( tmp->data );
			g_free( tmp->data );
		}
		g_slist_free( files );
	}
	g_free( (gchar*)treepaths[ 1 ] );

	return NULL;
}

static void add_widget( EggMenuMerge *merge, GtkWidget *widget, ScreemTagTree *tree )
{
	ScreemTagTreePrivate *priv;
	GtkWidget *box;
	
	priv = tree->private;
	box = priv->box;
	
	if( EGG_IS_TOOLBAR( widget ) ) {
		egg_toolbar_set_show_arrow( EGG_TOOLBAR( widget ), TRUE );
		egg_toolbar_set_tooltips( EGG_TOOLBAR( widget ), TRUE );
		
		gtk_box_pack_start( GTK_BOX( box ), widget,
				    FALSE, TRUE, 0 );
		gtk_widget_set_sensitive( widget, FALSE );
		priv->toolbar = widget;
	}
}

static void screem_tag_tree_tag_help( EggAction *action, ScreemTagTree *tree )
{
	ScreemTagTreePrivate *priv;
	GtkTreeView *view;
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter it;
	gint rows;
	NodeInfo *info;
	
	priv = tree->private;

	view = GTK_TREE_VIEW( priv->view );
	selection = gtk_tree_view_get_selection( view );
	
	rows = gtk_tree_selection_count_selected_rows( selection );
	if( rows > 0 ) {
		gtk_tree_selection_get_selected( selection, &model, &it );
		gtk_tree_model_get( model, &it, FILE_BROWSER_DATA_COL,
					 &info, -1 );
		if( info ) {
			GtkWidget *dialog;
			GtkWidget *box;
			GtkWidget *widget;
			GtkWidget *notebook;
			gchar *title;
			ScreemWindow *window;
			ScreemEditor *editor;
			GtkWidget *sw;
			GtkTextBuffer *buffer;
			gchar *txt;
			GtkSizeGroup *group;
			GSList *attrs;
			NodeAttribute *attr;
			GtkWidget *attrbox;
			gchar *attrtxt;
			
			g_object_get( G_OBJECT( tree ), "window", &window, NULL );
			editor = SCREEM_EDITOR( window->details->editor );
			
			box = gtk_vbox_new( FALSE, 0 );
			gtk_box_set_spacing( GTK_BOX( box ), 6 );
			gtk_container_set_border_width( GTK_CONTAINER( box ), 12 );
		
			if( ! info->isfunc ) {
				title = g_strconcat( "<span size=\"xx-large\">",
							info->name,
							"</span>",
							NULL );
			} else {
				title = g_strconcat( "<span size=\"xx-large\">",
							info->name,
							"()</span>",
							NULL );
			}
			widget = gtk_label_new( title );
			gtk_label_set_use_markup( GTK_LABEL( widget ), TRUE );
			g_free( title );
			gtk_label_set_justify( GTK_LABEL( widget ), GTK_JUSTIFY_LEFT );
			gtk_misc_set_alignment( GTK_MISC( widget ), 0.0, 0.0 );
			gtk_box_pack_start( GTK_BOX( box ), widget, FALSE, TRUE, 0 );
	

			widget = gtk_label_new( info->description );
			gtk_label_set_justify( GTK_LABEL( widget ), GTK_JUSTIFY_LEFT );
			gtk_misc_set_alignment( GTK_MISC( widget ), 0.0, 0.0 );
			gtk_label_set_line_wrap( GTK_LABEL( widget ), TRUE );
			gtk_box_pack_start( GTK_BOX( box ), widget, FALSE, TRUE, 0 );

			txt = NULL;
			if( ( ! info->text ) && info->file ) {
				GnomeVFSURI *vuri;
			
				vuri = gnome_vfs_uri_new( info->file );
				if( ! vuri ) {
					txt = g_strdup( _( "Invalid URI for requested resource" ) );
				} else if( gnome_vfs_uri_is_local( vuri ) ||
					  ! window->details->offline ) {
					GString *data;
				
					data = load_file( info->file );
					/* now set info->text to be the file contents,
				   	this way we don't refetch files we already have */
					info->text = data->str;
					g_string_free( data, FALSE );
				} else {
					txt = g_strdup( _( "Remote resources are not available in offline mode" ) );	
				}

				if( vuri ) {	
					gnome_vfs_uri_unref( vuri );
				}
			} 
			if( info->open ) {
				txt = g_strconcat( info->open, info->close, NULL );
			}
			if( info->text ) {
				g_free( txt );
				txt = g_strdup( info->text );
			}
			
			notebook = gtk_notebook_new();
			gtk_notebook_set_show_border( GTK_NOTEBOOK( notebook ),
							FALSE );
			gtk_notebook_set_tab_pos( GTK_NOTEBOOK( notebook ),
						GTK_POS_BOTTOM );
			gtk_box_pack_start( GTK_BOX( box ), notebook, TRUE, TRUE, 0 );
	

			if( info->attributes ) {
				sw = gtk_scrolled_window_new( NULL, NULL );
				gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
								GTK_POLICY_NEVER,
								GTK_POLICY_AUTOMATIC );
				gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
								GTK_SHADOW_IN );

				group = gtk_size_group_new( GTK_SIZE_GROUP_HORIZONTAL );

				attrbox = gtk_vbox_new( FALSE, 0 );

				for( attrs = info->attributes; attrs; attrs = attrs->next ) {
					attr = (NodeAttribute*)attrs->data;

					if( attr->type ) {
						attrtxt = g_strconcat( "<b><i>", attr->name, "(",
									attr->type,
									") - </i></b>",
									attr->description,
									NULL );
					} else {
						attrtxt = g_strconcat( "<b><i>", attr->name,
									" - </i></b>",
									attr->description,
									NULL );
					}
					widget = gtk_label_new( attrtxt );
					gtk_label_set_use_markup( GTK_LABEL( widget ), TRUE );
					g_free( attrtxt );
					gtk_size_group_add_widget( group, widget );
				
					gtk_label_set_justify( GTK_LABEL( widget ), GTK_JUSTIFY_LEFT );
					gtk_misc_set_alignment( GTK_MISC( widget ), 0.0, 0.0 );

					gtk_box_pack_start( GTK_BOX( attrbox ), widget, FALSE, TRUE, 0 );
				}
				gtk_container_set_border_width( GTK_CONTAINER( attrbox ), 6 );
				gtk_box_set_spacing( GTK_BOX( attrbox ), 6 );
				gtk_scrolled_window_add_with_viewport( GTK_SCROLLED_WINDOW( sw ),
								attrbox );
				
				if( info->isfunc ) {
					widget = gtk_label_new( _( "Parameters" ) );
				} else {
					widget = gtk_label_new( _( "Attributes" ) );
				}
				
				gtk_notebook_append_page( GTK_NOTEBOOK( notebook ),
							  sw, widget );
			}
			
			if( txt ) {
				sw = gtk_scrolled_window_new( NULL, NULL );
				gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
								GTK_POLICY_AUTOMATIC,
								GTK_POLICY_AUTOMATIC );
				gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
								GTK_SHADOW_IN );
				widget = gtk_source_view_new();
				gtk_text_view_set_editable( GTK_TEXT_VIEW( widget ), FALSE );
				gtk_text_view_set_wrap_mode( GTK_TEXT_VIEW( widget ), GTK_WRAP_WORD );
				buffer = gtk_text_view_get_buffer( GTK_TEXT_VIEW( widget ) );
				gtk_text_buffer_set_text( buffer, txt, -1 );
				gtk_container_add( GTK_CONTAINER( sw ), widget );

				widget = gtk_label_new( _( "Text to Insert" ) );
				gtk_notebook_append_page( GTK_NOTEBOOK( notebook ),
							 sw, widget );
				dialog = gtk_dialog_new_with_buttons( _( "Tag Help - Screem" ),
								GTK_WINDOW( window ),
								GTK_DIALOG_MODAL | 
								GTK_DIALOG_NO_SEPARATOR,
								GTK_STOCK_CLOSE,
								GTK_RESPONSE_CLOSE,
								GTK_STOCK_ADD,
								GTK_RESPONSE_APPLY,
								NULL);
			} else {
				dialog = gtk_dialog_new_with_buttons( _( "Tag Help - Screem" ),
								GTK_WINDOW( window ),
								GTK_DIALOG_MODAL | 
								GTK_DIALOG_NO_SEPARATOR,
								GTK_STOCK_CLOSE,
								GTK_RESPONSE_CLOSE,
								NULL );
			}
			gtk_widget_show_all( box );
			
			gtk_box_pack_start( GTK_BOX( GTK_DIALOG( dialog )->vbox ), box, TRUE, TRUE, 0 );

			if( gtk_dialog_run( GTK_DIALOG( dialog ) ) == GTK_RESPONSE_APPLY ) {
				/* insert txt into the editor */
				if( info->open ) {
					screem_editor_insert_markup( editor, info->open, info->close );
				} else {
					screem_editor_insert( editor, -1, txt );
				}
			}
			g_free( txt );
			
			
			gtk_widget_destroy( dialog );
		}
	} else {
		g_warning( "no item selected for info\n" );
	}
}

static gboolean screem_tag_tree_page_set( gpointer data )
{
	ScreemTagTree *tree;
	ScreemPage *page;
	ScreemDTD *dtd;
	
	tree = SCREEM_TAG_TREE( data );
	page = tree->private->page;

gdk_threads_enter();
	
	if( page ) {
		g_signal_connect(G_OBJECT( page ), "notify::dtd",
				 G_CALLBACK( screem_tag_tree_add_dtd ),
				 tree );
		dtd = screem_page_get_dtd( page );
		if( dtd != tree->private->dtd ) {
			tree->private->dtd = dtd;
			screem_tag_tree_add_dtd( page, NULL, tree );
		}
	} else {
		tree->private->dtd = NULL;
	}

	tree->private->timeout = 0;
	
	gdk_threads_leave();
	
	return FALSE;
}

static void screem_tag_tree_add_file( ScreemTagTree *tree,
					ScreemTagFile *file,
					gboolean prepend )
{
	GtkTreeModel *model;
	
	g_return_if_fail( SCREEM_IS_TAG_TREE( tree ) );
	g_return_if_fail( SCREEM_IS_TAGFILE( file ) );

	model = screem_tag_file_get_model( file );
	
	if( prepend ) {
		egg_tree_model_union_prepend( EGG_TREE_MODEL_UNION( tree->private->umodel ), model );
	} else {
		egg_tree_model_union_append( EGG_TREE_MODEL_UNION( tree->private->umodel ), model );

	}
}

static void screem_tag_tree_remove_file( ScreemTagTree *tree,
					ScreemTagFile *file )
{
	GtkTreeModel *model;
	
	g_return_if_fail( SCREEM_IS_TAG_TREE( tree ) );
	g_return_if_fail( SCREEM_IS_TAGFILE( file ) );

	model = screem_tag_file_get_model( file );
	
	egg_tree_model_union_remove( EGG_TREE_MODEL_UNION( tree->private->umodel ), model );

}

static void screem_tag_file_active_set( ScreemTagFile *file,
					GParamSpec *spec,
					gpointer data )
{
	gboolean active;
	const gchar *uri;
	GtkTreeIter *it;
	GSList *tmp;
	xmlNodePtr node;
	
	g_mutex_lock( mutex );

	uri = screem_tag_file_get_uri( file );
	g_object_get( G_OBJECT( file ), 
			"active", &active, 
			NULL );
	
	it = screem_support_find_in_list( filesstore, 
			TAG_FILE_URI_COL, uri );
	g_assert( it );

	gtk_list_store_set( filesstore, it,
			TAG_FILE_ACTIVE_COL, active, -1 );

	gtk_tree_iter_free( it );

	if( active ) {
		for( tmp = trees; tmp; tmp = tmp->next ) {
			screem_tag_tree_add_file( tmp->data,
						file, FALSE );
		}
	} else {
		for( tmp = trees; tmp; tmp = tmp->next ) {
			screem_tag_tree_remove_file( tmp->data,
							file );
		}
	}

	if( ( node = screem_tag_tree_get_file_node( uri ) ) ) {
		xmlSetProp( node, "enable", active ? "1" : "0" );
	}

	screem_tag_tree_save_config();
	
	g_mutex_unlock( mutex );
}

/* NOTE: do not call without the mutex locked */
static void screem_tag_tree_save_config( void )
{
	xmlNodePtr node;
	GSList *tmp;
	const gchar *uri;
	const gchar *name;
	const gchar *desc;
	
	for( tmp = treefiles; tmp; tmp = tmp->next ) {
		uri = screem_tag_file_get_uri( SCREEM_TAGFILE( tmp->data ) );
		node = screem_tag_tree_get_file_node( uri );

		if( node ) {
			name = screem_tag_file_get_name( SCREEM_TAGFILE( tmp->data ) );
			desc = screem_tag_file_get_desc( SCREEM_TAGFILE( tmp->data ) );
			xmlSetProp( node, "name", name );
			xmlSetProp( node, "description", desc );
		} else {
			g_warning( "ScreemTagFile %p not in config",
					tmp->data );
		}
	}
	
	xmlSaveFile( configpath, config );
}

/* NOTE: do not call without the mutex locked */
static xmlNodePtr screem_tag_tree_get_file_node( const gchar *url )
{
	xmlNodePtr ret;
	xmlChar *src;

	ret = NULL;
	
	ret = xmlDocGetRootElement( config );

	g_return_val_if_fail( ret != NULL, NULL );
	
	ret = ret->children;
	while( ret ) {
		src = xmlGetProp( ret, "src" );
		if( src ) {
			if( ! strcmp( src, url ) ) {
				xmlFree( src );
				break;
			}
			xmlFree( src );
		}
		ret = ret->next;
	}
	
	return ret;
}
/* NOTE: do not call without the mutex locked */
static gboolean screem_tag_tree_is_active( const gchar *url,
					gchar **name,
					gchar **desc )
{
	gboolean ret;
	xmlNodePtr node;
	xmlChar *enable;
	
	g_assert( name != NULL );
	g_assert( desc != NULL );
	
	ret = TRUE;

	*name = NULL;
	*desc = NULL;

	node = screem_tag_tree_get_file_node( url );
	if( node ) {
		enable = xmlGetProp( node, "enable" );
		if( enable ) {
			ret = ( *enable == '1' );
			xmlFree( enable );
		}
		*name = xmlGetProp( node, "name" );
		*desc = xmlGetProp( node, "description" );
	} 	

	return ret;
}

static void screem_tag_tree_search( EggAction *action,
				gpointer data )
{
	ScreemTagTree *tree;
	ScreemTagTreePrivate *priv;
	GConfClient *client;
	gchar *glade_path;
	GtkWidget *widget;
	GtkTreeModel *model;
	GladeXML *xml;
	GtkCellRenderer *rend;
	GtkCellRenderer *prend;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;
	GtkTreeIter it;
	GtkTreePath *path;
	gint *indices;
	gint i;
	NodeInfo *info;
	
	tree = SCREEM_TAG_TREE( data );
	priv = tree->private;
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( priv->view ) );
	i = -1;
	if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
		path = gtk_tree_model_get_path( model, &it );	
		indices = gtk_tree_path_get_indices( path );
		i = indices[ 0 ];
		gtk_tree_path_free( path );
	}
	
	client = gconf_client_get_default();
	glade_path = gconf_client_get_string( client,
					      "/apps/screem/general/glade_path",
					      NULL );
	xml = glade_xml_new( glade_path, "find_tagtree", NULL );
	g_free( glade_path );
	g_object_unref( client );
	
	widget = glade_xml_get_widget( xml, "tagtrees" );
	model = screem_top_level_model_new();
	screem_top_level_model_set_model( SCREEM_TOP_LEVEL_MODEL( model ), priv->umodel );
	
	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( widget ) );

	prend = gtk_cell_renderer_pixbuf_new();
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	
	gtk_tree_view_column_pack_start( col, prend, FALSE );
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	
	gtk_tree_view_column_set_title( col, _( "Resources" ) );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 
					     FILE_BROWSER_NAME_COL,
					     NULL );
	gtk_tree_view_column_set_attributes( col, prend,
					     "pixbuf",
					     FILE_BROWSER_ICON_COL,
					     NULL );

	/* HIG says 1 column lists should have no header */
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( widget ), 
					FALSE ); 

	
	gtk_tree_view_set_model( GTK_TREE_VIEW( widget ), model );
	g_object_unref( model );

	widget = glade_xml_get_widget( xml, "find_tagtree" );
	gtk_widget_show_all( widget );
	
	if( i != -1 ) {
		path = gtk_tree_path_new_from_indices( i, -1 );
		gtk_tree_selection_select_path( selection, path );
		gtk_tree_path_free( path );
	}
	
	if( gtk_dialog_run( GTK_DIALOG( widget ) ) == GTK_RESPONSE_OK ) {
		widget = glade_xml_get_widget( xml, "pattern" );
		priv->search_pattern = gtk_entry_get_text( GTK_ENTRY( widget ) );
		widget = glade_xml_get_widget( xml, "regexp" );
		priv->regexp = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
		if( gtk_tree_selection_get_selected( selection, &model, &it ) ) {
			path = gtk_tree_model_get_path( model, &it );
			gtk_tree_model_get( model, &it,
					FILE_BROWSER_DATA_COL, &info, -1 );
			if( info ) {
				model = screem_tag_file_get_model( SCREEM_TAGFILE( info->tfile ) );
				gtk_tree_model_foreach( model, 
					screem_tag_tree_search_func,
					tree );
			}

			if( priv->search_result ) {
				/* modify the path based on its position
				 * in the union model */
				gtk_tree_path_get_indices( priv->search_result )[ 0 ] = gtk_tree_path_get_indices( path )[ 0 ];

				selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( priv->view ) );
				gtk_tree_view_expand_to_path( GTK_TREE_VIEW( priv->view ), priv->search_result );	
				gtk_tree_selection_select_path( selection, priv->search_result );
				col = gtk_tree_view_get_column( GTK_TREE_VIEW( priv->view ), 0 );
				gtk_tree_view_scroll_to_cell( GTK_TREE_VIEW( priv->view ), priv->search_result, col, TRUE, 0.0, 0.5 );
				gtk_tree_path_free( priv->search_result );
				priv->search_result = NULL;
			} else {
				/* not found */
			}

			gtk_tree_path_free( path );
		}
	}
	
	widget = glade_xml_get_widget( xml, "find_tagtree" );
	gtk_widget_destroy( widget );	
	g_object_unref( G_OBJECT( xml ) );
}

static gboolean screem_tag_tree_search_func( GtkTreeModel *model,
					GtkTreePath *path,
					GtkTreeIter *iter,
					gpointer data )
{
	ScreemTagTree *tree;
	ScreemTagTreePrivate *priv;
	gchar *name;
	gint len;

	tree = SCREEM_TAG_TREE( data );
	priv = tree->private;

	gtk_tree_model_get( model, iter,
				FILE_BROWSER_NAME_COL, &name,
				-1 );

	priv->search_result = NULL;

	if( ! screem_search_static( name, priv->search_pattern,
				priv->regexp, 0, &len ) ) {
		priv->search_result = gtk_tree_path_copy( path );
	}

	g_free( name );
	
	return ( priv->search_result != NULL );
}

/* G Object stuff */
#define PARENT_TYPE GTK_TYPE_BIN

static gpointer parent_class;

static void screem_tag_tree_class_init( ScreemTagTreeClass *klass )
{
	GObjectClass *object_class;
	GtkWidgetClass *widget_class;

	GParamSpec *pspec;

	object_class = G_OBJECT_CLASS( klass );
	widget_class = (GtkWidgetClass *)klass;

	object_class->finalize = screem_tag_tree_finalize;
	parent_class = g_type_class_peek_parent( klass );

	object_class->get_property = screem_tag_tree_get_prop;
	object_class->set_property = screem_tag_tree_set_prop;

	pspec = g_param_spec_pointer( "window", "window",
				      "window object",
				      G_PARAM_READABLE |
				      G_PARAM_WRITABLE );
	g_object_class_install_property( object_class,
					 ARG_WINDOW, pspec );

	pspec = g_param_spec_object( "page", "page",
				     "page",
				     SCREEM_TYPE_PAGE,
				     G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( object_class ),
					 ARG_PAGE,
					 pspec );

	
	widget_class->size_request = screem_tag_tree_size_request;
	widget_class->size_allocate = screem_tag_tree_size_allocate;
}

static void screem_tag_tree_init( ScreemTagTree *tag_tree )
{
	GtkWidget *sw;
	GtkCellRenderer *rend;
	GtkCellRenderer *prend;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;
	GSList *tmp;
	gchar *dot;

	static EggActionGroupEntry action_entries[] = {
		{
		   "SearchTagTree", N_( "Find in Tag Tree..." ),
		   GTK_STOCK_FIND, NULL, N_( "Search the tag tree" ),
		   G_CALLBACK( screem_tag_tree_search ), NULL
		},
		{ 
		   "Tag Help", N_( "Tag Help" ),
		   GTK_STOCK_DIALOG_INFO, NULL, N_( "Help" ),
	  	   G_CALLBACK( screem_tag_tree_tag_help ), NULL 
		},
	};
	static guint action_n_entries=G_N_ELEMENTS( action_entries );
	EggActionGroup *group;
	EggMenuMerge *merge;
	gint i;
	
	ScreemTagTreePrivate *priv;

	if( ! mutex ) {
		mutex = g_mutex_new();
		filesstore = gtk_list_store_new( MAX_TAG_FILE_COLS,
						G_TYPE_STRING,
						G_TYPE_STRING,
						G_TYPE_STRING,
						G_TYPE_BOOLEAN,
						G_TYPE_POINTER );
		dot = screem_get_dot_dir();
		configpath = g_build_path( G_DIR_SEPARATOR_S,
				dot, "tagtreeconfig.xml",
				NULL );
		g_free( dot );
		config = xmlParseFile( configpath );
		if( ! config ) {
			config = xmlNewDoc( XML_DEFAULT_VERSION );
			xmlDocSetRootElement( config,
					xmlNewDocNode( config, NULL,
						"refconfig", NULL ) );
		}
	}
	
	priv = tag_tree->private = g_new0( ScreemTagTreePrivate, 1 );

	priv->box = gtk_vbox_new( FALSE, 0 );
	gtk_container_add( GTK_CONTAINER( tag_tree ), priv->box );
		
	sw = gtk_scrolled_window_new( NULL, NULL );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( sw ),
					GTK_POLICY_AUTOMATIC,
                                        GTK_POLICY_AUTOMATIC );
	gtk_scrolled_window_set_shadow_type( GTK_SCROLLED_WINDOW( sw ),
						GTK_SHADOW_IN );
	priv->view = gtk_tree_view_new();
	
        gtk_tree_view_set_rules_hint( GTK_TREE_VIEW( priv->view ), TRUE );

	selection = gtk_tree_view_get_selection( GTK_TREE_VIEW( priv->view ) );
	g_signal_connect( G_OBJECT( selection ), "changed",
			  G_CALLBACK( screem_tagtree_sel_change ), tag_tree );

	prend = gtk_cell_renderer_pixbuf_new();
	rend = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new();
	
	gtk_tree_view_column_pack_start( col, prend, FALSE );
	gtk_tree_view_column_pack_start( col, rend, TRUE );
	gtk_tree_view_column_set_resizable( col, TRUE );
	gtk_tree_view_append_column( GTK_TREE_VIEW( priv->view ), col );
	
	gtk_tree_view_column_set_title( col, _( "Resources" ) );
	gtk_tree_view_column_set_attributes( col, rend,
					     "text", 
					     FILE_BROWSER_NAME_COL,
					     NULL );
	gtk_tree_view_column_set_attributes( col, prend,
					     "pixbuf",
					     FILE_BROWSER_ICON_COL,
					     NULL );

	/* HIG says 1 column lists should have no header */
	gtk_tree_view_set_headers_visible( GTK_TREE_VIEW( priv->view ), FALSE ); 

	gtk_widget_show( priv->view );

	g_signal_connect( G_OBJECT( priv->view ), "row_activated",
			  G_CALLBACK( screem_tag_tree_click ),
			  tag_tree );

	g_signal_connect( G_OBJECT( priv->view ), "button_press_event",
			  G_CALLBACK( screem_tag_tree_press ),
			  tag_tree );

	gtk_container_add( GTK_CONTAINER( sw ), priv->view );

	gtk_box_pack_start( GTK_BOX( priv->box ), sw, TRUE, TRUE, 0 );


	group = egg_action_group_new( "ScreemTagTreeActions" );
	/* set user data field in action table */
	for( i = 0; i < action_n_entries; ++ i ) {
		action_entries[ i ].user_data = tag_tree;
	}
	egg_action_group_add_actions( EGG_ACTION_GROUP( group ),
					action_entries,
					action_n_entries );
	merge = egg_menu_merge_new ();
	egg_menu_merge_insert_action_group( merge, 
					    EGG_ACTION_GROUP( group ), 
					    0 );
	g_signal_connect( merge, "add_widget", G_CALLBACK( add_widget ), tag_tree );
  	egg_menu_merge_add_ui_from_file( merge, 
					 UIDATADIR"/screem-tag-tree-bar.xml", 
					 NULL );
	
	gtk_widget_show_all( GTK_WIDGET( tag_tree ) );

	screem_tag_tree_set_model( tag_tree );

	g_mutex_lock( mutex );
	trees = g_slist_prepend( trees, tag_tree );

	for( tmp = treefiles; tmp; tmp = tmp->next ) {
		screem_tag_tree_add_file( tag_tree, 
				SCREEM_TAGFILE( tmp->data ), FALSE );
	}
	g_mutex_unlock( mutex );
}

static void screem_tag_tree_finalize( GObject *object )
{
	ScreemTagTree *tree;
	ScreemTagTreePrivate *priv;
	gboolean last;
	GSList *tmp;
	gboolean active;
	
	tree = SCREEM_TAG_TREE( object );
	priv = tree->private;

	if( priv->timeout != 0 ) {
		g_source_remove( priv->timeout );
	}

	/* clear and unref union model */
	g_mutex_lock( mutex );
	
	for( tmp = treefiles; tmp; tmp = tmp->next ) {
		g_object_get( G_OBJECT( tmp->data ),
				"active", &active, NULL );
		if( active ) {
			screem_tag_tree_remove_file( tree,
					SCREEM_TAGFILE( tmp->data ) );
		}
	}
	screem_tag_tree_remove_file( tree, priv->dtdbranch );
	
	g_object_unref( priv->umodel );

	/* destroy dtd branch ScreemTagFile */
	g_object_unref( priv->dtdbranch );

	g_free( priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );

	trees = g_slist_remove( trees, tree );
	last = ( trees == NULL );
	if( last ) {
		gtk_list_store_clear( filesstore );
		g_object_unref( filesstore );
		for( tmp = treefiles; tmp; tmp = tmp->next ) {
			g_object_unref( G_OBJECT( tmp->data ) );
		}
	}
	
	g_mutex_unlock( mutex );

	if( last ) {
		g_mutex_free( mutex );
		g_free( configpath );
		xmlFreeDoc( config );
		mutex = NULL;
	}
}

static void screem_tag_tree_set_prop( GObject *object, guint property_id,
				    const GValue *value, GParamSpec *pspec )
{
	ScreemTagTree *tree;
	ScreemPage *page;
	
	tree = SCREEM_TAG_TREE( object );

	switch( property_id ) {
	case ARG_WINDOW:
		tree->private->window = g_value_get_pointer( value );
		break;
	case ARG_PAGE:
		page = tree->private->page;
		if( page ) {
			g_signal_handlers_disconnect_matched( G_OBJECT( page ),
							      G_SIGNAL_MATCH_DATA,
							      0, 0, NULL, NULL, tree );
		}
		page = g_value_get_object( value );

		tree->private->page = page;
		if( tree->private->timeout ) {
			g_source_remove( tree->private->timeout );
			tree->private->timeout = 0;
		}
		tree->private->timeout = g_timeout_add( 2000, 
				(GSourceFunc)screem_tag_tree_page_set, 
				tree );
		break;
	}
}

static void screem_tag_tree_get_prop( GObject *object, guint property_id,
				    GValue *value, GParamSpec *pspec)
{
	switch( property_id ) {
	case ARG_WINDOW:
		g_value_set_pointer( value,
				     (gpointer)
				     SCREEM_TAG_TREE( object )->private->window );
		break;
	}
}

static void screem_tag_tree_size_request( GtkWidget *widget,
                           	  	  GtkRequisition *requisition )
{
	GtkBin *bin;

	bin = GTK_BIN (widget);

	requisition->width = GTK_CONTAINER( widget )->border_width * 2;
	requisition->height = GTK_CONTAINER( widget )->border_width * 2;

	if( bin->child && GTK_WIDGET_VISIBLE( bin->child ) ) {
		GtkRequisition child_requisition;

		gtk_widget_size_request( bin->child, &child_requisition );
		requisition->width += child_requisition.width;
		requisition->height += child_requisition.height;
	}
}

static void screem_tag_tree_size_allocate( GtkWidget *widget,
                              		   GtkAllocation *allocation )
{
	GtkBin *bin;
	GtkAllocation child_allocation;

	bin = GTK_BIN( widget );
	widget->allocation = *allocation;
	
	if( bin->child ) {
	child_allocation.x = allocation->x + 
				GTK_CONTAINER( widget )->border_width; 
	child_allocation.y = allocation->y + 
				GTK_CONTAINER (widget)->border_width;
	child_allocation.width = MAX( allocation->width - 
				      GTK_CONTAINER( widget )->border_width * 2,
					0);
	child_allocation.height = MAX( allocation->height - 
				       GTK_CONTAINER (widget)->border_width * 2,
					0);
	gtk_widget_size_allocate( bin->child, &child_allocation );
	}
}

GType screem_tag_tree_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemTagTreeClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_tag_tree_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemTagTree ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_tag_tree_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemTagTree",
					       &info, 0 );
	}

	return type;
}
