/*  Screem:  css-wizard.c
 *
 *  A Wizard for building up CSS selector patterns
 * 
 *  (C) Copyright 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <config.h>

#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-entry.h>
#include <libgnomeui/gnome-file-entry.h>

#include <gtk/gtk.h>

#include <glade/glade.h>

#include <string.h>

#include "screem-plugin.h"

#include "screem-window.h"
#include "screem-editor.h"

#include "support.h"

/* per wizard struct */
typedef struct {
	ScreemWindow *window;
	ScreemEditor *editor;

} CSSWizard;


/* keep track of how many wizards we have */
static GList *wizards = NULL;


enum {
	ANYWHERE = 0,
	SOMEWHERE,
	CHILD_OF,
	AFTER
};

/* Required Plugin parts */

/* we don't want to try and load old plugins
   so this is a version symbol for screem to spot */
int screem_plugin_version = 4;

#define NAME "CSS Selector Wizard"
#define AUTHOR "David A Knight (david@screem.org)"
#define PLUGIN_VERSION "2.0"
#define TAG NULL


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module );
G_MODULE_EXPORT void g_module_unload( GModule *module );
G_MODULE_EXPORT void init( ScreemPlugin *plugin );
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view );
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view );

void css_selector_wizard_display( GtkAction *action, gpointer user_data );
void css_selector_tag_change( GtkWidget *widget );


G_MODULE_EXPORT const gchar* g_module_check_init( GModule *module )
{
	return NULL;
}

G_MODULE_EXPORT void g_module_unload( GModule *module )
{
}

G_MODULE_EXPORT void init( ScreemPlugin *plugin )
{
	plugin->name = NAME;
	plugin->author = AUTHOR;
	plugin->version = PLUGIN_VERSION;

	/* these can all be left out if not required */
	plugin->tag = TAG;

	g_module_symbol( plugin->module, "popup", 
			 (gpointer*)&plugin->popup );
	g_module_symbol( plugin->module, "add_ui", 
			 (gpointer*)&plugin->add_ui );
	g_module_symbol( plugin->module, "remove_ui", 
			 (gpointer*)&plugin->remove_ui );
}

/* not needed if nothing is being added to the UI (unusual) */
G_MODULE_EXPORT void add_ui( GtkWidget *window, GtkWidget *editor,
			     GtkWidget *preview, GtkWidget *link_view )
{
	CSSWizard *wizard;

	const gchar *ui = "\
<ui>\
<menubar>\
<menu action=\"Insert\">\
<menu action=\"Wizards\">\
<menuitem action=\"CSSSelectorWizard\"/>\
</menu>\
</menu>\
</menubar>\
<toolbar name=\"Wizards Toolbar\">\
<toolitem action=\"CSSSelectorWizard\" />\
</toolbar>\
</ui>";
	GtkAction *action;
	
	gchar *label;
	gchar *tip;
	GError *error;
	
	wizard = g_new0( CSSWizard, 1 );
	wizard->window = SCREEM_WINDOW( window );
	wizard->editor = SCREEM_EDITOR( editor );
	
	label = g_strdup( _( "CSS Selector" ) );
	tip = g_strdup( _( "A Wizard to help you construct selectors for applying css properties to " ) );
	action = gtk_action_new( "CSSSelectorWizard", label, tip,
			GTK_STOCK_EXECUTE );
	g_signal_connect( G_OBJECT( action ), "activate",
			G_CALLBACK( css_selector_wizard_display ), wizard );
	gtk_action_group_add_action( GTK_ACTION_GROUP( wizard->window->action_group ), action );
	g_free( label );
	g_free( tip );
	
	error = NULL;
	if( ! gtk_ui_manager_add_ui_from_string( GTK_UI_MANAGER( wizard->window->merge ),
						ui, strlen( ui ), &error ) ) {
		g_message( "%s ui error = %s", "css wizard",
				error->message );
		g_error_free( error );
	}

	wizards = g_list_append( wizards, wizard );
}

/* required it add_ui is present */
G_MODULE_EXPORT void remove_ui( GtkWidget *window, GtkWidget *editor,
				GtkWidget *preview, GtkWidget *link_view )
{
	GList *list;
	CSSWizard *wizard;

	for( list = wizards; list; list = list->next ) {
		wizard = (CSSWizard*)list->data;
		if( wizard->window == SCREEM_WINDOW( window ) ) {
			/* got it */
			break;
		}
	}
       
	g_return_if_fail( list != NULL );

	/* wizard is the one to erase */
	wizards = g_list_remove( wizards, wizard );
	g_free( wizard );
}

/* End of required section */

static void create_tag_menu( CSSWizard *wizard, GladeXML *xml )
{
	GtkWidget *widget;
	GtkWidget *entry;
	GtkListStore *store;
	GtkTreeIter it;
	ScreemPage *page;
	ScreemDTD *dtd;
	const GSList *elements;
	const ScreemDTDElement *elem;
	const gchar *name;
	gboolean set;
	
	GtkEntryCompletion *comp;
	
	widget = glade_xml_get_widget( xml, "tag_menu" );
	entry = GTK_BIN( widget )->child;
	
	page = screem_window_get_document( wizard->window );
	if( page ) {
		comp = gtk_entry_completion_new();
		store = gtk_list_store_new( 1, G_TYPE_STRING );

		dtd = screem_page_get_dtd( page );
		elements = screem_dtd_get_elements( dtd );
		name = "";
		set = FALSE;
		if( elements ) {
			set = TRUE;
			elem = (const ScreemDTDElement*)elements->data;
			name = screem_dtd_element_get_name( elem );
		}
		gtk_entry_set_text( GTK_ENTRY( entry ), name );
		while( elements ) {	
			elem = (const ScreemDTDElement*)elements->data;
			name = screem_dtd_element_get_name( elem );
			gtk_list_store_append( store, &it );
			gtk_list_store_set( store, &it, 0, name, -1 );

			elements = elements->next;
		}
		g_signal_connect_swapped( G_OBJECT( entry ),
				"changed",
				G_CALLBACK( css_selector_tag_change ),
				widget );
		gtk_entry_completion_set_model( GTK_ENTRY_COMPLETION( comp ), GTK_TREE_MODEL( store ) );
		gtk_entry_completion_set_text_column( GTK_ENTRY_COMPLETION( comp ), 0 );
		gtk_entry_set_completion( GTK_ENTRY( entry ), comp );
		gtk_combo_box_set_model( GTK_COMBO_BOX( widget ), 
				GTK_TREE_MODEL( store ) );
		gtk_combo_box_entry_set_text_column( GTK_COMBO_BOX_ENTRY( widget ), 0 );
		if( set ) {
			gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
		}
		g_object_unref( comp );
	}
}

static void create_action_menu( CSSWizard *wizard, GladeXML *xml )
{
	GtkWidget *widget;
	GtkListStore *store;
	GtkTreeIter it;
	gint i;
	
	const gchar *actions[] = {
		N_( "activated" ), "active",
		N_( "hovered" ), "hover",
		N_( "focused" ), "focus",
		NULL
	};

	widget = glade_xml_get_widget( xml, "action_menu" );

	store = gtk_list_store_new( 2, G_TYPE_STRING, G_TYPE_STRING );
	
	for( i = 0; actions[ i ]; i += 2 ) {
		gtk_list_store_append( store, &it );
		gtk_list_store_set( store, &it, 
				0, gettext( actions[ i ] ),
				1, actions[ i + 1 ],
				-1 );
	}
	gtk_combo_box_set_model( GTK_COMBO_BOX( widget ),
			GTK_TREE_MODEL( store ) );
	gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
}

static void add_page( GtkWidget *widget, const gchar *label_txt )
{
	GladeXML *xml;
	GtkWidget *notebook;
	GtkWidget *label;
	CSSWizard *wizard;
	
	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "match_box" );
	notebook = g_object_get_data( G_OBJECT( widget ), "notebook" );
	wizard = g_object_get_data( G_OBJECT( widget ), "wizard" );
	
	xml = glade_xml_new( GLADE_PATH"/css-wizard.glade",
			"match_box", NULL );
	widget = glade_xml_get_widget( xml, "match_box" );
	g_object_set_data( G_OBJECT( widget ),
			"notebook", notebook );
	g_object_set_data( G_OBJECT( widget ), "wizard",
			wizard );

	label = gtk_label_new( label_txt );
	
	create_tag_menu( wizard, xml );
	create_action_menu( wizard, xml );
	
	css_selector_tag_change( widget );
	widget = glade_xml_get_widget( xml, "hyperlink_menu" );
	gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
	widget = glade_xml_get_widget( xml, "location_menu" );
	gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
	
	widget = glade_xml_get_widget( xml, "match_box" );
	gtk_widget_show( widget );
	gtk_widget_show( label );
	gtk_notebook_append_page( GTK_NOTEBOOK( notebook ), widget,
				  label );

	glade_xml_signal_autoconnect( xml );
}

static gchar *create_selector( GtkWidget *widget )
{
	GladeXML *xml;
	GtkWidget *notebook;
	CSSWizard *wizard;
	gint pages;
	gint i;
	gint item;
	GString *pattern;
	gchar *ret;

	GtkTreeModel *model;
	GtkTreeIter it;
	gchar *action;
	
	pattern = g_string_new( "\n" );
	notebook = g_object_get_data( G_OBJECT( widget ), "notebook" );
	wizard = g_object_get_data( G_OBJECT( widget ), "wizard" );

	pages = gtk_notebook_get_n_pages( GTK_NOTEBOOK( notebook ) );
	for( i = 0; i < pages; ) {
		widget = gtk_notebook_get_nth_page( GTK_NOTEBOOK( notebook ), i );
		xml = glade_get_widget_tree( widget );

		widget = glade_xml_get_widget( xml, "action_menu" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			/* apply action */
			model = gtk_combo_box_get_model( GTK_COMBO_BOX( widget ) );
			gtk_combo_box_get_active_iter( GTK_COMBO_BOX( widget ), &it );
			gtk_tree_model_get( model, &it, 1, 
					&action, -1 );
			g_string_prepend( pattern, action );
			g_free( action );
			g_string_prepend( pattern, ":" );
		}
		widget = glade_xml_get_widget( xml, "hyperlink_menu" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			item = gtk_combo_box_get_active( GTK_COMBO_BOX( widget ) );
			if( item == 0 ) {
				g_string_prepend( pattern,
						":link" );
			} else {
				g_string_prepend( pattern,
						":visited" );
			}
		}
		widget = glade_xml_get_widget( xml, "id_entry" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			/* match id */
			widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
			g_string_prepend( pattern,
					gtk_entry_get_text( GTK_ENTRY( widget ) ) );
			g_string_prepend_c( pattern, '#' );
		}
		widget = glade_xml_get_widget( xml, "class_entry" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			/* match class */
			widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
			g_string_prepend( pattern,
					gtk_entry_get_text( GTK_ENTRY( widget ) ) );
			g_string_prepend_c( pattern, '.' );
		}
		widget = glade_xml_get_widget( xml, "tag_menu" );
		if( GTK_WIDGET_IS_SENSITIVE( widget ) ) {
			/* match tag */
			widget = GTK_BIN( widget )->child;
			g_string_prepend( pattern,
					gtk_entry_get_text( GTK_ENTRY( widget ) ) );
		}

		i ++;
		
		if( i != pages ) {
			widget = glade_xml_get_widget( xml, 
					"location_menu" );
			item = gtk_combo_box_get_active( GTK_COMBO_BOX( widget ) );
			switch( item ) {
				case SOMEWHERE:
					g_string_prepend_c( pattern, ' ' );
					break;
				case CHILD_OF:
					g_string_prepend( pattern, " > " );
					break;
				case AFTER:
					g_string_prepend( pattern, " + " );
					break;
				default:
					break;
			}
		}
	}
	ret = pattern->str;
	g_string_free( pattern, FALSE );

	return ret;
}

static gchar *create_styles( GladeXML *xml )
{
	GString *str;
	gchar *ret;
	GtkWidget *widget;

	const gchar *fields[] = {
		/* fonts */
		"font-family",
		"font-style",
		"font-variant",
		"font-weight",
		"font-size",
		"font-size-adjust",
		"font-stretch",
		"color",

		/* background */
		"background-color",
		"background-image",
		"background-repeat",
		"background-attachment",
		"background-position",

		/* text */
		"word-spacing",
		"letter-spacing",
		"text-decoration",
		"vertical-align",
		"text-transform",
		"text-align",
		"text-indent",
		"line-height",

		/* box */
		"margin-top",
		"margin-left",
		"margin-bottom",
		"margin-right",
		"padding-top",
		"padding-left",
		"padding-bottom",
		"padding-right",

		/* border */
		"border-top-color",
		"border-left-color",
		"border-bottom-color",
		"border-right-color",
		"border-top-width",
		"border-left-width",
		"border-bottom-width",
		"border-right-width",
		"border-top-style",
		"border-left-style",
		"border-bottom-style",
		"border-right-style",

		/* size / placement */
		"width",
		"height",
		"float",
		"clear",

		/* classification */
		"display",
		"white-space",
		"list-style-type",
		"list-style-image",
		"list-style-position"
	};
	static const guint n_fields = G_N_ELEMENTS( fields );
	guint i;
	const gchar *field;
	const gchar *value;
	
	str = g_string_new( "" );
	for( i = 0; i < n_fields; ++ i ) {
		field = fields[ i ];
		widget = glade_xml_get_widget( xml, field );
		if( GNOME_IS_ENTRY( widget ) ) {
			widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
		} else if( GNOME_IS_FILE_ENTRY( widget ) ) {
			widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
		} else if( GTK_IS_COMBO_BOX_ENTRY( widget ) ) {
			widget = GTK_BIN( widget )->child;
		} else if( GTK_IS_COMBO( widget ) ) {
			g_warning( "COMBO %s needs replacing\n",
					field );
			widget = GTK_COMBO( widget )->entry;
		}
		value = gtk_entry_get_text( GTK_ENTRY( widget ) );
		if( value && *value ) {
			g_string_append_printf( str,
					"\t%s: %s;\n", field, value );
		}
	}

	if( str->len > 0 ) {
		g_string_prepend( str, "{\n" );
		g_string_append( str, "}\n" );
	}
	
	ret = str->str;
	g_string_free( str, FALSE );
	
	return ret;
}

void css_selector_wizard_display( GtkAction *action, gpointer user_data )
{
	CSSWizard *wizard;
	ScreemPage *page;
	GtkWidget *widget;
	GtkWidget *notebook;
	GladeXML *xml;
	gint button;
	gchar *pattern;
	gchar *styles;
	ScreemApplication *app;
	ScreemSession *session;
	
	wizard = (CSSWizard*)user_data;
	page = screem_window_get_document( wizard->window );

	app = SCREEM_APPLICATION( wizard->window->application );
	session = screem_application_get_session( app );
	
	if( page ) {
		xml = glade_xml_new( GLADE_PATH"/css-wizard.glade",
				"csspattern", NULL );
		widget = glade_xml_get_widget( xml, "match_box" );
		notebook = glade_xml_get_widget( xml, "notebook" );
		g_object_set_data( G_OBJECT( widget ),
				"notebook", notebook );
		g_object_set_data( G_OBJECT( widget ), "wizard",
				wizard );
		create_tag_menu( wizard, xml );
		create_action_menu( wizard, xml );
	
		widget = glade_xml_get_widget( xml, "hyperlink_menu" );
		gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
		widget = glade_xml_get_widget( xml, "location_menu" );
		gtk_combo_box_set_active( GTK_COMBO_BOX( widget ), 0 );
		
		widget = glade_xml_get_widget( xml, "csspattern" );
		gtk_widget_show( widget );
		glade_xml_signal_autoconnect( xml );

		css_selector_tag_change( widget );
		
		screem_session_restore_dialog( session, widget );
		do {	
			button = gtk_dialog_run( GTK_DIALOG( widget ) );		} while( button == 0 );

		screem_session_store_dialog( session, widget );

		if( button == GTK_RESPONSE_APPLY ) {
			widget = glade_xml_get_widget( xml, 
					"match_box" );
			pattern = create_selector( widget );
			
			screem_editor_insert( wizard->editor, -1,
					pattern );
			g_free( pattern );

			styles = create_styles( xml );
			screem_editor_insert( wizard->editor, -1,
					styles );
			g_free( styles );
		}
		
		widget = glade_xml_get_widget( xml, "csspattern" );
		gtk_widget_destroy( widget );
		g_object_unref( G_OBJECT( xml ) );
	}
}

void css_selector_next_change( GtkComboBox *omenu )
{
	GladeXML *xml;
	GtkWidget *widget;
	GtkWidget *notebook;
	gint page;
	gint pages;
	gint item;
	static const gchar *labels[] = {
		N_( "Rule" ),
		N_( "Contained in" ),
		N_( "Child of" ),
		N_( "After a" )
	};
		
	xml = glade_get_widget_tree( GTK_WIDGET( omenu ) );
	widget = glade_xml_get_widget( xml, "match_box" );
	notebook = g_object_get_data( G_OBJECT( widget ), "notebook" );
	page = gtk_notebook_get_current_page( GTK_NOTEBOOK( notebook ) );

	item = gtk_combo_box_get_active( omenu );
	pages = gtk_notebook_get_n_pages( GTK_NOTEBOOK( notebook ) );

	switch( item ) {
		case ANYWHERE:
			/* remove all subsequent pages */
			while( pages > ( page + 1 ) ) {
				widget = gtk_notebook_get_nth_page( GTK_NOTEBOOK( notebook ), page + 1 );
				xml = glade_get_widget_tree( widget );
				gtk_notebook_remove_page( GTK_NOTEBOOK( notebook ), page + 1 );
				g_object_unref( xml );
				pages --;
			}
			break;
		case SOMEWHERE:
		case CHILD_OF:
		case AFTER:
			if( pages == ( page + 1 ) ) {
				add_page( notebook, 
					gettext( labels[ item ] ) );
			} else {
				/* change label */
				widget = gtk_notebook_get_nth_page( GTK_NOTEBOOK( notebook ), page + 1 );
				gtk_notebook_set_tab_label_text( GTK_NOTEBOOK( notebook ),
						widget, 
						gettext( labels[ item ] ) );
			}

			break;
	}
}

void css_selector_tag_change( GtkWidget *widget )
{
	GladeXML *xml;
	CSSWizard *wizard;
	const gchar *name;
	ScreemPage *page;
	ScreemDTD *dtd;
	const ScreemDTDElement *elem;
	const ScreemDTDAttribute *attr;
	const GSList *attrs;
	gboolean show;
	GtkWidget *entry;

	xml = glade_get_widget_tree( widget );
	widget = glade_xml_get_widget( xml, "match_box" );
	wizard = g_object_get_data( G_OBJECT( widget ), "wizard" );

	widget = glade_xml_get_widget( xml, "tag_menu" );
	show = ( ! GTK_WIDGET_IS_SENSITIVE( widget ) );
	entry = GTK_BIN( widget )->child;

	name = gtk_entry_get_text( GTK_ENTRY( entry ) );

	page = screem_window_get_document( wizard->window );
	if( page && ! show ) {
		dtd = screem_page_get_dtd( page );
		elem = screem_dtd_valid_element( dtd, name );
		attrs = NULL;
		if( elem ) {
			attrs = screem_dtd_element_get_attrs( elem );
		}
		while( attrs ) {
			attr = (const ScreemDTDAttribute*)attrs->data;
			name = screem_dtd_attribute_get_name( attr );
			if( ! g_strcasecmp( "href", name ) ) {
				show = TRUE;
				break;
			}
			attrs = attrs->next;
		}
	}
	widget = glade_xml_get_widget( xml, "hyperlink_button" );
	gtk_widget_set_sensitive( widget, show );
	
	show &= gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( widget ) );
	widget = glade_xml_get_widget( xml, "hyperlink_menu" );
	gtk_widget_set_sensitive( widget, show );
}

void css_wizard_color_set( GtkWidget *widget, GtkColorButton *cp )
{
	GdkColor c;
	gchar *text;
	
	gtk_color_button_get_color( cp, &c );
	text = screem_gdk_color_to_string( &c );

	if( GNOME_IS_ENTRY( widget ) ) {
		widget = gnome_entry_gtk_entry( GNOME_ENTRY( widget ) );
	} else if( GNOME_IS_FILE_ENTRY( widget ) ) {
		widget = gnome_file_entry_gtk_entry( GNOME_FILE_ENTRY( widget ) );
	} else if( GTK_IS_COMBO_BOX_ENTRY( widget ) ) {
		widget = GTK_BIN( widget )->child;
	} else if( GTK_IS_COMBO( widget ) ) {
		widget = GTK_COMBO( widget )->entry;
	}

	gtk_entry_set_text( GTK_ENTRY( widget ), text );
	
	g_free( text );
}

