/*  Screem:  screem-tag_file.c
 *
 *  Copyright (C) 2003 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */

#include "config.h"

#include "screem-tagfile.h"

#include <string.h>
#include <libgnome/gnome-macros.h>
#include <libgnome/gnome-i18n.h>
#include <gtk/gtktreestore.h>
#include <gtk/gtktreemodelsort.h>
#include <gdk/gdk.h>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>

#include "fileops.h"
#include "screem-file-browser.h"

#include "support.h"

static void screem_tag_file_class_init( ScreemTagFileClass *klass );
static void screem_tag_file_instance_init( ScreemTagFile *tag_file );
static void screem_tag_file_finalize( GObject *object );
static void screem_tag_file_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec );
static void screem_tag_file_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec );

static gint screem_tag_file_compare_func( GtkTreeModel *model, 
				    	  GtkTreeIter *a, 
					  GtkTreeIter *b,
				    	  gpointer data );
static void screem_tag_file_icon_change( ScreemFileBrowser *browser, 
					const gchar *uri,
					const gchar *mime_type,
					GtkTreeIter *it,
					gpointer data );
static void screem_tag_file_free_node_info( NodeInfo *info );
static void screem_tag_file_clear( ScreemTagFile *tagfile );
static gboolean screem_tag_file_clear_foreach( GtkTreeModel *model,
						GtkTreePath *path,
						GtkTreeIter *iter,
						gpointer data );


enum {
	PROP_0,
	PROP_ACTIVE
};

struct ScreemTagFilePrivate {
	ScreemTagFile *tfile;
	
	gchar *filename;

	gchar *name;
	gchar *description;

	gchar *type;
	
	ScreemFileBrowser *browser;
	/* the unsorted browser model */
	GtkTreeModel *model;
	
	gboolean active;

	gboolean loaded;

	/* current auto complete prefix */
	const gchar *prefix;

	ScreemTagFileFormat format;

	/* current parent path */
	GtkTreePath *path;
};
GNOME_CLASS_BOILERPLATE( ScreemTagFile, screem_tag_file,
			  GObject,
			  G_TYPE_OBJECT )

static void screem_tag_file_class_init( ScreemTagFileClass *klass )
{
	GObjectClass *obj_class;
	GParamSpec *spec;

	obj_class = G_OBJECT_CLASS( klass );
	obj_class->finalize = screem_tag_file_finalize;
	obj_class->get_property = screem_tag_file_get_prop;
	obj_class->set_property = screem_tag_file_set_prop;

	spec = g_param_spec_boolean( "active", "active",
				_( "Is the Tag File active or no" ),
				TRUE, 
				G_PARAM_READABLE | G_PARAM_WRITABLE );
	g_object_class_install_property( obj_class, PROP_ACTIVE,
					spec );
}

static void screem_tag_file_instance_init( ScreemTagFile *tag_file )
{
	ScreemTagFilePrivate *priv;
	
	priv = tag_file->priv = g_new0( ScreemTagFilePrivate, 1 );
	priv->tfile = tag_file;

	priv->browser = screem_file_browser_new();
	priv->model = screem_file_browser_get_model( priv->browser );
	priv->model = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( priv->model ) );

	g_object_set_data( G_OBJECT( priv->model ), "tagfile", tag_file );
	
	screem_file_browser_set_mode( priv->browser,
					FILE_BROWSE_RECURSE );
	screem_file_browser_set_sort_func( priv->browser,
					   screem_tag_file_compare_func,
					   NULL );
	
	g_signal_connect( G_OBJECT( priv->browser ),
			  "icon_change", 
			  G_CALLBACK( screem_tag_file_icon_change ),
			  NULL );

}

static void screem_tag_file_finalize( GObject *object )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	
	g_return_if_fail( object != NULL );
	g_return_if_fail( SCREEM_IS_TAGFILE( object ) );

	tag_file = SCREEM_TAGFILE( object );

	priv = tag_file->priv;

	screem_tag_file_clear( tag_file );

	g_free( priv->filename );
	g_free( priv->name );
	g_free( priv->description );
	g_free( priv->type );
	
	g_object_unref( priv->browser );
	
	g_free( priv );
	
	GNOME_CALL_PARENT( G_OBJECT_CLASS, finalize, (object) );
}

static void screem_tag_file_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	
	tag_file = SCREEM_TAGFILE( object );
	priv = tag_file->priv;

	switch( prop_id ) {
		case PROP_ACTIVE:
			priv->active = g_value_get_boolean( value );
			if( priv->active && ! priv->loaded ) {
				screem_tag_file_load( tag_file,
						priv->format );
			}
			break;
		default:
			break;
	}
}

static void screem_tag_file_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec )
{
	ScreemTagFile *tag_file;
	ScreemTagFilePrivate *priv;
	
	tag_file = SCREEM_TAGFILE( object );
	priv = tag_file->priv;

	switch( prop_id ) {
		case PROP_ACTIVE:
			g_value_set_boolean( value, priv->active );
			break;
		default:
			break;
	}
}

/* static stuff */
static gint screem_tag_file_compare_func( GtkTreeModel *model, 
				    	  GtkTreeIter *a, 
					  GtkTreeIter *b,
				    	  gpointer data )
{
        NodeInfo *ainfo;
        NodeInfo *binfo;
	int ret = 0;

	gtk_tree_model_get( model, a, 
			FILE_BROWSER_DATA_COL, &ainfo, -1 );
	gtk_tree_model_get( model, b, 
			FILE_BROWSER_DATA_COL, &binfo, -1 );

	if( ainfo && ! binfo ) {
		ret = 1;
	} else if( binfo && !ainfo ) {
		ret = -1;
	} else if( ! ainfo && ! binfo ) {
		ret = 0;
	} else {
		ret = strcmp( ainfo->name, binfo->name );
	}
	
	return ret;
}

static void screem_tag_file_icon_change( ScreemFileBrowser *browser, 
					const gchar *uri,
					const gchar *mime_type,
					GtkTreeIter *it,
					gpointer data )
{
	GtkTreeModel *model;
	NodeInfo *info;
	GdkPixbuf *pixbuf;
	
	model = screem_file_browser_get_model( browser );
	model = gtk_tree_model_sort_get_model( GTK_TREE_MODEL_SORT( model ) );

	gtk_tree_model_get( model, it, 
			FILE_BROWSER_DATA_COL, &info,
			-1 );
	if( info ) {
		pixbuf = NULL;
		if( info->file ) {
			pixbuf = screem_file_browser_get_icon( browser,
								info->file,
								16, 16, FALSE, NULL );
		} else if( gtk_tree_model_iter_has_child( model, it ) ) {
			pixbuf = screem_file_browser_get_icon( browser,
								"/",
								16, 16, FALSE, NULL );
		}
		gtk_tree_store_set( GTK_TREE_STORE( model ), 
				it, FILE_BROWSER_ICON_COL,
				pixbuf, -1 );
		if( pixbuf ) {
			g_object_unref( pixbuf );
		}
	}
}

static void screem_tag_file_free_node_info( NodeInfo *info )
{
	GSList *list;
	NodeAttribute *attr;
	
	g_free( info->name );
	g_free( info->open );
	g_free( info->close );
	g_free( info->file );
	g_free( info->text );
	g_free( info->description );

	for( list = info->attributes; list; list = list->next ) {
		attr = (NodeAttribute*)list->data;

		g_free( attr->name );
		g_free( attr->type );
		g_free( attr->defvalue );
		g_free( attr->description );
		g_free( attr );
	}
	if( info->attributes ) {
		g_slist_free( info->attributes );
	}
	
	g_free( info );
}

static void screem_tag_file_clear( ScreemTagFile *tagfile )
{
	ScreemTagFilePrivate *priv;
	
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );

	priv = tagfile->priv;
	
	gtk_tree_model_foreach( priv->model,
			screem_tag_file_clear_foreach,
			NULL );

	gtk_tree_store_clear( GTK_TREE_STORE( priv->model ) );
}

static gboolean screem_tag_file_clear_foreach( GtkTreeModel *model,
						GtkTreePath *path,
						GtkTreeIter *iter,
						gpointer data )
{
	NodeInfo *info;
	
	gtk_tree_model_get( model, iter, 
			FILE_BROWSER_DATA_COL, &info, 
			-1 );
	if( info ) {
		screem_tag_file_free_node_info( info );
	}

	return FALSE;
}

static gboolean screem_tag_file_autocomplete_fill( GtkTreeModel *model,
						GtkTreePath *path,
						GtkTreeIter *iter,
						gpointer data )
{
	GSList **ret;
	ScreemTagFile *tfile;
	NodeInfo *info;
	const gchar *str;
	const gchar *prefix;

	tfile = SCREEM_TAGFILE( g_object_get_data( G_OBJECT( model ),
						"tagfile" ) );
	prefix = tfile->priv->prefix;
	ret = (GSList**)data;

	gtk_tree_model_get( model, iter,
			FILE_BROWSER_DATA_COL, &info,
			-1 );

	str = info->open;
	if( ! str ) {
		str = info->close;
	}
	if( ! str ) {
		str = info->text;
	}
	if( str && 
	   ! strncmp( str, prefix, strlen( prefix ) ) ) {
		*ret = g_slist_prepend( *ret, (gchar*)str );
	}

	return FALSE;
}

static gboolean screem_tag_file_autocomplete_tip_fill( GtkTreeModel *model,
						GtkTreePath *path,
						GtkTreeIter *iter,
						gpointer data )
{
	GSList **ret;
	ScreemTagFile *tfile;
	NodeInfo *info;
	const gchar *str;
	const gchar *prefix;

	tfile = SCREEM_TAGFILE( g_object_get_data( G_OBJECT( model ),
						"tagfile" ) );
	prefix = tfile->priv->prefix;
	ret = (GSList**)data;

	gtk_tree_model_get( model, iter,
			FILE_BROWSER_DATA_COL, &info,
			-1 );

	str = info->tip;
	if( ! str ) {
		str = info->open;
	}
	if( ! str ) {
		str = info->close;
	}
	if( ! str ) {
		str = info->text;
	}
	if( str && 
	   ! strncmp( str, prefix, strlen( prefix ) ) ) {
		*ret = g_slist_prepend( *ret, (gchar*)str );
	}

	return FALSE;
}

static gint lookup_lang( gconstpointer a, gconstpointer b )
{
	gint ret;
	
	if( a && b ) {
		ret = strcmp( a, b );
	} else if( b ) {
		ret = -1;
	} else {
		ret = 1;
	}

	return ret;
}

static gboolean find_node( const gchar *expr, gboolean lang,
		xmlXPathContextPtr ctx, 
		xmlNodePtr *node, xmlXPathObjectPtr *o )
{
	xmlXPathObjectPtr obj;
	xmlNodeSetPtr nodes;
	xmlNodePtr n;
	const GList *langs;
	const GList *tmp;
	gint i;
	gchar *xmllang;
	gchar *lexpr;
	
	langs = gnome_i18n_get_language_list( "LC_COLLATE" );

	if( ! node ) {
		node = &n;
	}
	if( o ) {
		*o = NULL;
	}
	
	/* first we want a <ref> matching the locale */
	*node = NULL;
	obj = NULL;
	tmp = NULL;
	if( ! lang ) {
		lexpr = g_strconcat( expr, "[@xml:lang]", NULL );
		obj = xmlXPathEvalExpression( lexpr, ctx );
		g_free( lexpr );
	}
	if( obj ) {
		/* got some results */
		nodes = obj->nodesetval;
		for( i = 0; ( ( *node = xmlXPathNodeSetItem( nodes, i ) ) ); ++ i ) {
			xmllang = xmlGetNsProp( *node, "lang",
					XML_XML_NAMESPACE );
			tmp = g_list_find_custom( (GList*)langs, 
					(const gchar *)xmllang,
					(GCompareFunc)lookup_lang );
			g_free( xmllang );
			if( tmp ) {
				if( o ) {
					*o = obj;
				}
				break;
			}
		}
		if( ( ! *node ) || ! o ) {
			xmlXPathFreeObject( obj );
		}
	}
	if( ! *node ) {
		/* failed to find one, fall back */
		lexpr = g_strconcat( expr, "[not(@xml:lang)]", NULL );
		obj = xmlXPathEvalExpression( lexpr, ctx );
		g_free( lexpr );
		if( obj ) {
			nodes = obj->nodesetval;
			*node = xmlXPathNodeSetItem( nodes, 0 );
			if( o ) {
				*o = obj;
			} else {
				xmlXPathFreeObject( obj );
			}
		}
	}

	return ( tmp != NULL );
}

static void process_param( xmlXPathContextPtr ctx, xmlNodePtr node, 
		gboolean lang, gpointer userdata )
{
	ScreemTagFile *tfile;
	ScreemTagFilePrivate *priv;
	xmlNodePtr snode;
	gboolean slang;

	gchar *name;
	gchar *desc;
	gchar *required;
	gchar *vallist;
	gchar *defval;
	gchar *type;

	GtkTreeIter it;
	NodeInfo *info;
	NodeAttribute *attr;

	tfile = SCREEM_TAGFILE( userdata );
	priv = tfile->priv;
	
	/* process attrs */
	name = xmlGetProp( node, "name" );
	required = xmlGetProp( node, "required" );
	vallist = xmlGetProp( node, "vallist" );
	defval = xmlGetProp( node, "default" );
	type = xmlGetProp( node, "type" );
	desc = NULL;

	/* process description */
	ctx->node = node;
	slang = find_node( "text()", lang, ctx, &snode, NULL );
	if( snode ) {
		desc = g_strdup( snode->content );
	}

	gtk_tree_model_get_iter( priv->model, &it, priv->path );
	gtk_tree_model_get( priv->model, &it,
			FILE_BROWSER_DATA_COL, &info, -1 );
	
	attr = g_new0( NodeAttribute, 1 );
	if( name ) {
		attr->name = g_strstrip( name );
	}
	if( required ) {
		attr->required = ( *required == '1' );
		g_free( required );
	}
	if( vallist ) {
		attr->vallist = ( *vallist == '1' );
		g_free( vallist );
	}
	if( defval ) {
		attr->defvalue = g_strstrip( defval );
	}
	if( type ) {
		attr->type = type;
	}
	if( desc ) {
		attr->description = g_strstrip( desc );
	}

	info->attributes = g_slist_prepend( info->attributes, attr );
}

static void process_entry( xmlXPathContextPtr ctx, xmlNodePtr node, 
		gboolean lang, gpointer userdata )
{
	ScreemTagFile *tfile;
	ScreemTagFilePrivate *priv;
	xmlXPathObjectPtr obj;
	xmlNodeSetPtr set;
	xmlNodePtr snode;
	gboolean slang;
	gint i;

	gchar *name;
	gchar *desc;

	gchar *open;
	gchar *close;
	gchar *href;
	gchar *data;
	gchar *pathname;
	
	GtkTreeIter it;
	GtkTreeIter pit;
	GdkPixbuf *pixbuf;
	NodeInfo *info;
	GtkTreePath *path;

	gboolean isfunc;
	
	tfile = SCREEM_TAGFILE( userdata );
	priv = tfile->priv;
	
	isfunc = ( ! strcmp( "function", node->name ) );
	
	/* process attrs */
	name = xmlGetProp( node, "name" );
	desc = NULL;

	/* process description */
	ctx->node = node;
	slang = find_node( "description", lang, ctx, &snode, NULL );
	if( snode ) {
		/* get text */
		ctx->node = snode;
		slang = find_node( "text()", slang, ctx,
				&snode, NULL );
		if( snode ) {
			desc = g_strdup( snode->content );
		}
	}

	pixbuf = NULL;
	
	info = g_new0( NodeInfo, 1 );
	info->name = name;
	info->isfunc = isfunc;
	if( desc ) {
		info->description = desc;
	}
	
	gtk_tree_model_get_iter( priv->model, &pit, priv->path );
	gtk_tree_store_append( GTK_TREE_STORE( priv->model ), 
			&it, &pit );
	gtk_tree_store_set( GTK_TREE_STORE( priv->model ), &it,
			FILE_BROWSER_NAME_COL, name,
			FILE_BROWSER_ICON_COL, pixbuf,
			FILE_BROWSER_DATA_COL, info,
			-1 );
	path = gtk_tree_model_get_path( priv->model, &it );


	/* process insert */
	ctx->node = node;
	slang = find_node( "insert", lang, ctx, &snode, NULL );
	if( snode ) {
		/* process attrs */
		open = xmlGetProp( snode, "open" );
		close = xmlGetProp( snode, "close" );
		href = xmlGetProp( snode, "href" );

		/* get text */
		ctx->node = snode;
		slang = find_node( "text()", slang, ctx,
				&snode, NULL );
		data = NULL;
		if( snode ) {
			data = g_strdup( snode->content );
		}

		if( href ) {
			pathname = relative_to_full( href,
					RESOURCE_PATH );
			info->file = pathname;
			pixbuf = screem_file_browser_get_icon( priv->browser,
					pathname, 16, 16, FALSE, NULL );
		} else {
			if( open ) {
				/* open text */
				info->open = g_strdup( open );
			} 
			if( close ) {
				/* close text */
				info->close = g_strdup( close );
			}
		}
		if( isfunc ) {
			info->text = g_strconcat( info->name, "()",
					NULL );
			if( data ) {
				info->tip = g_strstrip( data );
			}
		} else if( data ) {
			info->text = g_strstrip( data );
		}
		g_free( open );
		g_free( close );
		g_free( href );
	}
	
	/* process property / param */
	ctx->node = node;
	slang = find_node( "(property|param)", lang, ctx, NULL, &obj );
	if( obj ) {
		set = obj->nodesetval;
		for( i = 0; i < set->nodeNr; ++ i ) {
			snode = set->nodeTab[ i ];

			/* process <property> / <param> */
			priv->path = path;
			process_param( ctx, snode, lang, userdata );
		}
		xmlXPathFreeObject( obj );
	}

	gtk_tree_store_set( GTK_TREE_STORE( priv->model ), &it,
			FILE_BROWSER_ICON_COL, pixbuf,
			-1 );

	if( pixbuf ) {
		g_object_unref( pixbuf );
	}
	gtk_tree_path_free( path );
}

static void process_group( xmlXPathContextPtr ctx, xmlNodePtr node, 
	gboolean lang, gpointer userdata )
{
	ScreemTagFile *tfile;
	ScreemTagFilePrivate *priv;
	xmlXPathObjectPtr obj;
	xmlNodeSetPtr set;
	xmlNodePtr snode;
	gboolean slang;
	gint i;

	gchar *name;
	gchar *desc;

	GtkTreeIter it;
	GtkTreeIter pit;
	GdkPixbuf *pixbuf;
	NodeInfo *info;
	GtkTreePath *path;
	
	tfile = SCREEM_TAGFILE( userdata );
	priv = tfile->priv;
	
	/* process attrs */
	name = xmlGetProp( node, "name" );
	desc = xmlGetProp( node, "description" );
	
	/* process description */
	if( ! desc ) {
		ctx->node = node;
		slang = find_node( "description", lang, ctx, 
				&snode, NULL );
		if( snode ) {
			/* get text */
			ctx->node = snode;
			slang = find_node( "text()", slang, ctx,
					&snode, NULL );
			if( snode ) {
				desc = g_strdup( snode->content );
			}
		}
	}

	pixbuf = screem_file_browser_get_icon( priv->browser,
			"/", 16, 16, FALSE, NULL );
	
	info = g_new0( NodeInfo, 1 );
	info->name = name;
	if( desc ) {
		info->description = desc;
	}
	
	gtk_tree_model_get_iter( priv->model, &pit, priv->path );
	gtk_tree_store_append( GTK_TREE_STORE( priv->model ), 
			&it, &pit );
	gtk_tree_store_set( GTK_TREE_STORE( priv->model ), &it,
			FILE_BROWSER_NAME_COL, name,
			FILE_BROWSER_ICON_COL, pixbuf,
			FILE_BROWSER_DATA_COL, info,
			-1 );
	if( pixbuf ) {
		g_object_unref( G_OBJECT( pixbuf ) );
	}
	path = gtk_tree_model_get_path( priv->model, &it );

	/* get subgroups */
	ctx->node = node;
	slang = find_node( "group", lang, ctx, NULL, &obj );
	if( obj && ( obj->nodesetval->nodeNr != 0 ) ) {
		set = obj->nodesetval;
		for( i = 0; i < set->nodeNr; ++ i ) {
			snode = set->nodeTab[ i ];

			/* process <group> */
			priv->path = path;
			process_group( ctx, snode, slang, userdata );
		}
		xmlXPathFreeObject( obj );
	} else {
		if( obj ) {
			xmlXPathFreeObject( obj );
		}
		/* get entries */
		ctx->node = node;
		slang = find_node( "(entry|function)", lang, 
				ctx, NULL, &obj );
		if( obj ) {
			set = obj->nodesetval;
			for( i = 0; i < set->nodeNr; ++ i ) {
				snode = set->nodeTab[ i ];

				/* process <entry> */
				priv->path = path;
				process_entry( ctx, snode, slang,
						userdata );
			}
			xmlXPathFreeObject( obj );
		}
	}

	gtk_tree_path_free( path );
}

static void process_ref( xmlXPathContextPtr ctx, xmlNodePtr node,
		gboolean lang, gpointer userdata )
{
	ScreemTagFile *tfile;
	ScreemTagFilePrivate *priv;
	xmlXPathObjectPtr obj;
	xmlNodeSetPtr set;
	xmlNodePtr gnode;
	gint i;

	gchar *name;
	gchar *desc;
	gchar *type;
	
	GtkTreeIter it;
	GdkPixbuf *pixbuf;
	NodeInfo *info;
	GtkTreePath *path;
	gboolean slang;
	
	tfile = SCREEM_TAGFILE( userdata );
	priv = tfile->priv;
	
	/* process attrs */
	name = xmlGetProp( node, "name" );
	desc = xmlGetProp( node, "description" );
	type = xmlGetProp( node, "type" );

	g_free( priv->name );
	g_free( priv->description );
	g_free( priv->type );
	priv->name = name;
	if( desc ) {
		priv->description = desc;
	}
	if( type ) {
		priv->type = type;
	}
	
	pixbuf = screem_file_browser_get_icon( priv->browser,
			"/", 16, 16, FALSE, NULL );
	
	info = g_new0( NodeInfo, 1 );
	info->name = g_strdup( name );
	info->tfile = tfile;
	if( desc ) {
		info->description = g_strdup( desc );
	}

	gtk_tree_store_append( GTK_TREE_STORE( priv->model ), &it,
			NULL );
	gtk_tree_store_set( GTK_TREE_STORE( priv->model ), &it,
			FILE_BROWSER_NAME_COL, name,
			FILE_BROWSER_ICON_COL, pixbuf,
			FILE_BROWSER_DATA_COL, info,
			-1 );
	
	if( pixbuf ) {
		g_object_unref( G_OBJECT( pixbuf ) );
	}
	path = gtk_tree_model_get_path( priv->model, &it );
	
	ctx->node = node;
	
	/* get groups */
	lang = find_node( "group", lang, ctx, NULL, &obj );
	if( obj  && ( obj->nodesetval->nodeNr != 0 ) ) {
		set = obj->nodesetval;
		for( i = 0; i < set->nodeNr; ++ i ) {
			gnode = set->nodeTab[ i ];
			
			/* process <group> */
			priv->path = path;
			process_group( ctx, gnode, lang, userdata );
		}
		xmlXPathFreeObject( obj );
	} else {
		/* hmm no groups, cheat an treat <ref> as a group */
		if( obj ) {
			xmlXPathFreeObject( obj );
		}
		
		/* get entries */
		ctx->node = node;
		slang = find_node( "(entry|function)", lang, 
				ctx, NULL, &obj );
		if( obj ) {
			set = obj->nodesetval;
			for( i = 0; i < set->nodeNr; ++ i ) {
				gnode = set->nodeTab[ i ];

				/* process <entry> */
				priv->path = path;
				process_entry( ctx, gnode, slang,
						userdata );
			}
			xmlXPathFreeObject( obj );
		}
	}

	gtk_tree_path_free( path );
	priv->path = NULL;
}

static gboolean parse_from_string( const gchar *buffer, 
		gpointer userdata )
{
	xmlDocPtr doc;
	xmlXPathContextPtr ctx;
	xmlNodePtr node;
	gboolean lang;

	doc = xmlParseMemory( buffer, strlen( buffer ) );
	
	if( doc ) {
		ctx = xmlXPathNewContext( doc );
		
		lang = find_node( "/refs/ref", FALSE, ctx, &node, NULL );
		if( ! node ) {
			/* bluefish doesn't have the <refs> as
			 * it doesn't translate <ref> */
			lang = find_node( "/ref", FALSE, ctx, 
					&node, NULL );
		}
		if( node ) {
			/* process <ref> */
			process_ref( ctx, node, lang, userdata );
		} 
	
		xmlXPathFreeContext( ctx );
		xmlFreeDoc( doc );
		if( ! node ) {
			doc = NULL;
		}
	}

	return ( doc != NULL );
}

/* public stuff */

ScreemTagFile *screem_tag_file_new( void )
{
	ScreemTagFile *tag_file;

	tag_file = g_object_new( SCREEM_TYPE_TAGFILE, NULL );

	return tag_file;
}

gboolean screem_tag_file_load( ScreemTagFile *tagfile,
		ScreemTagFileFormat format )
{
	GString *file;
	gboolean ret;
	gchar *type;
	
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), FALSE );

	if( format == SCREEM_TAG_UNKNOWN_FORMAT ) {
		type = screem_get_mime_type( tagfile->priv->filename,
				FALSE ); 
		if( ! strcmp( "application/x-screem-tag-tree",
					type ) ) {
			format = SCREEM_TAG_TREE_FORMAT;
		} else if( ! strcmp( "application/x-weml+xml",
					type ) ) {
			format = SCREEM_TAG_WEML_FORMAT;
		} else if( ! strcmp( "text/xml", type ) ) {
			/* treat text/xml as screem tag trees,
			 * this is as bluefish uses .xml and
			 * we support those files */
			format = SCREEM_TAG_TREE_FORMAT;
		} else {
			g_free( type );
			return FALSE;
		}
		g_free( type );
	}
	
	ret = FALSE;
	file = load_file( tagfile->priv->filename, NULL, NULL, NULL );
	if( file ) {
		tagfile->priv->format = format;
		ret = screem_tag_file_set_from_string( tagfile, 
						file->str,
						file->len );
		g_string_free( file, TRUE );
	}
	
	return ret;	
}

gboolean screem_tag_file_load_with_uri( ScreemTagFile *tagfile,
				const gchar *uri,
				ScreemTagFileFormat format )
{
	screem_tag_file_set_uri( tagfile, uri );

	return screem_tag_file_load( tagfile, format );
}

gboolean screem_tag_file_set_from_string( ScreemTagFile *tagfile,
					const gchar *str,
					guint length )
{
	ScreemTagFilePrivate *priv;
	gboolean ret;

	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), FALSE );
	g_return_val_if_fail( str != NULL, FALSE );
	
	priv = tagfile->priv;
	
	if( length == 0 ) {
		length = strlen( str );
	}
	
	ret = FALSE;

	screem_tag_file_clear( tagfile );
	
	switch( priv->format ) {
		case SCREEM_TAG_WEML_FORMAT:

			break;
		default:
			ret = parse_from_string( str, tagfile );
			break;
	}
	
	priv->loaded = ret;
	
	return ret;	
}

GtkTreeModel *screem_tag_file_get_model( ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->model;
}

const gchar *screem_tag_file_get_name( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->name;
}

const gchar *screem_tag_file_get_desc( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->description;
}

const gchar *screem_tag_file_get_uri( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->filename;
}

const gchar *screem_tag_file_for_type( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );

	return tagfile->priv->type;
}

ScreemTagFileFormat screem_tag_file_get_format( const ScreemTagFile *tagfile )
{
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), 0 );

	return tagfile->priv->format;
}

void screem_tag_file_set_name( ScreemTagFile *tagfile, 
				const gchar *name )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );
	g_return_if_fail( name != NULL );

	g_free( tagfile->priv->name );
	tagfile->priv->name = g_strdup( name );
}

void screem_tag_file_set_desc( ScreemTagFile *tagfile,
				const gchar *desc )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );
	g_return_if_fail( desc != NULL  );

	g_free( tagfile->priv->description );
	tagfile->priv->description = g_strdup( desc );
}

void screem_tag_file_set_uri( ScreemTagFile *tagfile,
				const gchar *uri )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );
	g_return_if_fail( uri != NULL );

	g_free( tagfile->priv->filename );
	tagfile->priv->filename = g_strdup( uri );
}

void screem_tag_file_set_format( ScreemTagFile *tagfile,
		ScreemTagFileFormat format )
{
	g_return_if_fail( SCREEM_IS_TAGFILE( tagfile ) );

	tagfile->priv->format = format;
}

GSList *screem_tag_file_autocomplete( ScreemTagFile *tagfile,
		const gchar *prefix, gboolean fortip )
{
	ScreemTagFilePrivate *priv;
	GSList *ret;
	GtkTreeModelForeachFunc func;
	
	g_return_val_if_fail( SCREEM_IS_TAGFILE( tagfile ), NULL );
	g_return_val_if_fail( prefix != NULL, NULL );

	priv = tagfile->priv;
	
	ret = NULL;
	priv->prefix = prefix;
	if( ! fortip ) {
		func = screem_tag_file_autocomplete_fill;
	} else {
		func = screem_tag_file_autocomplete_tip_fill;
	}
	gtk_tree_model_foreach( priv->model, func, &ret );

	return ret;			
}

