/*  Screem:  screem-ctags-model.c
 *
 *  Copyright (C) 2003 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */

#include "config.h"

#include <glib/gi18n.h>

#include <libgnomevfs/gnome-vfs-utils.h>
#include <libgnomevfs/gnome-vfs-ops.h>

#include <gdk/gdkpixbuf.h>
#include <gdk/gdk.h>

#include <string.h>

#include "screem-ctags-model.h"

#include "readtags.h"

#include "screem-application.h"
#include "screem-icon-cache.h"

#include "support.h"

extern ScreemApplication *app;

static void screem_ctags_model_pathname_set( ScreemCtagsModel *model );
static void monitor_cb( GnomeVFSMonitorHandle *handle,
			  const gchar *monitor_uri,
			  const gchar *info_uri,
			  GnomeVFSMonitorEventType type,
			  gpointer data );
static void screem_ctags_model_theme_change( ScreemIconCache *cache,
					gpointer data );
static gboolean screem_ctags_model_icon_change( GtkTreeModel *model,
					GtkTreePath *path,
					GtkTreeIter *it,
					gpointer data );
typedef struct {
	GtkTreePath *path;
	const gchar *kind;
	const gchar *name;
} Levels;

struct ScreemCtagsModelPrivate {
	ScreemIconCache *icache;
	gchar *pathname;
	GnomeVFSMonitorHandle *monitor;

	guint source;
	Levels *toplevels;

	/* file->icon cache */
	GHashTable *cache;
	/* file->mime cache */
	GHashTable *mime;

	tagFile *file;
	tagFileInfo *info;
	
	/* autocomplete */
	const gchar *mime_type;
	const gchar *prefix;
	guint prefix_len;
};

enum {
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_PATHNAME
};

/*static guint screem_ctags_model_signals[ LAST_SIGNAL ] = { 0 };*/
	
G_DEFINE_TYPE( ScreemCtagsModel, screem_ctags_model, GTK_TYPE_TREE_STORE )

static void screem_ctags_model_finalize( GObject *object );
static void screem_ctags_model_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec );
static void screem_ctags_model_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec );

static void screem_ctags_model_class_init( ScreemCtagsModelClass *klass )
{
	GObjectClass *obj_class;
	GParamSpec *pspec;
	
	obj_class = G_OBJECT_CLASS( klass );
	obj_class->finalize = screem_ctags_model_finalize;
	obj_class->get_property = screem_ctags_model_get_prop;
	obj_class->set_property = screem_ctags_model_set_prop;


	pspec = g_param_spec_string( "pathname", "pathname",
				      "pathname",
				      "",
				      G_PARAM_READABLE |
				      G_PARAM_WRITABLE );
	g_object_class_install_property( G_OBJECT_CLASS( obj_class ),
					 PROP_PATHNAME,
					 pspec );
}


static void screem_ctags_model_init( ScreemCtagsModel *ctags_model )
{
	ScreemCtagsModelPrivate *priv;
	GtkTreeStore *store;
	GType columns[] = {
		G_TYPE_STRING,
		GDK_TYPE_PIXBUF,
		G_TYPE_STRING,
		G_TYPE_STRING,
		G_TYPE_UINT
	};
	static const Levels levels [] = {
		{ NULL, "a", N_( "HTML Anchors" ) },
		{ NULL, "c", N_( "Classes" ) },
		{ NULL, "d", N_( "Defines" ) },
		{ NULL, "e", N_( "Enumerators" ) },
		{ NULL, "f", N_( "Functions" ) },
		{ NULL, "F", N_( "Filenames" ) },
		{ NULL, "g", N_( "Enumerator Names" ) },
		{ NULL, "m", N_( "Class/Struct Members" ) },
		{ NULL, "p", N_( "Function Prototypes" ) },
		{ NULL, "s", N_( "Structs" ) },
		{ NULL, "t", N_( "Typedefs" ) },
		{ NULL, "u", N_( "Unions" ) },
		{ NULL, "v", N_( "Variables" ) },

		/* Keep these 2 last */
		{ NULL, "", N_( "Other" ) },
		{ NULL, NULL, NULL }
	};
	gint i;
	
	priv = ctags_model->priv = g_new0( ScreemCtagsModelPrivate, 1 );
	priv->icache = screem_application_get_icon_cache( app );

	g_signal_connect( G_OBJECT( priv->icache ), "changed",
			G_CALLBACK( screem_ctags_model_theme_change ),
			ctags_model );
	
	priv->toplevels = g_new0( Levels, G_N_ELEMENTS( levels ) );
	memcpy( priv->toplevels, levels, sizeof( levels ) );
	
	for( i = 0; priv->toplevels[ i ].name; ++ i ) {
		priv->toplevels[ i ].name = gettext( levels[ i ].name );
		priv->toplevels[ i ].path = gtk_tree_path_new_from_indices( i, -1 ); 
	}
	
	store = GTK_TREE_STORE( ctags_model );
	
	gtk_tree_store_set_column_types( store, CTAGS_MAX_COLS, columns );

	priv->cache = g_hash_table_new_full( g_str_hash, g_str_equal,
			(GDestroyNotify)g_free,
			(GDestroyNotify)g_object_unref );
	priv->mime = g_hash_table_new_full( g_str_hash, g_str_equal,
			(GDestroyNotify)g_free,
			(GDestroyNotify)g_free );
		
	priv->info = g_new0( tagFileInfo, 1 );
}

static void screem_ctags_model_finalize( GObject *object )
{
	ScreemCtagsModel *ctags_model;
	ScreemCtagsModelPrivate *priv;
	guint i;
				
	g_return_if_fail( object != NULL );
	g_return_if_fail( SCREEM_IS_CTAGS_MODEL( object ) );

	ctags_model = SCREEM_CTAGS_MODEL( object );

	priv = ctags_model->priv;

	if( priv->source ) {
		g_source_remove( priv->source );
		tagsClose( priv->file );
	}
	for( i = 0; priv->toplevels[ i ].name; ++ i ) {
		gtk_tree_path_free( priv->toplevels[ i ].path );
	}
	g_free( priv->toplevels );

	if( priv->cache ) {
		g_hash_table_destroy( priv->cache );
	}
	if( priv->mime ) {
		g_hash_table_destroy( priv->mime );
	}

	g_free( priv->pathname );
	
	if( priv->monitor ) {
		gnome_vfs_monitor_cancel( priv->monitor );
	}
	g_free( priv->info );
	
	g_free( priv );
	
	G_OBJECT_CLASS( screem_ctags_model_parent_class )->finalize( object );
}

static void screem_ctags_model_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec )
{
	ScreemCtagsModel *ctags_model;
	ScreemCtagsModelPrivate *priv;
	
	ctags_model = SCREEM_CTAGS_MODEL( object );
	priv = ctags_model->priv;

	switch( prop_id ) {
		case PROP_PATHNAME:
			if( priv->monitor ) {
				gnome_vfs_monitor_cancel( priv->monitor );
				priv->monitor = NULL;
			}
			g_free( priv->pathname );
			priv->pathname = (gchar*)g_value_get_string( value );
			if( priv->pathname ) {
				priv->pathname = gnome_vfs_get_local_path_from_uri( priv->pathname );
			}
			screem_ctags_model_pathname_set( ctags_model );
		default:
			break;
	}
}

static void screem_ctags_model_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec )
{
	ScreemCtagsModel *ctags_model;
	ScreemCtagsModelPrivate *priv;
	
	ctags_model = SCREEM_CTAGS_MODEL( object );
	priv = ctags_model->priv;

	switch( prop_id ) {
		case PROP_PATHNAME:
			g_value_set_string( value, priv->pathname );
			break;
		default:
			break;
	}
}

/* static stuff */
static gboolean doremove( gpointer key, gpointer value, gpointer data )
{
	return TRUE;
}

static gboolean screem_ctags_model_build( gpointer data )
{
	ScreemCtagsModel *model;
	ScreemCtagsModelPrivate *priv;
	tagEntry entry;
	tagResult tres;
	GtkTreeStore *store;
	GtkTreePath *path;
	GtkTreeIter it;
	GtkTreeIter pit;
	GdkPixbuf *icon;
	gchar *ipath;
	const gchar *kind;
	const gchar *sname;
	const gchar *uname;
	const gchar *cname;
	guint i;
	gchar *mimetype;
	
	model = SCREEM_CTAGS_MODEL( data );
	priv = model->priv;
	store = GTK_TREE_STORE( data );
	
	tres = tagsNext( priv->file, &entry ); 

	if( tres != TagSuccess ) {
		g_hash_table_foreach_remove( priv->cache, 
					doremove, NULL );
		tagsClose( priv->file );
		priv->source = 0;

		return FALSE;
	}

	gdk_threads_enter();
	
	kind = tagsField( &entry, "kind" );
	sname = tagsField( &entry, "struct" );
	uname = tagsField( &entry, "union" );
	cname = tagsField( &entry, "class" );
			
	if( ! kind ) {
		kind = "";
	}
	path = NULL;
	for( i = 0; priv->toplevels[ i ].kind; ++ i ) {
		path = priv->toplevels[ i ].path;
		if( ! strcmp( kind, priv->toplevels[ i ].kind ) ) {
			break;
		}
	}
	g_assert( priv->pathname );
	g_assert( entry.file );
	ipath = g_build_path( G_DIR_SEPARATOR_S,
			priv->pathname,
			entry.file, NULL );
	icon = g_hash_table_lookup( priv->cache, ipath );
	if( ! icon ) {
		mimetype = screem_get_mime_type( ipath, FALSE );

		icon = screem_icon_cache_get_pixbuf( priv->icache,
				ipath, mimetype, 
				SCREEM_ICON_CACHE_DEFAULT_SIZE );
		g_hash_table_insert( priv->cache,
				g_strdup( ipath ),
				icon );
		g_hash_table_insert( priv->mime, 
				g_strdup( ipath ), 
				mimetype );
	}
	/* place in toplevel class */
	gtk_tree_model_get_iter( GTK_TREE_MODEL( model ), &pit, path );
	gtk_tree_store_append( store, &it, &pit );
	gtk_tree_store_set( store, &it,
			CTAGS_NAME_COL, entry.name,
			CTAGS_FILE_COL, ipath,
			CTAGS_ICON_COL, icon,
			CTAGS_PATTERN_COL, entry.address.pattern,
			CTAGS_LINE_COL, entry.address.lineNumber,
			-1 );
	/* place in relevant struct if needed */
	if( sname ) {

	}
	/* place in relevant union if needed */
	if( uname ) {

	}
	/* place in relevant class if needed */
	if( cname ) {

	}

	g_free( ipath );

	gdk_threads_leave();

	return TRUE;
}

static void screem_ctags_model_pathname_set( ScreemCtagsModel *model )
{
	ScreemCtagsModelPrivate *priv;
	GtkTreeStore *store;
	gchar *pathname;
	guint i;
	GtkTreeIter it;
	GdkPixbuf *icon;
	
	priv = model->priv;

	if( priv->source ) {
		g_source_remove( priv->source );
		priv->source = 0;
		tagsClose( priv->file );
		priv->file = NULL;
	}
	
	store = GTK_TREE_STORE( model );
	gtk_tree_store_clear( store );

	icon = screem_icon_cache_get_pixbuf( priv->icache,
				"/", "x-directory/normal",
				SCREEM_ICON_CACHE_DEFAULT_SIZE );

	for( i = 0; priv->toplevels[ i ].name; ++ i ) {
		gtk_tree_store_append( store, &it, NULL );
		gtk_tree_store_set( store, &it,
			CTAGS_NAME_COL, priv->toplevels[ i ].name,
			CTAGS_ICON_COL, icon,
			-1 );
	}
	
	if( icon ) {
		g_object_unref( G_OBJECT( icon ) );
	}
	
	pathname = g_build_path( G_DIR_SEPARATOR_S,
			priv->pathname,
			"tags",
			NULL );
	
	priv->file = tagsOpen( pathname, priv->info );

	if( ! priv->file ) {
		g_free( pathname );
		pathname = g_build_path( G_DIR_SEPARATOR_S,
			priv->pathname,
			"TAGS",
			NULL );
		priv->file = tagsOpen( pathname, priv->info );
	}

	if( priv->file ) {
		if( ! priv->monitor ) {
			gnome_vfs_monitor_add( &priv->monitor, pathname,
					       GNOME_VFS_MONITOR_DIRECTORY,
					       monitor_cb, model );
		}
		priv->source = g_idle_add_full( G_PRIORITY_LOW,
				screem_ctags_model_build,
				model, NULL );
	}
	g_free( pathname );
}

static void monitor_cb( GnomeVFSMonitorHandle *handle,
			  const gchar *monitor_uri,
			  const gchar *info_uri,
			  GnomeVFSMonitorEventType type,
			  gpointer data )
{
	ScreemCtagsModel *model;

	model = SCREEM_CTAGS_MODEL( data );

	switch( type ) {
		case GNOME_VFS_MONITOR_EVENT_CHANGED:
		case GNOME_VFS_MONITOR_EVENT_DELETED:
		case GNOME_VFS_MONITOR_EVENT_CREATED:
			screem_ctags_model_pathname_set( model );
			break;
		default:
			break;
	}
}

static void screem_ctags_model_theme_change( ScreemIconCache *cache,
					gpointer data )
{
	ScreemCtagsModel *model;
	ScreemCtagsModelPrivate *priv;

	model = SCREEM_CTAGS_MODEL( data );
	priv = model->priv;


	g_hash_table_destroy( priv->cache );
	priv->cache = g_hash_table_new_full( g_str_hash, g_str_equal,
			(GDestroyNotify)g_free,
			(GDestroyNotify)g_object_unref );

	gtk_tree_model_foreach( GTK_TREE_MODEL( data ),
			(GtkTreeModelForeachFunc)screem_ctags_model_icon_change,
			cache );
}

static gboolean screem_ctags_model_icon_change( GtkTreeModel *model,
					GtkTreePath *path,
					GtkTreeIter *it,
					gpointer data )
{
	ScreemIconCache *cache;
	gchar *file;
	GdkPixbuf *pixbuf;
	ScreemCtagsModel *cmodel;
	ScreemCtagsModelPrivate *priv;
	
	cache = SCREEM_ICON_CACHE( data );
	cmodel = SCREEM_CTAGS_MODEL( model );
	priv = cmodel->priv;
	
	gtk_tree_model_get( GTK_TREE_MODEL( model ), it,
			CTAGS_FILE_COL, &file,
			-1 );
	if( ! file ) {
		pixbuf = screem_icon_cache_get_pixbuf( cache,
				"/", "x-directory/normal",
				SCREEM_ICON_CACHE_DEFAULT_SIZE);
	} else {
		pixbuf = g_hash_table_lookup( priv->cache, file );
		if( ! pixbuf ) {
			pixbuf = screem_icon_cache_get_pixbuf( priv->icache,
				file, NULL,
				SCREEM_ICON_CACHE_DEFAULT_SIZE );
			g_object_ref( pixbuf );
			g_hash_table_insert( priv->cache,
					g_strdup( file ), 
					pixbuf );
		}
	}

	gtk_tree_store_set( GTK_TREE_STORE( model ), it,
			CTAGS_ICON_COL, pixbuf, -1 );
	
	if( pixbuf ) {
		g_object_unref( G_OBJECT( pixbuf ) );
	}
	g_free( file );

	return FALSE;
}

static gboolean screem_ctags_file_autocomplete_fill( GtkTreeModel *model,
		GtkTreePath *path, GtkTreeIter *iter, gpointer data )
{
	ScreemCtagsModel *cmodel;
	ScreemCtagsModelPrivate *priv;
	GSList **ret = (GSList**)data;
	const gchar *mime;
	const gchar *mime_type;
	const gchar *prefix;
	guint len;
	gchar *file;
	gchar *name;

	gboolean check;
	
	cmodel = SCREEM_CTAGS_MODEL( model );
	priv = cmodel->priv;

	prefix = priv->prefix;
	len = priv->prefix_len;
	mime_type = priv->mime_type;

	gtk_tree_model_get( GTK_TREE_MODEL( model ), iter,
			CTAGS_FILE_COL, &file,
			CTAGS_NAME_COL, &name,
			-1 );
	check = FALSE;
	if( name && file ) {
		check = ( ! strncmp( name, prefix, len ) );
	}
		
	if( check ) {
		mime = g_hash_table_lookup( priv->mime, file );

		if( mime && mime_type && ! strcmp( mime, mime_type ) ) {
			*ret = g_slist_prepend( *ret, g_strdup( name ) );
		}
	}
	g_free( file );
	g_free( name );

	return FALSE;
}

/* public stuff */

ScreemCtagsModel *screem_ctags_model_new( void )
{
	ScreemCtagsModel *ctags_model;

	ctags_model = g_object_new( SCREEM_TYPE_CTAGS_MODEL, NULL );

	return ctags_model;
}

void screem_ctags_model_set_pathname( ScreemCtagsModel *model,
					const gchar *pathname )
{
	g_return_if_fail( SCREEM_IS_CTAGS_MODEL( model ) );
	
	g_object_set( G_OBJECT( model ), "pathname", pathname, NULL );
}

gchar *screem_ctags_model_get_pathname( const ScreemCtagsModel *model )
{
	gchar *ret;
	
	g_return_val_if_fail( SCREEM_IS_CTAGS_MODEL( model ), NULL );

	g_object_get( G_OBJECT( model ), "pathname", &ret, NULL );

	return ret;
}

GSList *screem_ctags_model_autocomplete( const ScreemCtagsModel *model,
					const gchar *mime_type,	const gchar *prefix )
{
	ScreemCtagsModelPrivate *priv;
	GSList *ret;
	GtkTreeModelForeachFunc func;
	
	g_return_val_if_fail( SCREEM_IS_CTAGS_MODEL( model ), NULL );
	g_return_val_if_fail( prefix != NULL, NULL );

	priv = model->priv;
	
	ret = NULL;
	priv->prefix = prefix;
	priv->prefix_len = strlen( prefix );
	priv->mime_type = mime_type;
	func = screem_ctags_file_autocomplete_fill;
	
	gtk_tree_model_foreach( GTK_TREE_MODEL( model ), func, &ret );

	return ret;			
}
