/* This file is part of the SpeedCrunch project
   Copyright (C) 2004 Ariya Hidayat <ariya@kde.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either version 2
   of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include "insertvardlg.h"
#include "evaluator.h"
#include "hmath.h"
#include "settings.h"

#include <qdialog.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>
#include <qtimer.h>

class InsertVariableDlg::Private
{
public:
  Evaluator* eval;
  QListView* varList;
  QPushButton* insertButton;
  QPushButton* cancelButton;
};

InsertVariableDlg::InsertVariableDlg( Evaluator* e, QWidget* parent, const char* name ): 
QDialog( parent, name )
{
  d = new Private;
  d->eval = e;
  
  QGridLayout* layout = new QGridLayout( this, 2, 3, 6, 6 );

  d->varList = new QListView( this );
  d->varList->addColumn( tr("Name") );
  d->varList->addColumn( tr("Value") );
  d->varList->setResizeMode( QListView::LastColumn );
  d->varList->setAllColumnsShowFocus( true );
  d->varList->setColumnAlignment( 1, Qt::AlignRight );
    
  d->insertButton = new QPushButton( tr("&Insert"), this );
  d->cancelButton = new QPushButton( tr("Cancel"), this );

  layout->addMultiCellWidget( d->varList, 0, 0, 1, 3 );  
  layout->addItem( new QSpacerItem( 20, 20, QSizePolicy::Expanding ), 1, 1 );  
  layout->addWidget( d->insertButton, 1, 2 );
  layout->addWidget( d->cancelButton, 1, 3 );
  
  d->insertButton->setDefault( true );
  
  connect( d->insertButton, SIGNAL( clicked() ), this, SLOT( accept() ) );
  connect( d->cancelButton, SIGNAL( clicked() ), this, SLOT( reject() ) );
  connect( d->varList, SIGNAL( returnPressed( QListViewItem* ) ), this, SLOT( accept() ) );
  connect( d->varList, SIGNAL( doubleClicked( QListViewItem*, const QPoint&, int ) ), 
    this, SLOT( accept() ) );
  
  setCaption( tr("Insert Variable") );
  QTimer::singleShot( 0, this, SLOT( initUI() ) );  
}

InsertVariableDlg::~InsertVariableDlg()
{
  delete d;
}

void InsertVariableDlg::updateList()
{
  d->varList->clear();
  QValueVector<Variable> vars = d->eval->variables();
  for( unsigned i = 0; i < vars.count(); i++ )
  {
    char* ss = HMath::format( vars[i].value, 'g' );
    new QListViewItem( d->varList, vars[i].name, QString( ss ) );
    free( ss );  
  }
  
  d->varList->setFocus();
}

QString InsertVariableDlg::variableName() const
{
  QListViewItem* item = d->varList->selectedItem();
  return item ? item->text(0) : QString::null;
}


void InsertVariableDlg::initUI()
{
  updateList();
  adjustSize();
}
