/*
 * Copyright (C) 2005-2009 Martin Willi
 * Copyright (C) 2005 Jan Hutter
 * Hochschule fuer Technik Rapperswil
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  See <http://www.fsf.org/copyleft/gpl.txt>.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 */

/**
 * @defgroup generator generator
 * @{ @ingroup encoding
 */

#ifndef GENERATOR_H_
#define GENERATOR_H_

typedef struct generator_t generator_t;

#include <library.h>
#include <encoding/payloads/encodings.h>
#include <encoding/payloads/payload.h>

/**
 * A generator_t class used to generate IKEv2 payloads.
 *
 * After creation, multiple payloads can be generated with the generate_payload
 * method. The generated bytes are appended. After all payloads are added,
 * the write_to_chunk method writes out all generated data since
 * the creation of the generator.
 * The generater uses a set of encoding rules, which it can get from
 * the supplied payload. With this rules, the generater can generate
 * the payload and all substructures automatically.
 */
struct generator_t {

	/**
	 * Generates a specific payload from given payload object.
	 *
	 * @param payload 		interface payload_t implementing object
	 */
	void (*generate_payload) (generator_t *this,payload_t *payload);

	/**
	 * Return a chunk for the currently generated data.
	 *
	 * The returned length pointer must be filled in with the length of
	 * the generated chunk (in network order).
	 *
	 * @param lenpos		receives a pointer to fill in length value
	 * @param return		chunk to internal buffer.
	 */
	chunk_t (*get_chunk) (generator_t *this, uint32_t **lenpos);

	/**
	 * Destroys a generator_t object.
	 */
	void (*destroy) (generator_t *this);
};

/**
 * Constructor to create a generator.
 *
 * @return generator_t object.
 */
generator_t *generator_create(void);

/**
 * Constructor to create a generator that does not log any debug messages > 1.
 *
 * @return generator_t object.
 */
generator_t *generator_create_no_dbg(void);


#endif /** GENERATOR_H_ @}*/
