/*
 * Collection of filters to modify streams of tagged items
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/Filters.h>

#include <iostream>

namespace Tagcoll {

template<typename ITEM>
void UnfacetedRemover<ITEM>::consumeItemUntagged(const ITEM& item)
{
	this->consumer->consume(item);
}

template<typename ITEM>
void UnfacetedRemover<ITEM>::consumeItem(const ITEM& item, const OpSet<std::string>& tags)
{
	OpSet<std::string> patched;
	for (OpSet<std::string>::const_iterator i = tags.begin();
			i != tags.end(); i++)
		if (i->find(separator) != std::string::npos)
			patched += *i;

	if (patched.size())
		this->consumer->consume(item, patched);
	else
		this->consumer->consume(item);
}

template<typename ITEM>
void UnfacetedRemover<ITEM>::consumeItemsUntagged(const OpSet<ITEM>& items)
{
	this->consumer->consume(items);
}

template<typename ITEM>
void UnfacetedRemover<ITEM>::consumeItems(const OpSet<ITEM>& items, const OpSet<std::string>& tags)
{
	OpSet<std::string> patched;
	for (OpSet<std::string>::const_iterator i = tags.begin();
			i != tags.end(); i++)
		if (i->find(separator) != std::string::npos)
			patched += *i;

	if (patched.size())
		this->consumer->consume(items, patched);
	else
		this->consumer->consume(items);
}

}

#ifndef INSTANTIATING_TEMPLATES
#include <string>

namespace Tagcoll {
template class Substitutions<std::string>;
template class Substitute<std::string, std::string>;
template class UnfacetedRemover<std::string>;
}
#endif


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

#include <tagcoll/InputMerger.h>

namespace tut {
using namespace tut_tagcoll;

struct tagcoll_filters_shar {
};
TESTGRP(tagcoll_filters);

template<> template<>
void to::test<1>()
{
	std::string input_coll(
			"a: b, c\n"
			"b:\n"
			"c: \n"
			"d:  c::D, e::F,    f::g\n"
			);
	std::string input_subst(
			"b: c\n"
			"c::D: e::F\n"
			);
	std::string output_coll(
			"a: b, c\n"
			"b:\n"
			"c: \n"
			"d:  c::D, e::F,    f::g\n"
			);
	InputMerger<string, string> result;
	Substitute<string, string> filter(result);

	outputCollection(input_subst, filter.substitutions());
	outputCollection(input_coll, filter); 

	InputMerger<string, string> reference;
	outputCollection(output_coll, reference); 

	ensure_coll_equal(reference, result);
}

template<> template<>
void to::test<2>()
{
	std::string input_coll(
			"a: b, c, d::e\n"
			"b: c::D, e::F, f::g\n"
			"c: c::D, e, f::g\n"
			);
	std::string output_coll(
			"a: d::e\n"
			"b: c::D, e::F, f::g\n"
			"c: c::D, f::g\n"
			);
	InputMerger<string, string> result;
	UnfacetedRemover<string> filter(result);
	outputCollection(input_coll, filter); 

	InputMerger<string, string> reference;
	outputCollection(output_coll, reference); 

	ensure_coll_equal(reference, result);
}

template<> template<>
void to::test<3>()
{
	std::string input_coll(
			"a: b, c, d---e\n"
			"b: c---D, e---F, f---g\n"
			"c: c---D, e, f---g\n"
			);
	std::string output_coll(
			"a: d---e\n"
			"b: c---D, e---F, f---g\n"
			"c: c---D, f---g\n"
			);
	InputMerger<string, string> result;
	UnfacetedRemover<string> filter(result, "---");
	outputCollection(input_coll, filter); 

	InputMerger<string, string> reference;
	outputCollection(output_coll, reference); 

	ensure_coll_equal(reference, result);
}

}

#endif
// vim:set ts=4 sw=4:
