#ifndef TAGCOLL_FILTERS_H
#define TAGCOLL_FILTERS_H

/* \file
 * Collection of filters to modify streams of tagged items
 */

/*
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/Consumer.h>
#include <tagcoll/Filter.h>

#include <map>
#include <string>

namespace Tagcoll
{

/**
 * Store a list of substitutions to operate on OpSets
 *
 * It uses a Consumer interface to allow to be populated using a decoder for
 * collections such as TextFormat.
 */
template<typename T>
class Substitutions : public Consumer<T, T>
{
protected:
	typedef std::map<T, T> changes_t;
	changes_t changes;

	virtual void consumeItemUntagged(const T& item) {}
	virtual void consumeItem(const T& item, const OpSet<T>& tags)
	{
		for (typename OpSet<T>::const_iterator i = tags.begin();
				i != tags.end(); i++)
			changes.insert(make_pair(*i, item));
	}
	virtual void consumeItemsUntagged(const OpSet<T>& items) {}

public:
	virtual ~Substitutions() {}

	/// Change a single value
	T change(const T& v) const
	{
		typename changes_t::const_iterator i = changes.find(v);

		return (i == changes.end()) ? v : i->second;
	}

	/// Change all the items in a set
	OpSet<T> change(const OpSet<T>& values) const
	{
		OpSet<T> res;
		for (typename OpSet<T>::const_iterator t = values.begin();
				t != values.end(); t++)
			res += change(*t);
		return res;
	}
};

/**
 * Filter replacing tags according to a list of Substitutions
 *
 * Example:
 *
 * \code
 *   Substitute<Item, Tag> filter(consumer);
 *
 *   // Parse substitutions from a file into the filter
 *   TextFormat<Item, Tag>::parse(itemconv, tagconv, input, filter.substitutions());
 *
 *   // Filter the collection
 *   coll.output(filter);
 * \endcode
 */
template<typename ITEM, typename TAG>
class Substitute : public Filter<ITEM, TAG>
{
protected:
	Substitutions<TAG> changes;
	
	virtual void consumeItemUntagged(const ITEM& item)
	{
		this->consumer->consume(item);
	}
	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags)
	{
		this->consumer->consume(item, changes.change(tags));
	}
	virtual void consumeItemsUntagged(const OpSet<ITEM>& items)
	{
		this->consumer->consume(items);
	}
	virtual void consumeItem(const OpSet<ITEM>& items, const OpSet<TAG>& tags)
	{
		this->consumer->consume(items, changes.change(tags));
	}

public:
	Substitute() {}
	Substitute(Consumer<ITEM, TAG>& cons) : Filter<ITEM, TAG>(cons) {}
	Substitute(const Substitutions<TAG>& changes) : changes(changes) {}
	Substitute(Consumer<ITEM, TAG>& cons, const Substitutions<TAG>& changes) :
		Filter<ITEM, TAG>(cons), changes(changes) {}

	/**
	 * Access the internal Substitution list
	 */
	Substitutions<TAG>& substitutions() { return changes; }

	/**
	 * Access the internal Substitution list (const version)
	 */
	const Substitutions<TAG>& substitutions() const { return changes; }
};

/**
 * Remove packages with no tags.
 *
 * It can also be used in 'inverse' mode, where in removes the packages which
 * have tags and keeps the packages that have none.
 */
template <typename ITEM, typename TAG>
class UntaggedRemover : public Filter<ITEM, TAG>
{
protected:
	bool inverse;

	virtual void consumeItemUntagged(const ITEM& item)
	{
		if (inverse)
			this->consumer->consume(item);
	}
	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags)
	{
		if (inverse)
		{
			if (tags.empty())
				this->consumer->consume(item);
		} else {
			if (!tags.empty())
				this->consumer->consume(item, tags);
		}
	}
	virtual void consumeItemsUntagged(const OpSet<ITEM>& items)
	{
		if (inverse)
			this->consumer->consume(items);
	}
	virtual void consumeItemss(const OpSet<ITEM>& items, const OpSet<TAG>& tags)
	{
		if (inverse)
		{
			if (tags.empty())
				this->consumer->consume(items);
		} else {
			if (!tags.empty())
				this->consumer->consume(items, tags);
		}
	}

public:
	UntaggedRemover(bool inverse = false) : inverse(inverse) {}
	UntaggedRemover(Tagcoll::Consumer<ITEM, TAG>& cons, bool inverse = false)
		: Filter<ITEM, TAG>(cons), inverse(inverse) {}
	virtual ~UntaggedRemover() {}
};


/**
 * Removes tags which are not inside a facet
 */
template<typename ITEM>
class UnfacetedRemover : public Filter<ITEM, std::string>
{
protected:
	std::string separator;
		
	virtual void consumeItemUntagged(const ITEM& item);
	virtual void consumeItem(const ITEM& item, const OpSet<std::string>& tags);
	virtual void consumeItemsUntagged(const OpSet<ITEM>& items);
	virtual void consumeItems(const OpSet<ITEM>& items, const OpSet<std::string>& tags);

public:
	UnfacetedRemover() : separator("::") {}
	UnfacetedRemover(Consumer<ITEM, std::string>& cons)
		: Filter<ITEM, std::string>(cons), separator("::") {}
	UnfacetedRemover(const std::string& separator) : separator(separator) {}
	UnfacetedRemover(Consumer<ITEM, std::string>& cons, const std::string& separator)
		: Filter<ITEM, std::string>(cons), separator(separator) {}
	virtual ~UnfacetedRemover() {}
};

};

// vim:set ts=4 sw=4:
#endif
