/*
 * Auto-expanding trees and smart hierarchy interface
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <tagcoll/SmartHierarchy.h>

//#define VERB1 1
//#define VERB_FLATTEN 1

#if VERB1 || VERB_FLATTEN
#include <stdio.h>
#include "../common/stringf.h"
//using namespace stringf;

static void print_spaces(int n) throw ()
{
	for (int i = 0; i < n; i++)
		fputc(' ', stderr);
}
#endif

using namespace std;
using namespace Tagcoll;

//template class SmartHierarchyNode<string>;
//template class SmartHierarchyNode<int>;
//template class CleanSmartHierarchyNode<string>;
//template class CleanSmartHierarchyNode<int>;

//namespace Debtags { class Package; }
//template class SmartHierarchyNode<Debtags::Package*>;
//template class CleanSmartHierarchyNode<Debtags::Package*>;

template<class ITEM, class TAG>
HierarchyNode<ITEM, TAG>::~HierarchyNode()
{
	if (coll)
		delete coll;
	for (typename vector<HierarchyNode<ITEM, TAG>*>::iterator i = children.begin();
			i != children.end(); i++)
		delete *i;
}

template<class ITEM, class TAG>
void SmartHierarchyNode<ITEM, TAG>::expand()
{
	/*
Define the "cardinality" of a tag as the number of items in the collection
having it in the tag set
Add the items with no tags to the tree
Delete the items with no tags from the collection
while (there are items left)
Choose the tag T with the highest cardinality and add it to the nodes in
  the tree, as a new folder called with the name of the tag
Insert into the new folder all associated items
Remove from all the items in the collection the tag T
Remove from all the items in the collection all the items implied by tag T
Delete the items with no tags from the collection
For all newly added folders, repeat this algorithm on the collection of
their items
	*/
	
#ifdef VERB1
	//print_spaces(nest);
	fprintf(stderr, "%d tags, %d tagsets\n", coll->tagCount(), coll->tagsetCount());
#endif

	if (flattenThreshold > 0 && this->coll->itemCount() < flattenThreshold)
	{
		this->items = unexpandedItems + this->coll->getTaggedItems();
#ifdef VERB_FLATTEN
		fprintf(stderr, "Flattened: threshold: %d, items: %d\n", flattenThreshold, items.size());
#endif
	}
	else
	{
		this->items = unexpandedItems;
	
		OpSet<TAG> processed;
		CardinalityStore<ITEM, TAG> workcoll = *this->coll;

		while (1)
		{
			// Choose the tag T with the highest cardinality
			int card;
			TAG tag = workcoll.findTagWithMaxCardinalityNotIn(processed, &card);

			// Stop condition: there are no more tags left
			if (card == 0)
				break;

			processed.insert(tag);

#ifdef VERB1
			//print_spaces(nest);
			fprintf(stderr, "   Processed: ");
			for (OpSet<TAG>::const_iterator i = processed.begin();
					i != processed.end(); i++)
				if (i == processed.begin())
					fprintf(stderr, "%.*s", PFSTR(*i));
				else
					fprintf(stderr, ", %.*s", PFSTR(*i));
			fprintf(stderr, "\n");
			//print_spaces(nest);
			fprintf(stderr, "   Chosen %.*s[%d]\n", PFSTR(tag), card);
#endif

			// Append the child node
			this->children.push_back(new SmartHierarchyNode(this, tag, workcoll, flattenThreshold));
			
			// Remove all tags already fully implied by the chosen one
			workcoll = workcoll.getCollectionWithoutTags(processed);
		}
	}

	delete this->coll;
	this->coll = 0;
}

template<class ITEM, class TAG>
void CleanSmartHierarchyNode<ITEM, TAG>::expand()
{
	/* Same as SmartHierarchyNode, except it create CleanSmartHierarchyNode
	 * children and performs merging of equivalent tags*/
	// FIXME: eventually go really OO, using a factory class to create subnodes
	// FIXME: another way is to use a virtual HiearchyNode* createChildNode(parms) method

	//fprintf(stderr, "CSHN::expand\n");
	this->coll->mergeEquivalentTags();
	//fprintf(stderr, "CSHN::merged\n");
	
	if (this->flattenThreshold > 0 && this->coll->itemCount() < this->flattenThreshold)
		this->items = this->unexpandedItems + this->coll->getTaggedItems();
	else
	{
		this->items = this->unexpandedItems;
	
		OpSet<TAG> processed;
		CardinalityStore<ITEM, TAG> workcoll = *this->coll;

		while (1)
		{
			// Choose the tag T with the highest cardinality
			int card;
			TAG tag = workcoll.findTagWithMaxCardinalityNotIn(processed, &card);

			// Stop condition: there are no more tags left
			if (card == 0)
				break;

			processed.insert(tag);

			// Append the child node
			this->children.push_back(new CleanSmartHierarchyNode(this, tag, workcoll, this->flattenThreshold));

			// Remove all tags already fully implied by the chosen one
			workcoll = workcoll.getCollectionWithoutTags(processed);
		}
	}

	// If we contain only one child node, replace it with his expansion
	while (this->children.size() == 1 && this->items.empty())
	{
		CleanSmartHierarchyNode* child = static_cast<CleanSmartHierarchyNode*>(this->children[0]);
		this->children.clear();
		
		// Move the child's children to this node
		for (typename CleanSmartHierarchyNode::iterator i = child->begin();
				i != child->end(); i++)
		{
			// TODO: make a mergeTags method which in the general case keeps
			// the first tag, and in the string case it joins the tags with a
			// comma
			static_cast<CleanSmartHierarchyNode*>(*i)->setTag(child->tag() + ", " + (*i)->tag());
			static_cast<CleanSmartHierarchyNode*>(*i)->setParent(this);
			this->children.push_back(*i);
		}

		this->items = child->getItems();

		child->children.clear();
		delete child;
	}

	delete this->coll;
	this->coll = 0;
}

#ifndef INSTANTIATING_TEMPLATES
#include <string>

namespace Tagcoll {
	template class HierarchyNode<std::string, std::string>;
	template class SmartHierarchyNode<std::string, std::string>;
	template class CleanSmartHierarchyNode<std::string, std::string>;
}
#endif


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

#include <tagcoll/TextFormat.h>
#include <tagcoll/StringParserInput.h>

namespace tut {
using namespace tut_tagcoll;

struct tagcoll_smarthierarchy_shar {
};
TESTGRP(tagcoll_smarthierarchy);

template<> template<>
void to::test<1>()
{
	StringParserInput input_coll(
			"a: b, c\n"
			"b:\n"
			"c: \n"
			"d:  c::D, e::F,    f::g\n"
			);
	ensure(false);
}

}

#endif

// vim:set ts=4 sw=4:
