#ifndef TAGCOLL_TDB_FILE_H
#define TAGCOLL_TDB_FILE_H

/** \file
 * TDB file functions
 */

/*
 * Copyright (C) 2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/Exception.h>
#include <tagcoll/OpSet.h>
#include <set>
#include <string>
#include <fcntl.h>
#include <tdb.h>

namespace Tagcoll
{

/**
 * Access an on-disk binary index in TDB format
 */
class TDBFile
{
private:
	/* Disable implicit copies */
	TDBFile(const TDBFile&);
	TDBFile operator=(const TDBFile&);

protected:
	TDB_CONTEXT* db;
	std::string _filename;

public:
	/**
	 * Create a new TDBFile
	 *
	 * @param filename
	 *   The filename of the on-disk database file
	 */
	TDBFile(const std::string& filename);
	virtual ~TDBFile();

	/**
	 * Get the file name
	 */
	const std::string& filename() const { return _filename; }

	/** @{
	 * Access the internal TDB context
	 */
	const TDB_CONTEXT* context() const { return db; }
	TDB_CONTEXT* context() { return db; }
	/** @} */

	/**
	 * Open the file.
	 *
	 * \param tdb_flags
	 *   Database flags.  See man tdb_open(3) for a description.
	 * \param open_flags
	 *   File open flags.  See man tdb_open(3) for a description.
	 * \param mode
	 *   File permissions to use if it is created.  Umask is respected.
	 *   See man open(2) for a description.
	 * \throw SystemException if the operation fails
	 */
	void open(int tdb_flags, int open_flags, mode_t mode=0666);

	/**
	 * Utility function to deserialize a string list
	 *
	 * \param val
	 *   a TDB entry containing a list of strings separated by a null character ('\\0')
	 * \return
	 *   the strings parsed into an OpSet<std::string>
	 */
	static OpSet<std::string> deserialize_stringset(const TDB_DATA& val);

	/**
	 * Returns true if the database contains the given key
	 */
	bool has(const std::string& key) const;

	/**
	 * Generic way of retrieving the value for a key.
	 *
	 * \note in most cases, you want to use the get() method instead.
	 *
	 * \param key
	 *   The key to retrieve
	 * \param buf
	 *   The buffer where the key value is to be retrieved
	 * \param size
	 *   The size of buf.  Reading will fail if buf is not big enough to hold
	 *   the result.
	 * \return
	 *   true if the key exists in the database and its value fits in the
	 *   buffer; false otherwise.
	 */
	bool getGeneric(const std::string& key, void* buf, unsigned int size) const;
	
	/**
	 * Get the value of a key.
	 *
	 * \param key
	 *   The key to retrieve.
	 * \param item
	 *   The item to set.  The size of the item is automatically inferred by
	 *   its type.
	 * \return
	 *   true if the key exists in the database and its value fits in the
	 *   given variable; false otherwise.
	 */
	template<class ITEM>
	bool get(const std::string& key, ITEM& item) const
	{
		return getGeneric(key, (void*)&item, sizeof(ITEM));
	}
	
	/**
	 * Get the value of a key, parsed as a set of strings
	 * 
	 * \param key
	 *   The key to retrieve.
	 * \return
	 *   The value parsed as an OpSet<std::string>.  The strings are supposed
	 *   to be encoded one after the other, separated by null ('\\0')
	 *   characters.
	 */
	OpSet<std::string> getStringSet(const std::string& key) const;

	/**
	 * Generic way of setting the value for a key.
	 *
	 * \note in most cases, you want to use the set() method instead.
	 *
	 * \param key
	 *   The key to store
	 * \param buf
	 *   The buffer with the data to store
	 * \param size
	 *   The size of the data in buf
	 * \throw SystemException in case of error
	 */
	void setGeneric(const std::string& key, const void* buf, unsigned int size);

	/**
	 * Set the value of a key.
	 *
	 * \param key
	 *   The key to store
	 * \param item
	 *   The item to store.  The size of the item is automatically inferred
	 *   by its type.
	 * \throw SystemException in case of error
	 */
	template<class ITEM>
	void set(const std::string& key, const ITEM& item)
	{
		setGeneric(key, (void*)&item, sizeof(ITEM));
	}
	
	/**
	 * Set the value of a key using a set of strings
	 * 
	 * \param key
	 *   The key to store.
	 * \param vals
	 *   The values to store.  The strings are encoded one after the other,
	 *   separated by null ('\\0') characters.
	 * \throw SystemException in case of error
	 */
	void setStringSet(const std::string& key, OpSet<std::string> vals);

	/**
	 * Remove an item
	 *
	 * \param key
	 *   The key of the item to remove
	 * \throw SystemException in case of error
	 */
	void remove(const std::string& key);

	/**
	 * Iterate through all the items in the database
	 *
	 * \param fn
	 *   Callback function that gets invoked once for every item in the
	 *   database.
	 * \param state
	 *   Arbitrary value that is passed to fn as-is.
	 * \return
	 *   The number of items traversed
	 * \throw SystemException in case of error
	 */
	int traverse(int (*fn)(TDB_CONTEXT *,TDB_DATA,TDB_DATA,void *), void *state) const;
};

};

// vim:set ts=4 sw=4:
#endif
