/*
 * Test TDBDiskIndex
 *
 * Copyright (C) 2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <test-utils.h>
#include <tagcoll/TDBDiskIndex.h>
#include <tagcoll/Exception.h>
#include <unistd.h>

namespace tut {
using namespace tut_tagcoll;

struct tagcoll_tests_tdbdiskindex_shar {
		tagcoll_tests_tdbdiskindex_shar() {
			unlink("testpkg.tdb");
			unlink("testtag.tdb");
		}
		~tagcoll_tests_tdbdiskindex_shar() {
			unlink("testpkg.tdb");
			unlink("testtag.tdb");
		}
};
TESTGRP(tagcoll_tests_tdbdiskindex);

template<> template<>
void to::test<1> ()
{
	Converter<string, string> conv;

	try {
		// An empty database should return empty sets, but not fail
		Tagcoll::TDBDiskIndex<string, string> tfi(
				"testpkg.tdb", "testtag.tdb",
				conv, conv, conv, conv);
		gen_ensure(!tfi.hasTag("cippo"));
		gen_ensure(!tfi.hasTag("lippo"));
		gen_ensure(tfi.getItems("cippo") == Tagcoll::OpSet<string>());
		gen_ensure(tfi.getTags("cippo") == Tagcoll::OpSet<string>());
		gen_ensure(tfi.getItems("lippo") == Tagcoll::OpSet<string>());
		gen_ensure(tfi.getTags("lippo") == Tagcoll::OpSet<string>());
		// No autovivification should happen on gets
		gen_ensure(!tfi.hasTag("cippo"));
		gen_ensure(!tfi.hasTag("lippo"));

		// Adding an item with no tags should still return an empty set for
		// that item 
		tfi.consume("cippo");
		gen_ensure(!tfi.hasTag("cippo"));
		gen_ensure(tfi.getItems("cippo") == Tagcoll::OpSet<string>());
		gen_ensure(tfi.getTags("cippo") == Tagcoll::OpSet<string>());

		// Add some data and check that it comes back
		Tagcoll::OpSet<string> s;
		s += "lippo";
		s += "zippo";
		s += "rippo";

		tfi.consume("cippo", s);
		gen_ensure(tfi.hasTag("lippo"));
		gen_ensure(tfi.hasTag("zippo"));
		gen_ensure(tfi.hasTag("rippo"));
		s = tfi.getTags("cippo");
		gen_ensure(s.contains("lippo"));
		gen_ensure(s.contains("zippo"));
		gen_ensure(s.contains("rippo"));
		gen_ensure(!s.contains("bippo"));

		s = tfi.getItems("lippo");
		gen_ensure(s.contains("cippo"));
		gen_ensure(!s.contains("lippo"));

		gen_ensure(tfi.getItems("") == Tagcoll::OpSet<string>());
	} catch (Exception& e) {
		fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
		throw;
	}

	// Reopen the database to see if data is actually persisting
	try {
		Tagcoll::TDBDiskIndex<string, string> tfi(
				"testpkg.tdb", "testtag.tdb",
				conv, conv, conv, conv);
		Tagcoll::OpSet<string> s = tfi.getTags("cippo");

		// Repeat the same queries: they should work
		gen_ensure(s.contains("lippo"));
		gen_ensure(s.contains("zippo"));
		gen_ensure(s.contains("rippo"));
		gen_ensure(! s.contains("bippo"));

		s = tfi.getItems("lippo");
		gen_ensure(s.contains("cippo"));
		gen_ensure(!s.contains("lippo"));

		gen_ensure(tfi.getItems("") == Tagcoll::OpSet<string>());
	} catch (Exception& e) {
		fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
		throw;
	}
}

}

// vim:set ts=4 sw=4:
