#include <tagcoll/Commandline.h>
#include <config.h>
#include <locale.h>
#include <ctype.h>
#include <stdio.h>
#include <errno.h>
#include <set>

using namespace std;

static bool isSwitch(const char* str)
{
	// No empty strings
	if (str[0] == 0)
		return false;
	// Must start with a dash
	if (str[0] != '-')
		return false;
	// Must not be "-" (usually it means 'stdin' file argument)
	if (str[1] == 0)
		return false;
	// Must not be "--" (end of switches)
	if (strcmp(str, "--") == 0)
		return false;
	return true;
}

namespace Tagcoll {
namespace commandline {

int Option::intValue() const
{
	return strtoul(stringValue().c_str(), NULL, 10);
}

static string fmtshort(char c, const std::string& usage)
{
	if (usage.empty())
		return string("-") + c;
	else
		return string("-") + c + " " + usage;
}

static string fmtlong(const std::string& c, const std::string& usage)
{
	if (usage.empty())
		return string("--") + c;
	else
		return string("--") + c + "=" + usage;
}

static string manfmtshort(char c, const std::string& usage)
{
	if (usage.empty())
		return string("\\-") + c;
	else
		return string("\\-") + c + " \\fI" + usage + "\\fP";
}

static string manfmtlong(const std::string& c, const std::string& usage)
{
	if (usage.empty())
		return string("\\-\\-") + c;
	else
		return string("\\-\\-") + c + "=\\fI" + usage + "\\fP";
}

const std::string& Option::fullUsage() const
{
	if (m_fullUsage.empty())
	{
		for (vector<char>::const_iterator i = shortNames.begin();
				i != shortNames.end(); i++)
		{
			if (!m_fullUsage.empty())
				m_fullUsage += ", ";
			m_fullUsage += fmtshort(*i, usage);
		}

		for (vector<string>::const_iterator i = longNames.begin();
				i != longNames.end(); i++)
		{
			if (!m_fullUsage.empty())
				m_fullUsage += ", ";
			m_fullUsage += fmtlong(*i, usage);
		}
	}
	return m_fullUsage;
}
	
std::string Option::fullUsageForMan() const
{
	string res;

	for (vector<char>::const_iterator i = shortNames.begin();
			i != shortNames.end(); i++)
	{
		if (!res.empty()) res += ", ";
		res += manfmtshort(*i, usage);
	}

	for (vector<string>::const_iterator i = longNames.begin();
			i != longNames.end(); i++)
	{
		if (!res.empty()) res += ", ";
		res += manfmtlong(*i, usage);
	}

	return res;
}
	

bool StringOption::parse(const char* str)
{
	if (!str) throw BadOption("no string argument found");
	m_value = str;
	return true;
}

std::string IntOption::stringValue() const
{
	return stringf::fmt(m_value);
}

bool IntOption::parse(const char* str)
{
	if (!str) throw BadOption("no int argument found");
	// Ensure that we're all numeric
	for (const char* s = str; *s; ++s)
		if (!isdigit(*s))
			throw BadOption(string("value ") + str + " must be numeric");
	m_value = strtoul(str, 0, 10);
	return true;
}

bool ExistingFileOption::parse(const char* str)
{
	if (!str) throw BadOption("no file argument found");
	if (access(str, F_OK) == -1)
		throw BadOption(string("file ") + str + " must exist");
	m_value = str;
	return true;
}


void OptionParser::addWithoutAna(Option* o)
{
	const vector<char>& shorts = o->shortNames;
	for (vector<char>::const_iterator i = shorts.begin(); i != shorts.end(); i++)
	{
		map<char, Option*>::iterator j = m_short.find(*i);
		if (j != m_short.end())
			throw ConsistencyCheckException(
					string("short option ") + *i + " is already mapped to " + j->second->name());
		m_short[*i] = o;
	}

	const vector<string>& longs = o->longNames;
	for (vector<string>::const_iterator i = longs.begin(); i != longs.end(); i++)
	{
		map<string, Option*>::iterator j = m_long.find(*i);
		if (j != m_long.end())
			throw ConsistencyCheckException(
					string("long option ") + *i + " is already mapped to " + j->second->name());
		m_long[*i] = o;
	}
}

void OptionParser::add(Option* o)
{
	addWithoutAna(o);
	m_options.push_back(o);	
}

void OptionParser::add(OptionGroup* group)
{
	const vector<Option*>& v = group->options;
	for (vector<Option*>::const_iterator i = v.begin(); i != v.end(); i++)
		addWithoutAna(*i);
	m_groups.push_back(group);	
}

iter OptionParser::parseConsecutiveSwitches(arglist& list, iter begin)
{
	while (begin != list.end() && isSwitch(*begin))
	{
		const char* opt = *begin;

		// We take care of this switch, then
		arglist::iterator next = begin; ++next;
		list.erase(begin);
		begin = next;
		
		if (opt[1] != '-')
		{
			// Short option
			for (const char* s = opt + 1; *s; ++s)
			{
				map<char, Option*>::const_iterator parser = m_short.find(*s);
				if (parser == m_short.end())
					throw BadOption(string("unknown short option -") + *s);
				// Remove the argument if it gets used
				if (parser->second->parse(*begin))
				{
					next = begin;
					++next;
					list.erase(begin);
					begin = next;
				}
			}
		} else {
			// Long option

			// Split option and argument from "--foo=bar"
			const char* sep = strchr(opt, '=');
			string name;
			if (sep == NULL)
			{
				// No argument
				name = string(opt + 2);
			} else {
				name = string(opt, 2, sep-opt-2);
				++sep;
			}

			map<string, Option*>::const_iterator parser = m_long.find(name);
			if (parser == m_long.end())
				throw BadOption(string("unknown long option --") + name);
			parser->second->parse(sep);
		}
	}

	return begin;
}

iter OptionParser::parse(arglist& list, iter begin)
{
	bool foundNonSwitches = false;
	iter firstNonSwitch;
	while (begin != list.end())
	{
		// Parse a row of switches
		begin = parseConsecutiveSwitches(list, begin);

		// End of switches?
		if (begin == list.end())
			break;

		// If the end is the "--" marker, take it out of the list as well
		if (strcmp(*begin, "--") == 0)
		{
			iter next = begin; ++next;
			list.erase(begin);
			begin = next;
			break;
		}

		if (!foundNonSwitches)
		{
			// Mark the start of non-switches if we haven't done it already
			firstNonSwitch = begin;
			foundNonSwitches = true;
		}
		while (begin != list.end() && !isSwitch(*begin))
			++begin;
	}
	return foundNonSwitches ? firstNonSwitch : begin;
}



void CommandParser::add(const std::string& alias, OptionParser* o)
{
	map<string, OptionParser*>::iterator a = m_aliases.find(alias);
	if (a != m_aliases.end())
		throw ConsistencyCheckException("command " + alias + " has already been set to " + a->second->name());
	m_aliases[alias] = o;
}

void CommandParser::add(OptionParser& o)
{
	add(o.primaryAlias, &o);
	for (vector<string>::const_iterator i = o.aliases.begin();
			i != o.aliases.end(); ++i)
		add(*i, &o);
}

OptionParser* CommandParser::command(const std::string& name) const
{
	map<string, OptionParser*>::const_iterator i = m_aliases.find(name);
	if (i == m_aliases.end())
		return 0;
	else
		return i->second;
}

iter CommandParser::parse(arglist& list, iter begin)
{
	// Look for the first non-switch in the list
	iter cmd = begin;
	while (cmd != list.end() && isSwitch(*cmd))
		++cmd;

	map<string, OptionParser*>::iterator a;
	if (cmd == list.end())
	{
		// No command has been found.  Try to see if there is a 'generic'
		// OptionParser with the "" alias.
		if ((a = m_aliases.find(string())) != m_aliases.end())
			;
		else
			return begin;
	} else {
		// Handle the command
		a = m_aliases.find(*cmd);
		if (a == m_aliases.end())
			throw BadOption("unknown command " + string(*cmd));

		// Remove the command from list
		if (cmd == begin)
			++begin;
		list.erase(cmd);
	}

	m_last_command = a->second;

	// Invoke the selected parser on the list
	return a->second->parse(list, begin);
}

std::map<std::string, OptionParser*> CommandParser::getCommandInfo() const
{
	std::map<std::string, OptionParser*> info;
	
	// Dig informations from cp
	for(map<string, OptionParser*>::const_iterator i = m_aliases.begin();
			i != m_aliases.end(); i++)
	{
		if (i->first == string())
			continue;
		map<string, OptionParser*>::iterator j = info.find(i->second->name());
		if (j == info.end())
			info[i->second->name()] = i->second;
	}

	return info;
}



class WordWrapper
{
	const std::string& s;
	size_t cursor;

public:
	WordWrapper(const std::string& s) : s(s), cursor(0) {}

	void restart() { cursor = 0; }

	bool hasData() const { return cursor < s.size(); }

	string get(unsigned int width)
	{
		if (cursor >= s.size())
			return "";

		// Find the last work break before `width'
		unsigned int brk = cursor;
		for (unsigned int j = cursor; j < s.size() && j < cursor + width; j++)
		{
			if (s[j] == '\n')
			{
				brk = j;
				break;
			} else if (!isspace(s[j]) && (j + 1 == s.size() || isspace(s[j + 1])))
				brk = j + 1;
		}
		if (brk == cursor)
			brk = cursor + width;

		string res;
		if (brk >= s.size())
		{
			res = string(s, cursor, string::npos);
			cursor = s.size();
		} else {
			res = string(s, cursor, brk - cursor);
			cursor = brk;
			while (cursor < s.size() && isspace(s[cursor]))
				cursor++;
		}
		return res;
	}
};

class HelpWriter
{
	// Width of the console
	std::ostream& out;
	int m_width;

public:
	HelpWriter(std::ostream& out);

	// Write 'size' spaces to out
	void pad(size_t size);

	// Output an item from a list.  The first bulletsize columns will be used to
	// output bullet, the rest will have text, wordwrapped and properly aligned
	void outlist(const std::string& bullet, size_t bulletsize, const std::string& text);

	void outstring(const std::string& str);
};

HelpWriter::HelpWriter(std::ostream& out) : out(out)
{
	char* columns = getenv("COLUMNS");
	m_width = columns ? atoi(columns) : 80;
}

void HelpWriter::pad(size_t size)
{
	for (size_t i = 0; i < size; i++) out << " ";
}

void HelpWriter::outlist(const std::string& bullet, size_t bulletsize, const std::string& text)
{
	WordWrapper wrapper(text);
	size_t rightcol = m_width - bulletsize;

	out << bullet;
	pad(bulletsize - bullet.size());
	out << wrapper.get(rightcol);
	out << endl;

	while (wrapper.hasData())
	{
		pad(bulletsize);
		out << wrapper.get(rightcol);
		out << endl;
	}
}

void HelpWriter::outstring(const std::string& str)
{
	WordWrapper wrapper(str);

	while (wrapper.hasData())
	{
		out << wrapper.get(m_width);
		out << endl;
	}
}

void Help::outputVersion(std::ostream& out)
{
	out << m_app << " version " << m_ver << endl;
}

void Help::outputHelp(std::ostream& out, const CommandParser& cp)
{
	// Dig informations from cp
	std::map<std::string, OptionParser*> m_info = cp.getCommandInfo();
	
	HelpWriter writer(out);

	// Compute the maximum length of alias names
	size_t maxAliasSize = 0;
	for (map<string, OptionParser*>::const_iterator i = m_info.begin();
			i != m_info.end(); i++)
	{
		const string& str = i->second->primaryAlias;
		if (maxAliasSize < str.size())
			maxAliasSize = str.size();
	}

	out << "Usage: " << m_app << " [options] " << cp.usage << endl;
	out << endl;
	writer.outstring("Description: " + cp.description);
	out << endl;
	out << "Commands are:" << endl;
	out << endl;

	// Print the list
	for (map<string, OptionParser*>::const_iterator i = m_info.begin();
			i != m_info.end(); i++)
	{
		string aliases;
		const vector<string>& v = i->second->aliases;
		if (!v.empty())
		{
			aliases += "  May also be invoked as ";
			for (vector<string>::const_iterator j = v.begin();
					j != v.end(); j++)
				if (j == v.begin())
					aliases += *j;
				else
					aliases += " or " + *j;
			aliases += ".";
		}

		writer.outlist(" " + i->second->primaryAlias, maxAliasSize + 3, i->second->description + "." + aliases);
	}

	out << endl;
}

void Help::outputHelp(std::ostream& out, const OptionParser& o)
{
	HelpWriter writer(out);

	out << "Usage: " << m_app << " [options] " << o.primaryAlias << " [options] " << o.usage << endl;
	out << endl;
	if (!o.aliases.empty())
	{
		out << "Command aliases: ";
		for (vector<string>::const_iterator i = o.aliases.begin();
				i != o.aliases.end(); i++)
			if (i == o.aliases.begin())
				out << *i;
			else
				out << ", " << *i;
		out << "." << endl;
		out << endl;
	}
	writer.outstring("Description: " + o.description);

	// Compute size of option display
	size_t maxLeftCol = 0;
	for (vector<OptionGroup*>::const_iterator i = o.groups().begin();
			i != o.groups().end(); i++)
		for (vector<Option*>::const_iterator j = (*i)->options.begin();
				j != (*i)->options.end(); j++)
		{
			size_t w = (*j)->fullUsage().size();
			if (w > maxLeftCol)
				maxLeftCol = w;
		}
	for (vector<Option*>::const_iterator j = o.options().begin();
			j != o.options().end(); j++)
	{
		size_t w = (*j)->fullUsage().size();
		if (w > maxLeftCol)
			maxLeftCol = w;
	}

	if (maxLeftCol)
	{
		// Output the options
		out << endl;
		out << "Options are:" << endl;
		for (vector<OptionGroup*>::const_iterator i = o.groups().begin();
				i != o.groups().end(); i++)
		{
			if (!(*i)->description.empty())
			{
				out << endl;
				writer.outstring((*i)->description + ":");
				out << endl;
			}
			for (vector<Option*>::const_iterator j = (*i)->options.begin();
					j != (*i)->options.end(); j++)
				writer.outlist(" " + (*j)->fullUsage(), maxLeftCol + 3, (*j)->description);
		}
		if (!o.options().empty())
		{
			out << endl;
			writer.outstring("Other options:");
			out << endl;
			for (vector<Option*>::const_iterator j = o.options().begin();
					j != o.options().end(); j++)
				writer.outlist(" " + (*j)->fullUsage(), maxLeftCol + 3, (*j)->description);
		}
	}
	out << endl;
}

static string toupper(const std::string& str)
{
	string res;
	for (size_t i = 0; i < str.size(); i++)
		res += ::toupper(str[i]);
	return res;
}

static string man_date()
{
	time_t tnow = time(0);
	struct tm* now = gmtime(&tnow);
	char buf[20];
	const char* oldlocale = setlocale(LC_TIME, "C");
	string res(buf, strftime(buf, 20, "%B %d, %Y", now));
	setlocale(LC_TIME, oldlocale);
	return res;
}

void Manpage::outputParagraph(std::ostream& out, const std::string& str)
{
	for (size_t i = 0; i < str.size(); i++)
		switch (str[i])
		{
			case '-':
				out << "\\-";
				break;
			case '\n':
				out << "\n.br\n";
				break;
			default:
				out << str[i];
		}		
	out << '\n';
}

void Manpage::outputOption(std::ostream& out, const Option* o)
{
	out << ".TP" << endl;
	out << ".B " << o->fullUsageForMan() << endl;
	out << o->description << "." << endl;
}

void Manpage::runHooks(std::ostream& out, const std::string& section, where where)
{
	for (std::vector<Hook>::const_iterator i = hooks.begin();
			i != hooks.end(); i++)
		if (i->section == section && i->placement == where)
			out << i->text;
}

void Manpage::startSection(std::ostream& out, const std::string& name)
{
	runHooks(out, name, BEFORE);
	out << ".SH " << name << endl;
	runHooks(out, name, BEGINNING);
	lastSection = name;
}

void Manpage::endSection(std::ostream& out)
{
	runHooks(out, lastSection, END);
	lastSection.clear();
}

void Manpage::output(std::ostream& out, const CommandParser& cp, int section)
{
	std::map<std::string, OptionParser*> m_info = cp.getCommandInfo();

	// Manpage header
	out << ".TH " << toupper(m_app) << " " << section << " \"" << man_date() << "\" \"" << m_ver << "\"" << endl;

	startSection(out, "NAME");
	out << cp.name() << " \\- " << cp.description << endl;
	endSection(out);

	startSection(out, "SYNOPSIS");
	out << "\\fB" << cp.name() << "\\fP [options] " << cp.usage << endl;
	endSection(out);

	startSection(out, "DESCRIPTION");
	if (!cp.longDescription.empty())
		outputParagraph(out, cp.longDescription);
	endSection(out);

	startSection(out, "COMMANDS");
	out << "\\fB" << cp.name() << "\\fP always requires a non-switch argument, that indicates what is the operation that should be performed:" << endl;
	for (map<string, OptionParser*>::const_iterator i = m_info.begin();
			i != m_info.end(); i++)
	{
		out << ".TP" << endl;
		out << "\\fB" << i->second->primaryAlias << "\\fP";

		const vector<string>& v = i->second->aliases;
		for (vector<string>::const_iterator j = v.begin(); j != v.end(); j++)
			out << " or \\fB" << *j << "\\fP";
		
		out << " " << i->second->usage << endl;
		out << ".br" << endl;
		if (i->second->longDescription.empty())
			outputParagraph(out, i->second->description);
		else
			outputParagraph(out, i->second->longDescription);
	}
	endSection(out);

	startSection(out, "OPTIONS");
	out << "This program follows the usual GNU command line syntax, with long options starting with two dashes (`\\-')." << endl << endl;
	out << "Every one of the commands listed above has its own set of options.  To keep this manpage readable, all the options are presented together.  Please refer to \"\\fB" << cp.name() << "\\fP help \\fIcommand\\fP\" to see which options are accepted by a given command." << endl;

	// Harvest merged option informations
	set<OptionGroup*> groups;
	vector<Option*> options;
	for(map<string, OptionParser*>::const_iterator i = m_info.begin();
			i != m_info.end(); i++)
	{
		if (i->first == string())
			continue;
		OptionParser& o = *i->second;

		for (vector<OptionGroup*>::const_iterator i = o.groups().begin();
				i != o.groups().end(); i++)
			groups.insert(*i);
		for (vector<Option*>::const_iterator j = o.options().begin();
				j != o.options().end(); j++)
			options.push_back(*j);
	}

	for (set<OptionGroup*>::const_iterator i = groups.begin();
			i != groups.end(); i++)
	{
		if (!(*i)->description.empty())
			out << endl << (*i)->description << ":" << endl;
		for (vector<Option*>::const_iterator j = (*i)->options.begin();
				j != (*i)->options.end(); ++j)
			outputOption(out, *j);
		out << ".PP" << endl;
	}

	if (!options.empty())
	{
		out << endl;
		out << "Other options:" << endl;
		for (vector<Option*>::const_iterator j = options.begin();
				j != options.end(); ++j)
			outputOption(out, *j);
	}
	endSection(out);

	startSection(out, "AUTHOR");
	out << "\\fB" << cp.name() << "\\fP is maintained by " << PACKAGE_BUGREPORT << "." << endl << endl;
	out << "This manpage has been automatically generated by the " << m_app << " program." << endl;
	endSection(out);
}

void Manpage::output(std::ostream& out, const OptionParser& o, int section)
{
	// Manpage header
	out << ".TH " << toupper(m_app) << " " << section << " \"" << man_date() << "\" \"" << m_ver << "\"" << endl;

	startSection(out, "NAME");
	out << o.name() << " \\- " << o.description << endl;
	endSection(out);

	startSection(out, "SYNOPSIS");
	out << "\\fB" << m_app << "\\fP [options] " << o.usage << endl;
	endSection(out);

	startSection(out, "DESCRIPTION");
	if (!o.longDescription.empty())
		outputParagraph(out, o.longDescription);
	endSection(out);

	startSection(out, "OPTIONS");
	out << "This program follows the usual GNU command line syntax, with long options starting with two dashes (`\\-')." << endl << endl;

	// Harvest merged option informations
	for (vector<OptionGroup*>::const_iterator i = o.groups().begin();
			i != o.groups().end(); i++)
	{
		if (!(*i)->description.empty())
			out << endl << (*i)->description << ":" << endl;
		for (vector<Option*>::const_iterator j = (*i)->options.begin();
				j != (*i)->options.end(); ++j)
			outputOption(out, *j);
		out << ".PP" << endl;
	}

	if (!o.options().empty())
	{
		out << endl;
		out << "Other options:" << endl;
		for (vector<Option*>::const_iterator j = o.options().begin();
				j != o.options().end(); ++j)
			outputOption(out, *j);
	}
	endSection(out);

	startSection(out, "AUTHOR");
	out << "\\fB" << o.name() << "\\fP is maintained by " << PACKAGE_BUGREPORT << "." << endl << endl;
	out << "This manpage has been automatically generated by the " << m_app << " program." << endl;
	endSection(out);
}

static string readline(FILE* in)
{
	string res;
	int c;
	while ((c = getc(in)) != EOF && c != '\n')
		res += c;
	return res;
}

void Manpage::readHooks(const std::string& file)
{
	FILE* in = fopen(file.c_str(), "r");
	if (!in) throw SystemException(errno, "opening file " + file);
	string section;
	commandline::Manpage::where placement = commandline::Manpage::BEFORE;
	string text;
	while (!feof(in))
	{
		string line(readline(in));
		if (line.empty())
			continue;
		if (line[0] == '|')
		{
			text += line.substr(1) + "\n";
		}
		else if (isalpha(line[0]))
		{
			if (!section.empty())
			{
				addHook(section, placement, text);
				text.clear();
			}
			size_t sep = line.find(' ');
			if (sep == string::npos)
			{
				fclose(in);
				throw ConsistencyCheckException("expected two words in line: " + line);
			}
			section = line.substr(0, sep);
			string w(line, sep+1);
			if (w == "before")
			{
				placement = commandline::Manpage::BEFORE;
			} else if (w == "beginning") {
				placement = commandline::Manpage::BEGINNING;
			} else if (w == "end") {
				placement = commandline::Manpage::END;
			} else {
				fclose(in);
				throw ConsistencyCheckException("expected 'before', 'beginning' or 'end' in line: " + line);
			}
		}
	}
	if (!section.empty())
		addHook(section, placement, text);
	fclose(in);
}

}
}


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>

namespace tut {
using namespace tut_tagcoll;

struct tagcoll_commandline_shar {
};
TESTGRP(tagcoll_commandline);

using namespace Tagcoll::commandline;

// Test isSwitch
template<> template<>
void to::test<1>()
{
	gen_ensure_equals(isSwitch("-a"), true);
	gen_ensure_equals(isSwitch("-afdg"), true);
	gen_ensure_equals(isSwitch("--antani"), true);
	gen_ensure_equals(isSwitch("--antani-blinda"), true);
	gen_ensure_equals(isSwitch("-"), false);
	gen_ensure_equals(isSwitch("--"), false);
	gen_ensure_equals(isSwitch("antani"), false);
	gen_ensure_equals(isSwitch("a-ntani"), false);
	gen_ensure_equals(isSwitch("a--ntani"), false);
}

// Test BoolOption
template<> template<>
void to::test<2>()
{
	BoolOption opt("test");

	gen_ensure_equals(opt.name(), string("test"));
	gen_ensure_equals(opt.boolValue(), false);
	gen_ensure_equals(opt.stringValue(), string("false"));

	gen_ensure_equals(opt.parse(0), false);
	gen_ensure_equals(opt.boolValue(), true);
	gen_ensure_equals(opt.stringValue(), string("true"));
}

// Test StringOption
template<> template<>
void to::test<3>()
{
	StringOption opt("test");

	gen_ensure_equals(opt.name(), string("test"));
	gen_ensure_equals(opt.boolValue(), false);
	gen_ensure_equals(opt.stringValue(), string());

	gen_ensure_equals(opt.parse("-a"), true);
	gen_ensure_equals(opt.boolValue(), true);
	gen_ensure_equals(opt.stringValue(), "-a");
}

class TestParser : public OptionParser
{
public:
	TestParser() :
		OptionParser("test"),
		antani("antani"),
		blinda("blinda")
	{
		antani.addAlias('a');
		antani.addAlias("antani");
		antani.addAlias("an-tani");

		blinda.addAlias('b');
		blinda.addAlias("blinda");

		add(&antani);
		add(&blinda);
	}

	BoolOption antani;
	StringOption blinda;
};

// Test OptionParser
template<> template<>
void to::test<5>()
{
	{
		list<const char*> opts;
		opts.push_back("ciaps");
		opts.push_back("-b");
		opts.push_back("cippo");
		opts.push_back("foobar");

		TestParser parser;
		iter i = parser.parseList(opts);
		gen_ensure(i == opts.begin());
		gen_ensure_equals(opts.size(), 2u);
		gen_ensure_equals(string(*opts.begin()), string("ciaps"));
		gen_ensure_equals(string(*opts.rbegin()), string("foobar"));
		gen_ensure_equals(parser.antani.boolValue(), false);
		gen_ensure_equals(parser.blinda.stringValue(), "cippo");
	}
	{
		list<const char*> opts;
		opts.push_back("-a");
		opts.push_back("foobar");

		TestParser parser;
		iter i = parser.parseList(opts);
		gen_ensure(i == opts.begin());
		gen_ensure_equals(opts.size(), 1u);
		gen_ensure_equals(string(*opts.begin()), string("foobar"));
		gen_ensure_equals(parser.antani.boolValue(), true);
		gen_ensure_equals(parser.blinda.boolValue(), false);
	}
	{
		list<const char*> opts;
		opts.push_back("-ab");
		opts.push_back("cippo");

		TestParser parser;
		iter i = parser.parseList(opts);
		gen_ensure(i == opts.end());
		gen_ensure_equals(opts.size(), 0u);
		gen_ensure_equals(parser.antani.boolValue(), true);
		gen_ensure_equals(parser.blinda.stringValue(), "cippo");
	}
	{
		list<const char*> opts;
		opts.push_back("--an-tani");
		opts.push_back("foobar");

		TestParser parser;
		iter i = parser.parseList(opts);
		gen_ensure(i == opts.begin());
		gen_ensure_equals(opts.size(), 1u);
		gen_ensure_equals(string(*opts.begin()), string("foobar"));
		gen_ensure_equals(parser.antani.boolValue(), true);
		gen_ensure_equals(parser.blinda.boolValue(), false);
	}
	{
		list<const char*> opts;
		opts.push_back("--blinda=cippo");
		opts.push_back("foobar");
		opts.push_back("--antani");

		TestParser parser;
		iter i = parser.parseList(opts);
		gen_ensure(i == opts.begin());
		gen_ensure_equals(opts.size(), 1u);
		gen_ensure_equals(string(*opts.begin()), string("foobar"));
		gen_ensure_equals(parser.antani.boolValue(), true);
		gen_ensure_equals(parser.blinda.stringValue(), "cippo");
	}
}

class TestCParser : public CommandParser
{
public:
	class Scramble : public OptionParser
	{
	public:
		Scramble() :
			OptionParser("scramble"),
			random("random"),
			yell("yell")
		{
			random.addAlias('r');
			random.addAlias("random");
			yell.addAlias("yell");
			add(&random);
			add(&yell);
			aliases.push_back("mess");
		}

		BoolOption random;
		StringOption yell;
	};
	class Fix : public OptionParser
	{
	public:
		Fix() :
			OptionParser("fix"),
			quick("quick"),
			yell("yell")
		{
			quick.addAlias('Q');
			quick.addAlias("quick");
			yell.addAlias("yell");
			add(&quick);
			add(&yell);
		}

		BoolOption quick;
		StringOption yell;
	};

	TestCParser() :
		CommandParser("test")
	{
		add(scramble);
		add(fix);
	}

	Scramble scramble;
	Fix fix;
};

// Test CommandParser
template<> template<>
void to::test<6>()
{
	{
		list<const char*> opts;
		opts.push_back("--yell=foo");
		opts.push_back("mess");
		opts.push_back("-r");

		TestCParser parser;
		iter i = parser.parseList(opts);
		gen_ensure(i == opts.end());
		gen_ensure_equals(opts.size(), 0u);
		gen_ensure_equals(parser.lastCommand(), &parser.scramble);
		gen_ensure_equals(parser.scramble.yell.stringValue(), "foo");
		gen_ensure_equals(parser.scramble.random.boolValue(), true);
		gen_ensure_equals(parser.fix.yell.stringValue(), string());
		gen_ensure_equals(parser.fix.quick.boolValue(), false);
	}
	{
		list<const char*> opts;
		opts.push_back("--yell=foo");
		opts.push_back("fix");
		opts.push_back("-Q");

		TestCParser parser;
		iter i = parser.parseList(opts);
		gen_ensure(i == opts.end());
		gen_ensure_equals(opts.size(), 0u);
		gen_ensure_equals(parser.lastCommand(), &parser.fix);
		gen_ensure_equals(parser.scramble.yell.stringValue(), string());
		gen_ensure_equals(parser.scramble.random.boolValue(), false);
		gen_ensure_equals(parser.fix.yell.stringValue(), "foo");
		gen_ensure_equals(parser.fix.quick.boolValue(), true);
	}
}

}

#endif

// vim:set ts=4 sw=4:
