/*
 * Fast index for tag data, using a read-only database
 *
 * Copyright (C) 2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/TDBReadonlyDiskIndex.h>


#ifndef INSTANTIATING_TEMPLATES
#include <string>

namespace Tagcoll {
	template class TDBReadonlyDiskIndex<std::string, std::string>;
}
#endif


#ifdef COMPILE_TESTSUITE

#include <tests/test-utils.h>
#include <tagcoll/TDBIndexer.h>

namespace tut {
using namespace tut_tagcoll;

struct tagcoll_tdbreadonlydiskindex_shar {
};
TESTGRP(tagcoll_tdbreadonlydiskindex);

template<> template<>
void to::test<1>()
{
	TrivialConverter<string, string> a;

	// Create the collection on disk
	{
		TDBIndexer<string, string> rwcoll;
		output_test_collection(rwcoll);
		rwcoll.writeIndex(a, a, "pkgidx.test", "tagidx.test");
	}

	TDBReadonlyDiskIndex<string, string> coll("pkgidx.test", "tagidx.test", a, a, a, a);
	test_readonly_collection(coll);

	unlink("pkgidx.test");
	unlink("tagidx.test");
}

template<> template<>
void to::test<2>()
{
	TrivialConverter<string, string> a;

	{
		// Create a real disk index
		Tagcoll::TDBDiskIndex<string, string> di(
				"pkgidx.test", "tagidx.test",
				a, a, a, a);
	}

	TDBReadonlyDiskIndex<string, string> coll("pkgidx.test", "tagidx.test", a, a, a, a);

	test_collection(coll);

	unlink("pkgidx.test");
	unlink("tagidx.test");
}

template<> template<>
void to::test<3> ()
{
	TrivialConverter<string, string> conv;
	PatchList<string, string> changes;
	try {
		{
			// Create a real disk index
			Tagcoll::TDBDiskIndex<string, string> di(
					"test-ro-pkg.tdb", "test-ro-tag.tdb",
					conv, conv, conv, conv);
		}
		
		// An empty database should return empty sets, but not fail
		Tagcoll::TDBReadonlyDiskIndex<string, string> tfi(
				"test-ro-pkg.tdb", "test-ro-tag.tdb",
				conv, conv, conv, conv);
		gen_ensure(!tfi.hasTag("cippo"));
		gen_ensure(!tfi.hasTag("lippo"));
		gen_ensure(tfi.getItems("cippo") == Tagcoll::OpSet<string>());
		gen_ensure(tfi.getTags("cippo") == Tagcoll::OpSet<string>());
		gen_ensure(tfi.getItems("lippo") == Tagcoll::OpSet<string>());
		gen_ensure(tfi.getTags("lippo") == Tagcoll::OpSet<string>());
		// No autovivification should happen on gets
		gen_ensure(!tfi.hasTag("cippo"));
		gen_ensure(!tfi.hasTag("lippo"));

		// Adding an item with no tags should still return an empty set for
		// that item 
		tfi.consume("cippo");
		gen_ensure(!tfi.hasTag("cippo"));
		gen_ensure(tfi.getItems("cippo") == Tagcoll::OpSet<string>());
		gen_ensure(tfi.getTags("cippo") == Tagcoll::OpSet<string>());

		// Add some data and check that it comes back
		Tagcoll::OpSet<string> s;
		s += "lippo";
		s += "zippo";
		s += "rippo";

		tfi.consume("cippo", s);
		gen_ensure(tfi.hasTag("lippo"));
		gen_ensure(tfi.hasTag("zippo"));
		gen_ensure(tfi.hasTag("rippo"));
		s = tfi.getTags("cippo");
		gen_ensure(s.contains("lippo"));
		gen_ensure(s.contains("zippo"));
		gen_ensure(s.contains("rippo"));
		gen_ensure(!s.contains("bippo"));

		s = tfi.getItems("lippo");
		gen_ensure(s.contains("cippo"));
		gen_ensure(!s.contains("lippo"));

		tfi.consume("untagged");

		/* Save the patchlist */
		changes = tfi.getChanges();
	} catch (Exception& e) {
		fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
		throw;
	}

	// Reopen the database to see if data is actually persisting
	try {
		Tagcoll::TDBReadonlyDiskIndex<string, string> tfi(
				"test-ro-pkg.tdb", "test-ro-tag.tdb",
				conv, conv, conv, conv);

		/* Restore the patchlist */
		tfi.setChanges(changes);

		// This currently fails, until we find a way to store untagged
		// added items in the output patch (probably, that'd be allowing a
		// patch to ADD and REMOVE items)
		//gen_ensure(tfi.hasItem("untagged"));

		Tagcoll::OpSet<string> s = tfi.getTags("cippo");

		// Repeat the same queries: they should work
		gen_ensure(s.contains("lippo"));
		gen_ensure(s.contains("zippo"));
		gen_ensure(s.contains("rippo"));
		gen_ensure(!s.contains("bippo"));

		s = tfi.getItems("lippo");
		gen_ensure(s.contains("cippo"));
		gen_ensure(!s.contains("lippo"));
		//gen_ensure(s.contains("untagged"));
	} catch (Exception& e) {
		fprintf(stderr, "%s: %.*s\n", e.type(), PFSTR(e.desc()));
		throw;
	}
}

}

#endif
// vim:set ts=4 sw=4:
