/*
 * Classes handling tag patches
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifndef TAGCOLL_PATCHES_CPP
#define TAGCOLL_PATCHES_CPP

#include <tagcoll/patch.h>

using namespace std;
using namespace wibble::operators;

namespace tagcoll {

template <class ITEM, class TAG>
void PatchList<ITEM, TAG>::addPatch(const Patch<ITEM, TAG>& patch)
{
	// Filter out empty patches
	if (patch.added.empty() && patch.removed.empty())
		return;

	iterator i = this->find(patch.getItem());
	if (i == this->end())
		insert(make_pair<ITEM, Patch<ITEM, TAG> >(patch.getItem(), patch));
	else
		i->second.mergeWith(patch);
}

template <class ITEM, class TAG>
void PatchList<ITEM, TAG>::addPatch(const PatchList<ITEM, TAG>& patches)
{
	for (typename PatchList<ITEM, TAG>::const_iterator i = patches.begin();
			i != patches.end(); i++)
		addPatch(i->second);
}

template <class ITEM, class TAG> template<typename COLL1, typename COLL2>
void PatchList<ITEM, TAG>::addPatch(const COLL1& im1, const COLL2& im2)
{
	for (typename COLL1::const_iterator i1 = im1.begin();
			i1 != im1.end(); ++i1)
	{
		std::set<TAG> ts2 = im2.getTags(i1->first);
		std::set<TAG> added = ts2 - i1->second;
		std::set<TAG> removed = i1->second - ts2;
		if (!added.empty() || !removed.empty())
			addPatch(Patch<ITEM, TAG>(i1->first, added, removed));
	}
}

template <class ITEM, class TAG>
std::set<TAG> PatchList<ITEM, TAG>::patch(const ITEM& item, const std::set<TAG>& tagset) const
{
	// Find the patch record for this item
	const_iterator p = this->find(item);
	if (p == this->end())
		// If there are no patches, return the tagset unchanged
		return tagset;

	// There are patches: apply them:
	return p->second.apply(tagset);
}

template <class ITEM, class TAG>
PatchList<ITEM, TAG> PatchList<ITEM, TAG>::getReverse() const
{
	PatchList<ITEM, TAG> res;
	for (typename PatchList<ITEM, TAG>::const_iterator i = this->begin();
			i != this->end(); i++)
		res.addPatch(i->second.getReverse());
	return res;
}


/*
template <class ITEM>
void PatchList<ITEM>::consume(const ITEM& item, const std::set<string>& tags)
{
	patches.insert(make_pair(item, tags));
}

// Output the patch list to a TagcollConsumer
template <class ITEM>
void PatchList<ITEM>::output(TagcollConsumer<ITEM, std::string>& consumer) const
{
	for (typename map< ITEM, std::set<string> >::const_iterator i = patches.begin();
			i != patches.end(); i++)
		if (i->second.size() == 0)
			consumer.consume(i->first);
		else
			consumer.consume(i->first, i->second);
}
*/

}

#endif

// vim:set ts=4 sw=4:
