# -*- coding: utf-8 -*-
#
# Authors: Natalia B. Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# In addition, as a special exception, the copyright holders give
# permission to link the code of portions of this program with the
# OpenSSL library under certain conditions as described in each
# individual source file, and distribute linked combinations
# including the two.
# You must obey the GNU General Public License in all respects
# for all of the code used other than OpenSSL.  If you modify
# file(s) with this exception, you may extend this exception to your
# version of the file(s), but you are not obligated to do so.  If you
# do not wish to do so, delete this exception statement from your
# version.  If you delete this exception statement from all source
# files in the program, then also delete it here.
"""Platform independent tests for the XDG constants."""

import os

from twisted.trial.unittest import TestCase

from ubuntu_sso import xdg_base_directory


class TestBaseDirectory(TestCase):
    """Tests for the BaseDirectory module."""

    def assert_utf8_bytes(self, value):
        """Check that 'value' is a bytes sequence encoded with utf-8."""
        self.assertIsInstance(value, str)
        try:
            value.decode('utf-8')
        except UnicodeDecodeError:
            self.fail('%r should be a utf8 encoded string.' % value)

    def test_xdg_home_is_utf8_bytes(self):
        """The returned path is bytes."""
        actual = xdg_base_directory.xdg_home
        self.assert_utf8_bytes(actual)

    def test_xdg_cache_home_is_utf8_bytes(self):
        """The returned path is bytes."""
        actual = xdg_base_directory.xdg_cache_home
        self.assert_utf8_bytes(actual)

    def test_xdg_config_home_is_utf8_bytes(self):
        """The returned path is bytes."""
        actual = xdg_base_directory.xdg_config_home
        self.assert_utf8_bytes(actual)

    def test_xdg_config_dirs_are_bytes(self):
        """The returned path is bytes."""
        result = xdg_base_directory.xdg_config_dirs
        for actual in result:
            self.assert_utf8_bytes(actual)

    def test_xdg_data_home_is_utf8_bytes(self):
        """The returned path is bytes."""
        actual = xdg_base_directory.xdg_data_home
        self.assert_utf8_bytes(actual)

    def test_xdg_data_dirs_are_bytes(self):
        """The returned path is bytes."""
        result = xdg_base_directory.xdg_data_dirs
        for actual in result:
            self.assert_utf8_bytes(actual)

    def test_load_config_paths_filter(self):
        """Since those folders don't exist, this should be empty."""
        self.assertEqual(list(xdg_base_directory.load_config_paths("x")), [])

    def test_save_config_path(self):
        """The path should end with xdg_config/x (respecting the separator)."""
        self.patch(os, "makedirs", lambda *args: None)
        result = xdg_base_directory.save_config_path("x")
        self.assertEqual(result.split(os.sep)[-2:], ['xdg_config', 'x'])
