/*
 * UbuntuOne Nautilus plugin
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2009-2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <glib/gi18n-lib.h>
#include <libsyncdaemon/libsyncdaemon.h>
#include "share-dialog.h"

G_DEFINE_TYPE(ShareDialog, share_dialog, GTK_TYPE_DIALOG)

static void
share_dialog_finalize (GObject *object)
{
	ShareDialog *dialog = SHARE_DIALOG (object);

	if (dialog->path != NULL)
		g_free (dialog->path);

	G_OBJECT_CLASS (share_dialog_parent_class)->finalize (object);
}

static void
share_dialog_class_init (ShareDialogClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = share_dialog_finalize;
}

static void
dialog_response_cb (GtkDialog *gtk_dialog,
                    gint response,
                    gpointer user_data)
{
	ShareDialog *dialog = SHARE_DIALOG (gtk_dialog);

	switch (response) {
	case GTK_RESPONSE_ACCEPT: {
        const gchar *entry_data;

        entry_data = gtk_entry_get_text (GTK_ENTRY (dialog->user_picker));
        if (entry_data != NULL && strlen (entry_data) > 0) {
            SyncdaemonSharesInterface *interface;
            gchar **email_entries;
            GSList *emails = NULL;
            gboolean allow_mods = FALSE;
            gint i;

            email_entries = g_strsplit (entry_data, ",", -1);
            for (i = 0; email_entries[i] != NULL; i++) {
                const gchar *email = g_strstrip (email_entries[i]);
                emails = g_slist_prepend (emails, g_strdup (email));
            }
            g_strfreev (email_entries);

            if (emails != NULL && g_slist_length (emails) > 0) {
                allow_mods = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->allow_mods));

                interface = SYNCDAEMON_SHARES_INTERFACE (syncdaemon_daemon_get_shares_interface (dialog->uon->syncdaemon));
                if (interface != NULL) {
                    syncdaemon_shares_interface_create (interface,
                                                        dialog->path,
                                                        emails,
                                                        g_path_get_basename (dialog->path),
                                                        allow_mods);
                }
            }
        }
    }
	default:
		gtk_widget_destroy (GTK_WIDGET (dialog));
		break;
	}
}

static void
share_dialog_init (ShareDialog *dialog)
{
	GtkWidget *area, *table, *hbox, *label;

	gtk_window_set_title (GTK_WINDOW (dialog), _("Share on Ubuntu One"));
	gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
	gtk_dialog_add_buttons (GTK_DIALOG (dialog),
				GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				(_("Share")), GTK_RESPONSE_ACCEPT,
				NULL);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_ACCEPT);
	gtk_window_set_icon_name (GTK_WINDOW (dialog), "ubuntuone");
	g_signal_connect (G_OBJECT (dialog), "response",
			  G_CALLBACK (dialog_response_cb), NULL);

	area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));

    table = gtk_vbox_new (FALSE, 12);
    gtk_container_set_border_width (GTK_CONTAINER (table), 7);
    gtk_box_pack_start (GTK_BOX (area), table, TRUE, TRUE, 0);
    gtk_widget_show (table);

    /* entry here maybe */
    hbox = gtk_hbox_new (FALSE, 12);
    gtk_box_pack_start (GTK_BOX (table), hbox, FALSE, FALSE, 0);
    gtk_widget_show (hbox);

    label = gtk_label_new_with_mnemonic (_("_Email:"));
    gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    gtk_widget_show (label);

    dialog->user_picker = gtk_entry_new ();
    gtk_widget_set_tooltip_text (dialog->user_picker,
                                 _("Type an email address to share this "
                                   "folder with. Use a comma to separate "
                                   "multiple addresses."));
    gtk_label_set_mnemonic_widget (GTK_LABEL (label), dialog->user_picker);
    gtk_entry_set_activates_default (GTK_ENTRY (dialog->user_picker), TRUE);
    gtk_box_pack_start (GTK_BOX (hbox), dialog->user_picker, TRUE, TRUE, 0);
    gtk_widget_show (dialog->user_picker);

	dialog->allow_mods = gtk_check_button_new_with_mnemonic (_("_Allow Modification"));
    /* Default to RW */
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dialog->allow_mods),
                                  TRUE);
    gtk_box_pack_end (GTK_BOX (table), dialog->allow_mods, FALSE, FALSE, 0);
	gtk_widget_show (dialog->allow_mods);

	gtk_widget_set_size_request (GTK_WIDGET (dialog), 320, -1);
}

GtkWidget *
share_dialog_new (GtkWidget *parent, UbuntuOneNautilus *uon, const gchar *path)
{
	ShareDialog *dialog;
	
	dialog = (ShareDialog *) g_object_new (TYPE_SHARE_DIALOG, NULL);
	dialog->uon = uon;
	dialog->path = g_strdup (path);
	gtk_window_set_transient_for (GTK_WINDOW (dialog), GTK_WINDOW (parent));

	return (GtkWidget *) dialog;
}
