# -*- coding: utf-8 -*-
#
# Author: John R. Lenton <john.lenton@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" Tests for the action queue """
from __future__ import with_statement

import logging
import shutil
import unittest

import dbus
from dbus.mainloop.glib import DBusGMainLoop
from twisted.internet import defer

from contrib.testing.testcase import (
    BaseTwistedTestCase,
    MementoHandler,
)

from ubuntuone.syncdaemon.dbus_interface import DBusInterface
from ubuntuone.syncdaemon.main import Main
from ubuntuone.syncdaemon.action_queue import NoisyRequestQueue

DBusInterface.test = True


class AcionQueueTests(BaseTwistedTestCase):
    """ Basic tests to check ActionQueue """

    def setUp(self):
        '''
        prepare to run the test
        '''
        BaseTwistedTestCase.setUp(self)
        self.root = self.mktemp('root')
        self.shares = self.mktemp('shares')
        self.data = self.mktemp('data')
        self.partials = self.mktemp('partials')
        self.handler = MementoHandler()
        self.handler.setLevel(logging.ERROR)
        self.main = Main(root_dir=self.root,
                         shares_dir=self.shares,
                         data_dir=self.data,
                         partials_dir=self.partials,
                         host='localhost', port=0,
                         dns_srv=False, ssl=False,
                         disable_ssl_verify=True,
                         realm='fake.realm',
                         mark_interval=60,
                         handshake_timeout=2,
                         glib_loop=DBusGMainLoop(set_as_default=True))
        logging.getLogger('ubuntuone.SyncDaemon').addHandler(self.handler)
        dbus.service.BusName.__del__ = lambda _: None

    def tearDown(self):
        '''
        cleanup after the test
        '''
        self.main.shutdown()
        shutil.rmtree(self.root)
        shutil.rmtree(self.shares)
        shutil.rmtree(self.data)
        shutil.rmtree(self.partials)
        for record in self.handler.records:
            exc_info = getattr(record, 'exc_info', None)
            if exc_info is not None:
                raise exc_info[0], exc_info[1], exc_info[2]
        BaseTwistedTestCase.tearDown(self)

    @defer.inlineCallbacks
    def test_content_queue_has_only_one_op_per_node(self):
        '''
        Check that the content queue uniquifies operations per node.
        '''
        yield self.main.start()
        # totally fake, we don't care: the messages are only validated on run
        self.main.action_q.download('foo', 'bar', 0, 0)
        self.main.action_q.upload('foo', 'bar', 0, 0, 0, 0, 0)
        self.assertEqual(len(self.main.action_q.content_queue.waiting), 1)

    @defer.inlineCallbacks
    def test_content_queue_has_only_one_op_per_node_even_counting_markers(self):
        '''
        Check that the content queue uniquifies operations per node
        even when some of the operations were added using markers.
        '''
        yield self.main.start()
        self.main.action_q.download('foo', 'bar', 0, 0)
        self.main.action_q.uuid_map.set('foo', 'feh')
        self.main.action_q.uuid_map.set('bar', 'bah')
        self.main.action_q.upload('feh', 'bah', 0, 0, 0, 0, 0)
        self.assertEqual(len(self.main.action_q.content_queue.waiting), 1)

    @defer.inlineCallbacks
    def test_aq_resolve_uuid_maybe(self):
        '''
        Check action_q.resolve_uuid_maybe does what it's supposed to
        '''
        yield self.main.start()
        self.assertEqual(self.main.action_q.resolve_uuid_maybe('foo'), 'foo')
        self.main.action_q.uuid_map.set('foo', 'feh')
        self.assertEqual(self.main.action_q.resolve_uuid_maybe('foo'), 'feh')


class TestNoisyRQ(unittest.TestCase):
    '''
    Tests for NoisyRequestQueue
    '''

    def test_noisy_rq_blurts_about_head(self):
        '''
        Test NRQ calls its callback when head is set
        '''
        rq = NoisyRequestQueue('name', None,
                               lambda h, w:
                                   setattr(self, 'result', (h, tuple(w))))
        rq._head = 'blah'
        self.assertEqual(self.result, ('blah', ()))

    def test_noisy_rq_blurts_about_waiting(self):
        '''
        Test NRQ calls its callback when the waiting queue is altered.
        '''
        class BlackHole(object):
            '''The universal tool.'''
            __call__ = __getattr__ = lambda *_, **__: BlackHole()
        class FakeCommand(object):
            '''Yet another fake action queue command'''
            def run(self):
                '''run that just succeeds'''
                return defer.succeed(None)
            def is_runnable(self):
                """returns True"""
                return True
        def cb(head, waiting):
            '''NRQ testing callback'''
            evts.append((head, tuple(waiting)))
        evts = []
        cmd = FakeCommand()
        cmd2 = FakeCommand()
        rq = NoisyRequestQueue('name', BlackHole(), cb)
        rq.queue(cmd2)
        rq.queue_top(cmd)
        rq.run()
        self.assertEqual(evts, [(None, ()),           # __init__
                                (None, (cmd2,)),      # queue
                                (None, (cmd, cmd2)),  # queue_top
                                (cmd, (cmd2,)),       # run
                                (None, (cmd2,)),      # done
                                ])
