# ubuntuone.syncdaemon.logger - logging utilities
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" SyncDaemon logging utilities and config. """
from __future__ import with_statement

import contextlib
import logging
import sys
import os
import re
import weakref
import zlib
import functools
import xdg.BaseDirectory

from itertools import imap
from logging.handlers import TimedRotatingFileHandler

# extra levels
# be more verbose than logging.DEBUG(10)
TRACE = 5
# info that we almost always want to log (logging.ERROR - 1)
NOTE = logging.ERROR - 1

# map names to the extra levels
levels = {'TRACE':TRACE, 'NOTE':NOTE}
for k, v in levels.items():
    logging.addLevelName(v, k)


class Logger(logging.Logger):
    """Logger that support out custom levels."""

    def note(self, msg, *args, **kwargs):
        """log at NOTE level"""
        if self.isEnabledFor(NOTE):
            self._log(NOTE, msg, args, **kwargs)

    def trace(self, msg, *args, **kwargs):
        """log at TRACE level"""
        if self.isEnabledFor(TRACE):
            self._log(TRACE, msg, args, **kwargs)

# use our logger as the default Logger class
logging.setLoggerClass(Logger)


class DayRotatingFileHandler(TimedRotatingFileHandler):
    """A mix of TimedRotatingFileHandler and RotatingFileHandler configured for
    daily rotation but that uses the suffix and extMatch of Hourly rotation, in
    order to allow seconds based rotation on each startup.
    The log file is also rotated when the specified size is reached.
    """

    def __init__(self, *args, **kwargs):
        """ create the instance and override the suffix and extMatch.
        Also accepts a maxBytes keyword arg to rotate the file when it reachs
        maxBytes.
        """
        kwargs['when'] = 'D'
        kwargs['backupCount'] = LOGBACKUP
        # check if we are in 2.5, only for PQM
        if sys.version_info[:2] >= (2, 6):
            kwargs['delay'] = 1
        if 'maxBytes' in kwargs:
            self.maxBytes = kwargs.pop('maxBytes')
        else:
            self.maxBytes = 0
        TimedRotatingFileHandler.__init__(self, *args, **kwargs)
        # override suffix
        self.suffix = "%Y-%m-%d_%H-%M-%S"
        self.extMatch = re.compile(r"^\d{4}-\d{2}-\d{2}_\d{2}-\d{2}-\d{2}$")

    def shouldRollover(self, record):
        """
        Determine if rollover should occur.

        Basically, see if TimedRotatingFileHandler.shouldRollover and if it's
        False see if the supplied record would cause the file to exceed
        the size limit we have.

        The size based rotation are from logging.handlers.RotatingFileHandler
        """
        if TimedRotatingFileHandler.shouldRollover(self, record):
            return 1
        else:
            # check the size
            if self.stream is None:                 # delay was set...
                self.stream = self._open()
            if self.maxBytes > 0:                   # are we rolling over?
                msg = "%s\n" % self.format(record)
                self.stream.seek(0, 2)  #due to non-posix-compliant Windows feature
                if self.stream.tell() + len(msg) >= self.maxBytes:
                    return 1
            return 0


# pylint: disable-msg=C0103
class mklog(object):
    """
    Create a logger that keeps track of the method where it's being
    called from, in order to make more informative messages.
    """
    __slots__ = ('logger', 'zipped_desc')
    def __init__(self, _logger, _method, _share, _uid, *args, **kwargs):
        # args are _-prepended to lower the chances of them
        # conflicting with kwargs

        all_args = []
        for arg in args:
            all_args.append(
                repr(arg).decode('ascii', 'replace').encode('ascii', 'replace')
                )
        for k, v in kwargs.items():
            v = repr(v).decode('ascii', 'replace').encode('ascii', 'replace')
            all_args.append("%s=%r" % (k, v))
        args = ", ".join(all_args)

        desc = "%-28s share:%-40r node:%-40r %s(%s) " % (_method, _share,
                                                         _uid, _method, args)
        self.zipped_desc = zlib.compress(desc, 9)
        self.logger = _logger

    @property
    def desc(self):
        return zlib.decompress(self.zipped_desc)

    def _log(self, logger_func, *args):
        """
        Generalized form of the different logging functions.
        """
        logger_func(self.desc + " ".join(imap(str, args)))
    def debug(self, *args):
        """Log at level DEBUG"""
        self._log(self.logger.debug, *args)
    def info(self, *args):
        """Log at level INFO"""
        self._log(self.logger.info, *args)
    def warn(self, *args):
        """Log at level WARN"""
        self._log(self.logger.warn, *args)
    def error(self, *args):
        """Log at level ERROR"""
        self._log(self.logger.error, *args)
    def exception(self, *args):
        """Log an exception"""
        self._log(self.logger.exception, *args)
    def note(self, *args):
        """Log at NOTE level (high-priority info) """
        self._log(self.logger.high, *args)
    def trace(self, *args):
        """Log at level TRACE"""
        self._log(self.logger.trace, *args)

    def callbacks(self, success_message='success', success_arg='',
                  failure_message='failure'):
        """
        Return a callback and an errback that log success or failure
        messages.

        The callback/errback pair are pass-throughs; they don't
        interfere in the callback/errback chain of the deferred you
        add them to.
        """
        def callback(arg, success_arg=success_arg):
            "it worked!"
            if callable(success_arg):
                success_arg = success_arg(arg)
            self.debug(success_message, success_arg)
            return arg
        def errback(failure):
            "it failed!"
            self.error(failure_message, failure.getErrorMessage())
            self.debug('traceback follows:\n\n' + failure.getTraceback(), '')
            return failure
        return callback, errback
# pylint: enable-msg=C0103


class DebugCapture(logging.Handler):
    """
    A context manager to capture debug logs.
    """

    def __init__(self, logger, raise_unhandled=False, on_error=True):
        """Creates the instance.

        @param logger: the logger to wrap
        @param raise_unhandled: raise unhandled errors (which are alse logged)
        @param on_error: if it's True (default) the captured debug info is
        dumped if a record with log level >= ERROR is logged.
        """
        logging.Handler.__init__(self, logging.DEBUG)
        self.on_error = on_error
        self.dirty = False
        self.raise_unhandled = raise_unhandled
        self.records = []
        # insert myself as the handler for the logger
        self.logger = weakref.proxy(logger)
        # store the logger log level
        self.old_level = logger.level
        # remove us from the Handler list and dict
        self.close()

    def emit_debug(self):
        """emit stored records to the original logger handler(s)"""
        enable_debug = self.enable_debug
        for record in self.records:
            for slave in self.slaves:
                with enable_debug(slave):
                    slave.handle(record)

    @contextlib.contextmanager
    def enable_debug(self, obj):
        """context manager that temporarily changes the level attribute of obj
        to logging.DEBUG.
        """
        old_level = obj.level
        obj.level = logging.DEBUG
        yield obj
        obj.level = old_level

    def clear(self):
        """cleanup the captured records"""
        self.records = []

    def install(self):
        """Install the debug capture in the logger"""
        self.slaves = self.logger.handlers
        self.logger.handlers = [self]
        # set the logger level in DEBUG
        self.logger.setLevel(logging.DEBUG)

    def uninstall(self):
        """restore the logger original handlers"""
        # restore the logger
        self.logger.handlers = self.slaves
        self.logger.setLevel(self.old_level)
        self.clear()
        self.dirty = False
        self.slaves = []

    def emit(self, record):
        """A emit() that append the record to the record list"""
        self.records.append(record)

    def handle(self, record):
        """ handle a record """
        # if its a DEBUG level record then intercept otherwise
        # pass through to the original logger handler(s)
        if self.old_level <= logging.DEBUG:
            return sum(slave.handle(record) for slave in self.slaves)
        if record.levelno == logging.DEBUG:
            return logging.Handler.handle(self, record)
        elif self.on_error and record.levelno >= logging.ERROR and \
            record.levelno != NOTE:
            # if it's >= ERROR keep it, but mark the dirty falg
            self.dirty = True
            return logging.Handler.handle(self, record)
        else:
            return sum(slave.handle(record) for slave in self.slaves)

    def __enter__(self):
        """ContextManager API"""
        self.install()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        """ContextManager API"""
        if exc_type is not None:
            self.emit_debug()
            self.on_error = False
            self.logger.error('unhandled exception', exc_info=(exc_type,
                exc_value, traceback))
        elif self.dirty:
            # emit all debug messages collected after the error
            self.emit_debug()
        self.uninstall()
        if self.raise_unhandled and exc_type is not None:
            raise exc_type, exc_value, traceback
        else:
            return True


### configure the thing ###
# define the location of the log folder
home = xdg.BaseDirectory.xdg_cache_home
LOGFOLDER = os.path.join(home, 'ubuntuone','log')
if not os.path.exists(LOGFOLDER):
    os.makedirs(LOGFOLDER)

LOGFILENAME = os.path.join(LOGFOLDER, 'syncdaemon.log')
EXLOGFILENAME = os.path.join(LOGFOLDER, 'syncdaemon-exceptions.log')

LOGBACKUP = 5 # the number of log files to keep around

basic_formatter = logging.Formatter(fmt="%(asctime)s - %(name)s - " \
                                    "%(levelname)s - %(message)s")
debug_formatter = logging.Formatter(fmt="%(asctime)s %(name)s %(module)s " \
                                    "%(lineno)s %(funcName)s %(message)s")

# a constant to change the default DEBUG level value
_DEBUG_LOG_LEVEL = logging.DEBUG


# partial config of the handler to rotate when the file size is 1MB
CustomRotatingFileHandler = functools.partial(DayRotatingFileHandler,
                                              maxBytes=1048576)

# root logger
root_logger = logging.getLogger("ubuntuone.SyncDaemon")
root_logger.propagate = False
root_logger.setLevel(_DEBUG_LOG_LEVEL)
root_handler = CustomRotatingFileHandler(filename=LOGFILENAME)
root_handler.addFilter(logging.Filter("ubuntuone.SyncDaemon"))
root_handler.setFormatter(basic_formatter)
root_handler.setLevel(_DEBUG_LOG_LEVEL)
root_logger.addHandler(root_handler)
# exception logs
exception_handler = CustomRotatingFileHandler(filename=EXLOGFILENAME)
exception_handler.setFormatter(basic_formatter)
exception_handler.setLevel(logging.ERROR)
# add the exception handler to the root logger
logging.getLogger('').addHandler(exception_handler)
root_logger.addHandler(exception_handler)

# hook twisted.python.log with standard logging
from twisted.python import log
observer = log.PythonLoggingObserver('twisted')
observer.start()
# configure the logger to only show errors
twisted_logger = logging.getLogger('twisted')
twisted_logger.propagate = False
twisted_logger.setLevel(logging.ERROR)
twisted_handler = CustomRotatingFileHandler(filename=LOGFILENAME)
twisted_handler.addFilter(logging.Filter("twisted"))
twisted_handler.setFormatter(basic_formatter)
twisted_handler.setLevel(logging.ERROR)
twisted_logger.addHandler(twisted_handler)
twisted_logger.addHandler(exception_handler)


def set_level(level):
    """set 'level' as the level for all the logger/handlers"""
    root_logger.setLevel(level)
    root_handler.setLevel(level)


def set_debug(dest):
    """ Set the level to debug of all registered loggers, and replace their
    handlers. if debug_level is file, syncdaemon-debug.log is used. If it's
    stdout, all the logging is redirected to stdout. If it's stderr, to stderr.

    @param dest: a string with a one or more of 'file', 'stdout', and 'stderr'
                 e.g. 'file stdout'
    """
    if not [ v for v in ['file', 'stdout', 'stderr'] if v in dest]:
        # invalid dest value, let the loggers alone
        return
    sd_filter = logging.Filter('ubuntuone.SyncDaemon')
    if 'file' in dest:
        # setup the existing loggers in debug
        root_handler.setLevel(_DEBUG_LOG_LEVEL)
        twisted_handler.setLevel(_DEBUG_LOG_LEVEL)
        logfile = os.path.join(LOGFOLDER, 'syncdaemon-debug.log')
        root_handler.baseFilename = os.path.abspath(logfile)
        twisted_handler.baseFilename = os.path.abspath(logfile)
        # don't cap the file size
        root_handler.maxBytes = 0
        twisted_handler.maxBytes = 0
    for name in ['ubuntuone.SyncDaemon', 'twisted']:
        logger = logging.getLogger(name)
        logger.setLevel(_DEBUG_LOG_LEVEL)
        if 'stderr' in dest:
            stderr_handler = logging.StreamHandler()
            stderr_handler.setFormatter(basic_formatter)
            stderr_handler.setLevel(_DEBUG_LOG_LEVEL)
            stderr_handler.addFilter(sd_filter)
            logger.addHandler(stderr_handler)
        if 'stdout' in dest:
            stdout_handler = logging.StreamHandler(sys.stdout)
            stdout_handler.setFormatter(basic_formatter)
            stdout_handler.setLevel(_DEBUG_LOG_LEVEL)
            stdout_handler.addFilter(sd_filter)
            logger.addHandler(stdout_handler)


def set_server_debug(dest):
    """ Set the level to debug of all registered loggers, and replace their
    handlers. if debug_level is file, syncdaemon-debug.log is used. If it's
    stdout, all the logging is redirected to stdout.

    @param dest: a string containing 'file' and/or 'stdout', e.g: 'file stdout'
    """
    logger = logging.getLogger("storage.server")
    logger.setLevel(5) # this shows server messages
    if 'file' in dest:
        handler = DayRotatingFileHandler(filename=os.path.join(LOGFOLDER,
                                                'syncdaemon-debug.log'))
        handler.setFormatter(basic_formatter)
        handler.setLevel(5) # this shows server messages
        logger.addHandler(handler)
    if 'stdout' in dest:
        stdout_handler = logging.StreamHandler(sys.stdout)
        stdout_handler.setFormatter(basic_formatter)
        stdout_handler.setLevel(5) # this shows server messages
        logger.addHandler(stdout_handler)
    if 'stderrt' in dest:
        stdout_handler = logging.StreamHandler(sys.stdout)
        stdout_handler.setFormatter(basic_formatter)
        stdout_handler.setLevel(5) # this shows server messages
        logger.addHandler(stdout_handler)


# if we are in debug mode, replace/add the handlers
DEBUG = os.environ.get("DEBUG", None)
if DEBUG:
    set_debug(DEBUG)

# configure server logging if SERVER_DEBUG != None
SERVER_DEBUG = os.environ.get("SERVER_DEBUG", None)
if SERVER_DEBUG:
    set_server_debug(SERVER_DEBUG)


def rotate_logs():
    """do a rollover of the three handlers"""
    twisted_handler.close()
    # ignore the missing file error on a failed rollover
    # pylint: disable-msg=W0704
    try:
        root_handler.doRollover()
    except OSError:
        pass
    try:
        exception_handler.doRollover()
    except OSError:
        pass


