# ubuntuone.syncdaemon.volume_manager - manages volumes
#
# Author: Guillermo Gonzalez <guillermo.gonzalez@canonical.com>
#
# Copyright 2009 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.
""" The all mighty Volume Manager """
from __future__ import with_statement

import itertools
import functools
import cPickle
import logging
import os
import re
import shutil
import stat
import sys
from contextlib import contextmanager
from itertools import ifilter

from ubuntuone.storageprotocol import request
from ubuntuone.storageprotocol.volumes import (
    ShareVolume,
    UDFVolume,
    RootVolume,
)
from ubuntuone.syncdaemon.marker import MDMarker
from ubuntuone.syncdaemon.interfaces import IMarker
from ubuntuone.syncdaemon import file_shelf, config

from twisted.internet import defer


class _Share(object):
    """Represents a share or mount point"""

    def __init__(self, share_id=request.ROOT, node_id=None, path=None,
                 name=None, access_level='View', accepted=False,
                 other_username=None, other_visible_name=None):
        """ Creates the instance.

        The received path should be 'bytes'
        """
        if path is None:
            self.path = None
        else:
            self.path = os.path.normpath(path)
        self.id = str(share_id)
        self.access_level = access_level
        self.accepted = accepted
        self.name = name
        self.other_username = other_username
        self.other_visible_name = other_visible_name
        self.subtree = node_id
        self.free_bytes = None


class _UDF(object):
    """A representation of a User Defined Folder."""

    def __init__(self, udf_id, node_id, suggested_path,
                 path, subscribed=True):
        """Create the UDF, subscribed by default"""
        # id and node_id should be str or None
        assert isinstance(udf_id, basestring) or udf_id is None
        assert isinstance(node_id, basestring) or node_id is None
        self.id = udf_id
        self.node_id = node_id
        self.suggested_path = suggested_path
        self.path = path
        self.subscribed = subscribed


class Volume(object):
    """A generic volume."""

    def __init__(self, volume_id, node_id):
        """Create the volume."""
        # id and node_id should be str or None
        assert isinstance(volume_id, basestring) or volume_id is None
        assert isinstance(node_id, basestring) or node_id is None
        self.volume_id = volume_id
        self.node_id = node_id

    @property
    def id(self):
        return self.volume_id

    def can_write(self):
        raise NotImplementedError('Subclass responsability')

    def __eq__(self, other):
        result = (self.id == other.id and
                  self.node_id == other.node_id)
        return result


class Share(Volume):
    """A volume representing a Share."""

    def __init__(self, volume_id=None, node_id=None, path=None, name=None,
                 other_username=None, other_visible_name=None, accepted=False,
                 access_level='View', free_bytes=None):
        """Create the share."""
        super(Share, self).__init__(volume_id, node_id)
        self.__dict__['type'] = 'Share'
        if path is None:
            self.path = None
        else:
            self.path = os.path.normpath(path)
        self.name = name
        self.other_username = other_username
        self.other_visible_name = other_visible_name
        self.accepted = accepted
        self.access_level = access_level
        self.free_bytes = free_bytes

    @classmethod
    def from_response(cls, share_response, path):
        """ Creates a Share instance from a ShareResponse.

        The received path should be 'bytes'
        """
        share = cls(str(share_response.id), str(share_response.subtree),
                    path, share_response.name, share_response.other_username,
                    share_response.other_visible_name,
                    share_response.accepted, share_response.access_level)
        return share

    @classmethod
    def from_notify_holder(cls, share_notify, path):
        """ Creates a Share instance from a NotifyShareHolder.

        The received path should be 'bytes'
        """
        share = cls(volume_id=str(share_notify.share_id),
                    node_id=str(share_notify.subtree),
                    path=path, name=share_notify.share_name,
                    other_username=share_notify.from_username,
                    other_visible_name=share_notify.from_visible_name,
                    access_level=share_notify.access_level)
        return share

    @classmethod
    def from_share_volume(cls, share_volume, path):
        """Creates a Share instance from a volumes.ShareVolume.

        The received path should be 'bytes'

        """
        share = cls(str(share_volume.volume_id), str(share_volume.node_id),
                    path, share_volume.share_name,
                    share_volume.other_username,
                    share_volume.other_visible_name, share_volume.accepted,
                    share_volume.access_level)
        return share

    def can_write(self):
        """ check the access_level of this share,
        returns True if it's 'Modify'.
        """
        return self.access_level == 'Modify'

    @property
    def active(self):
        """Return True if this Share is accepted."""
        return self.accepted

    def __eq__(self, other):
        result = (super(Share, self).__eq__(other) and
                  self.path == other.path and
                  self.name == other.name and
                  self.other_username == other.other_username and
                  self.other_visible_name == other.other_visible_name and
                  self.accepted == other.accepted and
                  self.access_level == other.access_level)
        return result


class Shared(Share):

    def __init__(self, *args, **kwargs):
        super(Shared, self).__init__(*args, **kwargs)
        self.__dict__['type'] = 'Shared'


class Root(Volume):
    """A volume representing the root."""

    def __init__(self, node_id=None, path=None, free_bytes=None):
        """Create the Root."""
        super(Root, self).__init__(request.ROOT, node_id)
        self.__dict__['type'] = 'Root'
        self.path = path
        self.free_bytes = free_bytes

    def __eq__(self, other):
        result = (super(Root, self).__eq__(other) and
                  self.path == other.path)
        return result

    def can_write(self):
        return True

    @property
    def active(self):
        return True


class UDF(Volume):
    """A volume representing a User Defined Folder."""

    def __init__(self, volume_id=None, node_id=None,
                 suggested_path=None, path=None, subscribed=True):
        """Create the UDF, subscribed by default"""
        super(UDF, self).__init__(volume_id, node_id)
        self.__dict__['type'] = 'UDF'
        self.node_id = node_id
        self.suggested_path = suggested_path
        self.path = path
        self.subscribed = subscribed

    def __repr__(self):
        return "<UDF id %r, real path %r>" % (self.id, self.path)

    @property
    def ancestors(self):
        """Calculate all the ancestors for this UDF's path."""
        user_home = os.path.abspath(os.path.expanduser('~'))
        path_list = os.path.abspath(self.path).split(os.path.sep)
        common_prefix = os.path.commonprefix([self.path, user_home])
        common_list = common_prefix.split(os.path.sep)

        result = [user_home]
        for p in path_list[len(common_list):-1]:
            result.append(os.path.join(result[-1], p))

        return result

    def can_write(self):
        """We always can write in a UDF."""
        return True

    @property
    def active(self):
        """Returns True if the UDF is subscribed."""
        return self.subscribed

    @classmethod
    def from_udf_volume(cls, udf_volume, path):
        """Creates a UDF instance from a volumes.UDFVolume.

        The received path should be 'bytes'

        """
        return cls(str(udf_volume.volume_id), str(udf_volume.node_id),
                   udf_volume.suggested_path, path)

    def __eq__(self, other):
        result = (super(UDF, self).__eq__(other) and
                  self.suggested_path == other.suggested_path and
                  self.path == other.path and
                  self.subscribed == other.subscribed)
        return result


class VolumeManager(object):
    """Manages shares and mount points."""

    METADATA_VERSION = '6'

    def __init__(self, main):
        """Create the instance and populate the shares/d attributes
        from the metadata (if it exists).
        """
        self.log = logging.getLogger('ubuntuone.SyncDaemon.VM')
        self.m = main
        self._data_dir = os.path.join(self.m.data_dir, 'vm')
        self._shares_dir = os.path.join(self._data_dir, 'shares')
        self._shared_dir = os.path.join(self._data_dir, 'shared')
        self._udfs_dir = os.path.join(self._data_dir, 'udfs')

        md_upgrader = MetadataUpgrader(self._data_dir, self._shares_dir,
                                       self._shared_dir, self._udfs_dir,
                                       self.m.root_dir, self.m.shares_dir,
                                       self.m.shares_dir_link)
        md_upgrader.upgrade_metadata()

        # build the dir layout
        if not os.path.exists(self.m.root_dir):
            self.log.debug('creating root dir: %r', self.m.root_dir)
            os.makedirs(self.m.root_dir)
        if not os.path.exists(self.m.shares_dir):
            self.log.debug('creating shares directory: %r', self.m.shares_dir)
            os.makedirs(self.m.shares_dir)
        # create the shares symlink
        if not os.path.exists(self.m.shares_dir_link):
            self.log.debug('creating Shares symlink: %r -> %r',
                           self.m.shares_dir_link, self.m.shares_dir)
            os.symlink(self.m.shares_dir, self.m.shares_dir_link)
        # make the shares_dir read only
        os.chmod(self.m.shares_dir, 0555)
        # make the root read write
        os.chmod(self.m.root_dir, 0775)

        # create the metadata directories
        if not os.path.exists(self._shares_dir):
            os.makedirs(self._shares_dir)
        if not os.path.exists(self._shared_dir):
            os.makedirs(self._shared_dir)
        if not os.path.exists(self._udfs_dir):
            os.makedirs(self._udfs_dir)

        # load the metadata
        self.shares = VMFileShelf(self._shares_dir)
        self.shared = VMFileShelf(self._shared_dir)
        self.udfs = VMFileShelf(self._udfs_dir)
        if self.shares.get(request.ROOT) is None:
            self.root = Root(path=self.m.root_dir)
        else:
            self.root = self.shares[request.ROOT]
        self.root.access_level = 'Modify'
        self.root.path = self.m.root_dir
        self.shares[request.ROOT] = self.root
        self.marker_share_map = {}
        self.marker_udf_map = {}
        self.list_shares_retries = 0
        self.list_volumes_retries = 0
        self.retries_limit = 5

    def init_root(self):
        """ Creates the root mdid. """
        self.log.debug('init_root')
        self._create_share_dir(self.root)
        try:
            self.m.fs.get_by_path(self.root.path)
        except KeyError:
            self.m.fs.create(path=self.root.path,
                             share_id=request.ROOT, is_dir=True)

    def handle_SYS_ROOT_RECEIVED(self, root_id):
        """Got the root, map it to the  root share."""
        self.log.debug('handle_SYS_ROOT_RECEIVED(%s)', root_id)
        self._set_root(root_id)
        self.m.action_q.inquire_account_info()
        self.m.action_q.inquire_free_space(request.ROOT)
        self.refresh_volumes()
        self.refresh_shares()

    def _set_root(self, node_id):
        """Set the root node_id to the root share and mdobj."""
        # only set the root if we don't have it
        root = self.shares[request.ROOT]
        if not root.node_id:
            mdobj = self.m.fs.get_by_path(self.root.path)
            if getattr(mdobj, 'node_id', None) is None:
                self.m.fs.set_node_id(self.root.path, node_id)
            self.root.node_id = root.node_id = node_id
            self.shares[request.ROOT] = root
        elif root.node_id != node_id:
            self.m.event_q.push('SYS_ROOT_MISMATCH', root.node_id, node_id)

    def refresh_shares(self):
        """Request the list of shares to the server."""
        self.m.action_q.list_shares()

    def refresh_volumes(self):
        """Request the list of volumes to the server."""
        self.m.action_q.list_volumes()

    def handle_AQ_LIST_VOLUMES(self, volumes):
        """Handle AQ_LIST_VOLUMES event."""
        self.log.debug('handling volumes list')
        self.list_volumes_retries = 0
        shares = []
        udfs = []
        for volume in volumes:
            vol = self._handle_new_volume(volume)
            if isinstance(vol, Share):
                shares.append(vol.id)
            elif isinstance(vol, UDF):
                udfs.append(vol.id)
        self._cleanup_volumes(shares=shares, udfs=udfs)

    def _handle_new_volume(self, volume):
        """
        Handle a (probably) new volume, and call the specific method to
        handle this volume type (share/udf/root).

        Returns the created share/udf.

        """
        if isinstance(volume, ShareVolume):
            dir_name = self._build_share_path(volume.share_name,
                                              volume.other_visible_name)
            path = os.path.join(self.m.shares_dir, dir_name)
            share = Share.from_share_volume(volume, path)
            self.add_share(share)
            return share
        elif isinstance(volume, UDFVolume):
            path = self._build_udf_path(volume.suggested_path)
            udf = UDF.from_udf_volume(volume, path)
            subscribe = config.get_user_config().get_udf_autosubscribe()
            udf.subscribed = subscribe
            self.add_udf(udf)
            return udf
        elif isinstance(volume, RootVolume):
            self._set_root(str(volume.node_id))
        else:
            self.log.warning("Got a unkown type in the volmes list: %r",
                             volume)

    def handle_AQ_SHARES_LIST(self, shares_list):
        """ handle AQ_SHARES_LIST event """
        self.log.debug('handling shares list: ')
        self.list_shares_retries = 0
        shares = []
        shared = []
        for a_share in shares_list.shares:
            share_id = getattr(a_share, 'id',
                               getattr(a_share, 'share_id', None))
            self.log.debug('share %r: id=%s, name=%r', a_share.direction,
                           share_id, a_share.name)
            if a_share.direction == "to_me":
                dir_name = self._build_share_path(a_share.name,
                                                  a_share.other_visible_name)
                path = os.path.join(self.m.shares_dir, dir_name)
                share = Share.from_response(a_share, path)
                shares.append(share.volume_id)
                self.add_share(share)
            elif a_share.direction == "from_me":
                try:
                    mdobj = self.m.fs.get_by_node_id("", a_share.subtree)
                    path = self.m.fs.get_abspath(mdobj.share_id, mdobj.path)
                except KeyError:
                    # we don't have the file/md of this shared node_id yet
                    # for the moment ignore this share
                    self.log.warning("we got a share with 'from_me' direction,"
                            " but don't have the node_id in the metadata yet")
                    path = None
                share = Shared.from_response(a_share, path)
                shared.append(share.volume_id)
                self.add_shared(share)
        self._cleanup_shared(shared)
        self._cleanup_shares(shares)

    def _cleanup_volumes(self, shares=None, udfs=None):
        """Remove missing shares from the shares and shared shelfs."""
        # housekeeping of the shares, shared and udfs shelf's each time we
        # get the list of shares/volumes
        self.log.debug('deleting dead volumes')
        if shares is not None:
            for share in ifilter(lambda item: item and item not in shares,
                                 self.shares):
                self.log.debug('deleting share: id=%s', share)
                self.share_deleted(share)
        if udfs is not None:
            # cleanup missing udfs
            for udf in ifilter(lambda item: item and item not in udfs,
                               self.udfs):
                self.log.debug('deleting udfs: id=%s', udf)
                self.udf_deleted(udf)

    def _cleanup_shared(self, to_keep):
        """Cleanup shared Shares from the shelf."""
        self.log.debug('deleting dead shared')
        for share in ifilter(lambda item: item and item not in to_keep,
                             self.shared):
            self.log.debug('deleting shared: id=%s', share)
            del self.shared[share]

    def _cleanup_shares(self, to_keep):
        """Cleanup not-yet accepted Shares from the shares shelf."""
        self.log.debug('deleting dead shares')
        for share in ifilter(lambda item: item and item not in to_keep and \
                             not self.shares[item].accepted, self.shares):
            self.log.debug('deleting shares: id=%s', share)
            self.share_deleted(share)

    def _build_share_path(self, share_name, visible_name):
        """Builds the root path using the share information."""
        dir_name = share_name + u' from ' + visible_name
        # Unicode boundary! the name is Unicode in protocol and server,
        # but here we use bytes for paths
        dir_name = dir_name.encode("utf8")
        return dir_name

    def _build_udf_path(self, suggested_path):
        """Build the udf path using the suggested_path."""
        # Unicode boundary! the name is Unicode in protocol and server,
        # but here we use bytes for paths
        return os.path.expanduser(suggested_path).encode("utf8")

    def handle_AQ_LIST_VOLUMES_ERROR(self, error):
        """Handle AQ_LIST_VOLUMES_ERROR event."""
        # call list_volumes again, until we reach the retry limit
        if self.list_volumes_retries <= self.retries_limit:
            self.m.action_q.list_volumes()
            self.list_volumes_retries += 1

    def handle_AQ_LIST_SHARES_ERROR(self, error):
        """ handle AQ_LIST_SHARES_ERROR event """
        # just call list_shares again, until we reach the retry limit
        if self.list_shares_retries <= self.retries_limit:
            self.m.action_q.list_shares()
            self.list_shares_retries += 1

    def handle_SV_FREE_SPACE(self, share_id, free_bytes):
        """ handle SV_FREE_SPACE event """
        self.update_free_space(str(share_id), free_bytes)
        # check AQ wait conditions
        self.m.action_q.check_conditions()

    def handle_SV_SHARE_CHANGED(self, share_info):
        """ handle SV_SHARE_CHANGED event """
        if str(share_info.share_id) not in self.shares:
            self.log.debug("New share notification, share_id: %s",
                     share_info.share_id)
            dir_name = self._build_share_path(share_info.share_name,
                                              share_info.from_visible_name)
            path = os.path.join(self.m.shares_dir, dir_name)
            share = Share.from_notify_holder(share_info, path)
            self.add_share(share)
        else:
            self.log.debug('share changed! %s', share_info.share_id)
            self.share_changed(share_info)

    def handle_SV_SHARE_DELETED(self, share_id):
        """ handle SV_SHARE_DELETED event """
        self.log.debug('share deleted! %s', share_id)
        self.share_deleted(str(share_id))

    def handle_AQ_CREATE_SHARE_OK(self, share_id, marker):
        """ handle AQ_CREATE_SHARE_OK event. """
        share = self.marker_share_map.get(marker)
        if share is None:
            self.m.action_q.list_shares()
        else:
            share.volume_id = str(share_id)
            if IMarker.providedBy(share.node_id):
                mdobj = self.m.fs.get_by_mdid(str(share.node_id))
                share.node_id = mdobj.node_id
            self.add_shared(share)
            if marker in self.marker_share_map:
                del self.marker_share_map[marker]

    def handle_AQ_CREATE_SHARE_ERROR(self, marker, error):
        """ handle AQ_CREATE_SHARE_ERROR event. """
        if marker in self.marker_share_map:
            del self.marker_share_map[marker]

    def handle_SV_SHARE_ANSWERED(self, share_id, answer):
        """ handle SV_SHARE_ANSWERED event. """
        share = self.shared.get(share_id, None)
        if share is None:
            # oops, we got an answer for a share we don't have,
            # probably created from the web.
            # refresh the share list
            self.refresh_shares()
        else:
            share.accepted = True if answer == 'Yes' else False
            self.shared[share_id] = share

    def handle_AQ_ANSWER_SHARE_OK(self, share_id, answer):
        """ Handle successfully accepting a share """
        if answer == 'Yes':
            share = self.shares[share_id]
            self._create_fsm_object(share.path, share.volume_id, share.node_id)
            self._create_share_dir(share)
            self.m.action_q.query([(share.volume_id, str(share.node_id), "")])
            self.m.action_q.inquire_free_space(share.volume_id)

    def add_share(self, share):
        """ Add a share to the share list, and creates the fs mdobj. """
        self.log.info('Adding new share with id: %s - path: %r',
                      share.volume_id, share.path)
        if share.volume_id in self.shares:
            del self.shares[share.volume_id]
        self.shares[share.volume_id] = share
        if share.accepted:
            self._create_fsm_object(share.path, share.volume_id, share.node_id)
            self._create_share_dir(share)
            self.m.action_q.query([(share.volume_id, str(share.node_id), "")])
            self.m.action_q.inquire_free_space(share.volume_id)
            self.m.event_q.push('VM_SHARE_CREATED', share.volume_id)

    def update_free_space(self, volume_id, free_bytes):
        """ Update free space for a given share."""
        if volume_id in self.shares:
            share = self.shares[volume_id]
            share.free_bytes = free_bytes
            self.shares[volume_id] = share
        elif volume_id in self.udfs:
            root = self.shares[request.ROOT]
            root.free_bytes = free_bytes
            self.shares[request.ROOT] = root
        else:
            self.log.warning("Update of free space requested, but there is "
                             "no such volume_id: %s", volume_id)

    def get_free_space(self, volume_id):
        """Return the free_space for volume_id.

        If there is no such volume_id in the udfs or shares metadata,
        return 0.
        """
        if volume_id in self.shares:
            share = self.shares[volume_id]
            return share.free_bytes
        elif volume_id in self.udfs:
            root = self.shares[request.ROOT]
            return root.free_bytes
        else:
            self.log.warning("Requested free space of volume_id: %s, but"
                             " there is no such volume.", volume_id)
            return 0

    def accept_share(self, share_id, answer):
        """ Calls AQ.accept_share with answer ('Yes'/'No')."""
        self.log.debug("Accept share, with id: %s - answer: %s ",
                       share_id, answer)
        share = self.shares[share_id]
        share.accepted = answer
        self.shares[share_id] =  share
        answer_str = "Yes" if answer else "No"
        self.m.action_q.answer_share(share_id, answer_str)

    def share_deleted(self, share_id):
        """ process the share deleted event. """
        self.log.debug("Share (id: %s) deleted. ", share_id)
        share = self.shares.get(share_id, None)
        if share is None:
            # we don't have this share, ignore it
            self.log.warning("Got a share deleted notification (%r), "
                             "but don't have the share", share_id)
        else:
            if share.can_write():
                self._remove_watches(share.path)
            self._delete_fsm_object(share.path)
            del self.shares[share_id]
            self.m.event_q.push('VM_VOLUME_DELETED', share)

    def share_changed(self, share_holder):
        """ process the share changed event """
        # the holder id is a uuid, use the str
        share = self.shares.get(str(share_holder.share_id), None)
        if share is None:
            # we don't have this share, ignore it
            self.log.warning("Got a share changed notification (%r), "
                             "but don't have the share", share_holder.share_id)
        else:
            share.access_level = share_holder.access_level
            self.shares[share.volume_id] = share
            self.m.event_q.push('VM_SHARE_CHANGED', share.volume_id)

    def _create_share_dir(self, share):
        """ Creates the share root dir, and set the permissions. """
        # XXX: verterok: This is going to be moved into fsm
        # if the share don't exists, create it
        if not os.path.exists(share.path):
            with allow_writes(os.path.dirname(share.path)):
                os.mkdir(share.path)
            # add the watch after the mkdir
            if share.can_write():
                self.log.debug('adding inotify watch to: %s', share.path)
                self.m.event_q.inotify_add_watch(share.path)
        # if it's a ro share, change the perms
        if not share.can_write():
            os.chmod(share.path, 0555)

    def _create_fsm_object(self, path, volume_id, node_id):
        """ Creates the mdobj for this share in fs manager. """
        try:
            self.m.fs.get_by_path(path)
        except KeyError:
            self.m.fs.create(path=path, share_id=volume_id, is_dir=True)
            self.m.fs.set_node_id(path, node_id)

    def _delete_fsm_object(self, path):
        """Deletes the share and it files/folders metadata from fsm."""
        # XXX: partially implemented, this should be moved into fsm?.
        # should delete all the files in the share?
        # delete all the metadata but dont touch the files/folders
        # pylint: disable-msg=W0612
        for a_path, _ in self.m.fs.get_paths_starting_with(path):
            self.m.fs.delete_metadata(a_path)

    def _remove_watch(self, path):
        """Remove the inotify watch from path."""
        try:
            self.m.event_q.inotify_rm_watch(path)
        except (ValueError, RuntimeError, TypeError), e:
            # pyinotify has an ugly error management, if we can call
            # it that, :(. We handle this here because it's possible
            # and correct that the path is not there anymore
            self.log.warning("Error %s when trying to remove the watch"
                             " on %r", e, path)

    def _remove_watches(self, path):
        """Remove the inotify watches from path and it subdirs."""
        for a_path, is_dir in self.m.fs.get_paths_starting_with(path):
            if is_dir:
                self._remove_watch(a_path)

    def create_share(self, path, username, name, access_level):
        """ create a share for the specified path, username, name """
        self.log.debug('create share(%r, %s, %s, %s)',
                       path, username, name, access_level)
        mdobj = self.m.fs.get_by_path(path)
        mdid = mdobj.mdid
        marker = MDMarker(mdid)
        if mdobj.node_id is None:
            # we don't have the node_id yet, use the marker instead
            node_id = marker
        else:
            node_id = mdobj.node_id
        share = Shared(path=self.m.fs.get_abspath("", mdobj.path),
                      volume_id=marker,
                      name=name, access_level=access_level,
                      other_username=username, other_visible_name=None,
                      node_id=node_id)
        self.marker_share_map[marker] = share
        self.m.action_q.create_share(node_id, username, name,
                                     access_level, marker)

    def add_shared(self, share):
        """ Add a share with direction == from_me """
        self.log.info('New shared subtree: id: %s - path: %r',
                      share.volume_id, share.path)
        current_share = self.shared.get(share.volume_id)
        if current_share is None:
            self.shared[share.volume_id] = share
        else:
            for k in share.__dict__:
                setattr(current_share, k, getattr(share, k))
            self.shared[share.volume_id] = current_share

    def add_udf(self, udf):
        """Add the udf to the VM metadata if isn't there.

        If it's a new udf, create the directory, hook inotify
        and execute a query.

        """
        self.log.debug('add_udf: %s', udf)
        if self.udfs.get(udf.volume_id, None) is None:
            self.log.debug('udf not in metadata, adding it!')
            self.udfs[udf.volume_id] = udf
            self._create_fsm_object(udf.path, udf.volume_id, udf.node_id)
            # local and server rescan, this will add the inotify hooks
            # to the udf root dir and any child directory.
            if udf.subscribed:
                # temporary set to unsubscribed to do the local rescan
                udf.subscribed = False
                self.udfs[udf.volume_id] = udf
                if not os.path.exists(udf.path):
                    os.makedirs(udf.path)

                def subscribe(result):
                    """Subscribe the UDF after the local rescan.

                    As we don't wait for server rescan to finish, the udf is
                    subscribed just after the local rescan it's done.

                    """
                    udf.subscribed = True
                    self.udfs[udf.volume_id] = udf
                    return result

                d = self._scan_udf(udf)
                d.addCallback(subscribe)
            else:
                # don't scan the udf as we are not subscribed to it
                d = defer.succeed(None)

            d.addCallback(lambda _: self.m.event_q.push('VM_UDF_CREATED', udf))
            return d

    def udf_deleted(self, udf_id):
        """Delete the UDF from the VM and filesystem manager metadata"""
        self.log.info('udf_deleted: %r', udf_id)
        try:
            udf = self.udfs[udf_id]
        except KeyError:
            self.log.exception("UDF with id: %r does not exists", udf_id)
            raise
        # remove the watches
        self._remove_watches(udf.path)
        self._delete_fsm_object(udf.path)
        # remove the udf from VM metadata
        del self.udfs[udf_id]
        self.m.event_q.push('VM_VOLUME_DELETED', udf)

    def get_volume(self, id):
        """Returns the Share or UDF with the matching id."""
        volume = self.shares.get(id, None)
        if volume is None:
            volume = self.udfs.get(id, None)
        if volume is None:
            raise KeyError(id)
        return volume

    def get_volumes(self, all_volumes=False):
        """Return a generator for the list of 'active' volumes.

        This list contains subscribed UDFs and accepted Shares.

        """
        volumes = itertools.chain(self.shares.values(), self.udfs.values())
        for volume in volumes:
            if all_volumes or volume.active:
                yield volume

    def _is_nested_udf(self, path):
        """Check if it's ok to create a UDF in 'path'.

        Check if path is child or ancestor of another UDF or if
        it's inside the root.

        """
        volumes = itertools.chain([self.shares[request.ROOT]],
                                   self.udfs.values())
        for volume in volumes:
            vol_path = volume.path + os.path.sep
            if path.startswith(vol_path) or vol_path.startswith(path):
                return True
        return False

    def create_udf(self, path):
        """Request the creation of a UDF to AQ."""
        self.log.debug('create udf: %r', path)
        # check if path is the realpath, bail out if not
        if os.path.realpath(path) != path:
            self.m.event_q.push('VM_UDF_CREATE_ERROR', path,
                                "UDFs can not be a symlink")
            return
        # check if the path it's ok (outside root and
        # isn't a ancestor or child of another UDF)
        if self._is_nested_udf(path):
            self.m.event_q.push('VM_UDF_CREATE_ERROR', path,
                                "UDFs can not be nested")
            return

        try:
            udf_path, udf_name = self._get_udf_path_name(path)
        except ValueError, e:
            self.m.event_q.push('VM_UDF_CREATE_ERROR', path,
                                "INVALID_PATH: %s" % e)
        else:
            try:
                marker = MDMarker(path)
                if marker in self.marker_udf_map:
                    # ignore this request
                    self.log.warning('Duplicated create_udf request for '
                                     'path (ingoring it!): %r', udf_path)
                    return
                # the UDF it's in subscribed state by default
                subscribe = config.get_user_config().get_udf_autosubscribe()
                udf = UDF(None, None, os.path.join(udf_path, udf_name), path,
                          subscribed=subscribe)
                self.marker_udf_map[marker] = udf
                self.m.action_q.create_udf(udf_path, udf_name, marker)
            except Exception, e:
                self.m.event_q.push('VM_UDF_CREATE_ERROR', path,
                                    "UNKNOWN_ERROR: %s" % e)

    def _get_udf_path_name(self, path):
        """
        Return (path, name) from path.

        path must be a path inside the user home direactory, if it's not
        a ValueError is raised.

        """
        if not path:
            raise ValueError("no path specified")

        user_home = os.path.expanduser('~')
        start_list = os.path.abspath(user_home).split(os.path.sep)
        path_list = os.path.abspath(path).split(os.path.sep)

        # Work out how much of the filepath is shared by user_home and path.
        common_prefix = os.path.commonprefix([start_list, path_list])
        if os.path.sep + os.path.join(*common_prefix) != user_home:
            raise ValueError("path isn't inside user home: %r" % path)

        i = len(common_prefix)
        rel_list = [os.path.pardir] * (len(start_list)-i) + path_list[i:]
        relpath = os.path.join(*rel_list)
        head, tail = os.path.split(relpath)
        if head == '':
            head = '~'
        else:
            head = os.path.join('~', head)
        return head, tail

    def delete_volume(self, volume_id):
        """Request the deletion of a volume to AQ.

        if volume_id isn't in the shares or dufs shelf a KeyError is raised

        """
        self.log.info('delete_volume: %r', volume_id)
        try:
            volume = self.get_volume(volume_id)
        except KeyError:
            self.m.event_q.push('VM_VOLUME_DELETE_ERROR', volume_id,
                                "DOES_NOT_EXIST")
        else:
            self.m.action_q.delete_volume(volume.id)

    def subscribe_udf(self, udf_id):
        """Mark the UDF with id as subscribed.

        Also fire a local and server rescan.

        """
        push_error = functools.partial(self.m.event_q.push,
                                       'VM_UDF_SUBSCRIBE_ERROR', udf_id)
        self.log.info('subscribe_udf: %r', udf_id)
        try:
            udf = self.udfs[udf_id]
        except KeyError:
            push_error("DOES_NOT_EXIST")
        else:
            if not os.path.exists(udf.path):
                # the udf path isn't there, create it!
                os.makedirs(udf.path)
            def subscribe(result):
                """Subscribe the UDF after the local rescan.

                As we don't wait for server rescan to finish, the udf is
                subscribed just after the local rescan it's done.

                """
                udf.subscribed = True
                self.udfs[udf_id] = udf
                return result
            try:
                d = self._scan_udf(udf)
            except KeyError:
                push_error("METADATA_DOES_NOT_EXIST")
            else:
                d.addCallback(subscribe)
                d.addCallbacks(
                    lambda _: self.m.event_q.push('VM_UDF_SUBSCRIBED', udf),
                    lambda f: push_error(f.getErrorMessage()))
            return d

    def _scan_udf(self, udf):
        """Local and server rescan of a UDF."""
        mdobj = self.m.fs.get_by_path(udf.path)
        d = self.m.lr.scan_dir(mdobj.mdid, udf.path, udfmode=True)
        def server_rescan(_):
            """Do a query over all known nodes."""
            data = self.m.fs.get_for_server_rescan_by_path(udf.path)
            self.m.action_q.query(data)
            self.m.action_q.inquire_free_space(request.ROOT)
        d.addCallback(server_rescan)
        return d

    def unsubscribe_udf(self, udf_id):
        """Mark the UDF with udf_id as unsubscribed."""
        self.log.info('unsubscribe_udf: %r', udf_id)
        try:
            udf = self.udfs[udf_id]
        except KeyError:
            self.m.event_q.push('VM_UDF_UNSUBSCRIBE_ERROR', udf_id,
                                "DOES_NOT_EXIST")
        else:
            # remove the inotify watches, but don't delete the metadata
            self._remove_watches(udf.path)
            # makr the udf as unsubscribed
            udf.subscribed = False
            self.udfs[udf_id] = udf
            self.m.event_q.push('VM_UDF_UNSUBSCRIBED', udf)

    def handle_AQ_CREATE_UDF_OK(self, marker, volume_id, node_id):
        """Handle AQ_CREATE_UDF_OK."""
        udf = self.marker_udf_map.pop(marker)
        udf.volume_id = str(volume_id)
        udf.node_id = str(node_id)
        self.add_udf(udf)

    def handle_AQ_CREATE_UDF_ERROR(self, marker, error):
        """Handle AQ_CREATE_UDF_ERROR."""
        udf = self.marker_udf_map.pop(marker)
        self.m.event_q.push('VM_UDF_CREATE_ERROR', udf.path, str(error))

    def handle_AQ_DELETE_VOLUME_OK(self, volume_id):
        """Handle AQ_DELETE_VOLUME_OK."""
        self._handle_deleted_volume(volume_id)

    def handle_AQ_DELETE_VOLUME_ERROR(self, volume_id, error):
        """Handle AQ_DELETE_VOLUME_ERROR."""
        try:
            self.get_volume(str(volume_id))
        except KeyError:
            self.log.warning("Received a AQ_DELETE_VOLUME_ERROR of a missing"
                             "volume id")
        else:
            self.m.event_q.push('VM_VOLUME_DELETE_ERROR', volume_id, str(error))

    def handle_SV_VOLUME_CREATED(self, volume):
        """Handle SV_VOLUME_CREATED event."""
        self._handle_new_volume(volume)

    def handle_SV_VOLUME_DELETED(self, volume_id):
        """Handle SV_VOLUME_DELETED event."""
        self._handle_deleted_volume(volume_id)

    def _handle_deleted_volume(self, volume_id):
        """Handle a deleted volume.

        Call the specific method to
        handle this volume type (share/udf/root).

        """
        volume = self.get_volume(str(volume_id))
        if isinstance(volume, Share):
            self.log.debug('share deleted! %s', volume.id)
            self.share_deleted(volume.id)
        elif isinstance(volume, UDF):
            self.log.debug('udf deleted! %s', volume.id)
            self.udf_deleted(volume.id)
        else:
            self.log.warning("Tried to delete a missing volume id: %s",
                             volume_id)


class MetadataUpgrader(object):
    """A class that loads old metadata and migrate it."""

    def __init__(self, data_dir, shares_md_dir, shared_md_dir, udfs_md_dir,
                 root_dir, shares_dir, shares_dir_link):
        """Creates the instance"""
        self.log = logging.getLogger('ubuntuone.SyncDaemon.VM.MD')
        self._data_dir = data_dir
        self._shares_dir = shares_dir
        self._shares_md_dir = shares_md_dir
        self._shared_md_dir = shared_md_dir
        self._udfs_md_dir = udfs_md_dir
        self._root_dir = root_dir
        self._shares_dir_link = shares_dir_link
        self._version_file = os.path.join(self._data_dir, '.version')
        self.md_version = self._get_md_version()

    def upgrade_metadata(self):
        """Upgrade the metadata (only if it's needed)"""
        # upgrade the metadata
        if self.md_version != VolumeManager.METADATA_VERSION:
            upgrade_method = getattr(self, "_upgrade_metadata_%s" % \
                                     self.md_version)
            upgrade_method(self.md_version)

    def _get_md_version(self):
        """Returns the current md_version"""
        if not os.path.exists(self._data_dir):
            # first run, the data dir don't exist. No metadata to upgrade
            md_version = VolumeManager.METADATA_VERSION
            os.makedirs(self._data_dir)
            self.update_metadata_version()
        elif os.path.exists(self._version_file):
            with open(self._version_file) as fh:
                md_version = fh.read().strip()
            if not md_version:
                # we don't have a version of the metadata but a .version file?
                # assume it's None and do an upgrade from version 0
                md_version = self._guess_metadata_version()
        else:
            md_version = self._guess_metadata_version()
        self.log.debug('metadata version: %s', md_version)
        return md_version

    def _guess_metadata_version(self):
        """Try to guess the metadata version based on current metadata
        and layout, fallbacks to md_version = None if can't guess it.

        """
        md_version = None
        if os.path.exists(self._shares_md_dir) \
           and os.path.exists(self._shared_md_dir):
            # we have shares and shared dirs
            # md_version >= 1
            old_root_dir = os.path.join(self._root_dir, 'My Files')
            old_share_dir = os.path.join(self._root_dir, 'Shared With Me')
            if os.path.exists(old_share_dir) and os.path.exists(old_root_dir) \
               and not os.path.islink(old_share_dir):
                # md >= 1 and <= 3
                # we have a My Files dir, 'Shared With Me' isn't a
                # symlink and ~/.local/share/ubuntuone/shares doesn't
                # exists.
                # md_version <= 3, set it to 2 as it will migrate
                # .conflict to .u1conflict, and we don't need to upgrade
                # from version 1 any more as the LegacyShareFileShelf
                # takes care of that.
                md_version = '2'
            else:
                try:
                    target = os.readlink(self._shares_dir_link)
                except OSError:
                    target = None
                if os.path.islink(self._shares_dir_link) \
                   and os.path.normpath(target) == self._shares_dir_link:
                    # broken symlink, md_version = 4
                    md_version = '4'
                else:
                    # md_version >= 5
                    shelf = LegacyShareFileShelf(self._shares_md_dir)
                    # check a pickled value to check if it's in version
                    # 5 or 6
                    md_version = '5'
                    versions = {'5':0, '6':0}
                    for key in shelf:
                        share = shelf[key]
                        if isinstance(share, _Share):
                            versions['5'] += 1
                        else:
                            versions['6'] += 1
                    if versions['5'] > 0:
                        md_version = '5'
                    elif versions['6'] > 0:
                        md_version = '6'
        else:
            # this is metadata 'None'
            md_version = None
        return md_version

    def _upgrade_metadata_None(self, md_version):
        """Upgrade the shelf layout, for *very* old clients."""
        self.log.debug('Upgrading the share shelf layout')
        # the shelf already exists, and don't have a .version file
        # first backup the old data
        backup = os.path.join(self._data_dir, '0.bkp')
        if not os.path.exists(backup):
            os.makedirs(backup)
        # pylint: disable-msg=W0612
        # filter 'shares' and 'shared' dirs, in case we are in the case of
        # missing version but existing .version file
        filter_known_dirs = lambda d: d != os.path.basename(self._shares_md_dir)\
                and d != os.path.basename(self._shared_md_dir)
        for dirname, dirs, files in os.walk(self._data_dir):
            if dirname == self._data_dir:
                for dir in filter(filter_known_dirs, dirs):
                    if dir != os.path.basename(backup):
                        shutil.move(os.path.join(dirname, dir),
                                    os.path.join(backup, dir))
        # regenerate the shelf using the new layout using the backup as src
        old_shelf = LegacyShareFileShelf(backup)
        if not os.path.exists(self._shares_dir):
            os.makedirs(self._shares_dir)
        new_shelf = LegacyShareFileShelf(self._shares_md_dir)
        for key, share in old_shelf.iteritems():
            new_shelf[key] = share
        # now upgrade to metadata 2
        self._upgrade_metadata_2(md_version)

    def _upgrade_metadata_1(self, md_version):
        """Upgrade to version 2.

        Upgrade all pickled Share to the new package/module layout.

        """
        self.log.debug('upgrading share shelfs from metadata 1')
        shares = LegacyShareFileShelf(self._shares_md_dir)
        for key, share in shares.iteritems():
            shares[key] = share
        shared = LegacyShareFileShelf(self._shared_md_dir)
        for key, share in shared.iteritems():
            shared[key] = share
        # now upgrade to metadata 3
        self._upgrade_metadata_2(md_version)

    def _upgrade_metadata_2(self, md_version):
        """Upgrade to version 3

        Renames foo.conflict files to foo.u1conflict, foo.conflict.N
        to foo.u1conflict.N, foo.partial to .u1partial.foo, and
        .partial to .u1partial.

        """
        self.log.debug('upgrading from metadata 2 (bogus)')
        for top in self._root_dir, self._shares_dir:
            for dirpath, dirnames, filenames in os.walk(top):
                with allow_writes(dirpath):
                    for names in filenames, dirnames:
                        self._upgrade_names(dirpath, names)
        self._upgrade_metadata_3(md_version)

    def _upgrade_names(self, dirpath, names):
        """Do the actual renaming for _upgrade_metadata_2."""
        for pos, name in enumerate(names):
            new_name = name
            if re.match(r'.*\.partial$|\.u1partial(?:\..+)?', name):
                if name == '.partial':
                    new_name = '.u1partial'
                else:
                    new_name = re.sub(r'^(.+)\.partial$',
                                      r'.u1partial.\1', name)
                if new_name != name:
                    while os.path.lexists(os.path.join(dirpath, new_name)):
                        # very, very strange
                        self.log.warning('Found a .partial and .u1partial'
                                         ' for the same file: %s!', new_name)
                        new_name += '.1'
            elif re.search(r'\.(?:u1)?conflict(?:\.\d+)?$', name):
                new_name = re.sub(r'^(.+)\.conflict((?:\.\d+)?)$',
                                  r'\1.u1conflict\2', name)
                if new_name != name:
                    while os.path.lexists(os.path.join(dirpath, new_name)):
                        m = re.match(r'(.*\.u1conflict)((?:\.\d+)?)$', new_name)
                        base, num = m.groups()
                        if not num:
                            num = '.1'
                        else:
                            num = '.' + str(int(num[1:])+1)
                        new_name = base + num
            if new_name != name:
                old_path = os.path.join(dirpath, name)
                new_path = os.path.join(dirpath, new_name)
                self.log.debug('renaming %r to %r', old_path, new_path)
                os.rename(old_path, new_path)
                names[pos] = new_name


    def _upgrade_metadata_3(self, md_version):
        """Upgrade to version 4 (new layout!)

        move "~/Ubuntu One/Shared With" Me to XDG_DATA/ubuntuone/shares
        move "~/Ubuntu One/My Files" contents to "~/Ubuntu One"

        """
        self.log.debug('upgrading from metadata 3 (new layout)')
        old_share_dir = os.path.join(self._root_dir, 'Shared With Me')
        old_root_dir = os.path.join(self._root_dir, 'My Files')
        # change permissions
        os.chmod(self._root_dir, 0775)

        def move(src, dst):
            """Move a file/dir taking care if it's read-only."""
            prev_mode = stat.S_IMODE(os.stat(src).st_mode)
            os.chmod(src, 0755)
            shutil.move(src, dst)
            os.chmod(dst, prev_mode)

        # update the path's in metadata and move the folder
        if os.path.exists(old_share_dir) and not os.path.islink(old_share_dir):
            os.chmod(old_share_dir, 0775)
            if not os.path.exists(os.path.dirname(self._shares_dir)):
                os.makedirs(os.path.dirname(self._shares_dir))
            self.log.debug('moving shares dir from: %r to %r',
                           old_share_dir, self._shares_dir)
            for path in os.listdir(old_share_dir):
                src = os.path.join(old_share_dir, path)
                dst = os.path.join(self._shares_dir, path)
                move(src, dst)
            os.rmdir(old_share_dir)

        # update the shares metadata
        shares = LegacyShareFileShelf(self._shares_md_dir)
        for key, share in shares.iteritems():
            if share.path is not None:
                if share.path == old_root_dir:
                    share.path = share.path.replace(old_root_dir,
                                                    self._root_dir)
                else:
                    share.path = share.path.replace(old_share_dir,
                                                    self._shares_dir)
                shares[key] = share

        shared = LegacyShareFileShelf(self._shared_md_dir)
        for key, share in shared.iteritems():
            if share.path is not None:
                share.path = share.path.replace(old_root_dir, self._root_dir)
            shared[key] = share
        # move the My Files contents, taking care of dir/files with the same
        # name in the new root
        if os.path.exists(old_root_dir):
            self.log.debug('moving My Files contents to the root')
            # make My Files rw
            os.chmod(old_root_dir, 0775)
            path_join = os.path.join
            for relpath in os.listdir(old_root_dir):
                old_path = path_join(old_root_dir, relpath)
                new_path = path_join(self._root_dir, relpath)
                if os.path.exists(new_path):
                    shutil.move(new_path, new_path+'.u1conflict')
                if relpath == 'Shared With Me':
                    # remove the Shared with Me symlink inside My Files!
                    self.log.debug('removing shares symlink from old root')
                    os.remove(old_path)
                else:
                    self.log.debug('moving %r to %r', old_path, new_path)
                    move(old_path, new_path)
            self.log.debug('removing old root: %r', old_root_dir)
            os.rmdir(old_root_dir)

        # fix broken symlink (md_version 4)
        self._upgrade_metadata_4(md_version)

    def _upgrade_metadata_4(self, md_version):
        """Upgrade to version 5 (fix the broken symlink!)."""
        self.log.debug('upgrading from metadata 4 (broken symlink!)')
        if os.path.islink(self._shares_dir_link):
            target = os.readlink(self._shares_dir_link)
            if os.path.normpath(target) == self._shares_dir_link:
                # the symnlink points to itself
                self.log.debug('removing broken shares symlink: %r -> %r',
                               self._shares_dir_link, target)
                os.remove(self._shares_dir_link)
        self._upgrade_metadata_5(md_version)

    def _upgrade_metadata_5(self, md_version):
        """Upgrade to version 6 (plain dict storage)."""
        self.log.debug('upgrading from metadata 5')
        bkp_dir = os.path.join(os.path.dirname(self._data_dir), '5.bkp')
        new_md_dir = os.path.join(os.path.dirname(self._data_dir), 'md_6.new')
        new_shares_md_dir = os.path.join(new_md_dir, 'shares')
        new_shared_md_dir = os.path.join(new_md_dir, 'shared')
        new_udfs_md_dir = os.path.join(new_md_dir, 'udfs')
        try:
            # upgrade shares
            old_shares = LegacyShareFileShelf(self._shares_md_dir)
            shares = VMFileShelf(new_shares_md_dir)
            for key, share in old_shares.iteritems():
                shares[key] = self._upgrade_share_to_volume(share)
            # upgrade shared folders
            old_shared = LegacyShareFileShelf(self._shared_md_dir)
            shared = VMFileShelf(new_shared_md_dir)
            for key, share in old_shared.iteritems():
                shared[key] = self._upgrade_share_to_volume(share, shared=True)
            # upgrade the udfs
            old_udfs = LegacyShareFileShelf(self._udfs_md_dir)
            udfs = VMFileShelf(new_udfs_md_dir)
            for key, udf in old_udfs.iteritems():
                udfs[key] = UDF(udf.id, udf.node_id, udf.suggested_path,
                                udf.path, udf.subscribed)
            # move md dir to bkp
            os.rename(self._data_dir, bkp_dir)
            # move new to md dir
            os.rename(new_md_dir, self._data_dir)
            self.update_metadata_version()
        except Exception:
            # something bad happend, remove partially upgraded metadata
            shutil.rmtree(new_md_dir)
            raise

    def _upgrade_share_to_volume(self, share, shared=False):
        """Upgrade from _Share to new Volume hierarchy."""
        def upgrade_share_dict(share):
            """Upgrade share __dict__ to be compatible with the
            new Share.__init__.

            """
            if 'subtree' in share.__dict__:
                share.node_id = share.__dict__.pop('subtree')
            if 'id' in share.__dict__:
                share.volume_id = share.__dict__.pop('id')
            if 'free_bytes' in share.__dict__:
                # FIXME: REVIEWERS PLEASE CONFIRM THIS IS CORRECT
                share.free_bytes = share.__dict__.pop('free_bytes')
            else:
                share.free_bytes = None
            return share

        if isinstance(share, dict):
            # oops, we have mixed metadata. fix it!
            clazz = VMFileShelf.classes[share[VMFileShelf.TYPE]]
            share_dict = share.copy()
            del share_dict[VMFileShelf.TYPE]
            return clazz(**share_dict)
        elif share.path == self._root_dir or share.id == '':
            # handle the root special case
            return Root(share.subtree, share.path)
        else:
            share = upgrade_share_dict(share)
            if shared:
                return Shared(**share.__dict__)
            else:
                return Share(**share.__dict__)

    def update_metadata_version(self):
        """Write the version of the metadata."""
        if not os.path.exists(os.path.dirname(self._version_file)):
            os.makedirs(os.path.dirname(self._version_file))
        with open(self._version_file, 'w') as fd:
            fd.write(VolumeManager.METADATA_VERSION)
            # make sure the data get to disk
            fd.flush()
            os.fsync(fd.fileno())


@contextmanager
def allow_writes(path):
    """A very simple context manager to allow writting in RO dirs."""
    prev_mod = stat.S_IMODE(os.stat(path).st_mode)
    os.chmod(path, 0755)
    yield
    os.chmod(path, prev_mod)


class VMFileShelf(file_shelf.CachedFileShelf):
    """Custom file shelf that allow request.ROOT as key, it's replaced
    by the string: root_node_id.

    """

    TYPE = 'type'
    classes = dict((sub.__name__, sub) for sub in \
                   Volume.__subclasses__() + Share.__subclasses__())

    def __init__(self, *args, **kwargs):
        """Create the instance."""
        super(VMFileShelf, self).__init__(*args, **kwargs)
        self.key = 'root_node_id'

    def key_file(self, key):
        """Override default key_file, to handle key == request.ROOT."""
        if key == request.ROOT:
            key = self.key
        return super(VMFileShelf, self).key_file(key)

    def keys(self):
        """Override default keys, to handle key == request.ROOT."""
        for key in super(VMFileShelf, self).keys():
            if key == self.key:
                yield request.ROOT
            else:
                yield key

    def _unpickle(self, fd):
        """Custom _unpickle.

        Unpickle a dict and build the class instance specified in
        value['type'].

        """
        value = cPickle.load(fd)
        class_name = value[self.TYPE]
        clazz = self.classes[class_name]
        obj = clazz.__new__(clazz)
        obj.__dict__.update(value)
        return obj

    def _pickle(self, value, fd, protocol):
        """Pickle value in fd using protocol."""
        cPickle.dump(value.__dict__, fd, protocol=protocol)


class LegacyShareFileShelf(VMFileShelf):
    """A FileShelf capable of replacing pickled classes
    with a different class.

    upgrade_map attribute is a dict of (module, name):class

    """

    upgrade_map = {
        ('ubuntuone.syncdaemon.volume_manager', 'UDF'):_UDF,
        ('ubuntuone.syncdaemon.volume_manager', 'Share'):_Share,
        ('canonical.ubuntuone.storage.syncdaemon.volume_manager',
         'Share'):_Share
    }

    def _find_global(self, module, name):
        """Returns the class object for (module, name) or None."""
        # handle our 'migration types'
        if (module, name) in self.upgrade_map:
            return self.upgrade_map[(module, name)]
        else:
            # handle all other types
            __import__(module)
            return getattr(sys.modules[module], name)

    def _unpickle(self, fd):
        """Override default _unpickle with one capable of migrating pickled classes."""
        unpickler = cPickle.Unpickler(fd)
        unpickler.find_global = self._find_global
        value = unpickler.load()
        return value

    def _pickle(self, value, fd, protocol):
        """Pickle value in fd using protocol."""
        cPickle.dump(value, fd, protocol=protocol)
