/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*-
 *
 * Authors: Alejandro J. Cura <alecu@canonical.com>
 *
 * Copyright (C) 2010 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3,
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "config.h"

#include <glib/gi18n-lib.h>
#include <gmodule.h>
#include <gtk/gtk.h>

#include "gnome-settings-daemon/gnome-settings-plugin.h"
#include "gsd-ubuntuone.h"

#ifdef REGISTER_PLUGIN
GNOME_SETTINGS_PLUGIN_REGISTER (GsdUbuntuOne, gsd_ubuntuone)
#else
G_DEFINE_TYPE (GsdUbuntuOne, gsd_ubuntuone, G_TYPE_OBJECT);
#endif

#define NO_SPACE _("Your Ubuntu One storage is full. Follow the link below to upgrade your subscription.")
#define NO_SPACE_SHARE _("There is no available space on the folder:\n\"%s\" shared by %s")
#define NO_SPACE_TITLE _("Out of space")
#define UPGRADE_SUBSCRIPTION_TEXT _("Upgrade Subscription")
#define UPGRADE_SUBSCRIPTION_URI "http://one.ubuntu.com/plans/"

static void
dialog_closed_callback (GtkDialog *dialog,
                        gint response_id,
                        gpointer user_data)
{
	GsdUbuntuOne *plugin = GSD_UBUNTUONE (user_data);

	g_debug ("dialog closed %d", response_id);
	gtk_widget_destroy (GTK_WIDGET (plugin->out_of_space_dialog));
	plugin->out_of_space_dialog = NULL;
}

static void
show_out_of_space_dialog (GsdUbuntuOne *plugin,
			  const gchar *title,
			  const gchar *body,
			  gboolean show_upgrade_link)
{
	GError *error = NULL;

	if (plugin->out_of_space_dialog != NULL) {
		gtk_widget_destroy (GTK_WIDGET (plugin->out_of_space_dialog));
		plugin->out_of_space_dialog = NULL;
	}
	
	g_debug ("notification: %s - %s", title, body);

	plugin->out_of_space_dialog = gtk_message_dialog_new (NULL,
							      GTK_DIALOG_NO_SEPARATOR,
							      GTK_MESSAGE_WARNING,
							      GTK_BUTTONS_CLOSE,
							      "%s",
							      body);

	gtk_window_set_skip_taskbar_hint (GTK_WINDOW (plugin->out_of_space_dialog), FALSE);
	gtk_window_set_title (GTK_WINDOW (plugin->out_of_space_dialog), title);
	gtk_window_set_icon_name (GTK_WINDOW (plugin->out_of_space_dialog), "ubuntuone");
	gtk_window_set_position (GTK_WINDOW (plugin->out_of_space_dialog), GTK_WIN_POS_CENTER);

        if (show_upgrade_link) {
                GtkWidget *upgrade_link;

                upgrade_link = gtk_link_button_new_with_label (UPGRADE_SUBSCRIPTION_URI,
                                                               UPGRADE_SUBSCRIPTION_TEXT);
                gtk_widget_show (GTK_WIDGET (upgrade_link));
                gtk_dialog_add_action_widget (GTK_DIALOG (plugin->out_of_space_dialog),
                                              upgrade_link, 0);
        }

	g_signal_connect (G_OBJECT (plugin->out_of_space_dialog), "response",
                          G_CALLBACK (dialog_closed_callback), plugin);
	gtk_widget_show (plugin->out_of_space_dialog);
}


static void
quota_exceeded_callback (SyncdaemonDaemon *daemon,
			 GHashTable *file_info,
			 gpointer user_data)
{
	gchar * volume_type;
	GsdUbuntuOne *plugin = GSD_UBUNTUONE (user_data);

	volume_type = g_hash_table_lookup (file_info, "type");
	if (g_strcmp0 (volume_type, "Share") == 0) {
		gchar * other_visible_name, * path, * message;

		other_visible_name = g_hash_table_lookup (file_info, "other_visible_name");
		path = g_hash_table_lookup (file_info, "path");
		message = g_strdup_printf (NO_SPACE_SHARE, path, other_visible_name);
		show_out_of_space_dialog (plugin, NO_SPACE_TITLE, message, FALSE);
		g_free (message);
	} else {
		show_out_of_space_dialog (plugin, NO_SPACE_TITLE, NO_SPACE, TRUE);
	}
}

static gboolean
delayed_syncdaemon_start (gpointer data)
{
	GsdUbuntuOne *plugin;

	plugin = GSD_UBUNTUONE (data);
	g_debug ("Performing delayed syncdaemon init");

	plugin->syncdaemon = syncdaemon_daemon_new ();
	plugin->out_of_space_dialog = NULL;
	g_signal_connect (G_OBJECT (plugin->syncdaemon), "quota_exceeded",
			  G_CALLBACK (quota_exceeded_callback), plugin);
	return FALSE;
}

static void
gsd_ubuntuone_init (GsdUbuntuOne *plugin)
{
}

void
gsd_ubuntuone_activate (GnomeSettingsPlugin *gsp_object)
{
	GsdUbuntuOne *plugin = GSD_UBUNTUONE (gsp_object);

	g_timeout_add (DELAYED_START_TIMEOUT, delayed_syncdaemon_start, plugin);
}

static void
gsd_ubuntuone_dispose (GObject *object)
{
	GsdUbuntuOne *plugin = GSD_UBUNTUONE (object);
	GsdUbuntuOneClass *klass = GSD_UBUNTUONE_GET_CLASS (object);
	GObjectClass *parent_class = G_OBJECT_CLASS (klass);

	if (plugin->syncdaemon)
	      g_object_unref (plugin->syncdaemon);

	parent_class->dispose (object);
}

static void
gsd_ubuntuone_class_init (GsdUbuntuOneClass *klass)
{
	GObjectClass *g_class = G_OBJECT_CLASS (klass);
	GnomeSettingsPluginClass *gsp_class = (GnomeSettingsPluginClass *) klass;

	g_class->dispose = gsd_ubuntuone_dispose;
	gsp_class->activate = gsd_ubuntuone_activate;
}
