/*
 * Syncdaemon API
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * Copyright 2010 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "config.h"
#include "syncdaemon-publicfiles-interface.h"
#include "utils.h"

G_DEFINE_TYPE(SyncdaemonPublicfilesInterface, syncdaemon_publicfiles_interface, SYNCDAEMON_TYPE_INTERFACE)

struct _SyncdaemonPublicfilesInterfacePrivate {
	GObject *proxy;
};

static void
syncdaemon_publicfiles_interface_finalize (GObject *object)
{
	SyncdaemonPublicfilesInterface *interface = SYNCDAEMON_PUBLICFILES_INTERFACE (object);

	if (interface->priv != NULL) {
		g_free (interface->priv);
	}

	G_OBJECT_CLASS (syncdaemon_publicfiles_interface_parent_class)->finalize (object);
}

static void
syncdaemon_publicfiles_interface_class_init (SyncdaemonPublicfilesInterfaceClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = syncdaemon_publicfiles_interface_finalize;
}

static void
file_published_cb (DBusGProxy *proxy, GHashTable *hash, gpointer user_data)
{
	SyncdaemonDaemon *daemon = NULL;
	SyncdaemonPublicfilesInterface *interface = SYNCDAEMON_PUBLICFILES_INTERFACE (user_data);

	g_object_get (G_OBJECT (interface), "daemon", &daemon, NULL);
	if (daemon != NULL) {
		SyncdaemonFileInfo *finfo;

		finfo = syncdaemon_file_info_new_from_hash_table (hash);
		g_signal_emit_by_name (daemon, "file_published", TRUE, finfo);

		g_object_unref (G_OBJECT (finfo));
	}
}

static void
got_public_files_cb (DBusGProxy *proxy, GPtrArray *files, gpointer user_data)
{
	SyncdaemonDaemon *daemon = NULL;
	SyncdaemonPublicfilesInterface *interface = SYNCDAEMON_PUBLICFILES_INTERFACE (user_data);

	g_object_get (G_OBJECT (interface), "daemon", &daemon, NULL);
	if (daemon != NULL) {
		GSList *finfo_list = NULL;
		guint i;

		for (i = 0; i < files->len; i++) {
			SyncdaemonFileInfo *finfo = syncdaemon_file_info_new_from_hash_table (g_ptr_array_index (files, i));

			finfo_list = g_slist_append (finfo_list, finfo);
		}

		g_signal_emit_by_name (daemon, "got_published_files", TRUE, finfo_list);

		g_slist_foreach (finfo_list, (GFunc) g_object_unref, NULL);
		g_slist_free (finfo_list);
	}
}

static void
public_access_change_error_cb (DBusGProxy *proxy, GHashTable *hash, gchar *error, gpointer user_data)
{
	SyncdaemonDaemon *daemon = NULL;
	SyncdaemonPublicfilesInterface *interface = SYNCDAEMON_PUBLICFILES_INTERFACE (user_data);

	g_object_get (G_OBJECT (interface), "daemon", &daemon, NULL);
	if (daemon != NULL) {
		SyncdaemonFileInfo *finfo;

		finfo = syncdaemon_file_info_new_from_hash_table (hash);
		g_signal_emit_by_name (daemon, "file_published", FALSE, finfo);

		g_object_unref (G_OBJECT (finfo));
	}
}

static void
public_files_list_error_cb (DBusGProxy *proxy, gchar *error, gpointer user_data)
{
	SyncdaemonDaemon *daemon = NULL;
	SyncdaemonPublicfilesInterface *interface = SYNCDAEMON_PUBLICFILES_INTERFACE (user_data);

	g_object_get (G_OBJECT (interface), "daemon", &daemon, NULL);
	if (daemon != NULL)
		g_signal_emit_by_name (daemon, "got_published_files", FALSE, NULL);
}

static void
syncdaemon_publicfiles_interface_init (SyncdaemonPublicfilesInterface *interface)
{
	interface->priv = g_new0 (SyncdaemonPublicfilesInterfacePrivate, 1);

	/* Setup DBus proxy */
	interface->priv->proxy = syncdaemon_interface_setup_proxy (SYNCDAEMON_INTERFACE (interface),
								   "com.ubuntuone.SyncDaemon",
								   "/publicfiles", "com.ubuntuone.SyncDaemon.PublicFiles");
	if (interface->priv->proxy != NULL) {
		/* Connect to DBus signals */
		dbus_g_proxy_add_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicAccessChanged",
					 dbus_g_type_get_map ("GHashTable",
							      G_TYPE_STRING,
							      G_TYPE_STRING),
					 G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicAccessChanged",
					     G_CALLBACK (file_published_cb), interface, NULL);

		dbus_g_proxy_add_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicAccessChangeError",
					 dbus_g_type_get_map ("GHashTable",
							      G_TYPE_STRING,
							      G_TYPE_STRING),
					 G_TYPE_STRING,
					 G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicAccessChangeError",
					     G_CALLBACK (public_access_change_error_cb), interface, NULL);

		dbus_g_proxy_add_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicFilesList",
					 dbus_g_type_get_collection ("GPtrArray",
								     dbus_g_type_get_map
								     ("GHashTable",
								      G_TYPE_STRING,
								      G_TYPE_STRING)),
					 G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicFilesList",
					     G_CALLBACK (got_public_files_cb), interface, NULL);

		dbus_g_proxy_add_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicFilesListError",
					 G_TYPE_STRING,
					 G_TYPE_INVALID);
		dbus_g_proxy_connect_signal (DBUS_G_PROXY (interface->priv->proxy), "PublicFilesListError",
					     G_CALLBACK (public_files_list_error_cb), interface, NULL);
	}
}

/**
 * syncdaemon_publicfiles_interface_new:
 */
SyncdaemonPublicfilesInterface *
syncdaemon_publicfiles_interface_new (SyncdaemonDaemon *daemon)
{
	g_return_val_if_fail (SYNCDAEMON_IS_DAEMON (daemon), NULL);

	return g_object_new (SYNCDAEMON_TYPE_PUBLICFILES_INTERFACE, "daemon", daemon, NULL);
}

/**
 * syncdaemon_publicfiles_interface_change_public_access:
 */
void
syncdaemon_publicfiles_interface_change_public_access (SyncdaemonPublicfilesInterface *interface,
						       const gchar *share_id,
						       const gchar *node_id,
						       gboolean is_public)
{
	g_return_if_fail (SYNCDAEMON_IS_PUBLICFILES_INTERFACE (interface));
	g_return_if_fail (share_id != NULL);
	g_return_if_fail (node_id != NULL);

	dbus_g_proxy_begin_call (DBUS_G_PROXY (interface->priv->proxy), "change_public_access",
				 no_output_dbus_call_ended_cb, interface, NULL,
				 G_TYPE_STRING, share_id,
				 G_TYPE_STRING, node_id,
				 G_TYPE_BOOLEAN, is_public,
				 G_TYPE_INVALID);
}

/**
 * syncdaemon_publicfiles_interface_get_public_files:
 */
void
syncdaemon_publicfiles_interface_get_public_files (SyncdaemonPublicfilesInterface *interface)
{
	g_return_if_fail (SYNCDAEMON_IS_PUBLICFILES_INTERFACE (interface));

	dbus_g_proxy_begin_call (DBUS_G_PROXY (interface->priv->proxy), "get_public_files",
				 no_output_dbus_call_ended_cb, interface, NULL,
				 G_TYPE_INVALID);
}
