# -*- coding: utf-8 *-*

# Copyright 2012-2013 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The user interface for the control panel for Ubuntu One."""

import os

from PyQt4 import QtGui, QtCore
from twisted.internet.defer import inlineCallbacks

from ubuntuone.platform import expand_user

from ubuntuone.controlpanel.logger import setup_logging
from ubuntuone.controlpanel.gui import (
    COPY_LINK,
    OPEN,
    SHARE_FILE,
    NO_RESULTS_LABEL,
    SEARCH_FILES,
    SEARCH_RESULTS,
    SHARED_FILES,
)

from ubuntuone.controlpanel.gui.qt.ui import (
    images_rc,
    share_links_ui,
)
from ubuntuone.controlpanel.gui.qt.share_file import ShareFileWidget
from ubuntuone.controlpanel.gui.qt.share_links_search import (
    get_system_icon_for_filename,
)
from ubuntuone.controlpanel.gui.qt.ubuntuonebin import UbuntuOneBin

# Avoid unused import warnings
assert(images_rc)

logger = setup_logging('qt.share_links')


# Table items index
FILE_NAME_COL = 0
PUBLIC_LINK_COL = 1
ACTIONS_COL = 2
# StackedLayout widgets index
SHARED_FILES_INDEX = 0
FILE_DETAILS_INDEX = 1
SEARCH_RESULTS_INDEX = 2


class ShareLinksPanel(UbuntuOneBin):
    """The Share Links Tab Panel widget"""

    ui_class = share_links_ui
    logger = logger
    _enhanced_line = None
    home_dir = ''
    _shared_files = {}

    def _setup(self):
        """Do some extra setupping for the UI."""
        super(ShareLinksPanel, self)._setup()
        self.home_dir = ''
        self.ui.search_files_lbl.setText(SEARCH_FILES)
        self.ui.shared_group.setTitle(SHARED_FILES)
        self.ui.btn_search.setIcon(QtGui.QIcon(":/search.png"))
        self.ui.label_no_results.setText(NO_RESULTS_LABEL)
        self.ui.label_no_results.hide()
        self.ui.page_search.layout().setAlignment(QtCore.Qt.AlignTop)

        # Set enhanced line edits
        self._enhanced_line = EnhancedLineEdit(self.ui.line_search,
            self._line_close_btn, icon=":/delete_search.png",
            style='enhanced_borderless')
        self.ui.line_search.popup.popupShown.connect(
            self._hide_line_btn_close_show)
        EnhancedLineEdit(self.ui.line_copy_link, self._copy_link_from_line,
            text=COPY_LINK)

        # Set cursor type
        self.ui.back_to_file_list.setCursor(QtCore.Qt.PointingHandCursor)
        self.ui.back_to_file_list_2.setCursor(QtCore.Qt.PointingHandCursor)
        self.ui.open_in_browser.setCursor(QtCore.Qt.PointingHandCursor)
        self._enhanced_line.btn_operation.setCursor(
            QtCore.Qt.ArrowCursor)

        self.ui.line_search.itemSelected.connect(self.share_file)
        self.ui.line_search.filesFound.connect(self._load_search_results)
        self.ui.back_to_file_list.clicked.connect(self._move_to_main_list)
        self.ui.back_to_file_list_2.clicked.connect(self._move_to_main_list)
        self.ui.open_in_browser.clicked.connect(self._open_in_browser)
        self.ui.tree_shared_files.itemDoubleClicked.connect(
            self._open_properties_selected_file)

        self.get_public_files()
        self._enhanced_line.btn_operation.hide()

    def handle_current_tab_changed(self, index):
        """Workaround for bug LP: 1152388"""
        self.ui.line_search.clearFocus()
        self.ui.line_search.popup.hide()

    @inlineCallbacks
    def share_file(self, file_path):
        """Clean the previous file share details and publish file_path."""
        self.backend.set_public_access_changed_handler(self._file_shared)
        self.backend.set_public_access_change_error_handler(
            lambda: self._set_is_processing(False))
        file_path = unicode(file_path)
        if self.ui.hbox_share_file.count() > 0:
            widget = self.ui.hbox_share_file.takeAt(0).widget()
            widget.close()
        full_path = expand_user(file_path.encode('utf-8')).decode('utf-8')
        if full_path not in self._shared_files:
            self.is_processing = True
            share_file_widget = ShareFileWidget(file_path)
            self.ui.hbox_share_file.addWidget(share_file_widget)
            share_file_widget.linkDisabled.connect(
                lambda: self.ui.line_copy_link.setText(''))
            yield self.backend.change_public_access(
                full_path, True)
        else:
            self.open_file_properties(file_path, full_path)

    def _open_properties_selected_file(self, item):
        """Open the file properties on double click."""
        file_path = unicode(item.toolTip(FILE_NAME_COL))
        full_path = expand_user(file_path.encode('utf-8')).decode('utf-8')
        if self.ui.hbox_share_file.count() > 0:
            widget = self.ui.hbox_share_file.takeAt(0).widget()
            widget.close()
        self.open_file_properties(file_path, full_path)

    def open_file_properties(self, file_path, full_path):
        """Move the stacked widget to the properties section."""
        share_file_widget = ShareFileWidget(file_path)
        self.ui.hbox_share_file.addWidget(share_file_widget)
        share_file_widget.linkDisabled.connect(
            lambda: self.ui.line_copy_link.setText(''))
        self.ui.line_copy_link.setText(self._shared_files[full_path])
        self.ui.stacked_widget.setCurrentIndex(FILE_DETAILS_INDEX)

    def _file_shared(self, info):
        """Receive the notification that the file has been published."""
        url = info.get("public_url")
        self.ui.line_copy_link.setText(url)
        self.ui.stacked_widget.setCurrentIndex(FILE_DETAILS_INDEX)
        self.is_processing = False

    def _open_in_browser(self):
        """Open the link in line_copy_link in the browser."""
        url = self.ui.line_copy_link.text()
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(url))

    def _copy_link_from_line(self):
        """Copy link into clipboard from line edit."""
        app = QtGui.QApplication.instance()
        app.clipboard().setText(self.ui.line_copy_link.text())

    def _move_to_main_list(self):
        """Set the share files list as current widget."""
        self.ui.stacked_widget.setCurrentIndex(SHARED_FILES_INDEX)
        self.get_public_files()

    @inlineCallbacks
    def get_public_files(self):
        """Request the list of public files."""
        self.backend.set_public_files_list_handler(self._load_public_files)
        self.is_processing = True
        self.home_dir = yield self.backend.get_home_dir()
        yield self.backend.get_public_files()

    def _load_public_files(self, publicfiles):
        """Load the list of public files."""
        self.ui.tree_shared_files.clear()
        self._shared_files = {}
        for pfile in publicfiles:
            item = QtGui.QTreeWidgetItem()
            path = pfile['path']
            public_url = pfile['public_url']
            self._shared_files[path] = public_url
            name = os.path.basename(path)
            item.setText(FILE_NAME_COL, name)
            tooltip = path
            if tooltip.startswith(self.home_dir):
                tooltip = tooltip.replace(self.home_dir, '~', 1)
            item.setToolTip(FILE_NAME_COL, tooltip)
            icon = get_system_icon_for_filename(path.encode('utf-8'))
            item.setIcon(FILE_NAME_COL, icon)

            self.ui.tree_shared_files.setColumnWidth(PUBLIC_LINK_COL, 300)
            item.setSizeHint(FILE_NAME_COL, QtCore.QSize(-1, 35))
            self.ui.tree_shared_files.addTopLevelItem(item)

            link = ('<a href="%s"><span style="font-size: 12px;'
                    'color: #dd4814";>%s</span></a>'
                    % (public_url, public_url))
            label = QtGui.QLabel(link, self.ui.tree_shared_files)
            label.setOpenExternalLinks(True)
            self.ui.tree_shared_files.setItemWidget(item, PUBLIC_LINK_COL,
                label)

            actions = ActionsButtons(path, public_url,
                parent=self.ui.tree_shared_files)
            self.ui.tree_shared_files.setItemWidget(item, ACTIONS_COL, actions)
        self.is_processing = False

    def _load_search_results(self, results, prefix):
        """Load the search results from the search box."""
        self.ui.tree_search_results.clear()
        self.ui.label_search_results.setText(SEARCH_RESULTS % prefix)
        if len(results) > 0:
            for result in results:
                item = QtGui.QTreeWidgetItem()
                name = os.path.basename(result)
                item.setText(FILE_NAME_COL, name)
                tooltip = result
                if tooltip.startswith(self.home_dir):
                    tooltip = tooltip.replace(self.home_dir, '~', 1)
                item.setToolTip(FILE_NAME_COL, tooltip)
                icon = get_system_icon_for_filename(result.encode('utf-8'))
                item.setIcon(FILE_NAME_COL, icon)
                self.ui.tree_search_results.setColumnWidth(FILE_NAME_COL, 300)

                item.setSizeHint(FILE_NAME_COL, QtCore.QSize(-1, 35))
                self.ui.tree_search_results.addTopLevelItem(item)

                actions = ActionsResultsButtons(result, self.share_file,
                    self.ui.tree_search_results)
                self.ui.tree_search_results.setItemWidget(item,
                    PUBLIC_LINK_COL, actions)
            self.ui.tree_search_results.show()
            self.ui.label_no_results.hide()
        else:
            self.ui.tree_search_results.hide()
            self.ui.label_no_results.show()
        self.ui.stacked_widget.setCurrentIndex(SEARCH_RESULTS_INDEX)

    def _line_close_btn(self):
        """Close button in the line edit was pressed, hide the popup."""
        self.ui.line_search.setText('')
        self.ui.line_search.popup.hide()
        self.ui.line_search.setFocus()
        self._enhanced_line.btn_operation.hide()

    def _hide_line_btn_close_show(self):
        """Show the button inside the search line edit-"""
        self._enhanced_line.btn_operation.show()


class ActionsButtons(QtGui.QWidget):
    """Widget that contains the open and copy link actions on the list."""

    def __init__(self, path, link, parent=None, both_buttons=True):
        super(ActionsButtons, self).__init__(parent)
        self.path = path
        self.link = link

        self.hbox = QtGui.QHBoxLayout(self)
        self.hbox.addSpacerItem(QtGui.QSpacerItem(1, 0,
            QtGui.QSizePolicy.Expanding))

        btn_open = QtGui.QPushButton(OPEN)
        btn_open.setObjectName('action_button')
        self.hbox.addWidget(btn_open)
        btn_open.clicked.connect(self.open)
        if both_buttons:
            btn_copy = QtGui.QPushButton(COPY_LINK)
            btn_copy.setObjectName('action_button')
            self.hbox.addWidget(btn_copy)
            btn_copy.clicked.connect(self.copy)

    def open(self):
        """Open the file."""
        file_path = u'file://%s' % self.path
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(file_path))

    def copy(self):
        """Copy the public link of the file in the clipboard."""
        app = QtGui.QApplication.instance()
        app.clipboard().setText(self.link)


class ActionsResultsButtons(ActionsButtons):
    """Widget that contains: open and publish actions on the results list."""

    def __init__(self, path, publish_function=None, parent=None):
        super(ActionsResultsButtons, self).__init__(path, '', parent, False)
        self.publish_function = publish_function

        btn_publish = QtGui.QPushButton(SHARE_FILE)
        btn_publish.setObjectName('action_button')
        self.hbox.addWidget(btn_publish)
        btn_publish.clicked.connect(self.publish_file)

    def publish_file(self):
        """Publish the file specified."""
        if self.publish_function is not None:
            self.publish_function(self.path)


class EnhancedLineEdit(object):
    """Add a button inside the QLineEdit received."""

    def __init__(self, line_edit, operation, text=None, icon=None, style=None):
        hbox = QtGui.QHBoxLayout(line_edit)
        hbox.setMargin(0)
        line_edit.setLayout(hbox)
        hbox.addStretch()
        self.btn_operation = QtGui.QPushButton(line_edit)
        if text:
            self.btn_operation.setText(text)
        if icon:
            self.btn_operation.setIcon(QtGui.QIcon(icon))
        if style:
            self.btn_operation.setObjectName(style)
        hbox.addWidget(self.btn_operation)
        self.btn_operation.clicked.connect(operation)
