# -*- coding: utf-8 -*-
#
# Copyright 2011-2012 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The base widget for the Control Panel tabs."""

from PyQt4 import QtGui, QtCore

from twisted.internet import defer
from ubuntu_sso.qt import LINK_STYLE
from ubuntu_sso.qt.sso_wizard_page import BaseWizardPage
from ubuntu_sso.utils.ui import CLOSE_AND_SETUP_LATER

from ubuntuone.controlpanel import cache, utils
from ubuntuone.controlpanel.logger import log_call, setup_logging
from ubuntuone.controlpanel.gui import (
    APP_NAME,
    ARE_YOU_SURE_HELP,
    ARE_YOU_SURE_NO,
    ARE_YOU_SURE_SUBTITLE,
    ARE_YOU_SURE_TITLE,
    ARE_YOU_SURE_YES,
    CLOUD_TO_COMPUTER_SUBTITLE,
    CLOUD_TO_COMPUTER_TITLE,
    COMPUTER_TO_CLOUD_SUBTITLE,
    COMPUTER_TO_CLOUD_TITLE,
    LICENSE_AGREE,
    LICENSE_AGREEMENT,
    LICENSE_BASIC,
    LICENSE_DISAGREE,
    LICENSE_GPL3,
    LICENSE_LINK,
    UBUNTUONE_LINK,
)
from ubuntuone.controlpanel.gui.qt import handle_errors
from ubuntuone.controlpanel.gui.qt.folders import (
    RemoteFoldersPanel,
    LocalFoldersPanel,
)
from ubuntuone.controlpanel.gui.qt.preferences import PreferencesPanel
from ubuntuone.controlpanel.gui.qt.signin import SignInPanel
from ubuntuone.controlpanel.gui.qt.side_widget import SideWidget
from ubuntuone.controlpanel.gui.qt.ui import are_you_sure_ui


logger = setup_logging('qt.wizard')


GPL_URL = u'http://www.gnu.org/licenses'
GPL_LINK = LINK_STYLE.format(link_url=GPL_URL, link_text=GPL_URL)
LICENSE_CONTENT = u"""<html>
    <body style="font-family:Ubuntu">
        <h2>{license_agreement}</h2>
        <p>{license_gpl3}</p>
        <p>{license_basic}</p>
        <p>{license_link}</p>
    </body>
<html>""".format(license_agreement=LICENSE_AGREEMENT,
    license_gpl3=LICENSE_GPL3, license_basic=LICENSE_BASIC,
    license_link=LICENSE_LINK.format(license_link=GPL_LINK),
)


class AreYouSure(QtGui.QDialog):

    """A 'Are you sure?' dialog."""

    def __init__(self, *args, **kwargs):
        super(AreYouSure, self).__init__(*args, **kwargs)
        self.ui = are_you_sure_ui.Ui_Dialog()
        self.ui.setupUi(self)
        self.setWindowTitle(APP_NAME)

        self.ui.title_label.setText(ARE_YOU_SURE_TITLE)

        link = LINK_STYLE.format(link_url=UBUNTUONE_LINK,
                                 link_text=UBUNTUONE_LINK)
        msg = u'%s<p>%s' % (ARE_YOU_SURE_SUBTITLE,
                            ARE_YOU_SURE_HELP.format(support_url=link))
        self.ui.message_label.setText(msg)

        self.ui.yes_button.setText(ARE_YOU_SURE_YES)
        self.ui.no_button.setText(ARE_YOU_SURE_NO)


class UbuntuOnePage(BaseWizardPage):
    """A generic page for the UbuntuOneWizard."""

    is_final = False
    main_title = None
    max_width = 5000
    panel_class = QtGui.QFrame
    sub_title = None

    def __init__(self, *args, **kwargs):
        super(UbuntuOnePage, self).__init__(*args, **kwargs)

        self.panel = None
        if self.panel_class is not None:
            self.panel = self.panel_class()
            self.layout().addWidget(self.panel)

        if self.main_title is not None:
            self.setTitle(self.main_title)

        if self.sub_title is not None:
            self.setSubTitle(self.sub_title)

        self.form_errors_label.hide()
        self.setFinalPage(self.is_final)


class LicensePage(UbuntuOnePage):
    """The page to show the license."""

    panel_class = QtGui.QTextBrowser

    def __init__(self, *args, **kwargs):
        super(LicensePage, self).__init__(*args, **kwargs)
        self.panel.setHtml(LICENSE_CONTENT)


class SignInPage(UbuntuOnePage):
    """The page to signin to Ubuntu One."""

    panel_class = SignInPanel


class CloudToComputerPage(UbuntuOnePage):
    """The page to choose cloud folders to sync locally."""

    main_title = CLOUD_TO_COMPUTER_TITLE
    panel_class = RemoteFoldersPanel
    sub_title = CLOUD_TO_COMPUTER_SUBTITLE

    def __init__(self, *args, **kwargs):
        super(CloudToComputerPage, self).__init__(*args, **kwargs)
        self.panel.ui.add_folder_button.hide()


class SettingsPage(UbuntuOnePage):
    """The page to adjust the service settings."""

    panel_class = PreferencesPanel


class ComputerToCloudPage(UbuntuOnePage):
    """The page to choose local folders to sync remotly."""

    is_final = True
    main_title = COMPUTER_TO_CLOUD_TITLE
    panel_class = LocalFoldersPanel
    sub_title = COMPUTER_TO_CLOUD_SUBTITLE


class UbuntuOneWizard(cache.Cache, QtGui.QWizard):
    """The Ubuntu One wizard."""

    def __init__(self, *args, **kwargs):
        super(UbuntuOneWizard, self).__init__(*args, **kwargs)
        self.pages = {}

        self.setOption(self.NoBackButtonOnStartPage, True)
        self.setOption(self.HaveFinishButtonOnEarlyPages, False)
        self.setWizardStyle(self.ModernStyle)

        self.confirm_dialog = AreYouSure(self)

        self.side_widget = SideWidget()
        self.side_widget.stage = self.side_widget.signin_stage
        self.setSideWidget(self.side_widget)

        # license
        self.license_page = LicensePage()
        self.license_page.setCommitPage(True)
        self.addPage(self.license_page)

        # sign in
        self.signin_page = SignInPage()
        self.addPage(self.signin_page)

        self.signin_page.panel.ui.login_button.clicked.connect(self.login)
        self.signin_page.panel.ui.register_button.clicked.connect(
            self.register)

        # cloud to compuer
        self.cloud_folders_page = CloudToComputerPage()
        self.addPage(self.cloud_folders_page)

        self.cloud_folders_page.panel.ui.check_settings_button.clicked.connect(
            self.check_settings)

        # settings
        self.settings_page = SettingsPage()
        self.addPage(self.settings_page)

        # computer to cloud
        self.local_folders_page = ComputerToCloudPage()
        self.addPage(self.local_folders_page)

        self._next_id = self.pages[self.signin_page]
        self.setStartId(self._next_id)

    # pylint: disable=C0103

    def addPage(self, page):
        """Add 'page' to this wizard, and keep a reference of its ID."""
        page_id = super(UbuntuOneWizard, self).addPage(page)
        self.pages[page] = page_id

    @log_call(logger.info)
    def initializePage(self, page_id):
        """The wizard will show the page 'page_id'."""
        page = self.page(page_id)
        logger.debug('UbuntuOneWizard.initializePage: page is %r.', page)

        button_layout = button_to = button = stage = None

        if page is self.license_page:
            button_layout = [self.Stretch, self.CancelButton,
                self.CommitButton]
            button = self.button(self.CommitButton)
            button_to = self.button(self.CancelButton)
            stage = self.side_widget.install_stage
            self._next_id = self.pages[self.signin_page]

            self.setButtonText(self.CommitButton, LICENSE_AGREE)
            self.setButtonText(self.CancelButton, LICENSE_DISAGREE)

        elif page is self.signin_page:
            button_layout = [self.Stretch, self.CancelButton]
            button = self.signin_page.panel.ui.register_button
            button_to = self.button(self.CancelButton)
            stage = self.side_widget.signin_stage
            self._next_id = self.pages[self.cloud_folders_page]

            self.setButtonText(self.CancelButton, CLOSE_AND_SETUP_LATER)
            self.setTabOrder(self.signin_page.panel.ui.login_button, button)

        elif page is self.cloud_folders_page:
            button_layout = [self.Stretch, self.NextButton]
            button = self.cloud_folders_page.panel.ui.check_settings_button
            button_to = self.button(self.NextButton)
            stage = self.side_widget.folders_stage
            self._next_id = self.pages[self.local_folders_page]
        elif page is self.settings_page:
            button_layout = [self.Stretch, self.BackButton]
            button = self.settings_page.panel.ui.apply_changes_button
            button_to = self.button(self.BackButton)
            stage = self.side_widget.folders_stage
            self._next_id = self.pages[self.cloud_folders_page]
        elif page is self.local_folders_page:
            button_layout = [self.Stretch, self.BackButton, self.FinishButton]
            button = self.local_folders_page.panel.ui.add_folder_button
            button_to = self.button(self.BackButton)
            stage = self.side_widget.sync_stage
        else:
            logger.error('UbuntuOneWizard.initializePage was called with an'
                         'unknown page: %r (page_id was %r).', page, page_id)

        logger.info('UbuntuOneWizard.initializePage: new page is %r, '
                    'new button layout is %r, '
                    'new side widget stage is %r.', page, button_layout, stage)

        if button is not None and button_to is not None:
            self.setTabOrder(button, button_to)
        if button_layout is not None:
            self.setButtonLayout(button_layout)
        if stage is not None:
            self.side_widget.stage = stage

    @log_call(logger.info)
    def cleanupPage(self, page_id):
        """Called clean up 'page_id' just before the user leaves it."""
        page = self.page(page_id)
        logger.debug('UbuntuOneWizard.cleanupPage: page is %r.', page)
        if page is self.settings_page or page is self.local_folders_page:
            self.initializePage(self.pages[self.cloud_folders_page])

    def nextId(self):
        """Return the nextId to show."""
        return self._next_id

    # pylint: enable=C0103

    def _process_credentials(self, credentials=None):
        """Confirm which is the next step after analyzing 'credentials'."""
        if credentials:
            self.next()

    @QtCore.pyqtSlot()
    @handle_errors(logger=logger)
    @defer.inlineCallbacks
    def login(self):
        """Show the login dialog."""
        self.setEnabled(False)
        try:
            credentials = yield self.backend.login()
            self._process_credentials(credentials)
        finally:
            self.setEnabled(True)

    @QtCore.pyqtSlot()
    @handle_errors(logger=logger)
    @defer.inlineCallbacks
    def register(self):
        """Show the register dialog."""
        self.setEnabled(False)
        try:
            credentials = yield self.backend.register()
            self._process_credentials(credentials)
        finally:
            self.setEnabled(True)

    @QtCore.pyqtSlot()
    def check_settings(self):
        """Show the check settings page."""
        self._next_id = self.pages[self.settings_page]
        self.next()

    def done(self, result):
        """The main window is being closed, call any custom callback."""
        if result == QtGui.QDialog.Accepted:
            parent_done = super(UbuntuOneWizard, self).done
            f = lambda: parent_done(QtGui.QDialog.Accepted)
            self.local_folders_page.panel.changesApplied.connect(f)
            # commit local_folders_page's changes
            self.local_folders_page.panel.apply_changes()
        elif not self.page(self.currentId()).isFinalPage():
            response = self.confirm_dialog.exec_()
            if response == QtGui.QDialog.Accepted:
                logger.warning('UbuntuOneWizard: user canceled setup.')
                if self.currentId() == self.pages[self.license_page]:
                    logger.warning('UbuntuOneWizard: user wants to uninstall.')
                    utils.uninstall_application()
                self.rejected.emit()
        else:
            super(UbuntuOneWizard, self).done(result)
