# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2010 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The test suite for the control panel utilities."""

import logging
import sys

from twisted.internet import defer

from ubuntuone.devtools.handlers import MementoHandler

from ubuntuone.controlpanel import utils
from ubuntuone.controlpanel.tests import TestCase


CONSTANTS_MODULE = 'ubuntuone.controlpanel.constants'
NOT_DEFINED = object()


class FakedConstantsModule(object):
    """Fake the 'ubuntuone.controlpanel.constants' module."""

    PROJECT_DIR = '/tmp/foo/bar'


class FakedFailure(object):
    """Fake a twisted Failure."""

    def __init__(self, value):
        self.value = value


class GetProjectDirTestCase(TestCase):
    """Test case for get_project_dir when constants module is not defined."""

    # pylint: disable=E1101

    @defer.inlineCallbacks
    def setUp(self):
        yield super(GetProjectDirTestCase, self).setUp()
        self._constants = sys.modules.get(CONSTANTS_MODULE, NOT_DEFINED)
        sys.modules[CONSTANTS_MODULE] = None  # force ImportError

        self.memento = MementoHandler()
        self.memento.setLevel(logging.DEBUG)
        utils.logger.addHandler(self.memento)

    @defer.inlineCallbacks
    def tearDown(self):
        yield super(GetProjectDirTestCase, self).tearDown()
        if self._constants is not NOT_DEFINED:
            sys.modules[CONSTANTS_MODULE] = self._constants
        else:
            sys.modules.pop(CONSTANTS_MODULE)

    def test_get_project_dir_relative(self):
        """The relative path for the data directory is correctly retrieved."""
        module = utils.os.path.dirname(utils.__file__)
        rel_data = utils.os.path.join(module, utils.os.path.pardir,
                                      utils.os.path.pardir,
                                      utils.os.path.pardir,
                                      utils.DATA_SUFFIX)
        expected_dir = utils.os.path.abspath(rel_data)

        # ensure expected_path exists at os level
        self.patch(utils.os.path, 'exists', lambda path: path == expected_dir)

        result = utils.get_project_dir()
        self.assertEqual(expected_dir, result)

    def test_get_project_dir_none_exists(self):
        """No data directory exists, return None and log as error."""
        self.patch(utils.os.path, 'exists', lambda path: False)
        sys.modules[CONSTANTS_MODULE] = None

        result = utils.get_project_dir()
        self.assertTrue(result is None)
        msg = 'get_project_dir: can not build a valid path.'
        self.assertTrue(self.memento.check_error(msg))


class GetProjectDirWithConstantsTestCase(GetProjectDirTestCase):
    """Test case for get_project_dir when constants module is defined."""

    @defer.inlineCallbacks
    def setUp(self):
        yield super(GetProjectDirWithConstantsTestCase, self).setUp()
        self.patch(utils.os.path, 'exists', lambda path: False)
        self._constants = sys.modules.get(CONSTANTS_MODULE, NOT_DEFINED)
        sys.modules[CONSTANTS_MODULE] = FakedConstantsModule()

    def test_get_project_dir(self):
        """If the constants.py module exists, use PROJECT_DIR from it."""
        result = utils.get_project_dir()
        self.assertEqual(sys.modules[CONSTANTS_MODULE].PROJECT_DIR, result)


class GetDataFileTestCase(TestCase):
    """Test cases for get_data_file."""

    def test_get_data_file(self):
        """The path for a data file is correctly retrieved."""
        dummy_dir = '/yadda/yadda'
        dummy_file = 'test.png'
        self.patch(utils, 'get_project_dir', lambda: dummy_dir)
        result = utils.get_data_file(dummy_file)
        expected = utils.os.path.join(dummy_dir, dummy_file)
        self.assertEqual(expected, result)


class ExceptionHandlingTestCase(TestCase):
    """Test cases for exception handling."""

    def test_exception_to_error_dict(self):
        """Transform a regular Exception into a string-string dictionary."""
        msg = 'Something went wrong.'
        exc = AssertionError(msg)

        result = utils.exception_to_error_dict(exc)
        expected = {utils.ERROR_TYPE: exc.__class__.__name__,
                    utils.ERROR_MESSAGE: unicode(exc)}

        self.assertEqual(expected, result)

    def test_failure_to_error_dict(self):
        """Transform a Failure into a string-string dictionary."""
        msg = 'Something went wrong.'
        exc = AssertionError(msg)
        failure = FakedFailure(value=exc)

        result = utils.failure_to_error_dict(failure)
        expected = {utils.ERROR_TYPE: exc.__class__.__name__,
                    utils.ERROR_MESSAGE: unicode(exc)}

        self.assertEqual(expected, result)
