/*
 * Copyright (C) 2010 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Mikkel Kamstrup Erlandsen <mikkel.kamstrup@canonical.com>
 *
 */
using Dee;
using Zeitgeist;
using Zeitgeist.Timestamp;
using Config;
using Gee;

namespace Unity.FilesLens {
  
  const string ICON_PATH = Config.DATADIR + "/icons/unity-icon-theme/places/svg/";
  
  public class Daemon : GLib.Object
  {
    private Zeitgeist.Log log;
    private Zeitgeist.Index index;
    private Zeitgeist.Monitor monitor;

    private Bookmarks bookmarks;
    private UrlChecker urls;

    private Unity.Lens lens;
    private Unity.Scope scope;

    /* For each section we have a set of Zeitgeist.Event templates that
     * we use to query Zeitgeist */
    private HashTable<string, Event> type_templates;

    /* Keep track of the previous search, so we can determine when to
     * filter down the result set instead of rebuilding it */
    private LensSearch? previous_search;
    
    private bool is_dirty;
    
    construct
    {
      prepare_type_templates();

      scope = new Unity.Scope ("/com/canonical/unity/scope/files");
      scope.search_in_global = true;
      scope.activate_uri.connect (activate);

      lens = new Unity.Lens("/com/canonical/unity/lens/files", "files");
      lens.search_in_global = true;
      lens.search_hint = _("Search Files & Folders");
      lens.visible = true;
      populate_categories ();
      populate_filters();
      lens.add_local_scope (scope);
      
      previous_search = null;
      
      is_dirty = true;
      
      /* Bring up Zeitgeist interfaces */
      log = new Zeitgeist.Log();
      index = new Zeitgeist.Index();
      
      /* Listen for all file:// related events from Zeitgeist */
      var templates = new PtrArray();
      var event = new Zeitgeist.Event ();
      var subject = new Zeitgeist.Subject ();
      subject.set_uri ("file://*");
      event.add_subject (subject);
      templates.add (event.ref ());
      monitor = new Zeitgeist.Monitor (new Zeitgeist.TimeRange.from_now (),
                                       (owned) templates);
      monitor.events_inserted.connect (on_zeitgeist_changed);
      monitor.events_deleted.connect (on_zeitgeist_changed);
      log.install_monitor (monitor);
      
      bookmarks = new Bookmarks ();
      urls = new UrlChecker ();

      /* Listen for filter changes */
      scope.filters_changed.connect (
        () => {          
          if (scope.active_search != null)
            {
              is_dirty = true;
              scope.notify_property ("active-search");
            }
        }
      );

      /* Listen for changes to the lens entry search */
      scope.notify["active-search"].connect (
        (obj, pspec) => {
          var search = scope.active_search;
          
          if (!(Utils.search_has_really_changed (previous_search, search) || is_dirty))
            return;
          
          is_dirty = false;
          
          if (search_is_invalid (search))
            {
              update_without_search_async.begin(search);
            }
          else
            {
              update_search_async.begin (search);
             }
          previous_search = search;
        }
      );

      /* Listen for changes to the global search */
      scope.notify["active-global-search"].connect (
        (obj, pspec) => {

          var search = scope.active_global_search;
          
          if (search_is_invalid (search))
            return;
          
          if (!Utils.search_has_really_changed (previous_search, search))
            return;
          
          update_global_search_async.begin (search);
          previous_search = search;
        }
      );
     
      lens.export ();
    }

    private void populate_filters ()
    {
      var filters = new GLib.List<Unity.Filter> ();

      /* Last modified */
      {
        var filter = new RadioOptionFilter ("modified", _("Last modified"));

        filter.add_option ("last-7-days", _("Last 7 days"));
        filter.add_option ("last-30-days", _("Last 30 days"));
        filter.add_option ("last-year", _("Last year"));

        filters.append (filter);
      }

      /* Type filter */
      {
        var filter = new RadioOptionFilter ("type", _("Type"));
        filter.sort_type = OptionsFilter.SortType.DISPLAY_NAME;

        filter.add_option ("documents", _("Documents"));
        filter.add_option ("folders", _("Folders"));
        filter.add_option ("images", _("Images"));
        filter.add_option ("audio", _("Audio"));
        filter.add_option ("videos", _("Videos"));
        filter.add_option ("presentations", _("Presentations"));
        filter.add_option ("other", _("Other"));

        filters.append (filter);
      }

      /* Size filter */
      {
        var filter = new MultiRangeFilter ("size", _("Size"));

        filter.add_option ("1KB", _("1KB"));
        filter.add_option ("100KB", _("100KB"));
        filter.add_option ("1MB", _("1MB"));
        filter.add_option ("10MB", _("10MB"));
        filter.add_option ("100MB", _("100MB"));
        filter.add_option ("1GB", _("1GB"));
        filter.add_option (">1GB", _(">1GB"));

        filters.append (filter);
      }

      lens.filters = filters;
    }

    private void populate_categories ()
    {
      var categories = new GLib.List<Unity.Category> ();
      var icon_dir = File.new_for_path (ICON_PATH);

      var cat = new Unity.Category (_("Recent"),
                                    new FileIcon (icon_dir.get_child ("group-recent.svg")));
      categories.append (cat);

      cat =  new Unity.Category (_("Downloads"),
                                 new FileIcon (icon_dir.get_child ("group-downloads.svg")));
      categories.append (cat);

      cat = new Unity.Category (_("Folders"),
                                new FileIcon (icon_dir.get_child ("group-folders.svg")));
      categories.append (cat);

      lens.categories = categories;
    }

    private void prepare_type_templates ()
    {
      type_templates = new HashTable<string, Event> (str_hash, str_equal);
      Event event;

      /* HACK ALERT: All the (event as GLib.Object).ref() are needed because
       *             GPtrArray doesn't grab a ref to the event objects */

      /* Section.ALL_FILES */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               "", "", "", "", "", ""));
      type_templates.insert ("all", (event as GLib.Object).ref() as Event);
      
      /* Section.DOCUMENTS */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_DOCUMENT,
                                               "", "", "", "", ""),
                             new Subject.full ("file:*",
                                               "!"+NFO_PRESENTATION,
                                               "", "", "", "", ""));
      type_templates.insert ("documents", (event as GLib.Object).ref() as Event);

      /* Section.FOLDERS
       * - we're using special ORIGIN queries here */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               "", "", "", "", "", ""));
      type_templates.insert ("folders", (event as GLib.Object).ref() as Event);

      /* Section.IMAGES */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_IMAGE, "", "", "", "", ""));
      type_templates.insert ("images", (event as GLib.Object).ref() as Event);
      
      /* Section.AUDIO */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_AUDIO, "", "", "", "", ""));
      type_templates.insert ("audio", (event as GLib.Object).ref() as Event);

      /* Section.VIDEOS */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_VIDEO, "", "", "", "", ""));
      type_templates.insert ("videos", (event as GLib.Object).ref() as Event);

      /* Section.PRESENTATIONS
       * FIXME: Zeitgeist logger needs to user finer granularity
       *        on classification as I am not sure it uses
       *        NFO_PRESENTATION yet */
      event = new Event.full("", ZG_USER_ACTIVITY, "",
                             new Subject.full ("file:*",
                                               NFO_PRESENTATION, "", "", "", "", ""));
      type_templates.insert ("presentations", (event as GLib.Object).ref() as Event);

      /* Section.OTHER 
       * Note that subject templates are joined with logical AND */
      event = new Event.full("", ZG_USER_ACTIVITY, "");
      event.add_subject (new Subject.full ("file:*",
                                           "!"+NFO_DOCUMENT, "", "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_IMAGE,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_AUDIO,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_VIDEO,
                                           "",
                                           "", "", "", ""));
      event.add_subject (new Subject.full ("",
                                           "!"+NFO_PRESENTATION,
                                           "",
                                           "", "", "", ""));
      type_templates.insert ("other", (event as GLib.Object).ref() as Event);
    }

    private bool search_is_invalid (LensSearch? search)
    {
      /* This boolean expression is unfolded as we seem to get
       * some null dereference if we join them in a big || expression */
      if (search == null)
        return true;
      else if (search.search_string == null)
        return true;
      
      return search.search_string.strip() == "";
    }

    private string prepare_search_string (LensSearch? search)
    {
      var s = search.search_string;

      if (s.has_suffix (" "))
        s = s.strip ();

      if (!s.has_suffix ("*"))
        s = s + "*";

      /* The Xapian query parser (used by the Zeitgeist FTS Extension) seems to
       * handle hyphens in a special way, namely that it forces the joined
       * tokens into a phrase query no matter if it appears as the last word
       * in a query and we have the PARTIAL flag set on the query parser.
       * This makes 'unity-p' not match 'unity-package-search.cc' etc. */
      s = s.delimit ("-", ' ');

      return s;
    }
    
    private async void update_global_search_async (LensSearch search)
    {
      var results_model = scope.global_results_model;

      if (search_is_invalid (search))
        {
          results_model.clear ();
          return;
        }
      
      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      scope.freeze_notify ();
      
      var search_string = prepare_search_string (search);

      var templates = new PtrArray.sized(1);
      templates.add (type_templates.lookup ("all"));

      try {
        /* Get results ranked by recency */
        var timer = new Timer ();
        var results = yield index.search (search_string,
                                          new Zeitgeist.TimeRange.anytime(),
                                          templates,
                                          0,
                                          20,
                                          ResultType.MOST_RECENT_SUBJECTS,
                                          null);

        timer.stop ();
        debug ("Found %u/%u global results for search '%s' in %fms",
               results.size (), results.estimated_matches (),
               search_string, timer.elapsed()*1000);


        results_model.clear ();

        var checked_url = urls.check_url (search.search_string);
        if (checked_url != null)
          {
            results_model.append (checked_url, urls.icon, Categories.RECENT,
                                  "text/html", search.search_string,
                                  checked_url, checked_url);
          }
        var bookmark_matches = bookmarks.prefix_search (search.search_string);
        append_bookmarks (bookmark_matches, results_model, Categories.FOLDERS);
        
        int64 min_size, max_size;
        get_current_size_limits (out min_size, out max_size);

        Unity.FilesLens.append_events_sorted (results, results_model,
                                              min_size, max_size, false);

      } catch (GLib.Error e) {
        warning ("Error performing global search '%s': %s",
                 search.search_string, e.message);
      }
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        scope.thaw_notify ();
        return false;
      });
      
      search.finished ();
    }
    
    private async void update_search_async  (LensSearch search)
    {
      if (search_is_invalid (search))
        {
          update_without_search_async.begin (search);
          return;
        }

      var results_model = scope.results_model;

      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      scope.freeze_notify ();
      
      var search_string = prepare_search_string (search);

      string type_id = get_current_type ();

      bool origin_grouping = type_id == "folders";
      var result_type = origin_grouping ? ResultType.MOST_RECENT_ORIGIN
                                        : ResultType.MOST_RECENT_SUBJECTS;

      /* Grab the pre-compiled template we're going to use */
      var templates = new PtrArray.sized(1);
      templates.add (type_templates.lookup (type_id));


      try {
        /* Get results ranked by recency */
        var timer = new Timer ();
        var results = yield index.search (search_string,
                                          get_current_timerange (),
                                          templates,
                                          0,
                                          50,
                                          result_type,
                                          null);

        timer.stop ();
        debug ("Found %u/%u results for search '%s' in %fms",
               results.size (), results.estimated_matches (),
               search_string, timer.elapsed()*1000);

        results_model.clear ();

        var checked_url = urls.check_url (search.search_string);
        if (checked_url != null)
          {
            results_model.append (checked_url, urls.icon, Categories.RECENT,
                                  "text/html", search.search_string,
                                  checked_url, checked_url);
          }

        var bookmark_matches = bookmarks.prefix_search (search.search_string);
        append_bookmarks (bookmark_matches, results_model, Categories.FOLDERS);

        int64 min_size, max_size;
        get_current_size_limits (out min_size, out max_size);

        Unity.FilesLens.append_events_sorted (results, results_model,
                                              min_size, max_size,
                                              origin_grouping);

        yield update_downloads_async (results_model, search.search_string);
        
      } catch (GLib.Error e) {
        warning ("Error performing global search '%s': %s",
                 search.search_string, e.message);
      }
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        scope.thaw_notify ();
        return false;
      });
      
      search.finished ();
    }

    private string get_current_type ()
    {
      /* Get the current type to filter by */
      var filter = scope.get_filter ("type") as RadioOptionFilter;
      Unity.FilterOption? option =  filter.get_active_option ();
      return option == null ? "all" : option.id;
    }

    private TimeRange get_current_timerange ()
    {
      var filter = scope.get_filter ("modified") as RadioOptionFilter;
      Unity.FilterOption? option = filter.get_active_option ();

      string date = option == null ? "all" : option.id;

      if (date == "last-7-days")
        return new TimeRange (Timestamp.now() - Timestamp.WEEK, Timestamp.now ());
      else if (date == "last-30-days")
        return new TimeRange (Timestamp.now() - (Timestamp.WEEK * 4), Timestamp.now());
      else if (date == "last-year")
        return new TimeRange (Timestamp.now() - Timestamp.YEAR, Timestamp.now ());
      else
        return new TimeRange.anytime ();
    }

    private void get_current_size_limits (out int64 min_size, out int64 max_size)
    {
      var filter = scope.get_filter ("size") as MultiRangeFilter;
      Unity.FilterOption? min_opt = filter.get_first_active ();
      Unity.FilterOption? max_opt = filter.get_last_active ();

      if (min_opt == null || max_opt == null)
      {
        min_size = 0;
        max_size = int64.MAX;
        return;
      }

      int64[] sizes = 
      {
        0,
        1024,
        102400,
        1048576,
        10485760,
        104857600,
        1073741824,
        int64.MAX
      };

      switch (min_opt.id)
      {
        case "1KB":
          min_size = sizes[0]; break;
        case "100KB":
          min_size = sizes[1]; break;
        case "1MB":
          min_size = sizes[2]; break;
        case "10MB":
          min_size = sizes[3]; break;
        case "100MB":
          min_size = sizes[4]; break;
        case "1GB":
          min_size = sizes[5]; break;
        case ">1GB":
          min_size = sizes[6]; break;
        default:
          warn_if_reached ();
          min_size = 0;
          break;
      }

      switch (max_opt.id)
      {
        case "1KB":
          max_size = sizes[1]; break;
        case "100KB":
          max_size = sizes[2]; break;
        case "1MB":
          max_size = sizes[3]; break;
        case "10MB":
          max_size = sizes[4]; break;
        case "100MB":
          max_size = sizes[5]; break;
        case "1GB":
          max_size = sizes[6]; break;
        case ">1GB":
          max_size = sizes[7]; break;
        default:
          warn_if_reached ();
          max_size = int64.MAX;
          break;
      }
    }

    private async void update_without_search_async (LensSearch search)
    {
      var results_model = scope.results_model;

      /* Prevent concurrent searches and concurrent updates of our models,
       * by preventing any notify signals from propagating to us.
       * Important: Remeber to thaw the notifys again! */
      scope.freeze_notify ();
      
      string type_id = get_current_type ();

      bool origin_grouping = type_id == "folders";
      var result_type = origin_grouping ? ResultType.MOST_RECENT_ORIGIN
                                        : ResultType.MOST_RECENT_SUBJECTS;


      /* Grab the pre-compiled template we're going to use */
      var templates = new PtrArray.sized(1);
      templates.add (type_templates.lookup (type_id));

      try {
        /* Get results ranked by recency */
        var timer = new Timer ();
        var results = yield log.find_events (get_current_timerange (),
                                             templates,
                                             Zeitgeist.StorageState.ANY,
                                             100,
                                             result_type,
                                             null);

        timer.stop ();
        debug ("Found %u/%u no search results in %fms",
               results.size (), results.estimated_matches (),
               timer.elapsed()*1000);

        results_model.clear ();

        if (type_id == "all" || type_id == "folders")
        {
          append_bookmarks (bookmarks.list (), results_model, Categories.FOLDERS);
        }

        int64 min_size, max_size;
        get_current_size_limits (out min_size, out max_size);

        Unity.FilesLens.append_events_sorted (results, results_model,
                                              min_size, max_size,
                                              origin_grouping);

        yield update_downloads_async (results_model);

      } catch (GLib.Error e) {
        warning ("Error performing empty search: %s",
                 e.message);
      }
      
      /* Allow new searches once we enter an idle again.
       * We don't do it directly from here as that could mean we start
       * changing the model even before we had flushed out current changes
       */
      Idle.add (() => {
        scope.thaw_notify ();
        return false;
      });
      
      search.finished ();
    }

    private void append_bookmarks (GLib.List<Bookmark> bookmarks,
                                   Dee.Model results_model,
                                   Categories category)
    {
      foreach (var bookmark in bookmarks)
      {
        results_model.append (bookmark.uri, bookmark.icon, category,
                              bookmark.mimetype, bookmark.display_name,
                              bookmark.dnd_uri);
      }
    }

    private async void update_downloads_async (Dee.Model results_model,
                                               string? name_filter = null)
    {
      // FIXME: Store the Downloads folder and update on changes
      unowned string download_path =
                 Environment.get_user_special_dir (UserDirectory.DOWNLOAD);
      var download_dir = File.new_for_path (download_path);
      SList<FileInfo> downloads;

      try {
        if (name_filter != null)
          downloads = yield Utils.list_dir_filtered (download_dir, name_filter);
        else
          downloads = yield Utils.list_dir (download_dir);
      } catch (GLib.Error e) {
        warning ("Failed to list downloads from directory '%s': %s",
                 download_path, e.message);
        return;
      }
      
      /* Sort files by mtime, we do an ugly nested ternary
       * to avoid potential long/int overflow */
      downloads.sort ((CompareFunc) Utils.cmp_file_info_by_mtime);
      
      var timerange = get_current_timerange ();
      int64 min_size, max_size;
      get_current_size_limits (out min_size, out max_size);
      
      foreach (var info in downloads)
      {
        var uri = download_dir.get_child (info.get_name ()).get_uri ();
        var mimetype = info.get_content_type ();
        var icon_hint = Utils.check_icon_string (uri, mimetype, info);

        // check if we match the timerange
        uint64 atime = info.get_attribute_uint64 (FILE_ATTRIBUTE_TIME_ACCESS) * 1000;
        
        if (atime < timerange.get_start() || atime > timerange.get_end ())
          continue;

        // check if type matches
        var type_id = get_current_type ();
        if (type_id != "all" && !Utils.file_info_matches_type (info, type_id))
          continue;
        
        // check if size is within bounds
        int64 size = info.get_size ();
        if (size < min_size || size > max_size)
          continue;

        results_model.append (uri, icon_hint, Categories.DOWNLOADS,
                              mimetype, info.get_display_name (), uri);
      }
    }

    public Unity.ActivationResponse activate (string uri)
    {
      debug (@"Activating: $uri");
      try {
        if (!bookmarks.launch_if_bookmark (uri))
          AppInfo.launch_default_for_uri (uri, null);
        return new Unity.ActivationResponse(Unity.HandledType.HIDE_DASH);
      } catch (GLib.Error error) {
        warning ("Failed to launch URI %s", uri);
        return new Unity.ActivationResponse(Unity.HandledType.NOT_HANDLED);
      }
    }

    private void on_zeitgeist_changed ()
    {
      /* Emitting notify here will make us recheck if the search results
       * need update. In the negative case this is a noop */
      is_dirty = true;
      scope.notify_property ("active-search");
    }
    
  }  

  private const string ATTR_HIDDEN = FILE_ATTRIBUTE_STANDARD_IS_HIDDEN;
  private const string ATTR_SIZE_AND_HIDDEN = FILE_ATTRIBUTE_STANDARD_SIZE +
    "," + FILE_ATTRIBUTE_STANDARD_IS_HIDDEN;

  /* Appends a set of Zeitgeist.Events to our Dee.Model assuming that
   * these events are already sorted with descending timestamps */
  public void append_events_sorted (Zeitgeist.ResultSet events,
                                    Dee.Model results,
                                    int64 min_size, int64 max_size,
                                    bool use_origin)
  {
     foreach (var ev in events)
      {
        if (ev.num_subjects() > 0)
          {
            // FIXME: We only use the first subject...
            Zeitgeist.Subject su = ev.get_subject(0);

            string uri;
            string display_name;
            string mimetype;

            if (use_origin)
            {
              uri = su.get_origin ();
              display_name = "";
              mimetype = "inode/directory";
            }
            else
            {
              uri = su.get_uri ();
              display_name = su.get_text ();
              mimetype = su.get_mimetype ();
              mimetype = su.get_mimetype () != null ?
                         su.get_mimetype () : "application/octet-stream";
            }
            if (uri == null) continue;
            File file = File.new_for_uri (uri);

            if (display_name == null || display_name == "")
            {
              display_name = Path.get_basename (file.get_parse_name ());
            }

            bool check_size = min_size > 0 || max_size < int64.MAX;
            /* Don't check existence on non-native files as http:// and
             * friends are *very* expensive to query */
            if (file.is_native()) {
              // hidden files should be ignored
              try {
                FileInfo info = file.query_info (check_size ?
                  ATTR_SIZE_AND_HIDDEN : ATTR_HIDDEN, 0, null);
                if (info.get_is_hidden())
                  continue;
                if (check_size &&
                  (info.get_size () < min_size || info.get_size () > max_size))
                {
                  continue;
                }
              } catch (GLib.Error e) {
                // as error occurred file must be missing therefore ignoring it
                continue;
              }
            }
            string icon = Utils.get_icon_for_uri (uri, mimetype);

            uint category_id;
            string comment = file.get_parse_name ();
            
            category_id = file.query_file_type (0, null) == FileType.DIRECTORY ?
              Categories.FOLDERS : Categories.RECENT;
            results.append (uri, icon, category_id, mimetype,
                            display_name, comment);

          }
      }
   }
} /* namespace */

